/**
 * @file Bts2App_BaseMessage.h
 *
 * @par SW-Component
 * Messages
 *
 * @brief BtStackIf to Application messages.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Base message declaration for BtStackIf to Application messages.
 */

#ifndef _BTS_2_APP_BASE_MESSAGE_H_
#define _BTS_2_APP_BASE_MESSAGE_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class BtStackIfCallback;

/**
 *
 */
class Bts2App_BaseMessage
{
public:
   /**************************************************************************
    Constructor / destructor
    **************************************************************************/
   /**
    * Default constructor. Do not use!!!
    */
   Bts2App_BaseMessage();

   /**
    * Constructor.
    *
    * @param[in] messageType: message type
    */
   Bts2App_BaseMessage(IN const BTSFunctionBlock functionBlock, IN const BTSBts2AppOpcode opCode);

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   Bts2App_BaseMessage(IN const Bts2App_BaseMessage& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   Bts2App_BaseMessage& operator=(IN const Bts2App_BaseMessage& ref);

   /**
    * Destructor.
    */
   virtual ~Bts2App_BaseMessage();

   /**
    * Trace function for output parameters.
    */
   virtual void doOutputTrace(void);

   /**
    * Trigger callback message.
    *
    * @param[in] callback: callback reference
    */
   virtual void triggerCallback(IN BtStackIfCallback& callback);

   /**
    * Returns function block identifier.
    *
    * @return = function block identifier
    */
   inline BTSFunctionBlock getFunctionBlock(void) const { return _functionBlock; }

   /**
    * Returns message opcode.
    *
    * @return = message opcode
    */
   inline BTSBts2AppOpcode getOpCode(void) const { return _opCode; }

   /**
    * Returns message opcode for tracing.
    *
    * @return = message opcode
    */
   inline BTSTraceOpCode getTraceOpCode(void) const { return (BTSTraceOpCode)_opCode; }

   /**
    * Sets user reference. NOTE: replace by setCallback.
    *
    * @param[in] ptrUser: user reference
    */
   inline void setUser(IN BtStackIfCallback* ptrUser) { _callback = ptrUser; }

   /**
    * Returns user reference. NOTE: replace by getCallback.
    *
    * @return = user reference
    */
   inline BtStackIfCallback* getUser(void) { return _callback; }

   /**
    * Sets callback reference.
    *
    * @param[in] callback: callback reference
    */
   inline void setCallback(IN BtStackIfCallback* callback) { _callback = callback; }

   /**
    * Returns callback reference.
    *
    * @return = callback reference
    */
   inline BtStackIfCallback* getCallback(void) { return _callback; }

   /**
    * Sets session handle.
    *
    * @param[in] sessionHandle: session handle
    */
   inline void setSessionHandle(IN const BTSSessionHandle sessionHandle) { _sessionHandle = sessionHandle; }

   /**
    * Returns session handle.
    *
    * @return = session handle
    */
   inline BTSSessionHandle getSessionHandle(void) const { return _sessionHandle; }

   /**
    * Sets component.
    *
    * @param[in] component: component
    */
   inline void setComponent(IN const BTSFunctionBlock component) { _component = component; }

   /**
    * Returns component.
    *
    * @return = component
    */
   inline BTSFunctionBlock getComponent(void) const { return _component; }

   /**
    * Sets stack interface.
    *
    * @param[in] stackInterface: stack interface
    */
   inline void setStackInterface(IN const BTSInterfaceType stackInterface) { _stackInterface = stackInterface; }

   /**
    * Returns stack interface
    *
    * @return = stack interface
    */
   inline BTSInterfaceType getStackInterface(void) const { return _stackInterface; }

   /**
    * Sets BT device address.
    *
    * @param[in] deviceAddress: BT device address
    */
   inline void setBDAddress(IN const BTSBDAddress& deviceAddress) { _deviceAddress = deviceAddress; }

   /**
    * Gets BT device address.
    *
    * @param[out] deviceAddress: BT device address
    */
   inline void getBDAddress(OUT BTSBDAddress& deviceAddress) const { deviceAddress = _deviceAddress; }

   /**
    * Gets BT device address.
    *
    * @return = BT device address
    */
   inline const BTSBDAddress& getBDAddress(void) const { return _deviceAddress; }

   /**
    * Gets BT device address.
    *
    * @return = BT device address
    */
   inline BTSBDAddress& getBDAddressMutable(void) { return _deviceAddress; }

   /**
    * Returns copy of the message object.
    *
    * @return = copy of the message object (pointer, created by new)
    */
   virtual Bts2App_BaseMessage* clone(void) const = 0;

   /**
    * Get message trace id.
    *
    * @return = message trace id
    */
   uint32_t getMessageTraceId(void) const;

   /**
    * Returns number of existing BTS 2 APP messages.
    *
    * @return = number of existing BTS 2 APP messages
    */
   static inline unsigned int getBts2AppMessageCount(void) { return _countBts2AppMessages; }

protected:
   BTSFunctionBlock _functionBlock; /**< identifies the function block the message relates to (equals sub component) */

   BTSBts2AppOpcode _opCode; /**< identifies the message itself */

   BtStackIfCallback* _callback; /**< receiver of this answer */

   BTSSessionHandle _sessionHandle; /**< parameter session handle */

   BTSFunctionBlock _component; /**< component (function block) */

   BTSInterfaceType _stackInterface; /**< stack interface to be used (Bluetooth stack) */

   BTSBDAddress _deviceAddress; /**< parameter device address */

   static unsigned int _countBts2AppMessages; /**< number of existing BTS 2 APP messages */
};

} //btstackif

#endif //_BTS_2_APP_BASE_MESSAGE_H_
