/**
 * @file FC_Bluetooth_clienthandler_Diaglib.cpp
 * @author
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 *
 * @addtogroup FC_Bluetooth
 *
 * @brief ClientHandler provides Bluetooth specific functionality to diagnosis
 *
 * @{
 */
/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#define DIAGLIB_INCLUDE_ROUTINE_CONTROL
#define DIAGLIB_INCLUDE_DIAGNOSIS_DATA
#include "diaglib_if.h"

/* Needed for Trace */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
//Include FI interface of used service
#define MOST_FI_S_IMPORT_INTERFACE_MOST_TELFI_TYPES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_TELFI_FUNCTIONIDS
#define MOST_FI_S_IMPORT_INTERFACE_MOST_TELFI_ERRORCODES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_TELFI_SERVICEINFO
#include "conn_most_fi_if.h"

#include "FC_Bluetooth_main.h"
#include <FC_Bluetooth_rootdaemon_client.h>
#include "FC_Bluetooth_clienthandler_Phone.h"
#include "FC_Bluetooth_clienthandler_Diaglib.h"
#include "FC_Bluetooth_CCAService.h"
#include "IBmCoreRequestIf.h"
#include "FunctionTracer.h"

//TODO
#include <stdlib.h>          // for exit()
#include <dbus/dbus.h>       // for dbus_*
#include <dbus/dbus-glib.h>  // for dbus_g_*

#include <arpa/inet.h>
#include <sys/socket.h>
#include <netdb.h>
#include <ifaddrs.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <netpacket/packet.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APPLICATION
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FC_Bluetooth_clienthandler_Diaglib.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APPLICATION
#endif // VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

/*Clear Byte1 bit5 if song is not playing
 * Reseting the Byte1:Bit5 field to 0.Hex 0xDF: Bin 1101 1111*/
#define MASK_BIT_CLEAR_BYTE1_BIT5         0xDF
#define BDADDR_PVTEST_BYTE                0xCE
#define NUM_REMOTE_PROXY_DEVICES_DIAG     10
#define LEN_REMOTE_PROXY_DEVICES_NAMEPATH 30

#define BT_BUILD_ID_MARVELL "0735" //Marvell chip BT-WIFI
#define BT_CHIP_ID_MARVELL "0x9135"
#define BT_CHIP_ID_CSR "0x3a22"
#define BT_BUILD_ID_A11 "1404" //CSR BT-Wifi module
#define MAC_ADDRESS_ARRAY_SIZE 20

gboolean get_mib_config_string(char * p_key, char * p_value);
static void remove_special_char(char* szinputdata, const char cspl);

/******************************************************************
 Function Name:	vCheckDBusInit

 Description  : 	Check if init is done already and conduct init if necessary ...
 ******************************************************************/
void fc_bluetooth_tclClientHandler_Diaglib::vCheckDBusInit(void)
{
   ENTRY

   if (OSAL_ERROR == OSAL_s32EventCreate("FusionHandle", &m_hEvFusion))
   {
      /** Trace Event Handle create failed*/
      ETG_TRACE_FATAL(("BT Diag -> Event create Fail "));
   }
   else if (OSAL_C_INVALID_HANDLE == m_hEvFusion)
   {
      /** Trace Event Handle create failed*/
      ETG_TRACE_ERR(("BT Diag -> EVENT HANDLE INVALID "));
   }

   return;
}

fc_bluetooth_tclClientHandler_Diaglib* fc_bluetooth_tclClientHandler_Diaglib::m_poBtDiaglib = OSAL_NULL;

/******************************************************************
 Function Name:	fc_bluetooth_tclClientHandler_Diaglib

 Description: 	...
 ******************************************************************/
fc_bluetooth_tclClientHandler_Diaglib::fc_bluetooth_tclClientHandler_Diaglib(
      fc_bluetooth_tclApp* poMainAppl, diaglib::tclServiceDiaglib* pDiaglibService) : m_poDiaglibService(pDiaglibService)
{
   ENTRY

   _bluetoothApp = poMainAppl;
   m_u8TestmodeStatus_NotiType = (tU8) -1;
   m_u8TestmodeStatus_Status = (tU8) EN_BluetoothConnectionResultFAILED;
   m_hEvFusion = OSAL_C_INVALID_HANDLE;
   m_bFusionCBPending = FALSE;
   m_bEnableAnswer = TRUE;
   //!Fix for CID 22764: Uninitialized scalar field (UNINIT_CTOR)
   m_bLinkQuality_Status = BM_RESULT_OK;
   m_s8CallStatus = 0;

   fc_bluetooth_tclClientHandler_Diaglib::m_poBtDiaglib = this;

   if (m_poDiaglibService != OSAL_NULL)
   {
      // Routine Control
      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(ROUTINEID_BD_CONNECTION) "));
      m_poDiaglibService->poGetRoutineCtrl()->vRegisterListener(ROUTINEID_BD_CONNECTION, this);

      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(ROUTINEID_BD_ANTENNA_FIELDSTRENGTH) "));
      m_poDiaglibService->poGetRoutineCtrl()->vRegisterListener(ROUTINEID_BD_ANTENNA_FIELDSTRENGTH, this);

      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(ROUTINEID_BD_WIFI_ANTENNA_SNR) "));
      m_poDiaglibService->poGetRoutineCtrl()->vRegisterListener(ROUTINEID_BD_WIFI_ANTENNA_SNR, this);

      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(ROUTINEID_BD_SELFTEST_BLUETOOTH) "));
      m_poDiaglibService->poGetRoutineCtrl()->vRegisterListener(ROUTINEID_BD_SELFTEST_BLUETOOTH, this);

      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(ROUTINEID_BD_SELFTEST_WIFI) "));
      m_poDiaglibService->poGetRoutineCtrl()->vRegisterListener(ROUTINEID_BD_SELFTEST_WIFI, this);

      // Data ID
      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(DATAID_BD_ADDRESS_MAC) "));
      m_poDiaglibService->poGetDiagData()->vRegisterListener(DATAID_BD_ADDRESS_MAC, this);

      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(DATAID_BD_ADDRESS_BLUETOOTH_DEVICE) "));
      m_poDiaglibService->poGetDiagData()->vRegisterListener(DATAID_BD_ADDRESS_BLUETOOTH_DEVICE, this);

      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(DATAID_BD_CONNECTED_DEVICE_STATUS_1) "));
      m_poDiaglibService->poGetDiagData()->vRegisterListener(DATAID_BD_CONNECTED_DEVICE_STATUS_1, this);

      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(DATAID_BD_CONNECTED_DEVICE_STATUS_2) "));
      m_poDiaglibService->poGetDiagData()->vRegisterListener(DATAID_BD_CONNECTED_DEVICE_STATUS_2, this);

      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(DATAID_BD_GET_CURRENT_CONNECTION) "));
      m_poDiaglibService->poGetDiagData()->vRegisterListener(DATAID_BD_GET_CURRENT_CONNECTION, this);

      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(DATAID_BD_GET_LINKQUALITY_1) "));
      m_poDiaglibService->poGetDiagData()->vRegisterListener(DATAID_BD_GET_LINKQUALITY_1, this);

      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(DATAID_BD_GET_LINKQUALITY_2) "));
      m_poDiaglibService->poGetDiagData()->vRegisterListener(DATAID_BD_GET_LINKQUALITY_2, this);

      ETG_TRACE_USR3(("fc_bluetooth_tclClientHandler_Diaglib => vRegisterListener RoutineControl(DATAID_BD_GET_AUTHENTICATED_DEVICES) "));
      m_poDiaglibService->poGetDiagData()->vRegisterListener(DATAID_BD_GET_AUTHENTICATED_DEVICES, this);
   }
}

/******************************************************************
 Function Name:	~fc_bluetooth_tclClientHandler_Diaglib

 Description: 	...
 ******************************************************************/
fc_bluetooth_tclClientHandler_Diaglib::~fc_bluetooth_tclClientHandler_Diaglib()
{
   ENTRY
   _bluetoothApp = 0;
   m_poDiaglibService = OSAL_NULL;
}

/***********************************************************************
 * \brief CCA Service vOnIoControl
 *
 * Command from Diagnois for IOControl request
 *
 ************************************************************************/
tU32 fc_bluetooth_tclClientHandler_Diaglib::vOnRoutineControlStart(tU32 u32RoutineId, tU32 u32Bitmask,
      const diaglib::tclParameterVector& oParamVector, diaglib::tContext MsgContext)
{
   ENTRY
   vCheckDBusInit();

   tU32 u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK; // Return result for Diaglib, only for send routine
   tBool bSendResponseToDiaglib = FALSE; // internal Flag for big switch case to avoid double code

   OSAL_tEventMask oDummyMask, oEventMask;
   char outputstring[100];
   tS32 RetVal;

   // create object for response
   diaglib::tclParameterVector oTmpVec;
   diaglib::tenRoutineCtrlResult oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK;

   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32Bitmask);
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( MsgContext);

   // check for expected Routine ID
   switch (u32RoutineId)
   {
      //**********************************************
      //**********************************************
      case ROUTINEID_BD_CONNECTION:
      {
         ETG_TRACE_USR4(("vOnRoutineControlStart => ROUTINEID_BD_CONNECTION"));

         // check Parameter of incomming message
         if (diaglib::bCheckParameterSignature(oParamVector,
               ROUTINEID_BD_CONNECTION_START_LEN,
               ROUTINEID_BD_CONNECTION_START_PARAM1,
               ROUTINEID_BD_CONNECTION_START_PARAM2,
               ROUTINEID_BD_CONNECTION_START_PARAM2_LEN,
               ROUTINEID_BD_CONNECTION_START_PARAM3,
               ROUTINEID_BD_CONNECTION_START_PARAM3_LEN) == FALSE)
         {
            ETG_TRACE_ERR(("!!! vOnRoutineControlStart => ERROR: bCheckParameterSignature FAILED"));
            oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK;
            diaglib::trParameter paramError;
            paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
            paramError.u8ErrorValue = diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
            oTmpVec.push_back(paramError);
         } // if(diaglib::bCheckParameterSignature(oParamVector, ...
         else
         {
            unsigned int uiVecIndex, tempcheck = 0;
            unsigned int uiArrayIndex;
            tU8 au8BDAddress[BDADDR_LENGTH];

            tU32 u32Update = diaglib::EN_ROUTINE_CONTROL_OK;

            // Get Connect or Disconnect
            tenDiaglibConnection eType = static_cast<tenDiaglibConnection>(
                  oParamVector[ROUTINEID_BD_CONNECTION_START_u8Type].u8Value);

            // get Bluetooth Device address from Vector
            uiVecIndex = ROUTINEID_BD_CONNECTION_START_au8BDAddress;

            for (uiArrayIndex = 0;
                  uiArrayIndex < ROUTINEID_BD_CONNECTION_START_PARAM2_LEN;
                  ++uiArrayIndex)
            {
               au8BDAddress[uiArrayIndex] = oParamVector[uiVecIndex].u8Value;
               if (au8BDAddress[uiArrayIndex] != m_au8BDAddress[uiArrayIndex])
               {
                  tempcheck = 1;
               }
               ++uiVecIndex;
            }

            snprintf((char*) outputstring, sizeof(outputstring),
                  "vOnRoutineControlStart => Type =%x BD=%02x:%02x:%02x:%02x:%02x:%02x",
                  eType, au8BDAddress[0], au8BDAddress[1], au8BDAddress[2], au8BDAddress[3], au8BDAddress[4],
                  au8BDAddress[5]);

            ETG_TRACE_USR2(( "%s",outputstring));

            if ((eType != EN_ConnectionCONNECT) && (eType != EN_ConnectionDISCONECT))
            {
               ETG_TRACE_ERR(("ERROR: eType is neither CONNECT nor DISCONNECT"));
               u32Update = diaglib::EN_ROUTINE_CONTROL_NOT_OK;
            }

            // Check BDADDR address:
            // Conn-State=Active: New BDADDR must be equal to saved BDADDR (independent of CONNECT or
            // DISCONNECT cmd)
            // Conn-State=Idle:   Any new BDADDR allowed	(independent of CONNECT or DISCONNECT cmd)
            if (m_u8TestmodeStatus_Status == EN_BluetoothConnectionResultSUCCESFULL)
            {
               if (tempcheck == 1)
               {
                  ETG_TRACE_ERR(("ERROR: BDADDR wrong"));
                  u32Update = diaglib::EN_ROUTINE_CONTROL_NOT_OK;
               }
            }
            else if (m_u8TestmodeStatus_Status == EN_BluetoothConnectionResultCONNECTION_IN_PROGRESS)
            {
               // During this state another BD_CONNECTION command must not be sent!
               ETG_TRACE_ERR(("ERROR: Wrong state EN_BluetoothConnectionResultCONNECTION_IN_PROGRESS"));
               u32Update = diaglib::EN_ROUTINE_CONTROL_NOT_OK;
            }

            if (u32Update == diaglib::EN_ROUTINE_CONTROL_OK)
            {
               ETG_TRACE_USR4(("*********************CALLING START TESTMODE OF ALPS********************************"));

               memset(outputstring, 0, sizeof(outputstring));
               snprintf((char*) outputstring, sizeof(outputstring), "%02x%02x%02x%02x%02x%02x",
                     au8BDAddress[0], au8BDAddress[1], au8BDAddress[2], au8BDAddress[3], au8BDAddress[4], au8BDAddress[5]);

               BdAddress bdAddress = (const char *)outputstring;
               TargetSwitchState targetSwitchState;

               if (((eType == EN_ConnectionDISCONECT) && (m_u8TestmodeStatus_Status == EN_BluetoothConnectionResultFAILED))
                     || ((eType == EN_ConnectionCONNECT) && (m_u8TestmodeStatus_Status == EN_BluetoothConnectionResultSUCCESFULL)))
               {
                  ETG_TRACE_USR4(("Ok, we're already done. Do nothing!"));
               }
               else
               {
                   if (eType == EN_ConnectionDISCONECT)
                   { /* DISCONNECT */
                       targetSwitchState = TARGET_SWITCH_STATE_SWITCHED_OFF;
                   }
                   else /* CONNECT    */
                   {
                       m_u8TestmodeStatus_Status = EN_BluetoothConnectionResultFAILED;

                       targetSwitchState = TARGET_SWITCH_STATE_SWITCHED_ON;
                   }

                   _bluetoothApp->getBmCoreRequestIf().switchTestMode(targetSwitchState, bdAddress);

                   //...wait for immediate callback ...
                   ETG_TRACE_USR4(( " BT-TESTMODE: Entering OSAL_s32EventWait. " ));

                   m_bFusionCBPending = TRUE;
                   RetVal = OSAL_s32EventWait(m_hEvFusion, EVENT_FUSION, OSAL_EN_EVENTMASK_OR, BT_TESTMODE_WAIT_DURATION, &oDummyMask);

                   ETG_TRACE_USR4(( " BT-TESTMODE: Leaving OSAL_s32EventWait. " ));
                   if (RetVal != OSAL_OK)
                   {
                      tU32 u32ErrorCode = OSAL_u32ErrorCode();

                      ETG_TRACE_FATAL(( " BT-TESTMODE: OSAL_s32EventWait returned with Error code= %d", u32ErrorCode ));
                      ETG_TRACE_ERRMEM(( " BT-TESTMODE: OSAL_s32EventWait returned with Error code= %d", u32ErrorCode ));

                   }

                   // reset event!
                   oEventMask = (OSAL_tEventMask)(~EVENT_FUSION);
                   OSAL_s32EventPost(m_hEvFusion, oEventMask, OSAL_EN_EVENTMASK_AND);

                   // Update BDADDR ...
                   if ((eType == EN_ConnectionCONNECT) && (m_u8TestmodeStatus_Status ==  EN_BluetoothConnectionResultSUCCESFULL))
                   {
                      memcpy(&m_au8BDAddress[0], &au8BDAddress[0], sizeof(m_au8BDAddress));
                      ETG_TRACE_USR2(("vOnRoutineControlStart =>Copy to m_au8BDAddress BD=%02x:%02x:%02x:%02x:%02x:%02x",
                            m_au8BDAddress[0],m_au8BDAddress[1],m_au8BDAddress[2],m_au8BDAddress[3],m_au8BDAddress[4],m_au8BDAddress[5]));
                   }
               }

               oCtrlResult = diaglib::EN_ROUTINE_CONTROL_OK;
               diaglib::trParameter paramResult;

               tenDiaglibBluetoothConnectionResult eTestResult = (tenDiaglibBluetoothConnectionResult) m_u8TestmodeStatus_Status;

               paramResult.enType = static_cast<diaglib::tenParameterType>(ROUTINEID_BD_CONNECTION_RESULT_PARAM1);
               paramResult.u32Value = static_cast<tU32>(eTestResult);
               oTmpVec.push_back(paramResult);
            }
            else
            {
                // answer = ERROR !
                oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK;
                diaglib::trParameter paramError;
                paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
                paramError.u8ErrorValue = diaglib::EN_ERROR_FATAL_INTERNAL;
                oTmpVec.push_back(paramError);
            }
         }

         bSendResponseToDiaglib = TRUE;
         break;
      } // case ROUTINEID_BD_CONNECTION:

        //**********************************************
        //**********************************************
      case ROUTINEID_BD_ANTENNA_FIELDSTRENGTH:
      {
         ETG_TRACE_USR3(("vOnRoutineControlStart => ROUTINEID_BD_ANTENNA_FIELDSTRENGTH"));

         // check Parameter of incomming message
         if (diaglib::bCheckParameterSignature(oParamVector,
               ROUTINEID_BD_ANTENNA_FIELDSTRENGTH_START_LEN,
               ROUTINEID_BD_ANTENNA_FIELDSTRENGTH_START_PARAM1,
               ROUTINEID_BD_ANTENNA_FIELDSTRENGTH_START_PARAM1_LEN) == FALSE)
         {
            ETG_TRACE_ERR(("!!! vOnRoutineControlStart => ERROR: bCheckParameterSignature FAILED"));
            oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK;
            diaglib::trParameter paramError;
            paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
            paramError.u8ErrorValue = diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
            oTmpVec.push_back(paramError);
         } // if(diaglib::bCheckParameterSignature(oParamVector, ...
         else
         {
            // get lokal variables
            unsigned int uiVecIndex, tempcheck = 0;
            tU8 au8BDAddress[BDADDR_LENGTH];
            tU8 u8Pvtest = 0;
            tBool bRet = false;
            tU32 u32Update = diaglib::EN_ROUTINE_CONTROL_OK;

            // ToDo: BDADDR not necessary for LinkQuality
            //       -> check if this BDADDR matches BDADDR from TestMode Connection setup.
            uiVecIndex = ROUTINEID_BD_TESTMODE_BLUETOOTH_START_au8BDAddress;

            for (unsigned int uiArrayIndex = 0;
                  uiArrayIndex < ROUTINEID_BD_TESTMODE_BLUETOOTH_START_PARAM1_LEN;
                  ++uiArrayIndex)
            {
               au8BDAddress[uiArrayIndex] = oParamVector[uiVecIndex].u8Value;

               ETG_TRACE_USR2(("vOnRoutineControlStart => au8BDAddress BD=%02x m_au8BDAddress - %02x ",
                     au8BDAddress[uiArrayIndex] ,m_au8BDAddress[uiArrayIndex]));

               if (au8BDAddress[uiArrayIndex] != m_au8BDAddress[uiArrayIndex])
               {
                  tempcheck = 1;
               }
               if (au8BDAddress[uiArrayIndex] == BDADDR_PVTEST_BYTE)
               {
                  u8Pvtest++;
               }
               ++uiVecIndex;
            }

            ETG_TRACE_USR2(("vOnRoutineControlStart => BD=%02x:%02x:%02x:%02x:%02x:%02x",
                  au8BDAddress[0],au8BDAddress[1],au8BDAddress[2],au8BDAddress[3],au8BDAddress[4],au8BDAddress[5]));

            ETG_TRACE_USR2(("vOnRoutineControlStart => u8Pvtest - %d m_u8TestmodeStatus_Status - %d, tempcheck - %d",
                  u8Pvtest, m_u8TestmodeStatus_Status, tempcheck));

            // Check BDADDR address:
            // CONNECTION & ONGOING:    New BDADDR must be equal to saved BDADDR
            // All other states:        Error at all!
            if ((u8Pvtest != ROUTINEID_BD_TESTMODE_BLUETOOTH_START_PARAM1_LEN) &&
               ((m_u8TestmodeStatus_Status != EN_BluetoothConnectionResultSUCCESFULL) || (tempcheck == 1)))
            {
               ETG_TRACE_USR4(("ERROR: Wrong Testmode status or BDADDR "));
               u32Update = diaglib::EN_ROUTINE_CONTROL_NOT_OK;
            }
            else
            {
               m_bLinkQuality_Status = BM_RESULT_OK;
            }

            if (u32Update == diaglib::EN_ROUTINE_CONTROL_OK)
            {
               // special case for PVTEST -> might be uninitialized at this point, therefore conduct init now. No impact on other scenarios.
               vCheckDBusInit();

               ETG_TRACE_USR4(("*********************CALLING GET LINKQUALITY OF ALPS********************************"));

               _bluetoothApp->getBmCoreRequestIf().requestTestModeLinkQuality();

               if(m_bLinkQuality_Status == BM_RESULT_OK)
               {
                  // WAIT for event ...

                  ETG_TRACE_USR4(( " BT-LINKQUALITY: Entering OSAL_s32EventWait. " ));

                  m_bFusionCBPending = TRUE;
                  RetVal = OSAL_s32EventWait(m_hEvFusion, EVENT_FUSION,  OSAL_EN_EVENTMASK_OR, OSAL_C_TIMEOUT_FOREVER, &oDummyMask);

                  ETG_TRACE_USR4(( " BT-LINKQUALITY: Leaving OSAL_s32EventWait. " ));

                  if (RetVal != OSAL_OK)
                  {
                     tU32 u32ErrorCode = OSAL_u32ErrorCode();

                     ETG_TRACE_FATAL(( " BT-LINKQUALITY: OSAL_s32EventWait returned with Error code= %d", u32ErrorCode ));
                     ETG_TRACE_ERRMEM(( " BT-LINKQUALITY: OSAL_s32EventWait returned with Error code= %d", u32ErrorCode ));
                  }
                  // reset event!
                  oEventMask = (OSAL_tEventMask)(~EVENT_FUSION);
                  OSAL_s32EventPost( m_hEvFusion, oEventMask,  OSAL_EN_EVENTMASK_AND );

                  oCtrlResult = diaglib::EN_ROUTINE_CONTROL_OK;
                  diaglib::trParameter paramErrorBitRate, paramFieldstrength;
                  // ErrorBitRate
                  paramErrorBitRate.enType = static_cast<diaglib::tenParameterType>(ROUTINEID_BD_ANTENNA_FIELDSTRENGTH_RESULT_PARAM1);
                  paramErrorBitRate.u8Value = m_linkQualityInfo._linkQuality;
                  oTmpVec.push_back(paramErrorBitRate);
                  // Fieldstrength
                  paramFieldstrength.enType = static_cast<diaglib::tenParameterType>(ROUTINEID_BD_ANTENNA_FIELDSTRENGTH_RESULT_PARAM2);
                  paramFieldstrength.u8Value = (tU8) m_linkQualityInfo._rssi;
                  oTmpVec.push_back(paramFieldstrength);
               }
               else
               {
                  oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK;
                  diaglib::trParameter paramError;
                  paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
                  paramError.u8ErrorValue = diaglib::EN_ERROR_FATAL_INTERNAL;
                  oTmpVec.push_back(paramError);
               }
            }
            else
            {
               FC_Bluetooth_tclCCAService* poCCAService = FC_Bluetooth_tclCCAService::poGetInstance();

               if(poCCAService)
               {
                  memset(outputstring, 0, sizeof(outputstring));
                  snprintf((char*) outputstring, sizeof(outputstring), "%02x%02x%02x%02x%02x%02x",
                        au8BDAddress[0], au8BDAddress[1], au8BDAddress[2], au8BDAddress[3], au8BDAddress[4], au8BDAddress[5]);

                  BdAddress bdAddress = (const char *)outputstring;

                  bRet = poCCAService->bCheckLinkQualityStatus(bdAddress.c_str() ,m_linkQualityInfo);

                  if(bRet == true)
                  {
                     ETG_TRACE_USR1(("vOnRoutineControlStart: _linkQuality - %d _rssi - %d",
                           m_linkQualityInfo._linkQuality, m_linkQualityInfo._rssi));

                     oCtrlResult = diaglib::EN_ROUTINE_CONTROL_OK;
                     diaglib::trParameter paramErrorBitRate, paramFieldstrength;
                     // ErrorBitRate
                     paramErrorBitRate.enType = static_cast<diaglib::tenParameterType>(ROUTINEID_BD_ANTENNA_FIELDSTRENGTH_RESULT_PARAM1);
                     paramErrorBitRate.u8Value = m_linkQualityInfo._linkQuality;
                     oTmpVec.push_back(paramErrorBitRate);
                     // Fieldstrength
                     paramFieldstrength.enType = static_cast<diaglib::tenParameterType>(ROUTINEID_BD_ANTENNA_FIELDSTRENGTH_RESULT_PARAM2);
                     paramFieldstrength.u8Value = (tU8) m_linkQualityInfo._rssi;
                     oTmpVec.push_back(paramFieldstrength);
                  }
                  else
                  {
                     ETG_TRACE_USR1(("vOnRoutineControlStart: _linkQuality is Not OK"));

                     oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK;
                     diaglib::trParameter paramError;
                     paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
                     paramError.u8ErrorValue = diaglib::EN_ERROR_FATAL_INTERNAL;
                     oTmpVec.push_back(paramError);
                  }
               }
               else
               {
                  ETG_TRACE_USR1(("vOnRoutineControlStart: poCCAService is NULL"));

                  oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK;
                  diaglib::trParameter paramError;
                  paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
                  paramError.u8ErrorValue = diaglib::EN_ERROR_FATAL_INTERNAL;
                  oTmpVec.push_back(paramError);
               }
            }
         } //else // if(diaglib::bCheckParameterSignature(oParamVector, ...

         bSendResponseToDiaglib = TRUE;
         break;
      } // case ROUTINEID_BD_ANTENNA_FIELDSTRENGTH:

         //**********************************************
         //**********************************************
      case ROUTINEID_BD_WIFI_ANTENNA_SNR:
      {
         ETG_TRACE_USR3(("vOnRoutineControlStart => ROUTINEID_BD_WIFI_ANTENNA_SNR"));

         // check Parameter of incomming message
         if (diaglib::bCheckParameterSignature(oParamVector,
               ROUTINEID_BD_WIFI_ANTENNA_SNR_START_LEN,
               ROUTINEID_BD_WIFI_ANTENNA_SNR_START_PARAM1,
               ROUTINEID_BD_WIFI_ANTENNA_SNR_START_PARAM1_LEN) == FALSE)

         {
            ETG_TRACE_ERR(("!!! vOnRoutineControlStart => ERROR: bCheckParameterSignature FAILED"));
            oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK;
            diaglib::trParameter paramError;
            paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
            paramError.u8ErrorValue = diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
            oTmpVec.push_back(paramError);
         } // if(diaglib::bCheckParameterSignature(oParamVector, ...
         else
         {
            unsigned int uiVecIndex;
            tU8 au8BDAddress[BDADDR_LENGTH];
            // get Bluetooth Device address from Vector
            uiVecIndex = ROUTINEID_BD_WIFI_ANTENNA_SNR_START_au8BDAddress;

            for (unsigned int uiArrayIndex = 0;
                  uiArrayIndex < ROUTINEID_BD_WIFI_ANTENNA_SNR_START_PARAM1_LEN;
                  ++uiArrayIndex)
            {
               au8BDAddress[uiArrayIndex] = oParamVector[uiVecIndex].u8Value;
               ++uiVecIndex;
            }

            ETG_TRACE_USR2(("vOnRoutineControlStart => BD=%02x:%02x:%02x:%02x:%02x:%02x",
                  au8BDAddress[0],au8BDAddress[1],au8BDAddress[2],au8BDAddress[3],au8BDAddress[4],au8BDAddress[5]));

            // Check MAC address:
            // All addresses are allowed.
            // Return an error if counterdevice is not 'visible'

            tS32 tempSignal, tempNoise;
            char cmdstr[BD_CMD_STRING_LEN];
            CmdData signal_query_result;

            ETG_TRACE_USR4(("retrieve SignalNoise level ..."));

            memset(cmdstr, 0, BD_CMD_STRING_LEN);
            snprintf(cmdstr, BD_CMD_STRING_LEN, "%02x:%02x:%02x:%02x:%02x:%02x",
                  au8BDAddress[0], au8BDAddress[1], au8BDAddress[2], au8BDAddress[3], au8BDAddress[4], au8BDAddress[5]);

            signal_query_result = execRootCommand(FC_BT_ROOTDAEMON_CLIENT_NAME, GET_SIGNAL_NOISE, cmdstr);
            if (signal_query_result.errorNo == ERR_NONE)
            {
               memcpy(&tempSignal, signal_query_result.message, sizeof(tempSignal));
               memcpy(&tempNoise, signal_query_result.message + sizeof(tempSignal), sizeof(tempNoise));

               ETG_TRACE_USR2(("tempSignal=%d tempNoise=%d",tempSignal,tempNoise));
               if (tempSignal == 0)
               { // This indicates that the WiFi counterdevice is not visible, i.e. the iwlist tool didnt list the desired device
                  goto wifi_antenna_snr_failed; //lint !e801
               }

               oCtrlResult = diaglib::EN_ROUTINE_CONTROL_OK;
               diaglib::trParameter paramNoiseLevel, paramSignalLevel;
               // Update Noise Level
               paramNoiseLevel.enType = static_cast<diaglib::tenParameterType>(ROUTINEID_BD_WIFI_ANTENNA_SNR_RESULT_PARAM1);
               paramNoiseLevel.s8Value = (tS8) tempNoise;
               oTmpVec.push_back(paramNoiseLevel);
               // Update Signal Level
               paramSignalLevel.enType = static_cast<diaglib::tenParameterType>(ROUTINEID_BD_WIFI_ANTENNA_SNR_RESULT_PARAM2);
               paramSignalLevel.s8Value = (tS8) tempSignal;
               oTmpVec.push_back(paramSignalLevel);
            }
            else
            {
               ETG_TRACE_ERR(("execRootCommand returned error(%d)!", signal_query_result.errorNo));
               wifi_antenna_snr_failed:
               oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK;
               diaglib::trParameter paramError;
               paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
               paramError.u8ErrorValue = diaglib::EN_ERROR_FATAL_INTERNAL;
               oTmpVec.push_back(paramError);
            }
         } //else // if(diaglib::bCheckParameterSignature(oParamVector, ...

         bSendResponseToDiaglib = TRUE;
         break;

      } // case ROUTINEID_BD_WIFI_ANTENNA_SNR:

         //**********************************************
         //**********************************************
      case ROUTINEID_BD_SELFTEST_BLUETOOTH:
      {
         ETG_TRACE_USR3(("vOnRoutineControlStart => ROUTINEID_BD_SELFTEST_BLUETOOTH"));

         // ###################
         // TODO: add your implementation here
         // ###################

         // example for answer beginn
         oCtrlResult = diaglib::EN_ROUTINE_CONTROL_OK;
         diaglib::trParameter paramaStatus;
         paramaStatus.enType = static_cast<diaglib::tenParameterType>(ROUTINEID_BD_SELFTEST_BLUETOOTH_RESULT_PARAM1);
         paramaStatus.u8Value = static_cast<tU8>(EN_BluetoothSelftestResultFAILED);
         oTmpVec.push_back(paramaStatus);
         // example for answer end
         bSendResponseToDiaglib = TRUE;
         break;
      } // case ROUTINEID_BD_SELFTEST_BLUETOOTH:

        //**********************************************
        //**********************************************
      case ROUTINEID_BD_SELFTEST_WIFI:
      {
         ETG_TRACE_USR3(("vOnRoutineControlStart => ROUTINEID_BD_SELFTEST_WIFI"));

         // ###################
         // TODO: add your implementation here
         // ###################

         // example for answer beginn
         oCtrlResult = diaglib::EN_ROUTINE_CONTROL_OK;
         diaglib::trParameter paramaStatus;
         paramaStatus.enType = static_cast<diaglib::tenParameterType>(ROUTINEID_BD_SELFTEST_WIFI_RESULT_PARAM1);
         paramaStatus.u8Value = static_cast<tU8>(EN_BluetoothSelftestResultFAILED);
         oTmpVec.push_back(paramaStatus);
         // example for answer end
         bSendResponseToDiaglib = TRUE;
         break;
      } // case ROUTINEID_BD_SELFTEST_WIFI:

        //**********************************************
        //**********************************************
      default:
      {
         ETG_TRACE_ERR(("!!! vOnRoutineControlStart => ERROR : invalid Id: %d",u32RoutineId));
         break;
      }
   } // switch(u32RoutineId)

   // check debug mode
   if (m_bEnableAnswer == FALSE)
   {
      bSendResponseToDiaglib = FALSE;
   }

   // check now if we have to response to Diaglib now
   if (bSendResponseToDiaglib == TRUE)
   {
      // send message back to diagnosis
      if (m_poDiaglibService != OSAL_NULL)
      {
         if (TRUE == m_poDiaglibService->poGetRoutineCtrl()->bSendRoutineCtrlMethodResult(
               oCtrlResult, oTmpVec, MsgContext))
         {
            u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
            ETG_TRACE_USR4(("vOnRoutineControlStart => bSendRoutineCtrlMethodResult successfull"));
         } // if(TRUE == m_poDiaglibService->poGetRoutineCtrl()
         else
         {
            u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
            ETG_TRACE_ERR(("!!! vOnRoutineControlStart => ERROR : bSendRoutineCtrlMethodResult failed"));
         } // else // if(TRUE == m_poDiaglibService->poGetRoutineCtrl()
      } // if(m_poDiaglibService != OSAL_NULL)
      else
      {
         u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
         ETG_TRACE_ERR(("!!! vOnRoutineControlStart => ERROR : m_poDiaglibService == OSAL_NULL"));
      } // else // if(m_poDiaglibService != OSAL_NULL)
   } // if(bSendResponseToDiaglib == TRUE)

   ETG_TRACE_USR2(("<-- vOnRoutineControlStart(%x)",u32Return));

   return u32Return;
}

tU32 fc_bluetooth_tclClientHandler_Diaglib::vOnRoutineControlAbort(tU32 u32RoutineId, diaglib::tContext MsgContext)
{
ENTRY
	   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32RoutineId);
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( MsgContext);
   tU32 u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
   return u32Return;
}

tU32 fc_bluetooth_tclClientHandler_Diaglib::vOnGetRoutineResult(tU32 u32RoutineId, diaglib::tContext MsgContext)
{
   ENTRY
   tU32 u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
   tBool bSendResponseToDiaglib = FALSE; // internal Flag for big switch case to avoid double code
   // create object for response
   diaglib::tclParameterVector oTmpVec;
   diaglib::tenRoutineCtrlResult oCtrlResult = diaglib::EN_ROUTINE_CONTROL_NOT_OK;

   // check for expected Routine ID
   switch (u32RoutineId)
   {
      case ROUTINEID_BD_CONNECTION:
      {
         ETG_TRACE_USR2(("vOnGetRoutineResult => ROUTINEID_BD_CONNECTION"));

         // answer begin
         oCtrlResult = diaglib::EN_ROUTINE_CONTROL_OK; // data valid
         diaglib::trParameter paramaStatus;
         paramaStatus.enType = static_cast<diaglib::tenParameterType>(ROUTINEID_BD_CONNECTION_RESULT_PARAM1);
         paramaStatus.u32Value = static_cast<tU8>(m_u8TestmodeStatus_Status/*EN_BluetoothConnectionResult...XYZ...*/);
         oTmpVec.push_back(paramaStatus);
         // answer end

         bSendResponseToDiaglib = TRUE;
         break;
      } // case ROUTINEID_BD_TESTMODE_BLUETOOTH:

      default:
      {
         ETG_TRACE_ERR(("!!! vOnRoutineControlStart => ERROR : invalid Id: %d",u32RoutineId));
         break;
      }
   } // switch(u32RoutineId)

   // check now if we have to response to Diaglib now
   if (bSendResponseToDiaglib == TRUE)
   {
      // send message back to diagnosis
      if (m_poDiaglibService != OSAL_NULL)
      {
         if (TRUE == m_poDiaglibService->poGetRoutineCtrl()->bSendRoutineCtrlMethodResult(
               oCtrlResult, oTmpVec, MsgContext))
         {
            u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
            ETG_TRACE_USR4(("vOnRoutineControlStart => bSendRoutineCtrlMethodResult successfull"));
         } // if(TRUE == m_poDiaglibService->poGetRoutineCtrl()
         else
         {
            u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
            ETG_TRACE_ERR(("!!! vOnRoutineControlStart => ERROR : bSendRoutineCtrlMethodResult failed"));
         } // else // if(TRUE == m_poDiaglibService->poGetRoutineCtrl()
      } // if(m_poDiaglibService != OSAL_NULL)
      else
      {
         u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
         ETG_TRACE_ERR(("!!! vOnRoutineControlStart => ERROR : m_poDiaglibService == OSAL_NULL"));
      } // else // if(m_poDiaglibService != OSAL_NULL)
   } // if(bSendResponseToDiaglib == TRUE)

   ETG_TRACE_USR2(("<-- vOnGetRoutineResult(%x)",u32Return));

   return u32Return;
}

/***********************************************************************
 * \brief CCA Service vOnDiagnosisData
 *
 * Command from Diagnois for simple Data by Id request
 *
 ************************************************************************/
tU32 fc_bluetooth_tclClientHandler_Diaglib::vOnDiagnosisDataRead(tU32 u32DiagDataId, diaglib::tContext MsgContext)
{
   ENTRY

   tU32 u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
   tBool bSendResponseToDiaglib = FALSE;
   // create object for response
   diaglib::tclParameterVector oTmpVec;
   diaglib::tenDiagDataResult oDataResult = diaglib::EN_DIAGDATA_NOT_OK;
   BmResult result = BM_RESULT_ERR_GENERAL;
   BdAddress bdAddress = "";

   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( MsgContext);

   switch (u32DiagDataId)
   {
      /***************************************************************************/
      /* DATAID_BD_ADDRESS_MAC (read)                                            */
      /*                                                                         */
      /* \param[in]  none                                                        */
      /*                                                                         */
      /* \param[out] oDataResult                                                 */
      /*             u8MacAddress[6]                                             */
      /***************************************************************************/
      case DATAID_BD_ADDRESS_MAC:
      {
         ETG_TRACE_USR4(("vOnDiagnosisDataRead => DATAID_BD_ADDRESS_MAC"));

         //CMG3G-8820
         struct sockaddr_ll *stSockAddr = getWifiAddress();

         if (NULL != stSockAddr)
         {
            oTmpVec.resize(DATAID_BD_ADDRESS_MAC_READ_RESULT_LEN);
            oDataResult = diaglib::EN_DIAGDATA_OK; // Read Data succesfully

            unsigned int uiVecIndex = DATAID_BD_ADDRESS_MAC_READ_RESULT_au8MacAddress;

            for (unsigned int uiArrayIndex = 0; uiArrayIndex < stSockAddr->sll_halen; ++uiArrayIndex)
            {
               ETG_TRACE_USR4((" Wi-Fi Mac Address ::%02x", stSockAddr->sll_addr[uiArrayIndex]));

               oTmpVec[uiVecIndex].u8Value = (tU8) stSockAddr->sll_addr[uiArrayIndex];
               oTmpVec[uiVecIndex].enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8);
               ++uiVecIndex;
            }
         }
         else
         {
            oDataResult = diaglib::EN_DIAGDATA_NOT_OK; // Control failed
            diaglib::trParameter paramError;
            paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
            paramError.u8Value = (tU8) diaglib::EN_ERROR_FATAL_INTERNAL;
            oTmpVec.push_back(paramError);
         }
         // example for answer end
         bSendResponseToDiaglib = TRUE;

         break;
      } // case DATAID_BD_ADDRESS_MAC:

         /***************************************************************************/
         /* DATAID_BD_ADDRESS_BLUETOOTH_DEVICE (read)                               */
         /*                                                                         */
         /* \param[in]  none                                                        */
         /*                                                                         */
         /* \param[out] oDataResult                                                 */
         /*             au8BDAddress[6]                                             */
         /***************************************************************************/
      case DATAID_BD_ADDRESS_BLUETOOTH_DEVICE:
      {
         ETG_TRACE_USR4(("vOnDiagnosisDataRead => DATAID_BD_ADDRESS_BLUETOOTH_DEVICE"));
         tBool bReadBDAddressResult = FALSE;

         tChar szTemp[20];

         // Retrieve BLUETOOTH address from default_mib.txt data file and answer
         if (get_mib_config_string(const_cast<tChar*>("unifiBlueCorePSKey.1"), szTemp) == TRUE) //To Solve Lint Prio-2
         {
            remove_special_char(szTemp, '[');
            remove_special_char(szTemp, ']');
            remove_special_char(szTemp, ' ');
            oTmpVec.resize(DATAID_BD_ADDRESS_BLUETOOTH_DEVICE_READ_RESULT_LEN);
            oDataResult = diaglib::EN_DIAGDATA_OK; // Read Data succesfully

            tInt aiBDAddress[BDADDR_LENGTH];
            unsigned int uiVecIndex = DATAID_BD_ADDRESS_BLUETOOTH_DEVICE_READ_RESULT_au8BDAddress;
            tInt ignore;
            sscanf(szTemp, "%02x %02x %02x %02x %02x %02x %02x %02x", &ignore, &aiBDAddress[2], &aiBDAddress[1], &aiBDAddress[0], &ignore, &aiBDAddress[3], &aiBDAddress[5], &aiBDAddress[4]);
            for (int uiArrayIndex = BDADDR_LENGTH - 1; uiArrayIndex >= 0;
                  --uiArrayIndex)
                  {
               ETG_TRACE_USR4(("The address is %u",(tU8)aiBDAddress[uiArrayIndex]));
               oTmpVec[uiVecIndex].u8Value = (tU8) aiBDAddress[uiArrayIndex];
               oTmpVec[uiVecIndex].enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8);
               ++uiVecIndex;
            }
            bReadBDAddressResult = TRUE;
         }
         else
         {
            ETG_TRACE_USR4(("Cannot retrieve BD_Address from default_mib.txt"));

            FC_Bluetooth_tclCCAService* poCCAService = FC_Bluetooth_tclCCAService::poGetInstance();

            if(poCCAService)
            {
               poCCAService->vGetVehicleAddress(bdAddress);

               tU8 u8VehicleAddrLength = static_cast<tU8>(bdAddress.size());
               ETG_TRACE_USR4(("The length of Vehicle BD Address is %u",u8VehicleAddrLength));

               if (u8VehicleAddrLength == BDADDR_LENGTH * 2)
               {
                  oTmpVec.resize(DATAID_BD_ADDRESS_BLUETOOTH_DEVICE_READ_RESULT_LEN);
                  oDataResult = diaglib::EN_DIAGDATA_OK; // Read Data succesfully
                  unsigned int uiVecIndex = DATAID_BD_ADDRESS_BLUETOOTH_DEVICE_READ_RESULT_au8BDAddress;
                  tU8 u8HexValue;
                  char s8Temp[2] = {0};

                  /*lint -e{850} u8Index is modified in the body of the for loop */
                  for (tU8 u8Index = 0; u8Index < u8VehicleAddrLength - 1; ++u8Index)
                  {
                     s8Temp[0] = bdAddress[u8Index];
                     s8Temp[1] = '\0';
                     u8HexValue = static_cast<tU8>(strtol(s8Temp, OSAL_NULL, 16) * 16);
                     u8Index++;
                     s8Temp[0] = bdAddress[u8Index];
                     s8Temp[1] = '\0';
                     u8HexValue = (tU8)(u8HexValue + static_cast<tU8>(strtol(s8Temp, OSAL_NULL, 16)));

                     ETG_TRACE_USR4(("The address is %x",u8HexValue));
                     oTmpVec[uiVecIndex].u8Value = u8HexValue;
                     oTmpVec[uiVecIndex].enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8);
                     ++uiVecIndex;
                  }
                  bReadBDAddressResult = TRUE;
               }
               else
               {
                  ETG_TRACE_ERR(("Invalid address length from DB"));
               }
            }
            else
            {
               ETG_TRACE_ERR(("poCCAService is NULL"));
            }
         }

         if (bReadBDAddressResult == FALSE)
         {
            ETG_TRACE_ERR(("ERROR:No valid BD_ADDRESS"));
            oDataResult = diaglib::EN_DIAGDATA_NOT_OK; // Control failed
            diaglib::trParameter paramError;
            paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
            paramError.u8Value = (tU8) diaglib::EN_ERROR_FATAL_INTERNAL;
            oTmpVec.push_back(paramError);
         }
         // example for answer end
         bSendResponseToDiaglib = TRUE;

         break;
      } // case DATAID_BD_ADDRESS_BLUETOOTH_DEVICE:

         /***************************************************************************/
         /* DATAID_BD_GET_LINKQUALITY_[12] (read)                                      */
         /*                                                                         */
         /* \param[in]  none                                                        */
         /*                                                                         */
         /* \param[out] oDataResult                                                 */
         /*             u32 ID                                                      */
         /***************************************************************************/
      case DATAID_BD_GET_LINKQUALITY_1:
      case DATAID_BD_GET_LINKQUALITY_2:
      {
         tU16 u16DeviceHandle = 0;
         std::string szDeviceAddress = "";
         tU32 tempLq;
         char cmdstr[BD_CMD_STRING_LEN];
         char* ps8_deviceAddress;
         tBool bFind = false;
         CmdData link_quality_result;
         PairedDeviceList pairedDeviceList;
         DeviceConnectionStatusList deviceConnectionStatusList;

         FC_Bluetooth_tclCCAService* poCCAService = FC_Bluetooth_tclCCAService::poGetInstance();

         if(poCCAService)
         {
            poCCAService->vGetDeviceConnectionStatusList(deviceConnectionStatusList);

            if (u32DiagDataId == DATAID_BD_GET_LINKQUALITY_1)
            {
               ETG_TRACE_USR4(("vOnDiagnosisDataRead => DATAID_BD_GET_LINKQUALITY_1"));
               u16DeviceHandle = poCCAService->u16GetDeviceHandleForCurrentOutgoingSource(deviceConnectionStatusList);
            }
            else
            {
               ETG_TRACE_USR4(("vOnDiagnosisDataRead => DATAID_BD_GET_LINKQUALITY_2"));
               u16DeviceHandle = poCCAService->u16GetDeviceHandleForNonOutgoingSource(deviceConnectionStatusList);
            }

            ETG_TRACE_USR4(("The u16DeviceHandle value is %d",u16DeviceHandle));
            ETG_TRACE_USR4(("The m_u8TestmodeStatus_Status value is %d",m_u8TestmodeStatus_Status));
            memset(cmdstr, 0, BD_CMD_STRING_LEN);

            if (FC_BLUETOOTH_DEVICEHANDLE_UNKNOWN != u16DeviceHandle)
            {
               ETG_TRACE_USR4(("valid device handle for LinkQuality"));

               poCCAService->vGetPairedDeviceList(pairedDeviceList);

               for (std::list<DeviceBaseInfo>::iterator it = pairedDeviceList._deviceBaseInfoList.begin();
                     it != pairedDeviceList._deviceBaseInfoList.end(); it++)
               {
                  if (it->_deviceHandle == u16DeviceHandle)
                  {
                     if(poCCAService->_bBluetoothServiceIsAvail == true)
                     {
                        std::transform(it->_bdAddress.begin(), it->_bdAddress.begin(), it->_bdAddress.end(), ::tolower);
                     }

                     szDeviceAddress = const_cast<std::string &>(it->_bdAddress);

                     if (szDeviceAddress.length() == 0)
                     {
                        ETG_TRACE_ERR(("Error: szDeviceAddress is 0"));
                     }
                     else
                     {
                        ps8_deviceAddress = const_cast<char*>(szDeviceAddress.c_str()); //To Solve Lint Prio-2
                        snprintf(cmdstr,BD_CMD_STRING_LEN,
                              "%c%c:%c%c:%c%c:%c%c:%c%c:%c%c",
                              ps8_deviceAddress[0], ps8_deviceAddress[1],
                              ps8_deviceAddress[2], ps8_deviceAddress[3],
                              ps8_deviceAddress[4], ps8_deviceAddress[5],
                              ps8_deviceAddress[6], ps8_deviceAddress[7],
                              ps8_deviceAddress[8], ps8_deviceAddress[9],
                              ps8_deviceAddress[10],ps8_deviceAddress[11] );
                        bFind = true;
                     }
                     break;
                  }
               }

               if(bFind == false)
               {
                  ETG_TRACE_ERR(("Error: No valid device handle found"));
                  goto customer_diag_linkquality_failed; //lint !e801
               }
            }
            else if(EN_BluetoothConnectionResultSUCCESFULL == m_u8TestmodeStatus_Status)
            {
               ETG_TRACE_USR4((" Getting diagnosis connected device address!"));
               snprintf(cmdstr,BD_CMD_STRING_LEN,
                     "%02x:%02x:%02x:%02x:%02x:%02x",
                     m_au8BDAddress[0],
                     m_au8BDAddress[1],m_au8BDAddress[2],
                     m_au8BDAddress[3],m_au8BDAddress[4],
                     m_au8BDAddress[5]);
            }
            else
            {
               ETG_TRACE_ERR(("Error: No valid device handle found for Outgoing/Dev1 resp NonOutgoing/Dev2"));
               goto customer_diag_linkquality_failed; //lint !e801
            }

            tempLq = 0;
            link_quality_result = execRootCommand(FC_BT_ROOTDAEMON_CLIENT_NAME, GET_LINK_QUALITY, cmdstr);
            if (link_quality_result.errorNo == ERR_NONE)
            {
               memcpy(&tempLq, link_quality_result.message, sizeof(tempLq));
            }
            else
            {
               ETG_TRACE_FATAL(("Couldn't get link quality"));
               goto customer_diag_linkquality_failed; //lint !e801
            }

            oDataResult = diaglib::EN_DIAGDATA_OK;
            diaglib::trParameter paramId;
            paramId.enType = static_cast<diaglib::tenParameterType>(DATAID_BD_GET_LINKQUALITY_1_READ_RESULT_PARAM1);
            paramId.u32Value = tempLq;
            oTmpVec.push_back(paramId);
         }
         else
         {
            ETG_TRACE_ERR(("Error in :: poDevicemanager = FC_Bluetooth_DeviceManager::poGetInstance()"));
            customer_diag_linkquality_failed:
            oDataResult = diaglib::EN_DIAGDATA_NOT_OK; // Control failed
            diaglib::trParameter paramError;
            paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
            paramError.u8Value = (tU8) diaglib::EN_ERROR_FATAL_INTERNAL;
            oTmpVec.push_back(paramError);
         }
         // example for answer end
         bSendResponseToDiaglib = TRUE;
         break;
      } // case DATAID_BD_GET_LINKQUALITY_1:

         /***************************************************************************/
         /* DATAID_BD_GET_AUTHENTICATED_DEVICES (read)                              */
         /*                                                                         */
         /* \param[in]  none                                                        */
         /*                                                                         */
         /* \param[out] oDataResult                                                 */
         /*             u32 ID                                                      */
         /***************************************************************************/
      case DATAID_BD_GET_AUTHENTICATED_DEVICES:
      {
         ETG_TRACE_USR4(("vOnDiagnosisDataRead => DATAID_BD_GET_AUTHENTICATED_DEVICES"));
         tU32 u32NumPairedDevices = 0;
         PairedDeviceList pairedDeviceList;

         FC_Bluetooth_tclCCAService* poCCAService = FC_Bluetooth_tclCCAService::poGetInstance();

         if (poCCAService)
         {
            poCCAService->vGetPairedDeviceList(pairedDeviceList);

            // Number of bluetooth devices in database
            for (std::list<DeviceBaseInfo>::iterator it = pairedDeviceList._deviceBaseInfoList.begin();
                  it != pairedDeviceList._deviceBaseInfoList.end(); it++)
            {
               u32NumPairedDevices++;
            }

            oDataResult = diaglib::EN_DIAGDATA_OK;
            diaglib::trParameter paramId;
            paramId.enType = static_cast<diaglib::tenParameterType>(DATAID_BD_GET_AUTHENTICATED_DEVICES_READ_RESULT_PARAM1);
            paramId.u32Value = u32NumPairedDevices;
            oTmpVec.push_back(paramId);
         }
         else
         {
            ETG_TRACE_ERR(("Error in :: poCCAService is Null"));
            oDataResult = diaglib::EN_DIAGDATA_NOT_OK; // Control failed
            diaglib::trParameter paramError;
            paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
            paramError.u8Value = (tU8) diaglib::EN_ERROR_FATAL_INTERNAL;
            oTmpVec.push_back(paramError);
         }
         // example for answer end
         bSendResponseToDiaglib = TRUE;

         break;
      } // case DATAID_BD_GET_AUTHENTICATED_DEVICES:

         /***************************************************************************/
         /* DATAID_BD_CONNECTED_DEVICE_STATUS_1 (read)                              */
         /* DATAID_BD_CONNECTED_DEVICE_STATUS_2 (read)                              */
         /*                                                                         */
         /* \param[in]  none                                                        */
         /*                                                                         */
         /* \param[out] oDataResult                                                 */
         /*             tU8 Bitfield 1                                              */
         /*             tU8 Bitfield 2                                              */
         /*             tU8 Bitfield 3                                              */
         /*             tU8 Bitfield 4                                              */
         /*             tU8 Bitfield 5                                              */
         /***************************************************************************/
      case DATAID_BD_CONNECTED_DEVICE_STATUS_1:
      case DATAID_BD_CONNECTED_DEVICE_STATUS_2:
      {
         tU16 u16DeviceHandle = INVALID_DEVICE_HANDLE;
         std::string szDeviceAddress = "";
         FC_Bluetooth_tclCCAService* poCCAService = FC_Bluetooth_tclCCAService::poGetInstance();
         fc_bluetooth_tclClientHandler_Phone* poClienthandlerPhone = OSAL_NULL;
         //tenAlpsPlayStaus e_playbackStatus = ALPS_PLAYSTATUS_STOPPED;
         tU8 u8Ret1 = 0, u8Ret2 = 0, u8Ret3 = 0, u8Ret4 = 0, u8Ret5 = 0;
         DeviceConnectionStatusList deviceConnectionStatusList;
         RemoteDeviceInfo remoteDeviceInfo;
         tBool bPANSupport = false;
         //tBool bDUNSupport = false;

         if (poCCAService)
         {
            poCCAService->vGetDeviceConnectionStatusList(deviceConnectionStatusList);

            if (u32DiagDataId == DATAID_BD_CONNECTED_DEVICE_STATUS_1)
            {
               ETG_TRACE_USR4(("vOnDiagnosisDataRead => DATAID_BD_CONNECTED_DEVICE_STATUS_1"));
               u16DeviceHandle = poCCAService->u16GetDeviceHandleForCurrentOutgoingSource(deviceConnectionStatusList);
            }
            else
            {
               ETG_TRACE_USR4(("vOnDiagnosisDataRead => DATAID_BD_CONNECTED_DEVICE_STATUS_2"));
               u16DeviceHandle = poCCAService->u16GetDeviceHandleForNonOutgoingSource(deviceConnectionStatusList);
            }

            if (u16DeviceHandle != INVALID_DEVICE_HANDLE)
            {
               ETG_TRACE_USR4(("valid device handle for BD_CONNECTED_DEVICE_STATUS"));

               result = _bluetoothApp->getBmCoreRequestIf().getRemoteDeviceInfo(remoteDeviceInfo, u16DeviceHandle);

               if (BM_RESULT_OK != result)
               {
                  ETG_TRACE_ERR(("Error: getRemoteDeviceInfo"));
                  goto customer_diag_device_status_failed; //lint !e801
               }
               else
               {
                  if (poCCAService->_bBluetoothServiceIsAvail == true)
                  {
                     std::transform(remoteDeviceInfo._deviceBaseInfo._bdAddress.begin(), remoteDeviceInfo._deviceBaseInfo._bdAddress.end(),
                                             remoteDeviceInfo._deviceBaseInfo._bdAddress.begin(), ::tolower);
                  }

                  szDeviceAddress = const_cast<std::string &>(remoteDeviceInfo._deviceBaseInfo._bdAddress);

                  if (szDeviceAddress.length() == 0)
                  {
                     ETG_TRACE_ERR(("Error: szDeviceAddress is 0"));
                     goto customer_diag_device_status_failed; //lint !e801
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR(("Error: No valid device handle found for Outgoing/Dev1 resp NonOutgoing/Dev2"));
               goto customer_diag_device_status_failed; //lint !e801
            }
         }
         else
         {
            ETG_TRACE_ERR(("Error in :: poCCAService = FC_Bluetooth_tclCCAService::poGetInstance()"));
            goto customer_diag_device_status_failed; //lint !e801
         }

         //**********************
         //***   IMPL START   ***
         //**********************

         /* Get the coresponding Call Instance */
         //Update DeviceList: from DeviceList Manager to Most FI type.
         //---sbv2hi--- p_bt_devListManager = m_oDevManager;
         /*---sbv2hi--- TO DO:
          - Byte3,bit4/5 : UID Persistency
          - Byte1,bit4   : InternetStatus
          - Byte1,bit5   : PlayStatus
          */

         if (u16DeviceHandle != INVALID_DEVICE_HANDLE)
         {
            //GM Spec: Bluetooth Connected Device [12] Status: $1 = Connected
            u8Ret1 |= (1 << 7);
         }

         if (remoteDeviceInfo._deviceBaseInfo._majorDeviceClass == BTS_COD_MAJORDC_PHONE)
            u8Ret5 |= (1 << 7);
         else if (remoteDeviceInfo._deviceBaseInfo._majorDeviceClass == BTS_COD_MAJORDC_AUDIO)
            u8Ret5 |= (1 << 6);
         else if (remoteDeviceInfo._deviceBaseInfo._majorDeviceClass == BTS_COD_MAJORDC_COMPUTER)
            u8Ret5 |= (1 << 5);
         else if (remoteDeviceInfo._deviceBaseInfo._majorDeviceClass == BTS_COD_MAJORDC_LAN)
            u8Ret5 |= (1 << 4);

         for (ProtocolInfoMap::const_iterator it = remoteDeviceInfo._deviceBaseInfo._protocolInfo.begin();
               it != remoteDeviceInfo._deviceBaseInfo._protocolInfo.end(); it++)
         {
            ETG_TRACE_USR1(("ProtocolID = %i; isSupported = %d",
                       it->first, it->second._isSupported));

              switch (it->first)
              {
                 case BM_PROTOCOL_ID_HFP:
                    u8Ret2 |= (1 << 7);
                 break;
                 case BM_PROTOCOL_ID_AVP:
                    if (it->second._version == 0x0100)
                    {
                        // GM Spec: Bluetooth Connected Device 1 Supports Audio Streaming - AVRCP 1.0: $0 = No - $1 = Yes
                        u8Ret3 |= (1 << 7);
                     }
                     else if (it->second._version == 0x0103)
                     {
                        // GM Spec: Bluetooth Connected Device 1 Supports Audio Streaming - AVRCP 1.3: $0 = No - $1 = Yes
                        u8Ret3 |= (1 << 6);
                     }
                     else if (it->second._version == 0x0104)
                     {
                        //---sbv2hi--- TO DO ! Due to missing PlayerFeatureBitmask: BOTH 1.4-bits are set to indicate support of 1.4, but the detailed state is unknown!
                        // GM Spec: Bluetooth Connected Device 1 Supports Audio Streaming - AVRCP 1.4: Browsing/No UIDPersistency - $0 = No - $1 = Yes
                        u8Ret3 |= (1 << 5);
                        // GM Spec: Bluetooth Connected Device 1 Supports Audio Streaming - AVRCP 1.4: Browsing/UIDPersistency - $0 = No - $1 = Yes
                        u8Ret3 |= (1 << 4);
                     }
                 break;
                 case BM_PROTOCOL_ID_PAN:
                    bPANSupport = it->second._isSupported;
                    //bDUNSupport = false;
                 break;
                 case BM_PROTOCOL_ID_PBDL:
                    u8Ret2 |= (1 << 6);
                 break;
                 case BM_PROTOCOL_ID_MSG:
                    u8Ret2 |= (1 << 5);
                 break;
                 case BM_PROTOCOL_ID_SPP:
                    u8Ret2 |= (1 << 4);
                 break;
                 //if (poDevice->oGetBluetoothFeatureSupport().m_bPortableNavigationSupport)
                 //   u8Ret2 |= (1 << 3);
                 default:
                    ETG_TRACE_ERR(("Invalid supported profile"));
                 break;
              }
         }

         if(bPANSupport  == true)
         {
        	 u8Ret4 |= (1 << 6);
         }
         //!Fix for CID 20288: Logically dead code (DEADCODE)
         /*if((bDUNSupport  == true) && (bPANSupport == true))
         {
            u8Ret4 |= (1 << 5);
         }
         else if(bPANSupport  == true)
         {
            u8Ret4 |= (1 << 6);
         }
         else if(bDUNSupport  == true)
         {
            u8Ret4 |= (1 << 7);
         }*/


         //TODO
#if 0
         /*GMMY15- 4475: Audio Streaming Status : Not Playing when a Bluetooth Audio song is playing.
          * (Byte1-bit5 field is not updated to diagnostics)*/

         ETG_TRACE_USR4(("AVPPlaystatus in diaglib %d",poDevice->vGetAvpPlayStatus()));

         e_playbackStatus = poDevice->vGetAvpPlayStatus();

         if (e_playbackStatus == ALPS_PLAYSTATUS_PLAYING)
         {
            /* GM Spec: Bluetooth Connected Device 1 Status :$0 = Not Playing: $1 = Playing: */
            u8Ret1 |= (1 << 5);
         }
         else
         {
            u8Ret1 &= MASK_BIT_CLEAR_BYTE1_BIT5;
         }
#endif
         // START: call BT_APPL ...
         OSAL_tEventMask oDummyMask, oEventMask;

         ETG_TRACE_USR4(("*********************CALLING GET CALLSTATUS OF ALPS********************************"));
         poClienthandlerPhone = fc_bluetooth_tclClientHandler_Phone::poGetInstance();

         if(poClienthandlerPhone)
         {
            if((poClienthandlerPhone->vGetCallStatus()) == TRUE)
            {

               ETG_TRACE_USR4(( " BT-CALLSTATUS: Entering OSAL_s32EventWait. " ));

               m_bFusionCBPending = TRUE;
               // WAIT for event ...
               tS32 RetVal = OSAL_s32EventWait(m_hEvFusion, EVENT_FUSION, OSAL_EN_EVENTMASK_OR, OSAL_C_TIMEOUT_FOREVER, &oDummyMask);

               if (RetVal != OSAL_OK)
               {
            	   tU32 u32ErrorCode = OSAL_u32ErrorCode();

            	   ETG_TRACE_FATAL(( " BT-CALLSTATUS: OSAL_s32EventWait returned with Error code= %d", u32ErrorCode ));
            	   ETG_TRACE_ERRMEM(( " BT-CALLSTATUS: OSAL_s32EventWait returned with Error code= %d", u32ErrorCode ));
               }

               ETG_TRACE_USR4(( " BT-CALLSTATUS: Leaving OSAL_s32EventWait. " ));

               // reset event!
               oEventMask = (OSAL_tEventMask)(~EVENT_FUSION);
               OSAL_s32EventPost(m_hEvFusion, oEventMask, OSAL_EN_EVENTMASK_AND);

               // example for answer beginn
               if ((m_s8CallStatus == 0/*PHONE_CALLSTATE_IDLE*/) || (m_s8CallStatus == -1/*status is invalid*/))

               {
                  // Call idle
               }
               else
               {
                  // GM Spec: Bluetooth Connected Device 1 Hands Free Status: $0 = No Call Active - $1 = Call Active
                  u8Ret1 |= (1 << 6);
               }
            }
         }

         //dbus_g_proxy_disconnect_signal(remoteProxyDevice, "CallStatus", G_CALLBACK(_DiagCallStatusEvent_handler), OSAL_NULL);

         // END: call BT_APPL ...

         //********************
         //***   IMPL END   ***
         //********************

         // example for answer beginn
         if (1) //lint !e774 !e506 , eliminating lint warnings
         {
            oDataResult = diaglib::EN_DIAGDATA_OK;
            diaglib::trParameter paramBitfield;
            // Bitfield 1
            paramBitfield.enType = static_cast<diaglib::tenParameterType>(DATAID_BD_CONNECTED_DEVICE_STATUS_2_READ_RESULT_PARAM1);
            paramBitfield.u8Value = u8Ret1;
            oTmpVec.push_back(paramBitfield);
            // Bitfield 2
            paramBitfield.enType = static_cast<diaglib::tenParameterType>(DATAID_BD_CONNECTED_DEVICE_STATUS_2_READ_RESULT_PARAM2);
            paramBitfield.u8Value = u8Ret2;
            oTmpVec.push_back(paramBitfield);
            // Bitfield 3
            paramBitfield.enType = static_cast<diaglib::tenParameterType>(DATAID_BD_CONNECTED_DEVICE_STATUS_2_READ_RESULT_PARAM3);
            paramBitfield.u8Value = u8Ret3;
            oTmpVec.push_back(paramBitfield);
            // Bitfield 4
            paramBitfield.enType = static_cast<diaglib::tenParameterType>(DATAID_BD_CONNECTED_DEVICE_STATUS_2_READ_RESULT_PARAM4);
            paramBitfield.u8Value = u8Ret4;
            oTmpVec.push_back(paramBitfield);
            // Bitfield 5
            paramBitfield.enType = static_cast<diaglib::tenParameterType>(DATAID_BD_CONNECTED_DEVICE_STATUS_2_READ_RESULT_PARAM5);
            paramBitfield.u8Value = u8Ret5;
            oTmpVec.push_back(paramBitfield);

            ETG_TRACE_ERR(("paramBitfield1(dec) %d\n,paramBitfield2(dec) %d\n,paramBitfield3(dec) %d\n,paramBitfield4(dec) %d\n,paramBitfield5(dec) %d\n",u8Ret1,u8Ret2,u8Ret3,u8Ret4,u8Ret5));
            ETG_TRACE_ERR(("paramBitfield1(Hex) %x\n,paramBitfield2(Hex) %x\n,paramBitfield3(Hex) %x\n,paramBitfield4(Hex) %x\n,paramBitfield5(Hex) %x\n",u8Ret1,u8Ret2,u8Ret3,u8Ret4,u8Ret5));
         }
         else
         {
            customer_diag_device_status_failed: oDataResult = diaglib::EN_DIAGDATA_NOT_OK; // Control failed
            diaglib::trParameter paramError;
            paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
            paramError.u8Value = (tU8) diaglib::EN_ERROR_FATAL_INTERNAL;
            oTmpVec.push_back(paramError);
         }
         // example for answer end
         bSendResponseToDiaglib = TRUE;

         break;
      } // case DATAID_BD_CONNECTED_DEVICE_STATUS_1 AND DATAID_BD_CONNECTED_DEVICE_STATUS_2:

         /***************************************************************************/
         /* DATAID_BD_GET_CURRENT_CONNECTION (read)                                 */
         /*                                                                         */
         /* \param[in]  none                                                        */
         /*                                                                         */
         /* \param[out] oDataResult                                                 */
         /*             tU8[6] BD-Address                                           */
         /***************************************************************************/
      case DATAID_BD_GET_CURRENT_CONNECTION:
      {
         ETG_TRACE_USR4(("vOnDiagnosisDataRead => DATAID_BD_GET_CURRENT_CONNECTION"));

         // TODO: add your implementation here

         if (1) //lint !e774 !e506 , eliminating lint warnings
         {
            oTmpVec.resize(DATAID_BD_GET_CURRENT_CONNECTION_READ_RESULT_LEN);
            oDataResult = diaglib::EN_DIAGDATA_OK; // Read Data succesfully

            // fill BD-Address
            tU8 au8MacAddress[6] =
            { 0x11, 0x22, 0x33, 0x44, 0x55, 0x66 };
            unsigned int uiVecIndex = DATAID_BD_ADDRESS_MAC_READ_RESULT_au8MacAddress;
            for (unsigned int uiArrayIndex = 0;
                  uiArrayIndex < DATAID_BD_ADDRESS_MAC_READ_RESULT_PARAM1_LEN;
                  ++uiArrayIndex)
            {
               oTmpVec[uiVecIndex].u8Value = au8MacAddress[uiArrayIndex];
               oTmpVec[uiVecIndex].enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8);
               ++uiVecIndex;
            }

            // fill Status Result
            tenDiaglibBluetoothConnectionResult eTestResult = EN_BluetoothConnectionResultSUCCESFULL;

            oTmpVec[DATAID_BD_GET_CURRENT_CONNECTION_READ_RESULT_u32TestResult].u32Value = static_cast<tU32>(eTestResult);
            oTmpVec[DATAID_BD_GET_CURRENT_CONNECTION_READ_RESULT_u32TestResult].enType = static_cast<diaglib::tenParameterType>(DATAID_BD_GET_CURRENT_CONNECTION_READ_RESULT_PARAM2);
         }
         else
         {
            oDataResult = diaglib::EN_DIAGDATA_NOT_OK; // Control failed
            diaglib::trParameter paramError;
            paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
            paramError.u8Value = (tU8) diaglib::EN_ERROR_FATAL_INTERNAL;
            oTmpVec.push_back(paramError);
         }
         // example for answer end
         bSendResponseToDiaglib = TRUE;

         break;
      } // case DATAID_BD_CONNECTED_DEVICE_STATUS_2:

      default:
      {
         ETG_TRACE_USR2(("!!! vOnDiagnosisDataRead => ERROR : invalid u32DiagDataId: %d",u32DiagDataId));
         break;
      } // default:
   } // switch(u32DiagDataId)

   if (bSendResponseToDiaglib == TRUE)
   {
      // send message back to diagnosis
      if (m_poDiaglibService != OSAL_NULL)
      {
         if (TRUE == m_poDiaglibService->poGetDiagData()->bSendDiagnosisDataResult(oDataResult, oTmpVec, MsgContext))
         {
            u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
            ETG_TRACE_USR4(("vOnDiagnosisDataRead => bSendDiagnosisDataResult successfull"));
         } // if(TRUE == m_poDiaglibService->poGetIoCtrl()
         else
         {
            u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
            ETG_TRACE_ERR(("!!! vOnDiagnosisDataRead => ERROR : bSendDiagnosisDataResult failed"));
         } // else // if(TRUE == m_poDiaglibService->poGetIoCtrl()
      } // if(m_poDiaglibService != OSAL_NULL)
      else
      {
         u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
         ETG_TRACE_ERR(("!!! vOnDiagnosisDataRead => ERROR : m_poDiaglibService == OSAL_NULL"));
      } // else // if(m_poDiaglibService != OSAL_NULL)
   }

   ETG_TRACE_USR2(("<-- vOnDiagnosisDataRead(%x)",u32Return));

   return u32Return;
}

/***********************************************************************
 * \brief CCA Service vOnDiagDataWrite
 *
 * Command from Diagnois for simple Write request
 *
 ************************************************************************/
tU32 fc_bluetooth_tclClientHandler_Diaglib::vOnDiagnosisDataWrite(tU32 u32DiagDataId, const diaglib::tclParameterVector& oParamVector, diaglib::tContext MsgContext)
{
ENTRY
	   ETG_TRACE_ERR(("!!! fc_bluetooth_tclClientHandler_Diaglib::vOnDiagDataWrite => ERROR : not supported"));
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32DiagDataId);
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( oParamVector);
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( MsgContext);
   return diaglib::U32_DIAGLIB_RETURN_NOT_OK;
}

/*******************************************************************************
 * FUNCTION:   poGetInstance(..)
 *
 * DESCRIPTION:
 *   Get the instance of BT Diag service
 *  PARAMETERS:
 *     IN:   None
 *
 *  RETURNVALUE: fc_bluetooth_tclClientHandler_Diaglib* : Pointer to BT Diag service
 ********************************************************************************/
fc_bluetooth_tclClientHandler_Diaglib* fc_bluetooth_tclClientHandler_Diaglib::poGetInstance(tVoid)
{
   ENTRY
   return(m_poBtDiaglib);
}
/*******************************************************************************/

tVoid fc_bluetooth_tclClientHandler_Diaglib::vUpdateWifiDtc(tBool& bUpdate, tBool& bMacAddressfound)
{
   tChar  szChipID[MAC_ADDRESS_ARRAY_SIZE] = "";
   tChar  szBuildID[MAC_ADDRESS_ARRAY_SIZE] = "";

   ETG_TRACE_USR1(("vUpdateWifiDtc: entered"));

   bUpdate = FALSE;
   bMacAddressfound = FALSE;

   // Get the value for the ChipID
   if(TRUE == get_mib_config_string(const_cast<tChar*>("ChipID"), szChipID)) //To Solve Lint Prio-2
   {
      ETG_TRACE_USR4(("vUpdateWifiDtc: ChipID is %s",szChipID));
   }

   // Get the value for the BuildID
   if(TRUE == get_mib_config_string(const_cast<tChar*>("BuildID"), szBuildID)) //To Solve Lint Prio-2
   {
      ETG_TRACE_USR4(("vUpdateWifiDtc: BuildID is %s",szBuildID));
   }

   //For CSR A11 and Marvell chip is used BT-Wifi combo
   if(((0 == s32StringCompare(szChipID, BT_CHIP_ID_MARVELL)) && (0 == s32StringCompare(szBuildID, BT_BUILD_ID_MARVELL))) ||
         ((0 == s32StringCompare(szChipID, BT_CHIP_ID_CSR)) && (0 == s32StringCompare(szBuildID, BT_BUILD_ID_A11))))
   {
      ETG_TRACE_USR4(("vUpdateWifiDtc: Chip is belongs to BT-Wifi Combo"));
      bUpdate = TRUE;

      struct sockaddr_ll *stSockAddr = getWifiAddress();

      if(NULL != stSockAddr)
      {
         for (unsigned int uiArrayIndex = 0; uiArrayIndex < stSockAddr->sll_halen; ++uiArrayIndex)
         {
            ETG_TRACE_USR4(("vUpdateWifiDtc: Wi-Fi Mac Address ::%02x", stSockAddr->sll_addr[uiArrayIndex]));
         }

         bMacAddressfound = TRUE;
      }
   }

   ETG_TRACE_USR4(("vUpdateWifiDtc: bUpdate = %d, bMacAddressfound = %d", bUpdate, bMacAddressfound));
}

struct sockaddr_ll* fc_bluetooth_tclClientHandler_Diaglib::getWifiAddress()
{
   struct ifaddrs *ifaddr = NULL;
   struct sockaddr_ll *stSockAddr = NULL;

   ETG_TRACE_USR1(("getWifiAddress: entered"));

   //getifaddrs() function stores a reference to a linked list of ifaddrs
   //structures,one structure per interface. The end of the linked list of
   //structures is indicated by a structure with an ifa_next of NULL
   if (getifaddrs(&ifaddr) == -1)
   {
      ETG_TRACE_ERR(("getWifiAddress: getifaddrs error !"));
      return stSockAddr;
   }

   for (struct ifaddrs *ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next)
   {
      if (ifa->ifa_addr == NULL)
         continue;

      ETG_TRACE_USR4(("getWifiAddress: list of interface name: %s", ifa->ifa_name));

      //Compare the interface name with "wlan0", if it matches check
      //for the family AF_PACKET. AF_PACKET family provides the mac
      //address as AF_INET provides the IP address of the interface.
      if(((0 == s32StringCompare(ifa->ifa_name, "wlan0")) && (AF_PACKET == ifa->ifa_addr->sa_family)))
      {
         stSockAddr = (struct sockaddr_ll*)ifa->ifa_addr;   //lint !e826
         ETG_TRACE_USR4(("getWifiAddress: wla0 interface is available"));
         break;
      }
   }

   freeifaddrs(ifaddr);
   return stSockAddr;
}

tVoid fc_bluetooth_tclClientHandler_Diaglib::vTestmodeStatus(const S_TestmodeStatus_t* const prCallData)
{
   ENTRY
   m_u8TestmodeStatus_NotiType = prCallData->u8NotiType;

   ETG_TRACE_USR2(( " vTestmodeStatus ENTERED: m_Notitype= %d, m_Status= %d, _TMComp= %d", m_u8TestmodeStatus_NotiType, m_u8TestmodeStatus_Status, prCallData->u8Status_TMComp ));

   if (m_u8TestmodeStatus_NotiType == (tU8) -1) // Immediate response received (BT_APPL_TestMode_CFM)
   {
      // update status only in case of successful immediatate response.
      // Otherwise something went wrong and we have to keep the previous state!
      if (prCallData->u8Status_TMComp != EN_BluetoothConnectionResultFAILED)
      {
         m_u8TestmodeStatus_Status = prCallData->u8Status_TMComp;
         ETG_TRACE_USR2(("Waiting for the Validation complete."));
      }
      else
      {
         ETG_TRACE_ERR(("CFM is received with FAILURE "));
         m_u8TestmodeStatus_Status = prCallData->u8Status_TMComp;

         ETG_TRACE_USR2(( " BeforePOST : m_Notitype=%d m_Status=%d _TMComp=%d ", m_u8TestmodeStatus_NotiType, m_u8TestmodeStatus_Status, prCallData->u8Status_TMComp ));
         ETG_TRACE_USR4(( " BT-TESTMODE: [final result] OSAL_s32EventPost EVENT_FUSION. " ));
         OSAL_s32EventPost(m_hEvFusion, EVENT_FUSION, OSAL_EN_EVENTMASK_OR);
         m_bFusionCBPending = FALSE;
      }
   }
   else
   {
      // Ok, final result is available (BT_APPL_TestMode_Complete_IND)
      // In case of CONNECTION we can just take the given state
      // In case of DISCONNECTION we have to convert OK->NOK, because successfully disconnected means "No connection" :-)
      if (prCallData->u8NotiType == 0)
      { // disconnection
         m_u8TestmodeStatus_Status =
               (prCallData->u8Status_TMComp == EN_BluetoothConnectionResultSUCCESFULL) ? EN_BluetoothConnectionResultFAILED : EN_BluetoothConnectionResultSUCCESFULL;
      }
      else
      {
         m_u8TestmodeStatus_Status = prCallData->u8Status_TMComp;
      }

      ETG_TRACE_USR2(( " BeforePOST : m_Notitype=%d m_Status=%d _TMComp=%d ", m_u8TestmodeStatus_NotiType, m_u8TestmodeStatus_Status, prCallData->u8Status_TMComp ));
      ETG_TRACE_USR4(( " BT-TESTMODE: [final result] OSAL_s32EventPost EVENT_FUSION. " ));
      OSAL_s32EventPost(m_hEvFusion, EVENT_FUSION, OSAL_EN_EVENTMASK_OR);
      m_bFusionCBPending = FALSE;
   }

   ETG_TRACE_USR2(( " vTestmodeStatus RETURN: m_Notitype= %d, m_Status= %d, _TMComp= %d ", m_u8TestmodeStatus_NotiType, m_u8TestmodeStatus_Status, prCallData->u8Status_TMComp));
}

tVoid fc_bluetooth_tclClientHandler_Diaglib::vLinkQualityStatus(const BmResult bmResult, const LinkQualityInfo& linkQualityInfo)
{
   ETG_TRACE_USR2(( "vLinkQualityStatus ENTERED"));

   m_bLinkQuality_Status = bmResult;
   m_linkQualityInfo._rssi = linkQualityInfo._rssi;
   m_linkQualityInfo._linkQuality = linkQualityInfo._linkQuality;

   ETG_TRACE_USR2(( "vLinkQualityStatus: _rssi - %d _linkQuality - %d", m_linkQualityInfo._rssi, m_linkQualityInfo._linkQuality));

   if (m_bFusionCBPending)
   {
      OSAL_s32EventPost( m_hEvFusion,EVENT_FUSION, OSAL_EN_EVENTMASK_OR );
      m_bFusionCBPending = FALSE;
   }
}

/***********************************************************************
 *
 * Helper function to retrieve the value string of a given key from the MIB data file
 *
 ************************************************************************/
gboolean get_mib_config_string(char * p_key, char * p_value)
{
   ENTRY
#define MIB_STR_MAX 204
   static FILE *fp_EepromDataFile = OSAL_NULL;

   gboolean found = FALSE;

   tChar p_temp[MIB_STR_MAX];
   tChar *p_string;

   // Open the config file which contains EEPROM data from UGKZ2 module
   fp_EepromDataFile = fopen("/var/opt/bosch/dynamic/connectivity/bt_module/default_mib.txt", "r");

   if (fp_EepromDataFile == OSAL_NULL)
   {
      /* ERROR*/
      return FALSE;
   }
   while ((fgets(p_temp, MIB_STR_MAX, fp_EepromDataFile)) && (found == FALSE))
   {
      /*check for comment as first char*/
      if (p_temp[0] == '#')
      {
         continue;
      }

      p_temp[strlen(p_temp)] = '\0';

      p_string = strstr(p_temp, p_key);

      if (p_string != p_temp)
      {
         /*Key not found at start of string*/
         continue;
      }

      if (strstr(p_string + strlen(p_key), " ::= ") == OSAL_NULL)
      {
         /*Key isn't exactly same in the file or isn't followed by an " ::= " string*/
         continue;
      }

      /*Move to after the equals sign*/
      p_string += strlen(p_key) + strlen(" ::= ");
      /*Return string, first clear string array and add string termination byte... */
      memset(p_value, '\0', strlen(p_string));

      //!Fix for CID 47369: Integer overflowed argument (INTEGER_OVERFLOW)
      (0u == strlen(p_string))? (strncpy(p_value, p_string, strlen(p_string)))
    		  : (strncpy(p_value, p_string, strlen(p_string) - 1));

      ETG_TRACE_USR4(("The length of value is %d",strlen(p_value)));
      found = TRUE;
   }
   if (fp_EepromDataFile != OSAL_NULL)
   {
      fclose(fp_EepromDataFile);
   }
   return found;
}

tVoid fc_bluetooth_tclClientHandler_Diaglib::vDiagCallStatus(const tU8 u8CallState)
{
   ENTRY_INTERNAL

   ETG_TRACE_USR4((" vDiagCallStatus:u8CallState - %d", u8CallState));

   m_s8CallStatus = (tS8)u8CallState;

   OSAL_s32EventPost(m_hEvFusion, EVENT_FUSION, OSAL_EN_EVENTMASK_OR);
   m_bFusionCBPending = FALSE;

   return;
}

/**************************************************************************
 *  Helper function to remove a special character from a cstring
 *
 **************************************************************************/
void remove_special_char(char* szinputdata, const char spl)
{
   ENTRY

   if (OSAL_NULL == szinputdata)
   {
      ETG_TRACE_ERR((" The input string to remove_special_char is OSAL_NULL"));
      return;
   }
   unsigned short loop_index = 0;
   unsigned short write_index = 0;
   char c;
   while ((c = szinputdata[loop_index++]) != '\0')
   {
      if (c != spl)
      {
         szinputdata[write_index++] = c;
      }
   }
   szinputdata[write_index] = '\0';

   return;
}
/**************************************************************************
 *  Helper function to reset m_hEvFusion & clear event queue
 *  in case CB is not received , on Ignition OFF
 *
 **************************************************************************/

tVoid fc_bluetooth_tclClientHandler_Diaglib::vReleaseEventFusionOnShutdown()
{
   ENTRY
   if (m_bFusionCBPending)
   {
      ETG_TRACE_USR4(("Callback is pending,releasing m_hEvFusion"));
      OSAL_s32EventPost(m_hEvFusion, EVENT_FUSION, OSAL_EN_EVENTMASK_OR);
      m_bFusionCBPending = FALSE;
   }
   else
   {
      ETG_TRACE_USR4(("No callback pending, m_hEvFusion has been reset"));
   }

   return;
}
