/**
 * @file FC_Bluetooth_TestCommand.cpp
 *
 * @par SW-Component
 * Communication service layer
 *
 * @brief Communication service layer: test command handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Communication service layer: test command handling.
 */

#include "FC_Bluetooth_TestCommand.h"
#include "FC_Bluetooth_main.h"
#include "IBmCoreRequestIf.h"
#include "FwTestCommandServer.h"
#include "FwTestCommandData.h"
#include "FwStringUtils.h"
#include <string>

#include "FC_Bluetooth_clienthandler_Diaglib.h"

#define DIAGLIB_INCLUDE_ROUTINE_CONTROL
#define DIAGLIB_INCLUDE_DIAGNOSIS_DATA
#include "diaglib_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APPLICATION
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FC_Bluetooth_TestCommand.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APPLICATION
#endif
#endif

FC_Bluetooth_TestCommand::FC_Bluetooth_TestCommand() :
_channel(0)
{
   _bluetoothApp = OSAL_NULL;
}

FC_Bluetooth_TestCommand::FC_Bluetooth_TestCommand(fc_bluetooth_tclApp* poMainAppl) :
_channel(0)
{
   _bluetoothApp = poMainAppl;
}

FC_Bluetooth_TestCommand::~FC_Bluetooth_TestCommand()
{
   _bluetoothApp = OSAL_NULL;
}

void FC_Bluetooth_TestCommand::handleTestCommand(const unsigned char* data)
{
   if(0 == data)
   {
      return;
   }

   // data[0]: ignore
   // data[1]: component id (CCA layer, BM Core, BtStackIf, ...); use same ranges as used for tracing; check https://hi-dms.de.bosch.com/docushare/dsweb/ServicesLib/Document-770581/History
   // data[2]: command itself

   const unsigned char start = (unsigned char)(BM_APP_OFFSET);
   const unsigned char end = (unsigned char)(BM_APP_RANGE_END);

   if((start <= data[1]) && (data[1] <= end))
   {
      handleCommand(&data[2]);
   }
}

void FC_Bluetooth_TestCommand::create(const unsigned int channel)
{
   if(0 == channel)
   {
      return;
   }

   _channel = channel;

   // register test command callback
   ::fw::TestCommandServer::getInstance().create(_channel);
   ::fw::TestCommandServer::getInstance().registerClient(this);
}

void FC_Bluetooth_TestCommand::destroy(void)
{
   // deregister test command callback
   ::fw::TestCommandServer::getInstance().deregisterClient(this);
   ::fw::TestCommandServer::getInstance().destroy();
}

void FC_Bluetooth_TestCommand::handleCommand(const unsigned char* testData)
{
   if(0 == testData)
   {
      return;
   }

   ::fw::TestCommandData cmd;

   // use 2 bytes as command id
   const unsigned short int opcode = cmd.getU16Value(testData);
   // data starts with data[2]
   const unsigned char* data = &testData[2];

   ETG_TRACE_USR1((" TTFis command: opcode=0x%04X data=0x%02X %02X %02X %02X", opcode, data[0], data[1], data[2], data[3]));

   switch(opcode)
   {
      case 0x0000:
      {
         ETG_TRACE_USR1((" dummy command"));
         break;
      }
      case 0x0001: // to enable/disable sending answer for vOnRoutineControlStart()
      {
         if(0 != fc_bluetooth_tclClientHandler_Diaglib::poGetInstance())
         {
            fc_bluetooth_tclClientHandler_Diaglib::poGetInstance()->setEnableAnswer((tBool)(0x01 == cmd.getU8Value(data)));
         }
         break;
      }
      case 0x0002: // ROUTINEID_BD_CONNECTION: EN_ConnectionCONNECT
      {
         if(0 != fc_bluetooth_tclClientHandler_Diaglib::poGetInstance())
         {
            diaglib::tContext context = 0;
            diaglib::tclParameterVector param(ROUTINEID_BD_CONNECTION_START_LEN);
            param[ROUTINEID_BD_CONNECTION_START_u8Type].enType  = static_cast<diaglib::tenParameterType>(ROUTINEID_BD_CONNECTION_START_PARAM1);
            param[ROUTINEID_BD_CONNECTION_START_u8Type].u8Value = (tU8)EN_ConnectionCONNECT;
            const ::std::string addressString(cmd.getStringValue(data, 13));
            ::std::vector< unsigned char > address;
            ::fw::convertHexString2Binary(address, addressString);
            tU8 iVecIndex = ROUTINEID_BD_CONNECTION_START_au8BDAddress;
            for(tU8 u8ArrayIndex=0; u8ArrayIndex < ROUTINEID_BD_CONNECTION_START_PARAM2_LEN; ++u8ArrayIndex)
            {
               param[iVecIndex].enType  = diaglib::EN_PARAMETER_TYPE_U8;
               param[iVecIndex].u8Value = address[u8ArrayIndex];
               ++iVecIndex;
            }

            (void)fc_bluetooth_tclClientHandler_Diaglib::poGetInstance()->vOnRoutineControlStart(ROUTINEID_BD_CONNECTION, 0, param, context);
         }
         break;
      }
      case 0x0003: // ROUTINEID_BD_CONNECTION: EN_ConnectionDISCONECT
      {
         if(0 != fc_bluetooth_tclClientHandler_Diaglib::poGetInstance())
         {
            diaglib::tContext context = 0;
            diaglib::tclParameterVector param(ROUTINEID_BD_CONNECTION_START_LEN);
            param[ROUTINEID_BD_CONNECTION_START_u8Type].enType  = static_cast<diaglib::tenParameterType>(ROUTINEID_BD_CONNECTION_START_PARAM1);
            param[ROUTINEID_BD_CONNECTION_START_u8Type].u8Value = (tU8)EN_ConnectionDISCONECT;
            const ::std::string addressString(cmd.getStringValue(data, 13));
            ::std::vector< unsigned char > address;
            ::fw::convertHexString2Binary(address, addressString);
            tU8 iVecIndex = ROUTINEID_BD_CONNECTION_START_au8BDAddress;
            for(tU8 u8ArrayIndex=0; u8ArrayIndex < ROUTINEID_BD_CONNECTION_START_PARAM2_LEN; ++u8ArrayIndex)
            {
               param[iVecIndex].enType  = diaglib::EN_PARAMETER_TYPE_U8;
               param[iVecIndex].u8Value = address[u8ArrayIndex];
               ++iVecIndex;
            }

            (void)fc_bluetooth_tclClientHandler_Diaglib::poGetInstance()->vOnRoutineControlStart(ROUTINEID_BD_CONNECTION, 0, param, context);
         }
         break;
      }
      case 0x0004: // ROUTINEID_BD_ANTENNA_FIELDSTRENGTH
      {
         if(0 != fc_bluetooth_tclClientHandler_Diaglib::poGetInstance())
         {
            diaglib::tContext context  = 0;
            diaglib::tclParameterVector param(ROUTINEID_BD_ANTENNA_FIELDSTRENGTH_START_LEN);
            const ::std::string addressString(cmd.getStringValue(data, 13));
            ::std::vector< unsigned char > address;
            ::fw::convertHexString2Binary(address, addressString);
            tU8 iVecIndex = ROUTINEID_BD_ANTENNA_FIELDSTRENGTH_START_au8BDAddress;
            for(tU8 u8ArrayIndex=0; u8ArrayIndex < ROUTINEID_BD_ANTENNA_FIELDSTRENGTH_START_PARAM1_LEN; ++u8ArrayIndex)
            {
               param[iVecIndex].enType  = diaglib::EN_PARAMETER_TYPE_U8;
               param[iVecIndex].u8Value = address[u8ArrayIndex];
               ++iVecIndex;
            }

            (void)fc_bluetooth_tclClientHandler_Diaglib::poGetInstance()->vOnRoutineControlStart(ROUTINEID_BD_ANTENNA_FIELDSTRENGTH, 0, param, context);
         }
         break;
      }
      case 0x0005: // ROUTINEID_BD_WIFI_ANTENNA_SNR
      {
         if(0 != fc_bluetooth_tclClientHandler_Diaglib::poGetInstance())
         {
            diaglib::tContext context = 0;
            diaglib::tclParameterVector param(ROUTINEID_BD_WIFI_ANTENNA_SNR_START_LEN);
            const ::std::string addressString(cmd.getStringValue(data, 13));
            ::std::vector< unsigned char > address;
            ::fw::convertHexString2Binary(address, addressString);
            tU8 iVecIndex = ROUTINEID_BD_WIFI_ANTENNA_SNR_START_au8BDAddress;
            for(tU8 u8ArrayIndex=0; u8ArrayIndex < ROUTINEID_BD_WIFI_ANTENNA_SNR_START_PARAM1_LEN; ++u8ArrayIndex)
            {
               param[iVecIndex].enType  = diaglib::EN_PARAMETER_TYPE_U8;
               param[iVecIndex].u8Value = address[u8ArrayIndex];
               ++iVecIndex;
            }

            (void)fc_bluetooth_tclClientHandler_Diaglib::poGetInstance()->vOnRoutineControlStart(ROUTINEID_BD_WIFI_ANTENNA_SNR, 0, param, context);
         }
         break;
      }

      case 0x0006: //BM_CORE_CMD_STATE_CHANGE_NORMAL
      {
         ETG_TRACE_USR1(("stateChangeNormal"));
         //stateChangeNormal
         if(_bluetoothApp)
         {
            _bluetoothApp->getBmCoreRequestIf().stateChangeNormal();
         }
      }
      break;

      case 0x0007: //BM_CORE_CMD_STATE_CHANGE_OFF
      {
         ETG_TRACE_USR1(("stateChangeOff"));
         //stateChangeOff
         if(_bluetoothApp)
         {
            _bluetoothApp->getBmCoreRequestIf().stateChangeOff();
         }
      }
      break;

      case 0x0008: //BM_CORE_CMD_SET_BT_SYSTEM_STATE
      {
         ETG_TRACE_USR1(("setBtSystemState"));
         //setBtSystemState
         if(_bluetoothApp)
         {
            BtSystemState state((BluetoothSystemState) cmd.getU32Value(&data[0]));
            ETG_TRACE_USR1((" setBtSystemState to: %d", cmd.getU32Value(&data[0])));
            (void)_bluetoothApp->getBmCoreRequestIf().setBtSystemState(state);
         }
      }
      break;

      case 0x0009: //BM_CORE_CMD_GET_BT_SYSTEM_STATE
      {
         ETG_TRACE_USR1(("getBtSystemState"));
         //getBtSystemState
         if(_bluetoothApp)
         {
            BtSystemState state;
            (void)_bluetoothApp->getBmCoreRequestIf().getBtSystemState(state);
            ETG_TRACE_USR1((" getBtSystemState: %d", state._bluetoothSystemState));
         }
      }
      break;

      case 0x000A: //BM_CORE_CMD_SWITCH_BT_STATUS
      {
         ETG_TRACE_USR1(("switchBtStatus"));
         //switchBtStatus
         if(_bluetoothApp)
         {
             TargetSwitchState state((TargetSwitchState) cmd.getU32Value(&data[0]));
             ETG_TRACE_USR1((" switchBtStatus to: %d", cmd.getU32Value(&data[0])));
            (void)_bluetoothApp->getBmCoreRequestIf().switchBtStatus(state);
         }
      }
      break;

      case 0x000B: //BM_CORE_CMD_GET_BT_STATUS
      {
         ETG_TRACE_USR1(("getBtStatus"));
         //getBtStatus
         if(_bluetoothApp)
         {
            SwitchStatus state;
            (void)_bluetoothApp->getBmCoreRequestIf().getBtStatus(state);
            ETG_TRACE_USR1((" getBtStatus state: %d reason: %d", state._switchState, state._switchedOffReason));
         }
      }
      break;

      case 0x000C: //BM_CORE_CMD_SWITCH_LOCAL_PAIRABLE_MODE
      {
         ETG_TRACE_USR1(("switchLocalPairableMode"));
         //switchLocalPairableMode
         if(_bluetoothApp)
         {
            TargetSwitchState state((TargetSwitchState) cmd.getU32Value(&data[0]));
            ETG_TRACE_USR1((" switchLocalPairableMode to: %d", cmd.getU32Value(&data[0])));
            const unsigned char* str = &data[4];
            const ::std::string addressString(cmd.getStringValue(str, 13));
            ETG_TRACE_USR1((" bdaddress: %s", addressString.c_str()));
            (void)_bluetoothApp->getBmCoreRequestIf().switchLocalPairableMode(state, addressString);
         }
      }
      break;

      case 0x000D: //BM_CORE_CMD_GET_LOCAL_PAIRABLE_MODE
      {
         ETG_TRACE_USR1(("getLocalPairableMode"));
         //getLocalPairableMode
         if(_bluetoothApp)
         {
            SwitchStatus state;
            (void)_bluetoothApp->getBmCoreRequestIf().getLocalPairableMode(state);
            ETG_TRACE_USR1((" getLocalPairableMode state: %d reason: %d", state._switchState, state._switchedOffReason));
         }
      }
      break;

      case 0x000E: //BM_CORE_CMD_SWITCH_LOCAL_CONNECTABLE_MODE
      {
         ETG_TRACE_USR1(("switchLocalConnectableMode"));
         //switchLocalConnectableMode
         if(_bluetoothApp)
         {
            TargetSwitchState state((TargetSwitchState) cmd.getU32Value(&data[0]));
            ETG_TRACE_USR1((" switchLocalConnectableMode to: %d", cmd.getU32Value(&data[0])));
            const unsigned char* str = &data[4];
            const ::std::string addressString(cmd.getStringValue(str, 13));
            ETG_TRACE_USR1((" bdaddress: %s", addressString.c_str()));
            (void)_bluetoothApp->getBmCoreRequestIf().switchLocalConnectableMode(state, addressString);
         }
      }
      break;

      case 0x000F: //BM_CORE_CMD_GET_LOCAL_CONNECTABLE_MODE
      {
         ETG_TRACE_USR1(("getLocalConnectableMode"));
         //getLocalConnectableMode
         if(_bluetoothApp)
         {
            SwitchStatus status;
            (void)_bluetoothApp->getBmCoreRequestIf().getLocalConnectableMode(status);
            ETG_TRACE_USR1((" getLocalConnectableMode state: %d reason: %d", status._switchState, status._switchedOffReason));
         }
      }
      break;

      case 0x0010: //BM_CORE_CMD_SWITCH_DISCOVERY_STATUS
      {
         ETG_TRACE_USR1(("switchDiscoveryStatus"));
         //switchDiscoveryStatus
         if(_bluetoothApp)
         {
            TargetSwitchState state((TargetSwitchState) cmd.getU32Value(&data[0]));
            (void)_bluetoothApp->getBmCoreRequestIf().switchDiscoveryStatus(state);
         }
      }
      break;

      case 0x0011: //BM_CORE_CMD_GET_DISCOVERY_STATUS
      {
         ETG_TRACE_USR1(("getDiscoveryStatus"));
         //getDiscoveryStatus
         if(_bluetoothApp)
         {
            SwitchStatus status;
            (void)_bluetoothApp->getBmCoreRequestIf().getDiscoveryStatus(status);
            ETG_TRACE_USR1((" getDiscoveryStatus state: %d reason: %d", status._switchState, status._switchedOffReason));
         }
      }
      break;

      case 0x0012: //BM_CORE_CMD_SET_BT_LOCAL_FRIENDLY_NAME
      {
         ETG_TRACE_USR1(("setBtLocalFriendlyName"));
         //setBtLocalFriendlyName
         if(_bluetoothApp)
         {
            const ::std::string name(cmd.getStringValue(data, 35));
            (void)_bluetoothApp->getBmCoreRequestIf().setBtLocalFriendlyName(name);
         }
      }
      break;

      case 0x0013: //BM_CORE_CMD_GET_BT_LOCAL_FRIENDLY_NAME
      {
         ETG_TRACE_USR1(("getBtLocalFriendlyName"));
         //getBtLocalFriendlyName
         if(_bluetoothApp)
         {
            LocalFriendlyName name;
            (void)_bluetoothApp->getBmCoreRequestIf().getBtLocalFriendlyName(name);
            ETG_TRACE_USR1((" getBtLocalFriendlyName name: %s", name._localFriendlyName.c_str()));
         }
      }
      break;

      case 0x0014: //BM_CORE_CMD_GET_BT_LOCAL_INFO
      {
         ETG_TRACE_USR1(("getBtLocalInfo"));
         //getBtLocalInfo
         if(_bluetoothApp)
         {
            LocalInfo info;
            (void)_bluetoothApp->getBmCoreRequestIf().getBtLocalInfo(info);
            ETG_TRACE_USR1((" getBtLocalInfo BtModuleName: %s", info._moduleName.c_str()));
            ETG_TRACE_USR1((" getBtLocalInfo BdAddress: %s", info._bdAddress.c_str()));
            ETG_TRACE_USR1((" getBtLocalInfo LocalFriendlyNameType: %s", info._bdName.c_str()));
            for(ProtocolInfoMap::const_iterator it = info._protocolInfos.begin(); it != info._protocolInfos.end(); ++it)
            {
                ETG_TRACE_USR1((" getBtLocalInfo ProtocolInfoMap: Id: %d _name: %s  ", it->first, it->second._name.c_str()));
                ETG_TRACE_USR1((" getBtLocalInfo ProtocolInfoMap: _version:%d", it->second._version));
                ETG_TRACE_USR1((" getBtLocalInfo ProtocolInfoMap: _isSupported: %d", it->second._isSupported));
            }

         }
      }
      break;

      case 0x0015: //BM_CORE_CMD_START_PAIRING
      {
         ETG_TRACE_USR1(("startPairing"));
         //startPairing
         if(_bluetoothApp)
         {
            const ::std::string addr(cmd.getStringValue(data, 13));
            (void)_bluetoothApp->getBmCoreRequestIf().startPairing(addr);
         }
      }
      break;

      case 0x0016: //BM_CORE_CMD_CANCEL_PAIRING
      {
         ETG_TRACE_USR1(("cancelPairing"));
         //cancelPairing
         if(_bluetoothApp)
         {
        	 (void)_bluetoothApp->getBmCoreRequestIf().cancelPairing();
         }
      }
      break;

      case 0x0017: //BM_CORE_CMD_ACCEPT_PAIRING_REQUEST
      {
         ETG_TRACE_USR1(("acceptPairingRequest"));
         //acceptPairingRequest
         if(_bluetoothApp)
         {
        	 (void)_bluetoothApp->getBmCoreRequestIf().acceptPairingRequest();
         }
      }
      break;

      case 0x0018: //BM_CORE_CMD_REJECT_PAIRING_REQUEST
      {
         ETG_TRACE_USR1(("rejectPairingRequest"));
         //rejectPairingRequest
         if(_bluetoothApp)
         {
        	 (void)_bluetoothApp->getBmCoreRequestIf().rejectPairingRequest();
         }
      }
      break;

      case 0x0019: //BM_CORE_CMD_GET_PAIRING_STATUS
      {
         ETG_TRACE_USR1(("getPairingStatus"));
         //getPairingStatus
         if(_bluetoothApp)
         {
            PairingStatus state;
            (void)_bluetoothApp->getBmCoreRequestIf().getPairingStatus(state);
            ETG_TRACE_USR1((" getPairingStatus state: _state: %d _remoteBdAddress: %s ", state._state, state._remoteBdAddress.c_str()));
            ETG_TRACE_USR1((" getPairingStatus state: _remoteBdName: %s ", state._remoteBdName.c_str()));
            ETG_TRACE_USR1((" getPairingStatus state: _pairingType: %d ", state._pairingType));
            ETG_TRACE_USR1((" getPairingStatus state: _pin: %s ", state._pin.c_str()));
            ETG_TRACE_USR1((" getPairingStatus state: _origin: %d _userConfirmationRequired: %d ", state._origin, state._userConfirmationRequired));
            ETG_TRACE_USR1((" getPairingStatus state: _result: %d ", state._result));
         }
      }
      break;

      case 0x001A: //BM_CORE_CMD_SET_PAIRING_PIN
      {
         ETG_TRACE_USR1(("setPairingPin"));
         //setPairingPin
         if(_bluetoothApp)
         {
            const ::std::string pin(cmd.getStringValue(data, 35));
            PairingPin setPin(pin);
            (void)_bluetoothApp->getBmCoreRequestIf().setPairingPin(setPin);
         }
      }
      break;

      case 0x001B: //BM_CORE_CMD_GET_PAIRING_PIN
      {
         ETG_TRACE_USR1(("getPairingPin"));
         //getPairingPin
         if(_bluetoothApp)
         {
            PairingPin getPin;
            (void)_bluetoothApp->getBmCoreRequestIf().getPairingPin(getPin);
            ETG_TRACE_USR1(("getPairingPin is: %s", getPin._pin.c_str()));
         }
      }
      break;

      case 0x001C: //BM_CORE_CMD_GET_PAIRED_DEVICE_LIST
      {
         ETG_TRACE_USR1(("getPairedDeviceList"));
         //getPairedDeviceList
         if(_bluetoothApp)
         {
            PairedDeviceList pairedList;
            (void)_bluetoothApp->getBmCoreRequestIf().getPairedDeviceList(pairedList);
/*            for(DeviceBaseInfoList::const_iterator it = pairedList.begin(); it != pairedList.end(); ++it)
            {
               ETG_TRACE_USR1((" getPairedDeviceList - "));
               ETG_TRACE_USR1((" getPairedDeviceList %d", it->_deviceHandle));
            }*/
            ETG_TRACE_USR1((" getPairedDeviceList - "));
            for(DeviceBaseInfoList::iterator iter = pairedList._deviceBaseInfoList.begin(); iter != pairedList._deviceBaseInfoList.end(); iter++)
            {

               ETG_TRACE_USR1((" getPairedDeviceList DeviceId: %d BdAddress: %s", iter->_deviceHandle, iter->_bdAddress.c_str()));
               ETG_TRACE_USR1((" getPairedDeviceList BTSMajorDeviceClass: %d BdName: %s", iter->_majorDeviceClass, iter->_bdName.c_str()));
               ETG_TRACE_USR1((" getPairedDeviceList _specificationId: %d _vendorId: %d", iter->_deviceIdentification._specificationId, iter->_deviceIdentification._vendorId));
               ETG_TRACE_USR1((" getPairedDeviceList _vendorIdSource: %d _productId: %d", iter->_deviceIdentification._vendorIdSource, iter->_deviceIdentification._productId));
               ETG_TRACE_USR1((" getPairedDeviceList _version: %d", iter->_deviceIdentification._version));
               //TODO: ProtocolInfoMap
               ETG_TRACE_USR1((" ProtocolInfoMap - "));
               for(ProtocolInfoMap::const_iterator it = iter->_protocolInfo.begin();it != iter->_protocolInfo.end();++it)
               {
                  ETG_TRACE_USR1((" getBtLocalInfo ProtocolInfoMap: Id: %d _name: %s  ", it->first, it->second._name.c_str()));
                  ETG_TRACE_USR1((" getBtLocalInfo ProtocolInfoMap: _version:%d", it->second._version));
                  ETG_TRACE_USR1((" getBtLocalInfo ProtocolInfoMap: _isSupported: %d", it->second._isSupported));
               }

               ETG_TRACE_USR1((" getPairedDeviceList InbandRingtoneSupport: %d ConnectionStatus: %d", iter->_inbandRingtoneSupport, iter->_connectionStatus));
               ETG_TRACE_USR1((" getPairedDeviceList DisconnectedReason: %d", iter->_disconnectedReason));
               ETG_TRACE_USR1((" getPairedDeviceList ConnectionOrderInfo: _index: %d _seconds: %d", iter->_connectionOrderInfo._index, iter->_connectionOrderInfo._seconds));
            }
         }
      }
      break;

      case 0x001D: //BM_CORE_CMD_DELETE_DEVICE
      {
         ETG_TRACE_USR1(("deleteDevice"));
         //deleteDevice
         if(_bluetoothApp)
         {
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to delete device: %s", addr.c_str()));
               break;
            }
            (void)_bluetoothApp->getBmCoreRequestIf().deleteDevice(id, BM_DEVICE_HANDLE_TYPE_SINGLE);
         }

      }
      break;

      case 0x001E: //BM_CORE_CMD_CONNECT_DEVICE
      {
         ETG_TRACE_USR1(("connectDevice"));
         //connectDevice
         if(_bluetoothApp)
         {
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to connect device: %s", addr.c_str()));
               break;
            }
            (void)_bluetoothApp->getBmCoreRequestIf().connectDevice(id);
         }
      }
      break;

      case 0x001F: //BM_CORE_CMD_DISCONNECT_DEVICE
      {
         ETG_TRACE_USR1(("disconnectDevice"));
         //disconnectDevice
         if(_bluetoothApp)
         {
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to disconnect device: %s", addr.c_str()));
               break;
            }
            (void)_bluetoothApp->getBmCoreRequestIf().disconnectDevice(id, BM_DEVICE_HANDLE_TYPE_SINGLE);
         }
      }
      break;

      case 0x0020: //BM_CORE_CMD_CONNECT_PROFILE
      {
         ETG_TRACE_USR1(("connectProfiles"));
         //connectProfiles
         if(_bluetoothApp)
         {
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to connect profile for: %s", addr.c_str()));
               break;
            }

            const unsigned int proto = cmd.getU32Value(&data[13]);
            const ::std::string uuid(cmd.getStringValue(&data[17], 17));

            ProtocolList list;
            Protocol newProto((ProtocolId) proto,uuid);
            list.push_back(newProto);

            (void)_bluetoothApp->getBmCoreRequestIf().connectProfiles((DeviceId) id, list);
         }
      }
      break;

      case 0x0021: //BM_CORE_CMD_DISCONNECT_PROFILE
      {
         ETG_TRACE_USR1(("disconnectProfiles"));
         //disconnectProfiles
         if(_bluetoothApp)
         {
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to disconnect profile for: %s", addr.c_str()));
               break;
            }

            const unsigned int proto = cmd.getU32Value(&data[13]);

            ProtocolList list;
            Protocol newProto((ProtocolId) proto,"");
            list.push_back(newProto);

            (void)_bluetoothApp->getBmCoreRequestIf().disconnectProfiles((DeviceId) id, list);
         }
      }
      break;

      case 0x0022: //BM_CORE_CMD_STORE_DEVICE_RINGTONE_ID_IN_DB
      {
         ETG_TRACE_USR1(("storeDeviceRingtoneIdInDb"));
         //storeDeviceRingtoneIdInDb
      }
      break;

      case 0x0023: //BM_CORE_CMD_READ_DEVICE_RINGTONE_ID_FROM_DB
      {
         ETG_TRACE_USR1(("readDeviceRingtoneIdFromDb"));
         //readDeviceRingtoneIdFromDb
      }
      break;

      case 0x0024: //BM_CORE_CMD_STORE_SYSTEM_WIDE_RINGTONE_ID_IN_DB
      {
         ETG_TRACE_USR1(("storeSystemWideRingtoneIdInDb"));
         //storeSystemWideRingtoneIdInDb
      }
      break;

      case 0x0025: //BM_CORE_CMD_READ_SYSTEM_WIDE_RINGTONE_ID_FROM_DB
      {
         ETG_TRACE_USR1(("readSystemWideRingtoneIdFromDb"));
         //readSystemWideRingtoneIdFromDb
      }
      break;

      case 0x0026: //BM_CORE_CMD_GET_REMOTE_DEVICE_INFO
      {
         ETG_TRACE_USR1(("getRemoteDeviceInfo"));
         //getRemoteDeviceInfo
         if(_bluetoothApp)
         {
            RemoteDeviceInfo info;
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to disconnect profile for: %s", addr.c_str()));
               break;
            }

            (void)_bluetoothApp->getBmCoreRequestIf().getRemoteDeviceInfo(info, id);
            ETG_TRACE_USR1((" getRemoteDeviceInfo DeviceId: %d BdAddress: %s", info._deviceBaseInfo._deviceHandle, info._deviceBaseInfo._bdAddress.c_str()));
            ETG_TRACE_USR1((" getRemoteDeviceInfo BTSMajorDeviceClass: %d _bdName: %s", info._deviceBaseInfo._majorDeviceClass, info._deviceBaseInfo._bdName.c_str()));
            ETG_TRACE_USR1((" getRemoteDeviceInfo DeviceId: %d BdAddress: %s", info._deviceBaseInfo._deviceHandle, info._deviceBaseInfo._bdAddress.c_str()));
            ETG_TRACE_USR1((" getRemoteDeviceInfo DeviceId: %d BdAddress: %s", info._deviceBaseInfo._deviceHandle, info._deviceBaseInfo._bdAddress.c_str()));
            ETG_TRACE_USR1((" getRemoteDeviceInfo DeviceId: %d BdAddress: %s", info._deviceBaseInfo._deviceHandle, info._deviceBaseInfo._bdAddress.c_str()));
            ETG_TRACE_USR1((" getRemoteDeviceInfo DeviceId: %d BdAddress: %s", info._deviceBaseInfo._deviceHandle, info._deviceBaseInfo._bdAddress.c_str()));
            ETG_TRACE_USR1((" getRemoteDeviceInfo _linkKey: %s", info._linkKey.c_str()));
         }
      }
      break;

      case 0x0027: //BM_CORE_CMD_GET_DEVICE_CONNECTION_STATUS_LIST
      {
         ETG_TRACE_USR1(("getDeviceConnectionStatusList"));
         DeviceConnectionStatusList deviceConnectionStatusList;
         //getDeviceConnectionStatusList
         if(_bluetoothApp)
         {
        	 (void)_bluetoothApp->getBmCoreRequestIf().getDeviceConnectionStatusList(deviceConnectionStatusList);

            for(DeviceConnectionInfoList::iterator iter = deviceConnectionStatusList._deviceConnectionInfoList.begin(); iter != deviceConnectionStatusList._deviceConnectionInfoList.end(); iter++)
            {


               ETG_TRACE_USR1((" getDeviceConnectionStatusList DeviceId: %d ConnectionStatus: %d DisconnectedReason: %d ConnectionOrderInfo: _index: %d _seconds: %d ", iter->_deviceHandle,iter->_connectionStatus,iter->_disconnectedReason,iter->_connectionOrderInfo._index, iter->_connectionOrderInfo._seconds));
               ETG_TRACE_USR1((" ProtocolConnectionInfoMap - "));
               for(ProtocolConnectionInfoMap::const_iterator it = iter->_protocolConnectionInfo.begin();it != iter->_protocolConnectionInfo.end();++it)
               {
                  ETG_TRACE_USR1((" getDeviceConnectionStatusList ProtocolConnectionInfoMap: Id: %d _connectionstatus: %d _disconnectedReson: %d ", it->first, it->second._connectionStatus, it->second._disconnectedReason));
                  ETG_TRACE_USR1((" getDeviceConnectionStatusList ProtocolConnectionInfoMap: rfcommDevicePath: %s ", it->second._rfcommDevicePath.c_str()));
               }
            }

         }
      }
      break;

      case 0x0028: //BM_CORE_CMD_BLOCK_DEVICE
      {
         ETG_TRACE_USR1(("blockDevice"));
         //blockDevice
         if(_bluetoothApp)
         {
            RemoteDeviceInfo info;
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to blockDevice for: %s", addr.c_str()));
               break;
            }

            (void)_bluetoothApp->getBmCoreRequestIf().blockDevice(id, BM_DEVICE_HANDLE_TYPE_SINGLE);
         }
      }
      break;

      case 0x0029: //BM_CORE_CMD_UNBLOCK_DEVICE
      {
         ETG_TRACE_USR1(("unblockDevice"));
         //unblockDevice
         if(_bluetoothApp)
         {
            RemoteDeviceInfo info;
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to blockDevice for: %s", addr.c_str()));
               break;
            }

            (void)_bluetoothApp->getBmCoreRequestIf().unblockDevice(id, BM_DEVICE_HANDLE_TYPE_SINGLE);
         }
      }
      break;

      case 0x002A: //BM_CORE_CMD_BLOCK_PROFILE
      {
         ETG_TRACE_USR1(("blockProfiles"));
         //blockProfiles
         if(_bluetoothApp)
         {
            const ::std::string name(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(name);
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to blockProfiles for: %s", name.c_str()));
               break;
            }

            const unsigned int proto = cmd.getU32Value(&data[13]);

            ProtocolList list;
            Protocol newProto((ProtocolId) proto,"");
            list.push_back(newProto);

            (void)_bluetoothApp->getBmCoreRequestIf().blockProfiles((DeviceId) id, list);
         }
      }
      break;

      case 0x002B: //BM_CORE_CMD_UNBLOCK_PROFILES
      {
         ETG_TRACE_USR1(("unblockProfiles"));
         //unblockProfiles
         if(_bluetoothApp)
         {
            const ::std::string name(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(name);
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to unblockProfiles for: %s", name.c_str()));
               break;
            }

            const unsigned int proto = cmd.getU32Value(&data[13]);

            ProtocolList list;
            Protocol newProto((ProtocolId) proto,"");
            list.push_back(newProto);

            (void)_bluetoothApp->getBmCoreRequestIf().unblockProfiles((DeviceId) id, list);
         }
      }
      break;

      case 0x002C: //BM_CORE_CMD_GET_BLOCK_STATUS
      {
         ETG_TRACE_USR1(("getBlockStatus"));
         //getBlockStatus
         if(_bluetoothApp)
         {
            BlockStatus state;
            (void)_bluetoothApp->getBmCoreRequestIf().getBlockStatus(state);
            ETG_TRACE_USR1((" getRemoteDeviceInfo _blockState: %d", state._overallDeviceBlockStatus._blockState));
         }
      }
      break;

      case 0x002D: //BM_CORE_CMD_SET_AUTO_CONNECTION_TYPE
      {
         ETG_TRACE_USR1(("setAutoConnectionType"));
         //setAutoConnectionType
         if(_bluetoothApp)
         {
            AutoConnectionType type((StandardAutoConnectionType) cmd.getU32Value(&data[0]));
            (void)_bluetoothApp->getBmCoreRequestIf().setAutoConnectionType(type);
         }
      }
      break;

      case 0x002E: //BM_CORE_CMD_GET_AUTO_CONNECTION_TYPE
      {
         ETG_TRACE_USR1(("getAutoConnectionType"));
         //getAutoConnectionType
         if(_bluetoothApp)
         {
            AutoConnectionType type;;
            (void)_bluetoothApp->getBmCoreRequestIf().setAutoConnectionType(type);
            ETG_TRACE_USR1((" getRemoteDeviceInfo AutoConnectionType: %d", type._type));
         }
      }
      break;

      case 0x002F: //BM_CORE_CMD_START_AUTO_CONNECT_SEQUENCE
      {
         ETG_TRACE_USR1(("startAutoConnectSequence"));
         //startAutoConnectSequence
         if(_bluetoothApp)
         {
        	 (void)_bluetoothApp->getBmCoreRequestIf().startAutoConnectSequence(BM_TEMPORARY_AUTO_CONNECTION_DEFAULT);
         }
      }
      break;

      case 0x0030: //BM_CORE_CMD_STOP_AUTO_CONNECT_SEQUENCE
      {
         ETG_TRACE_USR1(("stopAutoConnectSequence"));
         //stopAutoConnectSequence
         if(_bluetoothApp)
         {
        	 (void)_bluetoothApp->getBmCoreRequestIf().stopAutoConnectSequence();
         }
      }
      break;

      case 0x0031: //BM_CORE_CMD_GET_BM_CONFIG_DATA
      {
         ETG_TRACE_USR1(("getBmConfigData"));
         //getBmConfigData
         if(_bluetoothApp)
         {
            BmConfigData config;
            (void)_bluetoothApp->getBmCoreRequestIf().getBmConfigData(config);
            ETG_TRACE_USR1(("getBmConfigData _maxNumPairedDevices: %d", config._maxNumPairedDevices));
         }
      }
      break;

      case 0x0032: //BM_CORE_CMD_UPDATE_LINK_QUALITY
      {
         ETG_TRACE_USR1(("updateLinkQuality"));
         //updateLinkQuality
         if(_bluetoothApp)
         {
            _bluetoothApp->getBmCoreRequestIf().updateLinkQuality();
         }
      }
      break;

      case 0x0033: //BM_CORE_CMD_GET_LINK_QUALITY
      {
         ETG_TRACE_USR1(("getLinkQuality"));
         //getLinkQuality
         if(_bluetoothApp)
         {
            LinkQuality quality;
            (void)_bluetoothApp->getBmCoreRequestIf().getLinkQuality(quality);
            //TODO
            //ETG_TRACE_USR1(("getLinkQuality _maxNumPairedDevices: %d", quality._linkQualityInfoList));
         }
      }
      break;

      case 0x0034: //BM_CORE_CMD_SET_BT_LIMITATION_MODE
      {
         ETG_TRACE_USR1(("setBtLimitationMode"));

         if(_bluetoothApp)
         {
            const BdAddress bdAddress(cmd.getStringValue(data, 13));
            const BdName bdName(cmd.getStringValue(&data[13], 20));
            const LimitationCommunicationIf communicationIf(static_cast<LimitationCommunicationIf>(cmd.getU32Value(&data[33])));
            const LimitationFeature feature(static_cast<LimitationFeature>(cmd.getU32Value(&data[37])));
            const LimitationAction action(static_cast<LimitationAction>(cmd.getU32Value(&data[41])));

            _bluetoothApp->getBmCoreRequestIf().setBtLimitationMode(bdAddress, bdName, LimitationMode(feature, communicationIf), action);
         }
      }
      break;

      case 0x0035: //BM_CORE_CMD_GET_BT_LIMITATION_MODE
      {
         ETG_TRACE_USR1(("getBtLimitationMode"));
         //getBtLimitationMode
         if(_bluetoothApp)
         {
            BtLimitationMode mode;
            (void)_bluetoothApp->getBmCoreRequestIf().getBtLimitationMode(mode);

            //TODO:
            //ETG_TRACE_USR1(("getBtLimitationMode ", mode._btLimitationModeInfoList));
         }
      }
      break;

      case 0x0036: //BM_CORE_CMD_RESET_TO_DEFAULT
      {
         ETG_TRACE_USR1(("resetToDefault"));
         //resetToDefault
         if(_bluetoothApp)
         {
        	 (void)_bluetoothApp->getBmCoreRequestIf().resetToDefault(false);
         }
      }
      break;

      case 0x0037: //BM_CORE_CMD_GET_RESET_TO_DEFAULT_STATUS
      {
         ETG_TRACE_USR1(("getResetToDefaultStatus"));
         //getResetToDefaultStatus
         if(_bluetoothApp)
         {
            ResetToDefaultStatus status;
            (void)_bluetoothApp->getBmCoreRequestIf().getResetToDefaultStatus(status);
            ETG_TRACE_USR1(("getResetToDefaultStatus _resetResult: %d _resetState: %d", status._resetResult, status._resetState ));
         }
      }
      break;

      case 0x0038: //BM_CORE_CMD_SWITCH_TEST_MODE
      {
         ETG_TRACE_USR1(("switchTestMode"));
         //switchTestMode
         if(_bluetoothApp)
         {
            TargetSwitchState state = TARGET_SWITCH_STATE_SWITCHED_ON;
            const ::std::string addr(cmd.getStringValue(data, 13));
            (void)_bluetoothApp->getBmCoreRequestIf().switchTestMode(state, addr);
         }
      }
      break;

      case 0x0039: //BM_CORE_CMD_GET_TEST_MODE
      {
         ETG_TRACE_USR1(("getTestMode"));
         //getTestMode
         if(_bluetoothApp)
         {
            SwitchStatus state;
            (void)_bluetoothApp->getBmCoreRequestIf().getTestMode(state);
            ETG_TRACE_USR1(("getTestMode state: SwitchState: %d SwitchedOffReason: %d", state._switchState, state._switchedOffReason));
         }
      }
      break;

      case 0x003A: //BM_CORE_CMD_REQUEST_TEST_MODE_LINK_QUALITY
      {
         ETG_TRACE_USR1(("requestTestModeLinkQuality"));
         //requestTestModeLinkQuality
         if(_bluetoothApp)
         {
        	 (void)_bluetoothApp->getBmCoreRequestIf().requestTestModeLinkQuality();
         }
      }
      break;

      case 0x003B: //BM_CORE_CMD_GET_LINK_QUALITY_TEST_MODE
      {
         ETG_TRACE_USR1(("getLinkQualityTestMode"));
         //getLinkQualityTestMode
         if(_bluetoothApp)
         {
            TestModeLinkQuality quality;
            (void)_bluetoothApp->getBmCoreRequestIf().getLinkQualityTestMode(quality);
            ETG_TRACE_USR1(("getLinkQualityTestMode LinkQualityType: %d rssi: %d", quality._linkQualityInfo._linkQuality, quality._linkQualityInfo._rssi));
         }
      }
      break;

      case 0x0042: //BM_CORE_CMD_ADD_OOB_PAIRED_DEVICE
      {
         ETG_TRACE_USR1(("addOobPairedDevice"));

         if(_bluetoothApp)
         {
            const BdAddress bdAddress(cmd.getStringValue(data, 13u));
            const BdName bdName(cmd.getStringValue(&data[13], 20));
            const LinkKey linkKey(cmd.getStringValue(&data[33], 33));
            const OobType oobType(static_cast<OobType>(cmd.getU8Value(&data[66])));
            const ActType act(cmd.getU8Value(&data[67]));

            (void)_bluetoothApp->getBmCoreRequestIf().addOobPairedDevice(bdAddress, bdName, linkKey, oobType, act);
         }
      }
      break;

      case 0x003C: //BM_CORE_CMD_DELETE_DEVICES
      {
         ETG_TRACE_USR1(("deleteDevices"));

         if(_bluetoothApp)
         {
            PairedDeviceList pairedList;
            (void)_bluetoothApp->getBmCoreRequestIf().getPairedDeviceList(pairedList);
            for(DeviceBaseInfoList::iterator iter = pairedList._deviceBaseInfoList.begin(); iter != pairedList._deviceBaseInfoList.end(); iter++)
            {
            	(void)_bluetoothApp->getBmCoreRequestIf().deleteDevice(iter->_deviceHandle, BM_DEVICE_HANDLE_TYPE_SINGLE);
            }
         }
      }
      break;

      case 0x003D: //BM_CORE_CMD_DISCONNECT_DEVICES
      {
         ETG_TRACE_USR1(("disconnectDevices"));
         //disconnectDevice
         if(_bluetoothApp)
         {

            PairedDeviceList pairedList;
            (void)_bluetoothApp->getBmCoreRequestIf().getPairedDeviceList(pairedList);
            for(DeviceBaseInfoList::iterator iter = pairedList._deviceBaseInfoList.begin(); iter != pairedList._deviceBaseInfoList.end(); iter++)
            {
            	(void)_bluetoothApp->getBmCoreRequestIf().disconnectDevice(iter->_deviceHandle, BM_DEVICE_HANDLE_TYPE_SINGLE);
            }
         }
      }
      break;


      case 0x0040: //BM_CORE_CMD_BLOCK_DEVICES
      {
         ETG_TRACE_USR1(("blockDevices"));
         //blockDevices
         if(_bluetoothApp)
         {
            PairedDeviceList pairedList;
            (void)_bluetoothApp->getBmCoreRequestIf().getPairedDeviceList(pairedList);
            for(DeviceBaseInfoList::iterator iter = pairedList._deviceBaseInfoList.begin(); iter != pairedList._deviceBaseInfoList.end(); iter++)
            {
            	(void)_bluetoothApp->getBmCoreRequestIf().blockDevice(iter->_deviceHandle, BM_DEVICE_HANDLE_TYPE_SINGLE);
            }
         }
      }
      break;
      case 0x0041: //BM_CORE_CMD_UNBLOCK_DEVICES
      {
         ETG_TRACE_USR1(("blockDevices"));

         if(_bluetoothApp)
         {
            PairedDeviceList pairedList;
            (void)_bluetoothApp->getBmCoreRequestIf().getPairedDeviceList(pairedList);
            for(DeviceBaseInfoList::iterator iter = pairedList._deviceBaseInfoList.begin(); iter != pairedList._deviceBaseInfoList.end(); iter++)
            {
            	(void)_bluetoothApp->getBmCoreRequestIf().unblockDevice(iter->_deviceHandle, BM_DEVICE_HANDLE_TYPE_SINGLE);
            }
         }
      }
      break;
      case 0x0043: //BM_CORE_CMD_SET_PRIMARY_HFP_DEVICE
      {
         ETG_TRACE_USR1(("setprimaryhfpdevice"));

         if(_bluetoothApp)
         {
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to setprimaryHFPdevice for: %s", addr.c_str()));
               break;
            }

            (void)_bluetoothApp->getBmCoreRequestIf().setPrimaryHfpDevice(id);
         }
      }
      break;
      case 0x0044: //BM_CORE_CMD_SET_FAVORITE
      {
         ETG_TRACE_USR1(("setFAVORITE"));

         if(_bluetoothApp)
         {
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);
            const FavoriteType favoriteType(static_cast<FavoriteType>(cmd.getU8Value(&data[13])));
            const bmcore::FavoriteIndex favoriteIndex(cmd.getU8Value(&data[14]));
            const ActType act(cmd.getU8Value(&data[15]));
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to setfavorite for: %s", addr.c_str()));
               break;
            }

            (void)_bluetoothApp->getBmCoreRequestIf().setFavorite(id,favoriteType,favoriteIndex,act);
         }
      }
      break;
      case 0x0045: //BM_CORE_CMD_CLEAR_FAVORITE
      {
         ETG_TRACE_USR1(("clearFavorite"));

         if(_bluetoothApp)
         {
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);
            const FavoriteType favoriteType(static_cast<FavoriteType>(cmd.getU8Value(&data[13])));

            const ActType act(cmd.getU8Value(&data[14]));
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to Clearfavorite for: %s", addr.c_str()));
               break;
            }

            (void)_bluetoothApp->getBmCoreRequestIf().clearFavorite(id,favoriteType,act);
         }
      }
      break;
      case 0x0046: //BM_CORE_CMD_START_SERVICE_SEARCH
      {
         ETG_TRACE_USR1(("StartServiceSearch"));

         if(_bluetoothApp)
         {
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);

            const ActType act(cmd.getU8Value(&data[13]));
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to StartServiceSearch for: %s", addr.c_str()));
               break;
            }

            (void)_bluetoothApp->getBmCoreRequestIf().startServiceSearch(id,act);
         }
      }
      break;
      case 0x0047: //BM_CORE_CMD_STOP_SERVICE_SEARCH
      {
         ETG_TRACE_USR1(("StopServiceSearch"));

         if(_bluetoothApp)
         {
            const ::std::string addr(cmd.getStringValue(data, 13));
            unsigned int id = getbdAddress2DeviceHandle(addr);

            const ActType act(cmd.getU8Value(&data[13]));
            if(id == 0)
            {
               ETG_TRACE_USR1(("not possible to StopServiceSearch for: %s", addr.c_str()));
               break;
            }

            (void)_bluetoothApp->getBmCoreRequestIf().stopServiceSearch(id,act);
         }
      }
      break;
      case 0x0048: //BM_CORE_CMD_SWITCH_MULTI_HFP_SUPPORT
      {
         ETG_TRACE_USR1(("switchMultiHFPSupport"));

         //switchMultiHFPSupport
         if(_bluetoothApp)
         {
            TargetSwitchState state((TargetSwitchState) cmd.getU32Value(&data[0]));
            ETG_TRACE_USR1((" switchMultiHFPSupport to: %d", cmd.getU32Value(&data[0])));

            (void)_bluetoothApp->getBmCoreRequestIf().switchMultiHFPSupport(state);
         }
      }
      break;
      case 0x0049: //BM_CORE_CMD_SET_PROFILE_USAGE
      {
         ETG_TRACE_USR1(("SetBTProfileUsage"));

         //SetBTProfileUsage
         if(_bluetoothApp)
         {
            _bluetoothApp->getBmCoreRequestIf().setBtProfileUsageInfo(_profileUsageList);

            _profileUsageList._btProfileUsageInfoList.clear();
         }
      }
      break;
      case 0x004A: //BM_CORE_CMD_SET_PROFILE_USAGE_ITEM
      {
         ETG_TRACE_USR1(("setBTProfileItem"));

         DeviceId deviceID (cmd.getU32Value(&data[0]));
         ProtocolId protocolID (static_cast<ProtocolId>(cmd.getU32Value(&data[4])));
         Uuid uuid(cmd.getStringValue(&data[8], 33));
         BTProfileUsageType profileUsageType (static_cast<BTProfileUsageType>(cmd.getU32Value(&data[42])));
         bool inUse (cmd.getU32Value(&data[46]));

         ETG_TRACE_USR4((" setBTProfileItem DeviceID: %d", deviceID));
         ETG_TRACE_USR4((" setBTProfileItem ProtocolID: %d", protocolID));
         ETG_TRACE_USR4((" setBTProfileItem uuid: %s", uuid));
         ETG_TRACE_USR4((" setBTProfileItem profileUsageType: %d, inUse %d", profileUsageType, inUse));

         Protocol protocol(protocolID, uuid);
         ProfileUsageInfo usageInfo(deviceID, protocol, profileUsageType, inUse);

         _profileUsageList._btProfileUsageInfoList.push_back(usageInfo);
      }
      break;
      case 0x004B: //BM_CORE_CMD_NOTIFY_USERDECISION
      {
         ETG_TRACE_USR1(("NotifyUserDecision"));

         ConflictTrigger conflictTrigger(static_cast<ConflictTrigger>(cmd.getU32Value(&data[0])));
         UserDecision decision(static_cast<UserDecision>(cmd.getU32Value(&data[4])));
         UserDecisionInfo userDecision(conflictTrigger, decision);

         if(_bluetoothApp)
         {
        	 _bluetoothApp->getBmCoreRequestIf().notifyUserDecision(userDecision);
         }
      }
      break;
      /*case 0x004C: //BM_CORE_CMD_RESOLVE_CONFLICTS
      {
         ETG_TRACE_USR1(("ResolveConflicts"));

         ConflictTrigger conflictTrigger(static_cast<ConflictTrigger>(cmd.getU32Value(&data[0])));
         DeviceId deviceID (cmd.getU32Value(&data[4]));

         //ConflictManager::resolveConflicts(conflictTrigger, deviceID);
      }
      break;*/
      case 0x004C: //BM_CORE_CMD_SET_PROTOCOL_USER_AVAILABILITY
      {
         ETG_TRACE_USR1(("setProtocolUserAvailability"));

         Protocol protocol(static_cast<ProtocolId>(cmd.getU32Value(&data[0])),
               static_cast<Uuid>(cmd.getStringValue(&data[4], 17)));
         bool available(static_cast<bool>(cmd.getU8Value(&data[21])));

         if(_bluetoothApp)
         {
            _bluetoothApp->getBmCoreRequestIf().setProtocolUserAvailability(protocol, available);
         }
      }
      break;

      case 0x004D: //BM_CORE_CMD_REPLACE_BT_LIMITATION_MODE
      {
         ETG_TRACE_USR1(("replaceBtLimitationMode"));

         if(_bluetoothApp)
         {
            const BdAddress bdAddress(cmd.getStringValue(data, 13));
            const BdName bdName(cmd.getStringValue(&data[13], 20));
            const LimitationCommunicationIf communicationIf(static_cast<LimitationCommunicationIf>(cmd.getU32Value(&data[33])));
            const LimitationFeature feature(static_cast<LimitationFeature>(cmd.getU32Value(&data[37])));
            const LimitationState state(static_cast<LimitationState>(cmd.getU32Value(&data[41])));

            const BtLimitationModeInfo currLimitationMode(bdAddress, bdName, LimitationMode(feature, communicationIf), state);

//            const BdAddress newbdAddress(cmd.getStringValue(&data[41], 13));
//            const BdName newbdName(cmd.getStringValue(&data[54], 20));
//            const LimitationCommunicationIf newCommunicationIf(static_cast<LimitationCommunicationIf>(cmd.getU32Value(&data[74])));
//            const LimitationFeature newFeature(static_cast<LimitationFeature>(cmd.getU32Value(&data[78])));
//            const LimitationAction newAction(static_cast<LimitationAction>(cmd.getU32Value(&data[82])));

            /*const BdAddress bdAddress("aaabbbcccddd");
             const LimitationCommunicationIf communicationIf(BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4);
             const LimitationFeature feature(BM_LIMITATION_FEATURE_CAR_PLAY);
             const LimitationState state(BM_LIMITATION_STATE_PREPARING);

             const BtLimitationModeInfo currLimitationMode(bdAddress, LimitationMode(feature, communicationIf), state);*/

             const BdAddress newbdAddress("aaabbbcccddd");
             const BdName newbdName("");
             const LimitationCommunicationIf newCommunicationIf(BM_LIMITATION_COMMUNICATION_IF_USB);
             const LimitationFeature newFeature(BM_LIMITATION_FEATURE_CAR_PLAY);
             const LimitationAction newAction(BM_LIMITATION_ACTION_ACTIVATE);

            _bluetoothApp->getBmCoreRequestIf().replaceBtLimitationMode(currLimitationMode, newbdAddress,
                  newbdName, LimitationMode(newFeature, newCommunicationIf), newAction);


         }
      }
      break;
      default:
      {
         break;
      }
   }
}

unsigned int FC_Bluetooth_TestCommand::getbdAddress2DeviceHandle(::std::string addr)
{
   unsigned int id = 0;
   if(_bluetoothApp)
   {
      PairedDeviceList pairedList;
      _bluetoothApp->getBmCoreRequestIf().getPairedDeviceList(pairedList);
      for(DeviceBaseInfoList::iterator iter = pairedList._deviceBaseInfoList.begin(); iter != pairedList._deviceBaseInfoList.end(); iter++)
      {
         if(addr.compare(iter->_bdAddress)==0)
         {
            id = iter->_deviceHandle;
         }
      }
   }
   return id;
}
