/**
 * @file FC_Bluetooth_AudioRouting.cpp
 * @author
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 *
 * @addtogroup FC_Bluetooth
 *
 * @brief Implementation of the interface class of FC_Bluetooth_RingtoneList
 * @{
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "FC_Bluetooth_main.h"
#include "FC_Bluetooth_CCAService.h"
#include "FC_Bluetooth_RingtoneList.h"
#include "FunctionTracer.h"
#include <string.h>
#include<unistd.h>
#include <sstream>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APPLICATION
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FC_Bluetooth_RingtoneList.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APPLICATION
#endif
#endif

tU8 FC_Bluetooth_RingtoneList::m_u8CurrentLanguageSetting = FC_BLUETOOTH_CCASERVICE_ZERO;

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::FC_Bluetooth_RingtoneList
 *
 * DESCRIPTION :  constructor
 *
 * PARAMETER   :  none
 *
 * RETURNVALUE :  none
 *
 *******************************************************************************/
FC_Bluetooth_RingtoneList::FC_Bluetooth_RingtoneList() :
      m_szFilePath(BT_RINGTONE_FILEPATH),
      m_u16PlayRingtoneID(INCOMING_CALL_RINGTONE_ID),
      m_u16InBandRingToneIndex(FC_BLUETOOTH_CCASERVICE_ZERO)
{
   ENTRY

   // Initialise ring tone list
   FC_Bluetooth_tclRingToneListItem oRingtoneListItem;

   for (int i = FC_BLUETOOTH_CCASERVICE_ZERO; i < BT_MAX_RINGTONE_SUPPORTED; i++)
   {
      m_oAvailableRingToneList.push_back(oRingtoneListItem);
   }

   //initialize inband ringtone
   m_oAvailableRingToneList[m_u16InBandRingToneIndex].u16RingtoneID = RINGTONE_INBAND_RINGTONEID;

   std::string qDeviceRingtoneName = "Bluetooth Device Ringtone";
   m_oAvailableRingToneList[m_u16InBandRingToneIndex].szRingtoneName = qDeviceRingtoneName.c_str();
   m_oAvailableRingToneList[m_u16InBandRingToneIndex].bRingtoneAvailable = FALSE;
   m_oAvailableRingToneList[m_u16InBandRingToneIndex].bPlayRingtoneAvailable = FALSE;

   //refresh the list from the directory
   vRequestRefreshRingToneList();

}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::~FC_Bluetooth_RingtoneList
 *
 * DESCRIPTION :  destructor
 *
 * PARAMETER   :  none
 *
 * RETURNVALUE :  none
 *
 *******************************************************************************/
FC_Bluetooth_RingtoneList::~FC_Bluetooth_RingtoneList()
{
   ENTRY

   m_oAvailableRingToneList.clear();
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::vInitializeVehicleRingToneList
 *
 * DESCRIPTION :  It used to initialize the VehicleRingtone list
 *
 * PARAMETER   :  none
 *
 * RETURNVALUE :  none
 *
 *******************************************************************************/
void FC_Bluetooth_RingtoneList::vInitializeVehicleRingToneList()
{

   ENTRY

   // Initialise ring tone list
   for (tU16 i = 1; i < m_oAvailableRingToneList.size(); i++)
   {
      //This condition will be true if the ringtonelist size is 4(Maximum)
      if (((tU16)(i + 1)) == m_oAvailableRingToneList.size())
      {
         //New Ringtone id is included as 0xFF for no ringtone.
         m_oAvailableRingToneList[i].u16RingtoneID = NO_RINGTONE;
         m_oAvailableRingToneList[i].szRingtoneName = "No Ringtone";
      }
      else
      {
         m_oAvailableRingToneList[i].u16RingtoneID = static_cast<tU16>(i + 1);

         std::string strRingtoneName;
         std::string sTemp(1, (char)(('A' + (i - 1))));

         strRingtoneName = "Ringtone";
         strRingtoneName.append(sTemp);

         m_oAvailableRingToneList[i].szRingtoneName = strRingtoneName;
      }

      m_oAvailableRingToneList[i].bRingtoneAvailable = FALSE;
      m_oAvailableRingToneList[i].bPlayRingtoneAvailable = FALSE;

      ETG_TRACE_USR4(("vInitializeVehicleRingToneList: szRingtoneName - %s", m_oAvailableRingToneList[i].szRingtoneName.c_str()));
   }
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::u16GetRingToneID
 *
 * DESCRIPTION :  It used to Browse and populate the get the ringtone ID
 *
 * PARAMETER   :  tU16
 *
 * RETURNVALUE :  tU16
 *
 *******************************************************************************/
tU16 FC_Bluetooth_RingtoneList::u16GetRingToneID(tU16 index)
{
   ENTRY

   tU16 u16RingToneID = INCOMING_CALL_RINGTONE_ID;

   if (index < m_oAvailableRingToneList.size())
   {
      u16RingToneID = m_oAvailableRingToneList[index].u16RingtoneID;
   }

   return u16RingToneID;
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::bIsPlayRingtoneAvailable
 *
 * DESCRIPTION :  It used to check ringtone is playable or not
 *
 * PARAMETER   :  tU16
 *
 * RETURNVALUE :  tBool
 *
 *******************************************************************************/
tBool FC_Bluetooth_RingtoneList::bIsPlayRingtoneAvailable(tU16 index)
{
   ENTRY

   tBool bPlayRingtoneAvailable = FALSE;

   if (index < m_oAvailableRingToneList.size())
   {
      bPlayRingtoneAvailable = m_oAvailableRingToneList[index].bPlayRingtoneAvailable;
   }

   return bPlayRingtoneAvailable;
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::bIsRingtoneAvailable
 *
 * DESCRIPTION :  It used to check ringtone is playable or not
 *
 * PARAMETER   :  tBool, tU16
 *
 * RETURNVALUE :  tBool
 *
 *******************************************************************************/
tBool FC_Bluetooth_RingtoneList::bIsRingtoneAvailable(tBool bInbandPhoneSupported, tU16 index)
{
   ENTRY

   tBool bRingtoneAvailable = FALSE;
   ETG_TRACE_USR2((" FeatureSupport: Phone InBand: %d index:%d", bInbandPhoneSupported, index));

   if (index < m_oAvailableRingToneList.size())
   {
      if (index == m_u16InBandRingToneIndex)
      {
         if(bInbandPhoneSupported == TRUE)
         {
            bRingtoneAvailable = TRUE;
         }
         else
         {
            bRingtoneAvailable = FALSE;
            ETG_TRACE_USR2(("In Band ringing is not supported"));
         }
      }
      else
      {
         bRingtoneAvailable = m_oAvailableRingToneList[index].bRingtoneAvailable;
      }

   }

   return bRingtoneAvailable;
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::bIsActiveRingtone
 *
 * DESCRIPTION :  It used to check ringtone is active or not
 *
 * PARAMETER   :  tBool, tU16
 *
 * RETURNVALUE :  tBool
 *
 *******************************************************************************/
tBool FC_Bluetooth_RingtoneList::bIsActiveRingtone(tU16 u16ActiveRingtoneID, tU16 index)
{
   ENTRY

   tBool bActiveRingtone = FALSE;

   if (u16ActiveRingtoneID == u16GetRingToneID(index))
   {
      bActiveRingtone = TRUE;
   }

   ETG_TRACE_USR2((" bActiveRingtone %d", bActiveRingtone));

   return bActiveRingtone;
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::vGetRingtoneName
 *
 * DESCRIPTION :  It used to get the ringtone name
 *
 * PARAMETER   :  tU16, std::string
 *
 * RETURNVALUE :  tBool
 *
 *******************************************************************************/
void FC_Bluetooth_RingtoneList::vGetRingtoneName(tU16 index, std::string& szRingtoneName)
{
   ENTRY

   if (index < m_oAvailableRingToneList.size())
   {
      szRingtoneName = m_oAvailableRingToneList[index].szRingtoneName;
   }
   else if(index == (NO_RINGTONE - 1))
   {
      szRingtoneName = m_oAvailableRingToneList[BT_MAX_RINGTONE_SUPPORTED - 1].szRingtoneName;
   }

   ETG_TRACE_USR2((" szRingtoneName = %s", szRingtoneName.c_str()));
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::szGetRingtoneNameWithRingtoneID
 *
 * DESCRIPTION :  It used to get the ringtone name with ringtone ID
 *
 * PARAMETER   :  tU16, std::string
 *
 * RETURNVALUE :  tBool
 *
 *******************************************************************************/
void FC_Bluetooth_RingtoneList::szGetRingtoneNameWithRingtoneID(tU16 u16RingtoneID, std::string& szRingtoneName)
{
   ENTRY

   ETG_TRACE_USR2((" szGetRingtoneNameWithRingtoneID %d ", u16RingtoneID));

   for (tU16 i = 0; i < m_oAvailableRingToneList.size(); i++)
   {
      if (m_oAvailableRingToneList[i].u16RingtoneID == u16RingtoneID)
      {
         szRingtoneName = m_oAvailableRingToneList[i].szRingtoneName;
         ETG_TRACE_USR2((" Found %d, Name: %s ",u16RingtoneID, szRingtoneName.c_str()));
         break;
      }

   }
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::szGetRingToneFilePath
 *
 * DESCRIPTION :  It used to get the ringtone file path
 *
 * PARAMETER   :
 *
 * RETURNVALUE :  std::string
 *
 *******************************************************************************/
const std::string& FC_Bluetooth_RingtoneList::szGetRingToneFilePath()
{
   ENTRY

   ETG_TRACE_USR2((" szFilePath %s", m_szFilePath.c_str() ));

   return m_szFilePath;
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::vGetRingToneNameWithFilePath
 *
 * DESCRIPTION :  It used to get the ringtone file path
 *
 * PARAMETER   :
 *
 * RETURNVALUE :  std::string
 *
 *******************************************************************************/
void FC_Bluetooth_RingtoneList::vGetRingToneNameWithFilePath(tU16 u16RingToneID, std::string& szFile)
{

   ENTRY

   std::string szRingtoneName = "";

   szGetRingtoneNameWithRingtoneID(u16RingToneID, szRingtoneName);

   if (NO_RINGTONE == u16RingToneID)
   {
      szFile = szRingtoneName;
   }
   else
   {
      szFile = m_szFilePath;
      szFile.append(szRingtoneName);
      szFile.append(".mp3");
   }

   ETG_TRACE_USR2((" RingTone file: %s", szFile.c_str()));
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::vSetPlayRingtoneID
 *
 * DESCRIPTION :  It used to set the Play ringtone ID
 *
 * PARAMETER   :
 *
 * RETURNVALUE :  std::string
 *
 *******************************************************************************/
tVoid FC_Bluetooth_RingtoneList::vSetPlayRingtoneID(tU16 playRingtoneID)
{
   ENTRY

   ETG_TRACE_USR2((" playRingtoneID =  %d", playRingtoneID));
   m_u16PlayRingtoneID = playRingtoneID;
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::vGetPlayRingtoneID
 *
 * DESCRIPTION :  It used to get the Play ringtone ID
 *
 * PARAMETER   :
 *
 * RETURNVALUE :  std::string
 *
 *******************************************************************************/
tU16 FC_Bluetooth_RingtoneList::vGetPlayRingtoneID()
{
   ENTRY

   ETG_TRACE_USR2((" u16PlayRingtoneID =  %d", m_u16PlayRingtoneID));
   return m_u16PlayRingtoneID;
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::vRequestRefreshRingToneList
 *
 * DESCRIPTION :  It used to Browse and populate the Ringtone list
 *
 * PARAMETER   :  none
 *
 * RETURNVALUE :  none
 *
 *******************************************************************************/
void FC_Bluetooth_RingtoneList::vRequestRefreshRingToneList()
{
   ENTRY

   // Browse through BT database path and update the ring tone list
   // maximum of 10 device are supported.
   vInitializeVehicleRingToneList();

   // Check if the directory with the ring tone files exist
   if (access(m_szFilePath.c_str(), F_OK) == -1)
   {
      ETG_TRACE_ERR(( "Ring tone directory [%s] does not exist ", m_szFilePath.c_str() ));
      return;
   }

   for (tU16 i = 1; i < (m_oAvailableRingToneList.size() - 1); i++)
   {
      std::string& szRingToneName = m_oAvailableRingToneList[i].szRingtoneName;

      std::string qszRingToneWithPath;
      qszRingToneWithPath = m_szFilePath.c_str();
      qszRingToneWithPath.append(szRingToneName.c_str());
      qszRingToneWithPath.append(".mp3");

      ETG_TRACE_USR2((" qszRingToneWithPath =  %s", qszRingToneWithPath.c_str()));

      if (access(qszRingToneWithPath.c_str(), F_OK) != -1)
      {
         ETG_TRACE_USR2((" RingToneFile: '%s' exists ", qszRingToneWithPath.c_str() ));

         m_oAvailableRingToneList[i].bRingtoneAvailable = TRUE;
         m_oAvailableRingToneList[i].bPlayRingtoneAvailable = TRUE;
      }
      else
      {
         ETG_TRACE_ERR((" RingToneFile: '%s' does not exist", qszRingToneWithPath.c_str() ));

         m_oAvailableRingToneList[i].bRingtoneAvailable = FALSE;
         m_oAvailableRingToneList[i].bPlayRingtoneAvailable = FALSE;
      }
   }
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::vPrintStatus
 *
 * DESCRIPTION :  It used to print the Ringtone list
 *
 * PARAMETER   :  none
 *
 * RETURNVALUE :  none
 *
 *******************************************************************************/
void FC_Bluetooth_RingtoneList::vPrintStatus()
{
   ENTRY

   for (tU16 i = 0; (i < m_oAvailableRingToneList.size()); i++)
   {
      ETG_TRACE_USR2(("%s",pszGetClassName()));
      ETG_TRACE_USR2(("RingID %d",m_oAvailableRingToneList[i].u16RingtoneID));
      ETG_TRACE_USR2(("Name: %s",m_oAvailableRingToneList[i].szRingtoneName.c_str()));
      ETG_TRACE_USR2(("RT avail %d",m_oAvailableRingToneList[i].bRingtoneAvailable));
      ETG_TRACE_USR2(("PlayAvai %d",m_oAvailableRingToneList[i].bPlayRingtoneAvailable));

   }
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::setLanguageSetting
 *
 * DESCRIPTION :  It used to set the language
 *
 * PARAMETER   :  none
 *
 * RETURNVALUE :  none
 *
 *******************************************************************************/
void FC_Bluetooth_RingtoneList::setLanguageSetting(tU8 u8LanType)
{
   ETG_TRACE_USR4(("Set Language Type: %d",u8LanType));
   m_u8CurrentLanguageSetting = u8LanType;
}

/*******************************************************************************
 *
 * FUNCTION    :  FC_Bluetooth_RingtoneList::getLanguageSetting
 *
 * DESCRIPTION :  It used to set the language
 *
 * PARAMETER   :  none
 *
 * RETURNVALUE :  none
 *
 *******************************************************************************/
tU8 FC_Bluetooth_RingtoneList::getLanguageSetting()
{
   ETG_TRACE_USR4(("Get Language Type: %d",m_u8CurrentLanguageSetting));
   return m_u8CurrentLanguageSetting;
}

