/**
 * @file CCAOutputWrapper.cpp
 * @author 
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 *
 * @addtogroup FC_Bluetooth
 *
 * @brief Public interface for CCABluetoothOutputWrapper
 * @{
 */
/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "FC_Bluetooth_main.h"
#include "FC_Bluetooth_CCAService.h"
#include "FC_Bluetooth_Loopback_Controller.h"
#include "FC_Bluetooth_clienthandler_Diaglib.h"
#include "CCAOutputWrapper.h"
#include "FunctionTracer.h"
#include "FwStringUtils.h"
#include "FwBoschRegistryWrapper.h"
#include <cmath>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APP_OUTPUTWRAPPER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CCAOutputWrapper.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_BLUETOOTH_BM_APP_OUTPUTWRAPPER
#endif // VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

#ifdef VARIANT_S_FTR_ENABLE_FW_MEMORY_CHECKER
#include "FwMemoryChecker.h"
#endif

/******************************************************************************/
/*                                                                            */
/*                                                                            */
/*                                                                            */
/******************************************************************************/

CCAOutputWrapper::CCAOutputWrapper(fc_bluetooth_tclApp* poMainAppl) :
   ::bmcore::IBmCoreCallbackIf()
{
   ENTRY

   _bluetoothApp = poMainAppl;
}

/******************************************************************************/
/*                                                                            */
/* PROPERTIES                                                                 */
/*                                                                            */
/******************************************************************************/

void CCAOutputWrapper::onBtSystemStateChanged(const BtSystemState& btSystemState)
{
   ENTRY
   (tVoid) btSystemState;

   //TODO (added by Stefan): new BM Core interface, add implementation
   return;
}

void CCAOutputWrapper::onLocalPairableModeChanged(const SwitchStatus& localPairableMode)
{
   ENTRY

   ETG_TRACE_USR1(("onLocalPairableModeChanged:entered - %d", localPairableMode._switchState));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_LOCALPAIRABLEMODE_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   SwitchStatus* pairableMode = OSAL_NEW SwitchStatus;

   if(pairableMode != OSAL_NULL)
   {
      pairableMode->_switchState = localPairableMode._switchState;
      pairableMode->_switchedOffReason = localPairableMode._switchedOffReason;

      l_oLoopBackMessageData.vpData = (tVoid*)pairableMode;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onLocalConnectableModeChanged(const SwitchStatus& localConnectableMode)
{
   ENTRY

   ETG_TRACE_USR1(("onLocalConnectableModeChanged:entered - %d", localConnectableMode._switchState));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_LOCALCONNECTABLEMODE_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   SwitchStatus* connectableMode = OSAL_NEW SwitchStatus;

   if(connectableMode != OSAL_NULL)
   {
      connectableMode->_switchState = localConnectableMode._switchState;
      connectableMode->_switchedOffReason = localConnectableMode._switchedOffReason;

      l_oLoopBackMessageData.vpData = (tVoid*)connectableMode;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onBtStatusChanged(const SwitchStatus& bluetoothStatus)
{
   ENTRY
   (tVoid) bluetoothStatus;

   ETG_TRACE_USR1(("onBtStatusChanged:entered - %d", bluetoothStatus._switchState));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SWITCHBTSTATUS_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   SwitchStatus* status = OSAL_NEW SwitchStatus;

   if(status != OSAL_NULL)
   {
      status->_switchState = bluetoothStatus._switchState;
      status->_switchedOffReason = bluetoothStatus._switchedOffReason;

      l_oLoopBackMessageData.vpData = (tVoid*)status;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onBtLocalFriendlyNameChanged(const LocalFriendlyName& btLocalFriendlyName)
{
   ENTRY
   (tVoid) btLocalFriendlyName;

   ETG_TRACE_USR1(("onBtLocalInfoChanged:FriendlyName - %s", btLocalFriendlyName._localFriendlyName.c_str()));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_LOCALFRIENDLYNAME_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   LocalFriendlyName* friendlyName = OSAL_NEW LocalFriendlyName;

   if(friendlyName != OSAL_NULL)
   {
      friendlyName->_localFriendlyName = btLocalFriendlyName._localFriendlyName;

      l_oLoopBackMessageData.vpData = (tVoid*)friendlyName;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onBtLocalInfoChanged(const LocalInfo& btLocalInfo)
{
   ENTRY
   (tVoid) btLocalInfo;

   ETG_TRACE_USR1(("onBtLocalInfoChanged:entered"));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_BTLOCALINFO_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   LocalInfo* localInfo = OSAL_NEW LocalInfo;

   if(localInfo != OSAL_NULL)
   {
      localInfo->_moduleName = btLocalInfo._moduleName;
      localInfo->_bdAddress = btLocalInfo._bdAddress;
      localInfo->_bdName = btLocalInfo._bdName;
      localInfo->_protocolInfos = btLocalInfo._protocolInfos;
      localInfo->_vehicleIdentification = btLocalInfo._vehicleIdentification;
      localInfo->_stackInitialized = btLocalInfo._stackInitialized;

      l_oLoopBackMessageData.vpData = (tVoid*)localInfo;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onPairingStatusChanged(const PairingStatus& btPairingStatus)
{
   ENTRY
   (tVoid) btPairingStatus;

   ETG_TRACE_USR1(("onPairingStatusChanged:entered"));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_PAIRINGSTATUS_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;

   PairingStatus* status = OSAL_NEW PairingStatus;

   if(status != OSAL_NULL)
   {
      status->_origin = btPairingStatus._origin;
      status->_pairingType = btPairingStatus._pairingType;
      status->_pin = btPairingStatus._pin;
      status->_result = btPairingStatus._result;
      status->_state = btPairingStatus._state;
      status->_userConfirmationRequired = btPairingStatus._userConfirmationRequired;
      status->_remoteBdAddress = btPairingStatus._remoteBdAddress;
      status->_remoteBdName = btPairingStatus._remoteBdName;

      l_oLoopBackMessageData.vpData = (tVoid*)status;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onPairingPinChanged(const PairingPin& pairingPin)
{
   ENTRY
   (tVoid) pairingPin;

   ETG_TRACE_USR1(("onPairingPinChanged:entered - %s", pairingPin._pin.c_str()));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_PAIRINGPIN;
   l_oLoopBackMessageData.result = BM_RESULT_OK;

   PairingPin* pin = OSAL_NEW PairingPin;

   if(pin != OSAL_NULL)
   {
      pin->_pin = pairingPin._pin;

      l_oLoopBackMessageData.vpData = (tVoid*)pin;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onPairedDeviceListChanged(const PairedDeviceList& pairedDeviceList)
{
   ENTRY
   (tVoid) pairedDeviceList;

   ETG_TRACE_USR1(("onPairedDeviceListChanged:entered"));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_PAIREDDEVICELIST_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;

   PairedDeviceList* devicelist = OSAL_NEW PairedDeviceList;

#ifdef VARIANT_S_FTR_ENABLE_FW_MEMORY_CHECKER
   FW_MEMORY_CHECK_NEW_OBJ(devicelist);
#endif

   if(devicelist != OSAL_NULL)
   {
      devicelist->_deviceBaseInfoList.clear();

      if(pairedDeviceList._deviceBaseInfoList.size() > FC_BLUETOOTH_CCASERVICE_ZERO)
      {
         for (std::list<DeviceBaseInfo>::const_iterator it = pairedDeviceList._deviceBaseInfoList.begin();
               it != pairedDeviceList._deviceBaseInfoList.end(); it++)
         {
            DeviceBaseInfo temp;

            temp._deviceHandle = it->_deviceHandle;
            temp._bdAddress  = it->_bdAddress;
            temp._bdName = it->_bdName;
            temp._majorDeviceClass = it->_majorDeviceClass;
            temp._inbandRingtoneSupport = it->_inbandRingtoneSupport;
            temp._connectionStatus = it->_connectionStatus;
            temp._disconnectedReason = it->_disconnectedReason;
            temp._deviceIdentification = it->_deviceIdentification;
            temp._connectionOrderInfo = it->_connectionOrderInfo;
            temp._favoriteIndex = it->_favoriteIndex;
            temp._absoluteVolumeSupport = it->_absoluteVolumeSupport;
            temp._remoteConnectable = it->_remoteConnectable;
            temp._usagePreference = it->_usagePreference;
            temp._pairingOrderInfo = it->_pairingOrderInfo;
            temp._iAPoverBTSupported = it->_iAPoverBTSupported;

            for (ProtocolInfoMap::const_iterator itPIM = it->_protocolInfo.begin(); itPIM != it->_protocolInfo.end(); itPIM++)
            {
               ProtocolId protocolId  = itPIM->first;

               ProtocolInfo protocolInfo;
               protocolInfo._name = itPIM->second._name;
               protocolInfo._version = itPIM->second._version;
               protocolInfo._isSupported = itPIM->second._isSupported;
               protocolInfo._favoriteIndex = itPIM->second._favoriteIndex;

               temp._protocolInfo[protocolId] = protocolInfo;
            }

            devicelist->_deviceBaseInfoList.push_back(temp);
         }
      }

      l_oLoopBackMessageData.vpData = (tVoid*)devicelist;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onDeviceConnectionStatusListChanged(const DeviceConnectionStatusList& deviceConnectionStatusList)
{
   ENTRY
   (tVoid) deviceConnectionStatusList;

   ETG_TRACE_USR1(("onDeviceConnectionStatusListChanged:entered"));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_CONNECTIONDEVICELIST_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;

   DeviceConnectionStatusList* devicelist = OSAL_NEW DeviceConnectionStatusList;

#ifdef VARIANT_S_FTR_ENABLE_FW_MEMORY_CHECKER
   FW_MEMORY_CHECK_NEW_OBJ(devicelist);
#endif

   if(devicelist != OSAL_NULL)
   {
      devicelist->_deviceConnectionInfoList.clear();

      if(deviceConnectionStatusList._deviceConnectionInfoList.size() > FC_BLUETOOTH_CCASERVICE_ZERO)
      {
         for (tU16 iCount = FC_BLUETOOTH_CCASERVICE_ZERO; iCount < deviceConnectionStatusList._deviceConnectionInfoList.size(); iCount++)
         {
            DeviceConnectionInfo temp;

            temp._deviceHandle = deviceConnectionStatusList._deviceConnectionInfoList[iCount]._deviceHandle;
            temp._connectionStatus = deviceConnectionStatusList._deviceConnectionInfoList[iCount]._connectionStatus;
            temp._disconnectedReason = deviceConnectionStatusList._deviceConnectionInfoList[iCount]._disconnectedReason;
            temp._connectionOrderInfo = deviceConnectionStatusList._deviceConnectionInfoList[iCount]._connectionOrderInfo;

            temp._protocolConnectionInfo.clear();

            for (ProtocolConnectionInfoMap::const_iterator itPIM = deviceConnectionStatusList._deviceConnectionInfoList[iCount]._protocolConnectionInfo.begin();
                  itPIM != deviceConnectionStatusList._deviceConnectionInfoList[iCount]._protocolConnectionInfo.end(); itPIM++)
            {
               ProtocolId protocolId = itPIM->first;

               ProtocolConnectionInfo protocolInfo;
               protocolInfo._connectionStatus = itPIM->second._connectionStatus;
               protocolInfo._disconnectedReason = itPIM->second._disconnectedReason;
               protocolInfo._rfcommDevicePath = itPIM->second._rfcommDevicePath;
               protocolInfo._primary = itPIM->second._primary;

               temp._protocolConnectionInfo[protocolId] = protocolInfo;
            }

            temp._sppConnectionInfo.clear();

            for (SppConnectionInfoMap::const_iterator itSCIM = deviceConnectionStatusList._deviceConnectionInfoList[iCount]._sppConnectionInfo.begin();
                  itSCIM != deviceConnectionStatusList._deviceConnectionInfoList[iCount]._sppConnectionInfo.end(); itSCIM++)
            {
               Uuid uuid= itSCIM->first;

               ProtocolConnectionInfo protocolInfo;
               protocolInfo._connectionStatus = itSCIM->second._connectionStatus;
               protocolInfo._disconnectedReason = itSCIM->second._disconnectedReason;
               protocolInfo._rfcommDevicePath = itSCIM->second._rfcommDevicePath;
               protocolInfo._primary = itSCIM->second._primary;

               temp._sppConnectionInfo[uuid] = protocolInfo;
            }

            devicelist->_deviceConnectionInfoList.push_back(temp);
         }
      }

      l_oLoopBackMessageData.vpData = (tVoid*)devicelist;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onDiscoveryStatusChanged(IN const SwitchStatus& discoveryStatus)
{
   ENTRY
   (tVoid) discoveryStatus;

   ETG_TRACE_USR1(("onDiscoveryStatusChanged:entered - %d", discoveryStatus._switchState));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SWITCHDISCOVERYSTATUS_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   SwitchStatus* status = OSAL_NEW SwitchStatus;

   if(status != OSAL_NULL)
   {
      status->_switchState = discoveryStatus._switchState;
      status->_switchedOffReason = discoveryStatus._switchedOffReason;

      l_oLoopBackMessageData.vpData = (tVoid*)status;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onDiscoveredDeviceChanged(IN const BdAddress& bdAddress, IN const BdName& btDeviceName)
{
   ENTRY
   (tVoid) bdAddress;
   (tVoid) btDeviceName;

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_DISCOVERDEVICE_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;

   DiscoveredDevice* temp = OSAL_NEW DiscoveredDevice;

   if (temp != OSAL_NULL)
   {
      temp->bdAddress = bdAddress;
      temp->btDeviceName = btDeviceName;

      l_oLoopBackMessageData.vpData = (tVoid*) temp;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if (poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onDiscoveredDeviceListChanged(IN const DiscoveredDeviceList& discoveredDeviceList)
{
   ENTRY

   (tVoid) discoveredDeviceList;

   ETG_TRACE_USR1(("onDiscoveredDeviceListChanged:entered - %d", discoveredDeviceList._discoveredDeviceInfoList.size()));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_DISCOVERDEVICELIST_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   DiscoveredDeviceList* temp = OSAL_NEW DiscoveredDeviceList;

   if(temp != OSAL_NULL)
   {
      temp->_discoveredDeviceInfoList.clear();

      if(discoveredDeviceList._discoveredDeviceInfoList.size() > FC_BLUETOOTH_CCASERVICE_ZERO)
      {
         for (std::list<DiscoveredDeviceInfo>::const_iterator it = discoveredDeviceList._discoveredDeviceInfoList.begin();
               it != discoveredDeviceList._discoveredDeviceInfoList.end(); it++)
         {
            DiscoveredDeviceInfo discoveredDeviceInfo;

            discoveredDeviceInfo._deviceHandle  = it->_deviceHandle;
            discoveredDeviceInfo._bdAddress = it->_bdAddress;
            discoveredDeviceInfo._bdName  = it->_bdName;
            discoveredDeviceInfo._majorDeviceClass  = it->_majorDeviceClass;
            discoveredDeviceInfo._minorDeviceClass  = it->_minorDeviceClass;
            discoveredDeviceInfo._connectionStatus  = it->_connectionStatus;
            discoveredDeviceInfo._lastDiscoveredTimestamp  = it->_lastDiscoveredTimestamp;

            for (ProtocolInfoMap::const_iterator itPIM = it->_protocolInfo.begin(); itPIM != it->_protocolInfo.end(); itPIM++)
            {
               ProtocolId protocolId  = itPIM->first;

               ProtocolInfo protocolInfo;
               protocolInfo._name = itPIM->second._name;
               protocolInfo._version = itPIM->second._version;
               protocolInfo._isSupported = itPIM->second._isSupported;

               discoveredDeviceInfo._protocolInfo[protocolId] = protocolInfo;
            }

            temp->_discoveredDeviceInfoList.push_back(discoveredDeviceInfo);
         }
      }

      l_oLoopBackMessageData.vpData = (tVoid*)temp;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onBlockStatusChanged(IN const BlockStatus& blockStatus)
{
   ENTRY
   (tVoid) blockStatus;

   ETG_TRACE_USR1(("onBlockStatusChanged:entered - %d", blockStatus._deviceBlockStatusList.size()));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_BLOCKSTATUS_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   BlockStatus* temp = OSAL_NEW BlockStatus;

   if(temp != OSAL_NULL)
   {
      temp->_overallDeviceBlockStatus._blockState = blockStatus._overallDeviceBlockStatus._blockState;
      temp->_overallDeviceBlockStatus._bdAddress = blockStatus._overallDeviceBlockStatus._bdAddress;

      temp->_deviceBlockStatusList.clear();

      if(blockStatus._deviceBlockStatusList.size() > FC_BLUETOOTH_CCASERVICE_ZERO)
      {
         for (std::vector<DeviceBlockStatus>::const_iterator it = blockStatus._deviceBlockStatusList.begin();
               it != blockStatus._deviceBlockStatusList.end(); it++)
         {
            DeviceBlockStatus deviceBlockStatus;

            deviceBlockStatus._deviceHandle = it->_deviceHandle;
            deviceBlockStatus._blockState  = it->_blockState;

            for (ProtocolBlockStatusMap::const_iterator itPIM = it->_protocolBlockStatusMap.begin(); itPIM != it->_protocolBlockStatusMap.end(); itPIM++)
            {
               BlockState blockstate = itPIM->second;

               Protocol protocol;
               protocol._protocolId = itPIM->first._protocolId;
               protocol._uuid = itPIM->first._uuid;

               deviceBlockStatus._protocolBlockStatusMap[protocol] = blockstate;
            }

            temp->_deviceBlockStatusList.push_back(deviceBlockStatus);
         }
      }

      l_oLoopBackMessageData.vpData = (tVoid*)temp;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onAutoConnectionTypeChanged(IN const AutoConnectionType& autoConnectionType)
{
   ENTRY
   (tVoid) autoConnectionType;

   ETG_TRACE_USR1(("onAutoConnectionTypeChanged:entered - %d", autoConnectionType._type));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_AUTOCONNECTIONTYPE_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   AutoConnectionType* connectionType = OSAL_NEW AutoConnectionType;

   if(connectionType != OSAL_NULL)
   {
      connectionType->_type = autoConnectionType._type;

      l_oLoopBackMessageData.vpData = (tVoid*)connectionType;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onBmConfigDataChanged(IN const BmConfigData& bmConfigData)
{
   ENTRY
   (tVoid) bmConfigData;

   ETG_TRACE_USR1(("onBmConfigDataChanged:entered - %d", bmConfigData._maxNumPairedDevices));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_BMCONFIGDATA_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   BmConfigData* configData = OSAL_NEW BmConfigData;

   if(configData != OSAL_NULL)
   {
      configData->_maxNumPairedDevices = bmConfigData._maxNumPairedDevices;

      l_oLoopBackMessageData.vpData = (tVoid*)configData;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onBmServiceAvailabilityChanged(IN const BmServiceAvailability& bmServiceAvailability)
{
   ENTRY
   (tVoid) bmServiceAvailability;

   ETG_TRACE_USR1(("onBmServiceAvailabilityChanged:entered - %d", bmServiceAvailability._serviceAvailability));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_BMSERVICEAVAILABILITY_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   BmServiceAvailability* serviceAvailability = OSAL_NEW BmServiceAvailability;

   if(serviceAvailability != OSAL_NULL)
   {
      serviceAvailability->_serviceAvailability = bmServiceAvailability._serviceAvailability;

      l_oLoopBackMessageData.vpData = (tVoid*)serviceAvailability;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }
}

void CCAOutputWrapper::onMultiHFPSupportStatusChanged(const SwitchStatus& multiHFPSupportStatus)
{
   ENTRY
   (tVoid) multiHFPSupportStatus;

   ETG_TRACE_USR1(("onMultiHFPSupportStatusChanged:entered - %d", multiHFPSupportStatus._switchState));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_MULTIHFPSUPPORT_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   SwitchStatus* status = OSAL_NEW SwitchStatus;

   if(status != OSAL_NULL)
   {
      status->_switchState = multiHFPSupportStatus._switchState;
      status->_switchedOffReason = multiHFPSupportStatus._switchedOffReason;

      l_oLoopBackMessageData.vpData = (tVoid*)status;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onAutoConnectionStatusChanged(IN const AutoConnectionStatus& autoConnectionStatus)
{
   ENTRY

   ETG_TRACE_USR1(("onAutoConnectionStatusChanged: autoConnectionState - %d", autoConnectionStatus._autoConnectionState));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_AUTOCONNECTIONSTATUS_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   AutoConnectionStatus* status = OSAL_NEW AutoConnectionStatus;

   if(status != OSAL_NULL)
   {
      status->_autoConnectionState = autoConnectionStatus._autoConnectionState;
      status->_deviceIdInProgress = autoConnectionStatus._deviceIdInProgress;

      l_oLoopBackMessageData.vpData = (tVoid*)status;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}


/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

void CCAOutputWrapper::updateConfiguration(INOUT BmCoreConfiguration& bmCoreConfiguration)
{
   ENTRY

   ETG_TRACE_USR1(("updateConfiguration: entered"));

   //TODO (added by Stefan, 20160511): new BM Core interface, add implementation

   //BM Core will get it's basic configuration from a project specific configuration file.
   //This configuration is given to CCA layer by calling CCAOutputWrapper::updateConfiguration(...).
   //In the implementation of CCAOutputWrapper::updateConfiguration(...) the configuration values can be adapted
   //depending on whether an additional project specific configuration has to be considered (KDS, EOL, ...).

   for (KdsInformation::const_iterator it = bmCoreConfiguration._kdsInformation.begin();
         it != bmCoreConfiguration._kdsInformation.end(); ++it)
   {
      KdsInfo kdsInfo = *it;

      // check the received UUID is matched with HU supported UUID's
      if((true == kdsInfo._read)&& ("CPW" == kdsInfo._serviceName))
      {
         vReadCarPlayWirelessSupported(kdsInfo, bmCoreConfiguration._carPlayWirelessSupported );
      }
      else if((true == kdsInfo._read)&& ("AAW" == kdsInfo._serviceName))
      {
         vReadAndroidAutoWirelessSupported(kdsInfo, bmCoreConfiguration._androidAutoWirelessSupported );
      }
      else if((true == kdsInfo._read)&& ("PAN" == kdsInfo._serviceName))
      {
         vReadPanProfileSupported(kdsInfo, bmCoreConfiguration._protocolSupportInformation[BM_PROTOCOL_ID_PAN]._maxNumInstances );
      }
      else if((true == kdsInfo._read)&& ("VehicleBtFriendlyName" == kdsInfo._serviceName))
      {
         vReadBTVehicleFriendlyName(kdsInfo, bmCoreConfiguration._btLocalFriendlyName );
      }
      else if((true == kdsInfo._read)&& ("VehicleBtOnOff" == kdsInfo._serviceName))
      {
         vReadVehicleBtOnOff(kdsInfo, bmCoreConfiguration._btOnOffSupport );
      }
   }

   vUpdateDidVersion(bmCoreConfiguration._didVersion);

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_UPDATE_CONFIGURATION;
   l_oLoopBackMessageData.result = BM_RESULT_OK;
   BmCoreConfiguration* configData = OSAL_NEW BmCoreConfiguration;

   if(configData != OSAL_NULL)
   {
      configData->_btModuleName = bmCoreConfiguration._btModuleName;
      configData->_btLocalFriendlyName = bmCoreConfiguration._btLocalFriendlyName;
      configData->_btLocalBdAddress = bmCoreConfiguration._btLocalBdAddress;
      configData->_didSpecificationId = bmCoreConfiguration._didSpecificationId;
      configData->_didVendorId = bmCoreConfiguration._didVendorId;
      configData->_didProductId = bmCoreConfiguration._didProductId;
      configData->_didVersion = bmCoreConfiguration._didVersion;
      configData->_didVendorIdSource = bmCoreConfiguration._didVendorIdSource;
      configData->_didClientExecutableUrl = bmCoreConfiguration._didClientExecutableUrl;
      configData->_didServiceDescription = bmCoreConfiguration._didServiceDescription;
      configData->_didClientDocumentationUrl = bmCoreConfiguration._didClientDocumentationUrl;
      configData->_carPlayWirelessSupported = bmCoreConfiguration._carPlayWirelessSupported;
      configData->_wideBandSpeechEnabled = bmCoreConfiguration._wideBandSpeechEnabled;
      configData->_pbdlViaPbapEnabled = bmCoreConfiguration._pbdlViaPbapEnabled;
      configData->_pbdlViaHfpEnabled = bmCoreConfiguration._pbdlViaHfpEnabled;
      configData->_pbdlViaSppEnabled = bmCoreConfiguration._pbdlViaSppEnabled;
      configData->_pbdlViaSyncMlEnabled = bmCoreConfiguration._pbdlViaSyncMlEnabled;
      configData->_voiceRecognitionEnabled = bmCoreConfiguration._voiceRecognitionEnabled;
      configData->_enhancedCallControlEnabled = bmCoreConfiguration._enhancedCallControlEnabled;
      configData->_avrcpBrowsingEnabled = bmCoreConfiguration._avrcpBrowsingEnabled;
      configData->_audioCodecMp3Enabled = bmCoreConfiguration._audioCodecMp3Enabled;
      configData->_audioCodecAacEnabled = bmCoreConfiguration._audioCodecAacEnabled;
      configData->_maxNumPairedDevices = bmCoreConfiguration._maxNumPairedDevices;
      configData->_maxNumConnectedDevices = bmCoreConfiguration._maxNumConnectedDevices;
      configData->_defaultBtStatus = bmCoreConfiguration._defaultBtStatus;
      configData->_defaultLocalPairableMode = bmCoreConfiguration._defaultLocalPairableMode;
      configData->_defaultLocalConnectableMode = bmCoreConfiguration._defaultLocalConnectableMode;
      configData->_localPairableTimeoutSeconds = bmCoreConfiguration._localPairableTimeoutSeconds;
      configData->_localConnectableTimeoutSeconds = bmCoreConfiguration._localConnectableTimeoutSeconds;
      configData->_useFixedPinLegacyPairing = bmCoreConfiguration._useFixedPinLegacyPairing;
      configData->_pinLengthLegacyPairing = bmCoreConfiguration._pinLengthLegacyPairing;
      configData->_defaultFixedPinLegacyPairing = bmCoreConfiguration._defaultFixedPinLegacyPairing;
      configData->_autoConfirmRemoteLegacyPairingRequest = bmCoreConfiguration._autoConfirmRemoteLegacyPairingRequest;
      configData->_autoConfirmLocalLegacyPairingRequest = bmCoreConfiguration._autoConfirmLocalLegacyPairingRequest;
      configData->_autoConfirmRemoteSecureSimplePairingRequest = bmCoreConfiguration._autoConfirmRemoteSecureSimplePairingRequest;
      configData->_autoConfirmLocalSecureSimplePairingRequest = bmCoreConfiguration._autoConfirmLocalSecureSimplePairingRequest;
      configData->_allowLegacyPairingRetry = bmCoreConfiguration._allowLegacyPairingRetry;
      configData->_lostDeviceConnectionRetryDelayMilliSeconds = bmCoreConfiguration._lostDeviceConnectionRetryDelayMilliSeconds;
      configData->_localProtocolConnectingDelaySeconds = bmCoreConfiguration._localProtocolConnectingDelaySeconds;
      configData->_protocolDisconnectingTimeoutSeconds = bmCoreConfiguration._protocolDisconnectingTimeoutSeconds;
      configData->_defaultStandardAutoConnectionType = bmCoreConfiguration._defaultStandardAutoConnectionType;
      configData->_startAutoConnectionOnStartUp = bmCoreConfiguration._startAutoConnectionOnStartUp;
      configData->_linkQualityRequestTimeOut = bmCoreConfiguration._linkQualityRequestTimeOut;
      configData->_blockAllProtocolsForNonProjectionDevices = bmCoreConfiguration._blockAllProtocolsForNonProjectionDevices;
      configData->_connectAfterPairing = bmCoreConfiguration._connectAfterPairing;
      configData->_iapEnabled = bmCoreConfiguration._iapEnabled;
      configData->_defaultmultiHFPSupport = bmCoreConfiguration._defaultmultiHFPSupport;
      configData->_systemWideRingtoneEnabled = bmCoreConfiguration._systemWideRingtoneEnabled;
      configData->_blockDeviceRemoteConnectionsTimeoutSeconds = bmCoreConfiguration._blockDeviceRemoteConnectionsTimeoutSeconds;
      configData->_updateConflictDuringCPWOOBPairing = bmCoreConfiguration._updateConflictDuringCPWOOBPairing;
      configData->_CPWReconnectTimeoutInSecs = bmCoreConfiguration._CPWReconnectTimeoutInSecs;
      configData->_enableDefaultActionAtEndOfBtLimitation = bmCoreConfiguration._enableDefaultActionAtEndOfBtLimitation;
      configData->_deleteDeviceDuringOOBTPairing = bmCoreConfiguration._deleteDeviceDuringOOBTPairing;
      configData->_serviceSearchTimeoutSeconds = bmCoreConfiguration._serviceSearchTimeoutSeconds;
      configData->_defaultConnectionPageTimeoutMilliSeconds = bmCoreConfiguration._defaultConnectionPageTimeoutMilliSeconds;
      configData->_delayAutoConnectionAtEndOfBtLimitationInSec = bmCoreConfiguration._delayAutoConnectionAtEndOfBtLimitationInSec;

      for (WifiHealthinessIndicatorInformation::const_iterator it = bmCoreConfiguration._wifiHealthinessIndicatorInformation.begin();
                  it != bmCoreConfiguration._wifiHealthinessIndicatorInformation.end(); ++it)
      {
         WifiHealthinessIndicatorInfo wifiHealthinessIndicatorInfo;

         wifiHealthinessIndicatorInfo._wifiHealthinessIndicatorThresholdValue = it->_wifiHealthinessIndicatorThresholdValue;
         wifiHealthinessIndicatorInfo._connectionPageTimeoutMilliSeconds = it->_connectionPageTimeoutMilliSeconds;

         configData->_wifiHealthinessIndicatorInformation.push_back(wifiHealthinessIndicatorInfo);
      }

      for (SppServiceInformation::const_iterator it = bmCoreConfiguration._sppServiceInformation.begin();
            it != bmCoreConfiguration._sppServiceInformation.end(); ++it)
      {
         SppServiceInfo sppServiceInfo;

         sppServiceInfo._maxNumInstances = it->_maxNumInstances;
         sppServiceInfo._serviceName = it->_serviceName;
         sppServiceInfo._localUuid = it->_localUuid;
         sppServiceInfo._remoteUuid = it->_remoteUuid;

         configData->_sppServiceInformation.push_back(sppServiceInfo);
      }

      for (ProtocolSupportInformation::const_iterator itPIM = bmCoreConfiguration._protocolSupportInformation.begin();
            itPIM != bmCoreConfiguration._protocolSupportInformation.end(); itPIM++)
      {
         ProtocolId protocolId  = itPIM->first;

         ProtocolSupportInfo protocolSupportInfo;
         protocolSupportInfo._name = itPIM->second._name;
         protocolSupportInfo._version = itPIM->second._version;
         protocolSupportInfo._maxNumInstances = itPIM->second._maxNumInstances;
         protocolSupportInfo._dependencyLevel = itPIM->second._dependencyLevel;
         protocolSupportInfo._masterProtocolsSelector = itPIM->second._masterProtocolsSelector;
         protocolSupportInfo._usedForDeviceConnection = itPIM->second._usedForDeviceConnection;

         configData->_protocolSupportInformation[protocolId] = protocolSupportInfo;
      }

      configData->_btOnOffSupport = bmCoreConfiguration._btOnOffSupport;

      l_oLoopBackMessageData.vpData = (tVoid*)configData;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }
}

void CCAOutputWrapper::stateChangeNormalResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY

   ETG_TRACE_USR1(("stateChangeNormalResponse:result - %d", bmResult));

   (tVoid) bmResult;
   (tVoid) act;

   return;
}

void CCAOutputWrapper::stateChangeOffResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY

   ETG_TRACE_USR1(("stateChangeOffResponse:result - %d", bmResult));

   (tVoid) bmResult;
   (tVoid) act;

   return;
}

void CCAOutputWrapper::setProtocolUserAvailabilityResponse(IN const BmResult bmResult, IN const ActType act)
{
   ETG_TRACE_USR1(("setProtocolUserAvailabilityResponse:result - %d", bmResult));

   (tVoid) act;

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SETPROFILEUSERAVAILABILITYRESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if (poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::switchLocalPairableModeResponse(const BmResult result, IN const ActType act)
{
   ENTRY
   (tVoid) result;
   (tVoid) act;

   ETG_TRACE_USR1(("switchLocalPairableModeResponse:result - %d",result));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SWITCHLOCALPAIRABLEMODERESPONSE;
   l_oLoopBackMessageData.result = result;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::switchLocalConnectableModeResponse(const BmResult result, IN const ActType act)
{
   ENTRY
   (tVoid) result;
   (tVoid) act;

   ETG_TRACE_USR1(("switchLocalConnectableModeResponse:result - %d", result));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SWITCHLOCALCONNECTABLEMODERESPONSE;
   l_oLoopBackMessageData.result = result;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::switchBtStatusResponse(const BmResult result, IN const ActType act)
{
   ENTRY
   (tVoid) result;
   (tVoid) act;

   ETG_TRACE_USR1(("switchBtStatusResponse:result - %d",result));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SWITCHBTSTATUS_RESPONSE;
   l_oLoopBackMessageData.result = result;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::rejectPairingRequestResponse(const BmResult bmResult, IN const ActType act)
{
   ENTRY
   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("rejectPairingRequestResponse:result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_PAIRINGRESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::acceptPairingRequestResponse(const BmResult bmResult, IN const ActType act)
{
   ENTRY
   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("acceptPairingRequestResponse:result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_PAIRINGRESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::cancelPairingResponse(const BmResult bmResult, IN const ActType act)
{
   ENTRY
   (tVoid) bmResult;
   (tVoid) act;


   ETG_TRACE_USR1(("cancelPairingResponse:result - %d", bmResult));

   if (BM_RESULT_OK != bmResult)
   {
      LoopBackMessageData l_oLoopBackMessageData;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SENDPAIRINGREQUESTRESPONSE;
      l_oLoopBackMessageData.result = bmResult;

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::setAutoConnectionTypeResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY

   ETG_TRACE_USR1(("setAutoConnectionTypeResponse: result - %d", bmResult));

   //TODO (added by Stefan, 20171110): new BM Core interface, add implementation

   (tVoid) act;
}

void CCAOutputWrapper::setPairingPinResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY

   ETG_TRACE_USR1(("setPairingPinResponse: result - %d", bmResult));

   //TODO (added by Stefan, 20171109): new BM Core interface, add implementation

   (tVoid) act;
}

void CCAOutputWrapper::addOobPairedDeviceResponse(const BmResult bmResult, IN const ActType act)
{
   ENTRY
   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("addOobPairedDeviceResponse:result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_ADDOOBPAIREDDEVICERESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }
}

void CCAOutputWrapper::startPairingResponse(const BmResult bmResult, IN const ActType act)
{
   ENTRY
   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("startPairingResponse:result - %d", bmResult));

   if (BM_RESULT_OK != bmResult)
   {
      LoopBackMessageData l_oLoopBackMessageData;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SENDPAIRINGREQUESTRESPONSE;
      l_oLoopBackMessageData.result = bmResult;

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::switchDiscoveryStatusResponse(IN const BmResult result, IN const ActType act)
{
   ENTRY

   (tVoid) act;

   ETG_TRACE_USR1(("switchDiscoveryStatusResponse:result - %d", result));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SWITCHDISCOVERYSTATUS_RESPONSE;
   l_oLoopBackMessageData.result = result;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::connectDeviceResponse(const DeviceId deviceHandle, const BmResult result, IN const ActType act)
{
   ENTRY
   (tVoid) deviceHandle;
   (tVoid) result;
   (tVoid) act;

   ETG_TRACE_USR1(("connectDeviceResponse: deviceHandle - %d result - %d", deviceHandle, result));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_CHANGEDEVICESTATERESPONSE;
   l_oLoopBackMessageData.result = result;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::disconnectDeviceResponse(const DeviceId deviceHandle, const BmResult result, IN const ActType act)
{
   ENTRY
   (tVoid) deviceHandle;
   (tVoid) result;
   (tVoid) act;

   ETG_TRACE_USR1(("disconnectDeviceResponse: deviceHandle - %dresult - %d", deviceHandle, result));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_CHANGEDEVICESTATERESPONSE;
   l_oLoopBackMessageData.result = result;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::deleteDeviceResponse(const DeviceId deviceHandle, const BmResult result, IN const ActType act)
{
   ENTRY
   (tVoid) deviceHandle;
   (tVoid) result;
   (tVoid) act;

   ETG_TRACE_USR1(("deleteDeviceResponse: deviceHandle - %dresult - %d", deviceHandle, result));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_CHANGEDEVICESTATERESPONSE;
   l_oLoopBackMessageData.result = result;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::blockDeviceResponse(const DeviceId deviceHandle, const BmResult result, IN const ActType act)
{
   ENTRY
   (tVoid) deviceHandle;
   (tVoid) result;
   (tVoid) act;

   ETG_TRACE_USR1(("blockDeviceResponse: deviceHandle - %dresult - %d", deviceHandle, result));

   // send the response to CCA thread context via Loopback
   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_BLOCKDEVICERESPONSE;
   l_oLoopBackMessageData.result = result;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::unblockDeviceResponse(const DeviceId deviceHandle, const BmResult result, IN const ActType act)
{
   ENTRY
   (tVoid) deviceHandle;
   (tVoid) result;
   (tVoid) act;

   ETG_TRACE_USR1(("unblockDeviceResponse: deviceHandle - %d result - %d", deviceHandle, result));

   // send the response to CCA thread context via Loopback
   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_UNBLOCKDEVICERSPONSE;
   l_oLoopBackMessageData.result = result;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::connectProfilesResponse(const DeviceId deviceHandle, const BmResult result, IN const ActType act)
{
   ENTRY
   (tVoid) deviceHandle;
   (tVoid) result;
   (tVoid) act;

   ETG_TRACE_USR1(("connectProfilesResponse:deviceHandle - %d result - %d", deviceHandle, result));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_CONNECTPROFILERESPONSE;
   l_oLoopBackMessageData.result = result;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::disconnectProfilesResponse(const DeviceId deviceHandle, const BmResult result, IN const ActType act)
{
   ENTRY
   (tVoid) deviceHandle;
   (tVoid) result;
   (tVoid) act;

   ETG_TRACE_USR1(("disconnectProfilesResponse:deviceHandle - %d result - %d", deviceHandle, result));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_DISCONNECTPROFILERESPONSE;
   l_oLoopBackMessageData.result = result;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::blockProfilesResponse(const DeviceId deviceHandle, const BmResult bmResult, IN const ActType act)
{
   ENTRY
   (tVoid) deviceHandle;
   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("blockProfilesResponse:deviceHandle - %d result - %d", deviceHandle, bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_BLOCKPROFILERESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::unblockProfilesResponse(const DeviceId deviceHandle, const BmResult bmResult, IN const ActType act)
{
   ENTRY
   (tVoid) deviceHandle;
   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("unblockProfilesResponse:deviceHandle - %d result - %d", deviceHandle, bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_UNBLOCKPROFILERESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   DeviceId* deviceId = OSAL_NEW DeviceId;

   if (deviceId != OSAL_NULL)
   {
      *deviceId = deviceHandle;

      l_oLoopBackMessageData.vpData = (tVoid*) deviceId;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if(poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::setFavoriteResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY

   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("setFavoriteResponse: result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SETFAVORITERESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if (poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }
}

void CCAOutputWrapper::clearFavoriteResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY

   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("clearFavoriteResponse: result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_CLEARFAVORITERESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if (poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }
}

void CCAOutputWrapper::startServiceSearchResponse (IN const BmResult bmResult, IN const ActType act )
{
   ENTRY

   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("startServiceSearchResponse: result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_STARTSERVICESEARCHRESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if (poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }
}

void CCAOutputWrapper::stopServiceSearchResponse (IN const BmResult bmResult, IN const ActType act )
{
   ENTRY

   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("stopServiceSearchResponse: result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_STOPSERVICESEARCHRESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if (poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }
}

void CCAOutputWrapper::setDeviceUsagePreferenceResponse (IN const BmResult bmResult, IN const ActType act )
{
   ENTRY

   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("setDeviceUsagePreferenceResponse: result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SETDEVICEUSAGEPREFERENCERESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if (poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }
}

void CCAOutputWrapper::switchMultiHFPSupportResponse (IN const BmResult bmResult, IN const ActType act )
{
   ENTRY

   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("switchMultiHFPSupportResponse: result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SWITCHMULTIHFPSUPPORTRESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if (poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }
}

void CCAOutputWrapper::startAutoConnectSequenceResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY
   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("startAutoConnectSequenceResponse: result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_STARTAUTOCONNECT;
   l_oLoopBackMessageData.result = bmResult;

   if (poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::stopAutoConnectSequenceResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY
   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("stopAutoConnectSequenceResponse: result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_STARTAUTOCONNECT;
   l_oLoopBackMessageData.result = bmResult;

   if (poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::setBtLimitationModeResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY

   ETG_TRACE_USR1(("setBtLimitationModeResponse:result - %d act - %d", bmResult, act));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_BTLIMITATIONMODERESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::replaceBtLimitationModeResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY

   ETG_TRACE_USR1(("replaceBtLimitationModeResponse: result - %d act - %d", bmResult, act));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller *poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   // Set the result and corresponding Id to LoopBackMessageData
   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_REPLACEBTLIMITATIONMODERESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if(poLoopbackController)
   {
      // Send the message to CCA layer Thread
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }
}

void CCAOutputWrapper::onBtLimitationModeChanged(IN const BtLimitationMode& btLimitationMode)
{
   ENTRY

   ETG_TRACE_USR1(("onBtLimitationModeChanged: entered "));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_BTLIMITATIONMODE_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;

   BtLimitationMode* limitationMode = OSAL_NEW BtLimitationMode;

   if (limitationMode != OSAL_NULL)
   {
      limitationMode->_btLimitationModeInfoList.clear();

      ETG_TRACE_USR1(("onBtLimitationModeChanged: size -%d ", btLimitationMode._btLimitationModeInfoList.size()));

      if (btLimitationMode._btLimitationModeInfoList.size() > FC_BLUETOOTH_CCASERVICE_ZERO)
      {
         for (std::vector<BtLimitationModeInfo>::const_iterator it = btLimitationMode._btLimitationModeInfoList.begin();
               it != btLimitationMode._btLimitationModeInfoList.end(); it++)
         {
            BtLimitationModeInfo temp;

            temp._bdAddress = it->_bdAddress;
            temp._bdName = it->_bdName;
            temp._limitationMode = it->_limitationMode;
            temp._limitationState = it->_limitationState;

            limitationMode->_btLimitationModeInfoList.push_back(temp);
         }
      }

      l_oLoopBackMessageData.vpData = (tVoid*) limitationMode;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if (poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::onBtProfileUsageInfoChanged(IN const ProfileUsageList& btProfileUsageList)
{
	ENTRY

	ETG_TRACE_USR1(("onBtProfileUsageInfoChanged: entered "));

	LoopBackMessageData l_oLoopBackMessageData;

	l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_BTPROFILEUSAGEINFO_CHANGED;
	l_oLoopBackMessageData.result = BM_RESULT_OK;

	ProfileUsageList* profileUsageList = OSAL_NEW ProfileUsageList;

	if (profileUsageList != OSAL_NULL)
	{
	   profileUsageList->_btProfileUsageInfoList.clear();

	   ETG_TRACE_USR1(("onBtProfileUsageInfoChanged: size -%d ", btProfileUsageList._btProfileUsageInfoList.size()));

	   if (btProfileUsageList._btProfileUsageInfoList.size() > FC_BLUETOOTH_CCASERVICE_ZERO)
	   {
	      for (std::list<ProfileUsageInfo>::const_iterator it = btProfileUsageList._btProfileUsageInfoList.begin();
	            it != btProfileUsageList._btProfileUsageInfoList.end(); it++)
	      {
	         ProfileUsageInfo temp;

	         temp._protocol = it->_protocol;
	         temp._deviceHandle = it->_deviceHandle;
	         temp._InUse = it->_InUse;
	         temp._ProfileUsageType = it->_ProfileUsageType;

	         profileUsageList->_btProfileUsageInfoList.push_back(temp);
	      }
	   }

	   l_oLoopBackMessageData.vpData = (tVoid*) profileUsageList;

	   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

	   if (poLoopbackController)
	   {
	      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
	   }
	}
}

void CCAOutputWrapper::setBtProfileUsageInfoResponse(IN const BmResult bmResult, IN const ActType act)
{
	ENTRY

	(tVoid)act;

	ETG_TRACE_USR1(("setBtProfileUsageInfoResponse:result - %d", bmResult));

	LoopBackMessageData l_oLoopBackMessageData;

	FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

	l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SETBTPROFILEUSAGEINFORESPONSE;
	l_oLoopBackMessageData.result = bmResult;

	if(poLoopbackController)
	{
	   poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
	}
}

void CCAOutputWrapper::onUserDecisionRequiredSignal(IN const IssueInfoList& issueInfoList)
{
   ENTRY

   ETG_TRACE_USR1(("onUserDecisionRequiredSignal: entered "));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_USERDECISIONREQUIRED_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;

   IssueInfoList* btissueInfoList = OSAL_NEW IssueInfoList;

   if (btissueInfoList != OSAL_NULL)
   {
      btissueInfoList->_conflictTrigger = issueInfoList._conflictTrigger;

      btissueInfoList->_conflictInfoList.clear();

      ETG_TRACE_USR1(("onUserDecisionRequiredSignal: size -%d ", issueInfoList._conflictInfoList.size()));

      if (issueInfoList._conflictInfoList.size() > FC_BLUETOOTH_CCASERVICE_ZERO)
      {
         for (std::vector<IssueInfo>::const_iterator it = issueInfoList._conflictInfoList.begin();
               it != issueInfoList._conflictInfoList.end(); it++)
         {
            IssueInfo temp;

            temp._conflictType = it->_conflictType;
            temp._deviceID = it->_deviceID;
            temp._protocol = it->_protocol;
            temp._profileUsageType = it->_profileUsageType;
            temp._conflictState = it->_conflictState;

            btissueInfoList->_conflictInfoList.push_back(temp);
         }
      }

      l_oLoopBackMessageData.vpData = (tVoid*) btissueInfoList;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if (poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }
}

void CCAOutputWrapper::notifyUserDecisionResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY

   (tVoid) act;

   ETG_TRACE_USR1(("notifyUserDecisionResponse:result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_NOTIFYUSERDECISIONRESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }
}

void CCAOutputWrapper::resetToDefaultResponse(const BmResult bmResult, IN const ActType act)
{
   ENTRY
   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("resetToDefaultResponse:result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_CLEARDEVICEINFORESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if(poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

   return;
}

void CCAOutputWrapper::onResetToDefaultStatusChanged(const ResetToDefaultStatus& status)
{
   ENTRY
   (tVoid) status;

   ETG_TRACE_USR1(("onResetToDefaultStatusChanged:resetResult - %d resetState - %d",
         status._resetResult, status._resetState));

   return;
}

void CCAOutputWrapper::switchTestModeResponse(const BmResult result, IN const ActType act)
{
   ENTRY

   (tVoid) act;

   ETG_TRACE_USR1(("switchTestModeResponse: result - %d ", result));

   S_TestmodeStatus_t rTemp;
   fc_bluetooth_tclClientHandler_Diaglib *poBluetoothDiaglib = OSAL_NULL;
   rTemp.u8NotiType = (tU8)-1; // indicate immediate callback

   if (result == BM_RESULT_OK)
   { // operation succeeded
      rTemp.u8Status_TMComp = (tU8)EN_BluetoothConnectionResultCONNECTION_IN_PROGRESS;
   }
   else
   { // operation failed
      rTemp.u8Status_TMComp = (tU8)EN_BluetoothConnectionResultFAILED;
   }

   poBluetoothDiaglib = fc_bluetooth_tclClientHandler_Diaglib::poGetInstance();
   if (OSAL_NULL != poBluetoothDiaglib)
   {
      poBluetoothDiaglib->vTestmodeStatus(&rTemp);
   }

   return;
}

void CCAOutputWrapper::storeDeviceRingtoneIdInDbResponse(IN const DeviceId deviceHandle, IN const BmResult bmResult,
      IN const ActType act)
{
   ENTRY

   (tVoid) deviceHandle;
   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("storeDeviceRingtoneIdInDbResponse:deviceHandle - %d bmResult - %d",
         deviceHandle, bmResult));

   return;
}

void CCAOutputWrapper::readDeviceRingtoneIdFromDbResponse(IN const DeviceId deviceHandle, IN const BmResult bmResult,
      IN const ActType act)
{
   ENTRY

   (tVoid) deviceHandle;
   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("readDeviceRingtoneIdFromDbResponse:deviceHandle - %d bmResult - %d",
            deviceHandle, bmResult));

   return;
}

void CCAOutputWrapper::storeSystemWideRingtoneIdInDbResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY

   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("storeSystemWideRingtoneIdInDbResponse: bmResult - %d", bmResult));

   return;
}

void CCAOutputWrapper::readSystemWideRingtoneIdFromDbResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY

   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("readSystemWideRingtoneIdFromDbResponse: bmResult - %d", bmResult));

   return;
}

void CCAOutputWrapper::onTestModeChanged(const SwitchStatus& testMode)
{
   ENTRY
   (tVoid) testMode;

   ETG_TRACE_USR1(("onTestModeChanged: _switchState - %d ", testMode._switchState));

   fc_bluetooth_tclClientHandler_Diaglib *poBluetoothDiaglib = OSAL_NULL;

   S_TestmodeStatus_t rTemp;

   if(testMode._switchState == SWITCH_STATE_SWITCHED_ON)
   {
      rTemp.u8NotiType = 1;
      rTemp.u8Status_TMComp = (tU8) EN_BluetoothConnectionResultSUCCESFULL;

      poBluetoothDiaglib = fc_bluetooth_tclClientHandler_Diaglib::poGetInstance();
      if (OSAL_NULL != poBluetoothDiaglib)
      {
         poBluetoothDiaglib->vTestmodeStatus(&rTemp);
      }
   }
   else if(testMode._switchState == SWITCH_STATE_SWITCHED_OFF)
   {
      rTemp.u8NotiType = FC_BLUETOOTH_CCASERVICE_ZERO;
      rTemp.u8Status_TMComp = (tU8) EN_BluetoothConnectionResultSUCCESFULL;

      poBluetoothDiaglib = fc_bluetooth_tclClientHandler_Diaglib::poGetInstance();
      if (OSAL_NULL != poBluetoothDiaglib)
      {
         poBluetoothDiaglib->vTestmodeStatus(&rTemp);
      }
   }

   return;
}

void CCAOutputWrapper::onLinkQualityChanged(const LinkQuality& linkQuality)
{
   ENTRY
   (tVoid) linkQuality;

   ETG_TRACE_USR1(("onLinkQualityChanged: entered "));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_LINKQUALITY_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;

   LinkQuality* temp = OSAL_NEW LinkQuality;

   if (temp != OSAL_NULL)
   {
      temp->_linkQualityInfoList.clear();

      for (LinkQualityInfoList::const_iterator itLQI = linkQuality._linkQualityInfoList.begin();
            itLQI != linkQuality._linkQualityInfoList.end(); itLQI++)
      {
         DeviceId deviceId;
         LinkQualityInfo linkQualityInfo;

         deviceId = itLQI->first;
         linkQualityInfo._rssi = itLQI->second._rssi;
         linkQualityInfo._linkQuality = itLQI->second._linkQuality;

         temp->_linkQualityInfoList.insert(std::make_pair(deviceId, linkQualityInfo));
      }

      l_oLoopBackMessageData.vpData = (tVoid*) temp;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if (poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}

void CCAOutputWrapper::updateLinkQualityResponse(IN const BmResult bmResult, IN const ActType act)
{
   ENTRY

   (tVoid) bmResult;
   (tVoid) act;

   ETG_TRACE_USR1(("updateLinkQualityResponse: bmResult - %d", bmResult));

   return;
}

void CCAOutputWrapper::requestTestModeLinkQualityResponse(IN const BmResult bmResult, IN const LinkQualityInfo& linkQualityInfo,
      IN const ActType act)
{
   ENTRY

   (tVoid) bmResult;
   (tVoid) linkQualityInfo;
   (tVoid) act;

   ETG_TRACE_USR1(("requestTestModeLinkQualityResponse: result - %d ", bmResult));

   fc_bluetooth_tclClientHandler_Diaglib *poBluetoothDiaglib = OSAL_NULL;

   poBluetoothDiaglib = fc_bluetooth_tclClientHandler_Diaglib::poGetInstance();
   if (OSAL_NULL != poBluetoothDiaglib)
   {
      poBluetoothDiaglib->vLinkQualityStatus(bmResult, linkQualityInfo);
   }

   return;
}

void CCAOutputWrapper::onLinkQualityTestModeChanged(const TestModeLinkQuality& linkQuality)
{
   ENTRY
   (tVoid) linkQuality;

   ETG_TRACE_USR1(("onLinkQualityTestModeChanged: entered "));

   LoopBackMessageData l_oLoopBackMessageData;

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_TESTMODE_LINKQUALITY_CHANGED;
   l_oLoopBackMessageData.result = BM_RESULT_OK;

   TestModeLinkQuality* temp = OSAL_NEW TestModeLinkQuality;

   if (temp != OSAL_NULL)
   {
      temp->_linkQualityInfo._rssi = linkQuality._linkQualityInfo._rssi;
      temp->_linkQualityInfo._linkQuality = linkQuality._linkQualityInfo._linkQuality;

      l_oLoopBackMessageData.vpData = (tVoid*) temp;

      FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

      if (poLoopbackController)
      {
         poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
      }
   }

   return;
}
void CCAOutputWrapper::setPrimaryHfpDeviceResponse(const BmResult bmResult)
{
   ENTRY
   (tVoid) bmResult;

   ETG_TRACE_USR1(("setPrimaryHfpDeviceResponse: result - %d", bmResult));

   LoopBackMessageData l_oLoopBackMessageData;

   FC_Bluetooth_Loopback_Controller * poLoopbackController = FC_Bluetooth_Loopback_Controller::poGetInstance();

   l_oLoopBackMessageData.u32MessageContentId = BTSET_BMCORE_MSG_SETPRIMARYRESPONSE;
   l_oLoopBackMessageData.result = bmResult;

   if (poLoopbackController)
   {
      poLoopbackController->vPostLoopBackMessageToCCA(l_oLoopBackMessageData);
   }

}

/*******************************************************************************
* FUNCTION:   vReadBTVehicleFriendlyName()
*
* DESCRIPTION: Read the BT Vehicle Friendly Name
*
*  PARAMETERS:
*     IN:   None
*
*  RETURNVALUE: None
********************************************************************************/
tVoid CCAOutputWrapper::vReadBTVehicleFriendlyName(KdsInfo& kdsInfo, std::string& btFriendlyName)
{
   ENTRY

   ETG_TRACE_USR1(("vReadBTVehicleFriendlyName: Values of Key = 0x%X, Value of Length = %d, Value of Byte = %d",
         kdsInfo._key, kdsInfo._length, kdsInfo._byte));

   if((0 != kdsInfo._length) && (0 != kdsInfo._key))
   {
      tU8 u8DeviceName[kdsInfo._length + 1];
      memset(u8DeviceName, 0, sizeof(u8DeviceName));

      if(bReadData(kdsInfo._key, kdsInfo._length, u8DeviceName))
      {
         ETG_TRACE_USR4(("bReadData for BTVehicleFriendlyName is successful"));
         u8DeviceName[kdsInfo._length] = 0;

         // check for valid UTF-8 string
         ::std::string tmpName((char*)u8DeviceName);
         if(true == ::fw::makeValidUtf8String(tmpName))
         {
            // at least one byte/character replaced (invalid UTF-8 string was given) => use internal default value
            ETG_TRACE_ERR(("makeValidUtf8String for BTVehicleFriendlyName failed: tmpName=%s", tmpName.c_str()));
         }
         else
         {
            ETG_TRACE_USR4(("makeValidUtf8String for BTVehicleFriendlyName is successful"));
            vCheckAndRemoveTrailingSpace(tmpName);
            btFriendlyName.swap(tmpName);
         }
      }
      else
      {
         ETG_TRACE_USR4(("bReadData for BTVehicleFriendlyName has failed"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("vReadBTVehicleFriendlyName: invalid length and key values"));
   }

   ETG_TRACE_USR2((" KDS BTFriendlyName is %s", btFriendlyName.c_str()));

   return;
}

/*******************************************************************************
* FUNCTION:   vCheckAndRemoveTrailingSpace()
*
* DESCRIPTION: Remove trailing space
*
*  PARAMETERS:
*     IN:   None
*
*  RETURNVALUE: None
********************************************************************************/
tVoid CCAOutputWrapper::vCheckAndRemoveTrailingSpace(std::string& btFriendlyName)
{
   ETG_TRACE_USR1(("vCheckAndRemoveTrailingSpace: btFriendlyName - %50s", btFriendlyName.c_str()));

   std::string whitespace(" ");

   std::size_t foundChar = btFriendlyName.find_last_not_of(whitespace);

   if (foundChar != std::string::npos)
   {
      btFriendlyName.erase(foundChar+1);
      ETG_TRACE_USR4(("vCheckAndRemoveTrailingSpace(btFriendlyName - %50s): after removal of trailing space", btFriendlyName.c_str()));
   }
   else
   {
      ETG_TRACE_USR4(("vCheckAndRemoveTrailingSpace: btFriendlyName is empty"));
   }
}

tVoid CCAOutputWrapper::vReadCarPlayWirelessSupported(KdsInfo& kdsInfo, bool& bCarPlayWirelessSupported)
{
   ENTRY

   ETG_TRACE_USR1(("vReadCarPlayWirelessSupported: Values of Key = 0x%X, Value of Length = %d, Value of Byte = %d, Value of Bit = %d",
         kdsInfo._key, kdsInfo._length, kdsInfo._byte, kdsInfo._bit ));

   // check the received UUID is matched with HU supported UUID's
   if((0 != kdsInfo._length) && (0 != kdsInfo._key))
   {
      tU8 u8CarPlayWirelessSupported[kdsInfo._length + 1];
      memset(u8CarPlayWirelessSupported, 0, sizeof(u8CarPlayWirelessSupported));

      if(bReadData(kdsInfo._key, kdsInfo._length, u8CarPlayWirelessSupported))
      {
         ETG_TRACE_USR4(("vReadCarPlayWirelessSupported: bReadData for CPW byte is %d", u8CarPlayWirelessSupported[kdsInfo._byte]));

         //2nd bit  will tell whether target supports CPW or not
         double kdsInfoBit = std::pow((double)FC_BT_KDS_BIT_CONVERT, (double)kdsInfo._bit);
         bCarPlayWirelessSupported = (bool)(u8CarPlayWirelessSupported[kdsInfo._byte] & ((tU8)kdsInfoBit));

         ETG_TRACE_USR4(("vReadCarPlayWirelessSupported - CPW supported is - %10s", bCarPlayWirelessSupported ? "true" : "false"));
      }
      else
      {
         ETG_TRACE_USR4(("vReadCarPlayWirelessSupported: bReadData for CPW has failed"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("vReadCarPlayWirelessSupported: invalid length and key values"));
   }

   return;
}

tVoid CCAOutputWrapper::vReadAndroidAutoWirelessSupported(KdsInfo& kdsInfo, bool& bAndroidAutoWirelessSupported)
{
   ENTRY

   ETG_TRACE_USR1(("vReadAndroidAutoWirelessSupported: Values of Key = 0x%X, Value of Length = %d, Value of Byte = %d, Value of Bit = %d",
         kdsInfo._key, kdsInfo._length, kdsInfo._byte, kdsInfo._bit ));

   // check the received UUID is matched with HU supported UUID's
   if((0 != kdsInfo._length) && (0 != kdsInfo._key))
   {
      tU8 u8AndroidAutoWirelessSupported[kdsInfo._length + 1];
      memset(u8AndroidAutoWirelessSupported, 0, sizeof(u8AndroidAutoWirelessSupported));

      if(bReadData(kdsInfo._key, kdsInfo._length, u8AndroidAutoWirelessSupported))
      {
         ETG_TRACE_USR4(("vReadAndroidAutoWirelessSupported: bReadData for AAW byte is %d", u8AndroidAutoWirelessSupported[kdsInfo._byte]));

         //2nd bit  will tell whether target supports AAW or not
         double KdsInfoBit = std::pow((double)FC_BT_KDS_BIT_CONVERT, (double)kdsInfo._bit);
         bAndroidAutoWirelessSupported = (bool)(u8AndroidAutoWirelessSupported[kdsInfo._byte] & ((tU8)KdsInfoBit));

         ETG_TRACE_USR4(("vReadAndroidAutoWirelessSupported - AAW supported is - %10s", bAndroidAutoWirelessSupported ? "true" : "false"));
      }
      else
      {
         ETG_TRACE_USR4(("vReadAndroidAutoWirelessSupported: bReadData for AAW has failed"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("vReadAndroidAutoWirelessSupported: invalid length and key values"));
   }

   return;
}

tVoid CCAOutputWrapper::vReadPanProfileSupported(KdsInfo& kdsInfo, uint8_t& u8MaxNumInstances)
{
   ENTRY

   ETG_TRACE_USR1(("vReadPanProfileSupported: Values of Key = 0x%X, Value of Length = %d, Value of Byte = %d, Value of Bit = %d",
         kdsInfo._key, kdsInfo._length, kdsInfo._byte, kdsInfo._bit ));

   if((0 != kdsInfo._length) && (0 != kdsInfo._key))
   {
      tU8 u8PanProfileSupported[kdsInfo._length + 1];
      bool bPANSupported = false;
      memset(u8PanProfileSupported, 0, sizeof(u8PanProfileSupported));

      // Read the byte
      if(bReadData(kdsInfo._key, kdsInfo._length, u8PanProfileSupported))
      {
         ETG_TRACE_USR4(("vReadPanProfileSupported: bReadData for PAN byte is %d", u8PanProfileSupported[kdsInfo._byte]));
         double KdsInfoBit = std::pow((double)FC_BT_KDS_BIT_CONVERT, (double)kdsInfo._bit);
         //2nd bit  will tell whether target supports PAN or not
         bPANSupported = (bool)(u8PanProfileSupported[kdsInfo._byte] & ((tU8)KdsInfoBit));

         ETG_TRACE_USR4(("vReadPanProfileSupported - PAN supported is - %10s", bPANSupported ? "true" : "false"));

         if(false == bPANSupported)
         {
            u8MaxNumInstances = 0u;
         }

      }
      else
      {
         ETG_TRACE_USR4(("vReadPanProfileSupported: bReadData for PAN has failed"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("vReadPanProfileSupported: invalid length and key values"));
   }

   return;
}

void CCAOutputWrapper::vReadVehicleBtOnOff(KdsInfo& kdsInfo, bool& btOnOffSupport)
{
   ENTRY

   ETG_TRACE_USR1(("vReadVehicleBtOnOff: Values of Key = 0x%X, Value of Length = %d, Value of Byte = %d, Value of Bit = %d",
         kdsInfo._key, kdsInfo._length, kdsInfo._byte, kdsInfo._bit ));

   if((0 != kdsInfo._length) && (0 != kdsInfo._key))
   {
      tU8 u8btOnOffSupport[kdsInfo._length + 1];
      memset(u8btOnOffSupport, 0, sizeof(u8btOnOffSupport));

      if(bReadData(kdsInfo._key, kdsInfo._length, u8btOnOffSupport))
      {
         ETG_TRACE_USR4(("vReadVehicleBtOnOff: bReadData for btOnOff byte is %d", u8btOnOffSupport[kdsInfo._byte]));

         //1st bit  will tell whether target supports BT or not
         double kdsInfoBit = std::pow((double)FC_BT_KDS_BIT_CONVERT, (double)kdsInfo._bit);
         btOnOffSupport = (bool)(u8btOnOffSupport[kdsInfo._byte] & ((tU8)kdsInfoBit));

         ETG_TRACE_USR4(("vReadVehicleBtOnOff - btOnOff supported is - %10s", btOnOffSupport ? "true" : "false"));
      }
      else
      {
         ETG_TRACE_USR4(("vReadVehicleBtOnOff: bReadData for btOnOff has failed"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("vReadVehicleBtOnOff: invalid length and key values"));
   }
}

/*******************************************************************************
* FUNCTION:   FC_Bluetooth_KdsConfig()
*
* DESCRIPTION: Interface to read data using a Key in KDS
*
*  PARAMETERS:
*     IN:   u16KdsKey : Key address at which data is stored in KDS
*           u16DataLen : Total size of the data, stored at u16KdsKey
*     OUT:  pu8DataBuffer : Pointer to buffer, to which read data should be copied.
*
*  RETURNVALUE: tBool: TRUE - if read is successful, else FALSE.
********************************************************************************/
tBool CCAOutputWrapper::bReadData(tU16 u16KdsKey, tU16 u16DataLen, tU8* pu8DataBuffer)
{
   ETG_TRACE_USR4(("CCAOutputWrapper::bReadData() entered: u16KdsKey = 0x%X, u16DataLen = 0x%x",
      u16KdsKey, u16DataLen));

   tBool bReadSuccess = FALSE;

   NORMAL_M_ASSERT(OSAL_NULL != pu8DataBuffer);
   if (OSAL_NULL != pu8DataBuffer)
   {
      //! Open KDS file
      OSAL_tIODescriptor tIoKdsHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READONLY);

      if (OSAL_ERROR != tIoKdsHandle)
      {
         tsKDSEntry rKDSEntryData;
         rKDSEntryData.u16Entry = u16KdsKey;
         rKDSEntryData.u16EntryLength = u16DataLen;
         rKDSEntryData.u16EntryFlags = M_KDS_ENTRY_FLAG_NONE;
         rKDSEntryData.au8EntryData[0] = 0;

         //! Read required data from file
         tS32 s32OsalReadErr = OSAL_s32IORead(tIoKdsHandle, (tPS8) &rKDSEntryData, (sizeof(rKDSEntryData)));
         if (OSAL_ERROR != s32OsalReadErr)
         {
            (tVoid) OSAL_pvMemoryCopy((tVoid*) (pu8DataBuffer), rKDSEntryData.au8EntryData, u16DataLen);
            bReadSuccess = TRUE;
         }
         else
         {
            ETG_TRACE_ERR(("CCAOutputWrapper::bReadData: Error reading KDS file: %d \n", OSAL_u32ErrorCode()));
         }

         //! Close KDS file
         (tVoid) OSAL_s32IOClose(tIoKdsHandle);

      }//if (OSAL_ERROR != tIoKdsHandle)
      else
      {
         ETG_TRACE_ERR(("CCAOutputWrapper::bReadData: Error opening KDS file: %d \n", OSAL_u32ErrorCode()));
      }
   }//if (OSAL_NULL != pu8DataBuffer)

   return bReadSuccess;
}

/*******************************************************************************
* FUNCTION:    vUpdateDidVersion()
*
* DESCRIPTION: Read SW version from registry and update DID version.
*
* PARAMETERS:
*     IN/OUT: version : DID version
*
* RETURNVALUE: none
********************************************************************************/
tVoid CCAOutputWrapper::vUpdateDidVersion(uint16_t& version)
{
   const ::std::string regPath("/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS");
   const ::std::string versionKey("BUILDVERSION_CUSTVERSTRING");
   ::fw::BoschRegistryWrapper registry;
   ::std::string swVersion;

   registry.open(regPath);

   if(true == registry.read(swVersion, versionKey))
   {
      ETG_TRACE_USR3(("CCAOutputWrapper::vUpdateDidVersion(): SW version=%s", swVersion.c_str()));

      uint64_t bcdCode(0);

      if(true == ::fw::convertHexString2BcdCode(bcdCode, swVersion, 4))
      {
         // SW version was successfully read and converted
         // check for value 0 because in this case we will fallback to value from config file
         if(0 < bcdCode)
         {
            version = (uint16_t)bcdCode;
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("CCAOutputWrapper::vUpdateDidVersion(): reading SW version failed"));
   }

   registry.close();
}
