#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_RESET_TO_DEFAULT_CONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ResetToDefaultController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_RESET_TO_DEFAULT_CONTROLLER
#endif
#endif

#include "LocalSpm.h"
#include "FunctionTracer.h"
#include "Dispatcher.h"
#include "BmVarTrace.h"
#include "ResetToDefaultController.h"
#include "BmGlobalLock.h"
#include "BmUtils.h"

namespace bmcore
{
   ResetToDefaultController::ResetToDefaultController(const ComponentId componentId) :
         ILocalSpm(componentId), _firstEntryInStateIdle(true), _factorySettingsChanged(false), _pairableModeReset(false),
         _connectableModeReset(false), _btStatusReset(false),_multiHFPSupportReset(false), _resetToDefaultResult(BM_RESET_RESULT_OK)
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("ResetToDefaultController: is being created"));

      // threadless SM
      disableAllLocks(true);

      /* Early initialization of variables */
      initMembers();
   }

   ResetToDefaultController::~ResetToDefaultController()
   {
      ENTRY_INTERNAL
   }

   Result ResetToDefaultController::resetToDefault(IN const bool factorySettingsChanged)
   {
      ENTRY

      ETG_TRACE_USR1(("resetToDefault"));

      Result result(CC_ERR_INT_NO_ERROR);

      _factorySettingsChanged = factorySettingsChanged;

      ResetToDefaultStatus resetToDefaultStatus;
      BmResult bmResult = LocalSpm::getBmCoreMainController().getResetToDefaultStatus(resetToDefaultStatus);

      if (BM_RESULT_OK == bmResult)
      {
         if (BM_RESET_STATE_IDLE == resetToDefaultStatus._resetState)
         {
            LocalSpm::getBmCoreMainController().updateResetToDefaultStatus(BM_RESET_STATE_IN_PROGRESS, BM_RESET_RESULT_NOT_APPLICABLE);

            //todo: handle an ongoing pairing

            ETG_TRACE_USR4(("resetToDefault: sending event DELETE_ALL_DEVICES"));
            result = this->SendEventByName("DELETE_ALL_DEVICES", 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("resetToDefault: could not send event DELETE_ALL_DEVICES (error = %d) ", ETG_CENUM(CcErrorInternal, result)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;

               LocalSpm::getBmCoreMainController().updateResetToDefaultStatus(BM_RESET_STATE_IDLE, _resetToDefaultResult);

               (void) LocalSpm::getBmCoreMainController().updateFunctionalityRestrictionInfo(BM_RESTRICTION_BIT_RESET_TO_DEFAULT_IN_PROGRESS, false);
            }
         }
         else if (BM_RESET_STATE_IN_PROGRESS == resetToDefaultStatus._resetState)
         {
            ETG_TRACE_USR1(("resetToDefault: resetting to default already in progress"));
         }
         else
         {
            result = CC_ERR_INT_GENERAL_ERROR;
            ETG_TRACE_ERR(("resetToDefault: unhandled reset state (error = %d) ", ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         result = CC_ERR_INT_NOT_ALLOWED;
         _resetToDefaultResult = BM_RESET_RESULT_ERROR;
      }

      return result;
   }

   Result ResetToDefaultController::initSm()
   {
      ENTRY

      ETG_TRACE_USR1(("initSm"));

      _firstEntryInStateIdle = true;

      return CC_ERR_INT_NO_ERROR;
   }

   Result ResetToDefaultController::handleStopSm()
   {
      ENTRY

      ETG_TRACE_USR1(("handleStopSm"));

      /* Send stop done to SPM in the transition to final state in state machine */
      return stopDone(0);
   }

   Result ResetToDefaultController::handleDone()
   {
      ENTRY

      ETG_TRACE_USR1(("handleDone"));

      /* Send done done to SPM in the transition to final state in state machine */
      return doneDone(0);

   }

   Result ResetToDefaultController::enterIdle()
   {
      ENTRY

      ETG_TRACE_USR1(("enterIdle"));

      if (false == _firstEntryInStateIdle)
      {
         LocalSpm::getBmCoreMainController().updateResetToDefaultStatus(BM_RESET_STATE_IDLE, _resetToDefaultResult);

         initMembers();
      }
      else
      {
         _firstEntryInStateIdle = false;
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ResetToDefaultController::enterDeletingAllDevices()
   {
      ENTRY

      ETG_TRACE_USR1(("enterDeletingAllDevices"));

      DeviceIdList deviceIdList;
      Result result = LocalSpm::getDbManager().getAllDeviceIds(deviceIdList);
      bool eventToBeSent(true);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR1(("enterDeletingAllDevices: number of paired devices: %u", deviceIdList.size()));

         if (false == deviceIdList.empty())
         {
            ETG_TRACE_USR1(("enterDeletingAllDevices: deleting all paired devices"));

            BmResult bmResult = LocalSpm::getBmCoreMainController().deleteDeviceInt(0u, BM_DEVICE_HANDLE_TYPE_ALL);

            if (BM_RESULT_OK == bmResult)
            {
               eventToBeSent = false;
            }
            else
            {
               ETG_TRACE_ERR(("enterDeletingAllDevices: could not delete all devices (bmResult = %d)", ETG_CENUM(BmResult, bmResult)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("enterDeletingAllDevices: could not get all device IDs from DB (error = %d)", ETG_CENUM(CcErrorInternal, result)));
         _resetToDefaultResult = BM_RESET_RESULT_ERROR;
      }

      if (true == eventToBeSent)
      {
         ETG_TRACE_USR4(("enterDeletingAllDevices: sending event NO_PAIRED_DEVICES to ResetToDefaultControllerSm"));
         result = this->SendEventByName("NO_PAIRED_DEVICES", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("enterDeletingAllDevices: could not send event NO_PAIRED_DEVICES (error = %d)", ETG_CENUM(CcErrorInternal, result)));
            _resetToDefaultResult = BM_RESET_RESULT_ERROR;
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ResetToDefaultController::enterResettingToDefault()
   {
      ENTRY

      ETG_TRACE_USR1(("enterResettingToDefault"));

      BmResult bmResult(BM_RESULT_OK);

      ETG_TRACE_USR1(("enterResettingToDefault: resetting PairingPin"));

      PairingPin pairingPin;

      if (true == _factorySettingsChanged)
      {
         pairingPin._pin = "";
      }
      else
      {
         pairingPin._pin = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultFixedPinLegacyPairing;
      }

      bmResult = LocalSpm::getBmCoreMainController().setPairingPinInt(pairingPin);

      if (BM_RESULT_OK != bmResult)
      {
         ETG_TRACE_ERR(("enterResettingToDefault: could not set PairingPin (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
         _resetToDefaultResult = BM_RESET_RESULT_ERROR;
      }

      ETG_TRACE_USR1(("enterResettingToDefault: resetting AutoConnectionType"));

      AutoConnectionType autoConnectionType;

      if (true == _factorySettingsChanged)
      {
         autoConnectionType._type = BM_STANDARD_AUTO_CONNECTION_UNKNOWN;
      }
      else
      {
         autoConnectionType._type = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultStandardAutoConnectionType;
      }

      bmResult = LocalSpm::getBmCoreMainController().setAutoConnectionTypeInt(autoConnectionType);

      if (BM_RESULT_OK != bmResult)
      {
         ETG_TRACE_ERR(("enterResettingToDefault: could not set AutoConnectionType (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
         _resetToDefaultResult = BM_RESET_RESULT_ERROR;
      }

      ETG_TRACE_USR1(("enterResettingToDefault: resetting BtLocalFriendlyName"));

      LocalFriendlyName localFriendlyName;

      if (true == _factorySettingsChanged)
      {
         localFriendlyName._localFriendlyName = "";
      }
      else
      {
         localFriendlyName._localFriendlyName = LocalSpm::getDataProvider().getBmCoreConfiguration()._btLocalFriendlyName;
      }

      bmResult = LocalSpm::getBmCoreMainController().setBtLocalFriendlyNameInt(localFriendlyName._localFriendlyName);

      if (BM_RESULT_OK != bmResult)
      {
         ETG_TRACE_ERR(("enterResettingToDefault: could not set BtLocalFriendlyName (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
         _resetToDefaultResult = BM_RESET_RESULT_ERROR;
      }

      ETG_TRACE_USR1(("enterResettingToDefault: resetting LocalPairableMode"));

      TargetSwitchState defaultTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode;

      SwitchStatus switchStatus;
      bmResult = LocalSpm::getBmCoreMainController().getLocalPairableMode(switchStatus);

      if (BM_RESULT_OK == bmResult)
      {
         if (((TARGET_SWITCH_STATE_SWITCHED_OFF == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_OFF == switchStatus._switchState))
               || ((TARGET_SWITCH_STATE_SWITCHED_ON == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_ON == switchStatus._switchState)))
         {
            _pairableModeReset = true;
            this->checkProgress();
         }
         else
         {
            bmResult = LocalSpm::getBmCoreMainController().switchLocalPairableModeInt(defaultTargetSwitchState, "");

            if (BM_RESULT_OK != bmResult)
            {
               ETG_TRACE_ERR(("enterResettingToDefault: could not switch LocalPairableMode to %d (bmResult = %d)", ETG_CENUM(TargetSwitchState, defaultTargetSwitchState), ETG_CENUM(BmResult, bmResult)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("enterResettingToDefault: could not get LocalPairableMode (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
         _resetToDefaultResult = BM_RESET_RESULT_ERROR;
      }

      ETG_TRACE_USR1(("enterResettingToDefault: resetting LocalConnectableMode"));

      defaultTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode;

      bmResult = LocalSpm::getBmCoreMainController().getLocalConnectableMode(switchStatus);

      if (BM_RESULT_OK == bmResult)
      {
         if (((TARGET_SWITCH_STATE_SWITCHED_OFF == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_OFF == switchStatus._switchState))
               || ((TARGET_SWITCH_STATE_SWITCHED_ON == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_ON == switchStatus._switchState)))
         {
            _connectableModeReset = true;
            this->checkProgress();
         }
         else
         {
            bmResult = LocalSpm::getBmCoreMainController().switchLocalConnectableModeInt(defaultTargetSwitchState, "");

            if (BM_RESULT_OK != bmResult)
            {
               ETG_TRACE_ERR(("enterResettingToDefault: could not switch LocalConnectableMode to %d (bmResult = %d)", ETG_CENUM(TargetSwitchState, defaultTargetSwitchState), ETG_CENUM(BmResult, bmResult)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("enterResettingToDefault: could not get LocalConnectableMode (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
         _resetToDefaultResult = BM_RESET_RESULT_ERROR;
      }

      ETG_TRACE_USR1(("enterResettingToDefault: resetting BtStatus"));

      defaultTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultBtStatus;

      bmResult = LocalSpm::getBmCoreMainController().getBtStatus(switchStatus);

      if (BM_RESULT_OK == bmResult)
      {
         if (((TARGET_SWITCH_STATE_SWITCHED_OFF == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_OFF == switchStatus._switchState))
               || ((TARGET_SWITCH_STATE_SWITCHED_ON == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_ON == switchStatus._switchState))
               || (false == LocalSpm::getDataProvider().getBmCoreConfiguration()._btOnOffSupport))
         {
            _btStatusReset = true;
            this->checkProgress();
         }
         else
         {
//            bmResult = LocalSpm::getBmController().doSwitchBtStatus(defaultTargetSwitchState, false, true);
            bmResult = LocalSpm::getBmCoreMainController().switchBtStatusInt(defaultTargetSwitchState, false, true);

            if (BM_RESULT_OK != bmResult)
            {
               ETG_TRACE_ERR(("enterResettingToDefault: could not switch BtStatus to %d (bmResult = %d)", ETG_CENUM(TargetSwitchState, defaultTargetSwitchState), ETG_CENUM(BmResult, bmResult)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("enterResettingToDefault: could not get BtStatus (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
         _resetToDefaultResult = BM_RESET_RESULT_ERROR;
      }

      ETG_TRACE_USR1(("enterResettingToDefault: resetting MultiHFPSupportStatus"));

      defaultTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultmultiHFPSupport;

      bmResult = LocalSpm::getBmCoreMainController().getMultiHFPSupportStatus(switchStatus);

      if (BM_RESULT_OK == bmResult)
      {
         if (((TARGET_SWITCH_STATE_SWITCHED_OFF == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_OFF == switchStatus._switchState))
               || ((TARGET_SWITCH_STATE_SWITCHED_ON == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_ON == switchStatus._switchState)))
         {
            _multiHFPSupportReset = true;
            this->checkProgress();
         }
         else
         {
            bmResult = LocalSpm::getBmCoreMainController().switchMultiHFPSupportStatusInt(defaultTargetSwitchState);

            if (BM_RESULT_OK != bmResult)
            {
               ETG_TRACE_ERR(("enterResettingToDefault: could not switch MultiHFPSupportStatus to %d (bmResult = %d)", ETG_CENUM(TargetSwitchState, defaultTargetSwitchState), ETG_CENUM(BmResult, bmResult)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("enterResettingToDefault: could not get MultiHFPSupportStatus (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
         _resetToDefaultResult = BM_RESET_RESULT_ERROR;
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ResetToDefaultController::handlePairableModeChange(IN const SwitchState localPairableModeSwitchState)
   {
      ENTRY

      ETG_TRACE_USR1(("handlePairableModeChange: localPairableModeSwitchState = %d", ETG_CENUM(SwitchState, localPairableModeSwitchState)));

      if (true == _pairableModeReset)
      {
         ETG_TRACE_USR1(("handlePairableModeChange(localPairableModeSwitchState = %d): pairable mode changed after it had been reset to default", ETG_CENUM(SwitchState, localPairableModeSwitchState)));
      }
      else
      {
         TargetSwitchState defaultTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode;

         if (((TARGET_SWITCH_STATE_SWITCHED_OFF == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_OFF == localPairableModeSwitchState))
               || ((TARGET_SWITCH_STATE_SWITCHED_ON == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_ON == localPairableModeSwitchState)))
         {
            _pairableModeReset = true;
            this->checkProgress();
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ResetToDefaultController::handleConnectableModeChange(IN const SwitchState localConnectableModeSwitchState)
   {
      ENTRY

      ETG_TRACE_USR1(("handleConnectableModeChange: localConnectableModeSwitchState = %d", ETG_CENUM(SwitchState, localConnectableModeSwitchState)));

      if (true == _connectableModeReset)
      {
         ETG_TRACE_USR1(("handleConnectableModeChange(localConnectableModeSwitchState = %d): connectable mode changed after it had been reset to default", ETG_CENUM(SwitchState, localConnectableModeSwitchState)));
      }
      else
      {
         TargetSwitchState defaultTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode;

         if (((TARGET_SWITCH_STATE_SWITCHED_OFF == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_OFF == localConnectableModeSwitchState))
               || ((TARGET_SWITCH_STATE_SWITCHED_ON == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_ON == localConnectableModeSwitchState)))
         {
            _connectableModeReset = true;
            this->checkProgress();
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ResetToDefaultController::handleBtStatusChange(IN const SwitchState btStatusSwitchState)
   {
      ENTRY

      ETG_TRACE_USR1(("handleBtStatusChange: btStatusSwitchState = %d", ETG_CENUM(SwitchState, btStatusSwitchState)));

      if (true == _btStatusReset)
      {
         ETG_TRACE_USR1(("handleBtStatusChange(btStatusSwitchState = %d): BT Status changed after it had been reset to default", ETG_CENUM(SwitchState, btStatusSwitchState)));
      }
      else
      {
         TargetSwitchState defaultTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultBtStatus;

         if (((TARGET_SWITCH_STATE_SWITCHED_OFF == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_OFF == btStatusSwitchState))
               || ((TARGET_SWITCH_STATE_SWITCHED_ON == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_ON == btStatusSwitchState)))
         {
            _btStatusReset = true;
            this->checkProgress();
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ResetToDefaultController::handleMultiHFPSupportStatusChange(IN const SwitchState multiHFPSupportStatusSwitchState)
   {
      ENTRY

      ETG_TRACE_USR1(("handleMultiHFPSupportStatusChange: multiHFPSupportStatusSwitchState = %d", ETG_CENUM(SwitchState, multiHFPSupportStatusSwitchState)));

      if (true == _multiHFPSupportReset)
      {
         ETG_TRACE_USR1(("handleMultiHFPSupportStatusChange(multiHFPSupportStatusSwitchState = %d): multiHFPSupport Status changed after it had been reset to default", ETG_CENUM(SwitchState, multiHFPSupportStatusSwitchState)));
      }
      else
      {
         TargetSwitchState defaultTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultmultiHFPSupport;

         if (((TARGET_SWITCH_STATE_SWITCHED_OFF == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_OFF == multiHFPSupportStatusSwitchState))
               || ((TARGET_SWITCH_STATE_SWITCHED_ON == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_ON == multiHFPSupportStatusSwitchState)))
         {
            _multiHFPSupportReset = true;
            this->checkProgress();
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   void ResetToDefaultController::create()
   {
      ENTRY

      ETG_TRACE_USR1(("create"));

      /* Create the state machine */
      ResetToDefaultControllerSm::Create();

      createDone(0);
   }

   Result ResetToDefaultController::init(InitReason reason)
   {
      ENTRY

      ETG_TRACE_USR1(("init: reason = %d", reason));

      (void) reason;

      /* Init the state machine */
      ResetToDefaultControllerSm::Init();
      SetAnswerTimeout(2000);

      /* Register state machine with dispatcher */
      Dispatcher::GetInstance().Register(IN this);

      return initDone(0);
   }

   Result ResetToDefaultController::run()
   {
      ENTRY

      ETG_TRACE_USR1(("run"));

      return runDone(0);
   }

   Result ResetToDefaultController::stop()
   {
      ENTRY

      ETG_TRACE_USR1(("stop"));

      Result result(CC_ERR_INT_NO_ERROR);

      /* Send STOP message to own SM */
      result = this->SendForceEvent(STOP_SM, (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("stop: could not send event STOP_SM (error = %d)", result));
      }

      return result;
   }

   Result ResetToDefaultController::done()
   {
      ENTRY

      ETG_TRACE_USR1(("done"));

      Result result(CC_ERR_INT_NO_ERROR);

      /* Deregister state machine with dispatcher */
      Dispatcher::GetInstance().DeRegister(IN this);

      /* Send DONE message to own SM */
      result = this->SendForceEvent(DONE, (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("done: could not send event DONE (error = %d)", result));
      }

      return result;
   }

   char* ResetToDefaultController::getSmStateName(OUT tGeneralString stateName, size_t size)
   {
      ENTRY

      ETG_TRACE_USR1(("getSmStateName"));

      this->GetCurrentState((char *) stateName, size);

      return stateName;
   }

   Result ResetToDefaultController::initMembers()
   {
      ENTRY

      ETG_TRACE_USR1(("initMembers"));

      _factorySettingsChanged = false;
      _pairableModeReset = false;
      _connectableModeReset = false;
      _btStatusReset = false;
      _multiHFPSupportReset = false;
      _resetToDefaultResult = BM_RESET_RESULT_OK;

      return CC_ERR_INT_NO_ERROR;
   }

   void ResetToDefaultController::checkProgress(void)
   {
      ENTRY

      ETG_TRACE_USR1(("checkProgress"));

      if ((true == _pairableModeReset) && (true == _connectableModeReset) && (true == _btStatusReset) && (true == _multiHFPSupportReset))
      {
         DeviceIdList deviceIdList;
         Result result = LocalSpm::getDbManager().getAllDeviceIds(deviceIdList);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR1(("checkProgress: number of paired devices: %u", deviceIdList.size()));

            if (false == deviceIdList.empty())
            {
               ETG_TRACE_ERR(("checkProgress: still paired devices available"));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkProgress: could not get all device IDs from DB (error = %d)", ETG_CENUM(CcErrorInternal, result)));
            _resetToDefaultResult = BM_RESET_RESULT_ERROR;
         }

         if (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._useFixedPinLegacyPairing)
         {
            PairingPinType legacyPairingPin("");
            result = LocalSpm::getDbManager().getLegacyPairingPin(legacyPairingPin);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               PairingPinType expectedPairingPin("");

               if (false == _factorySettingsChanged)
               {
                  expectedPairingPin = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultFixedPinLegacyPairing;
               }

               if (expectedPairingPin != legacyPairingPin)
               {
                  ETG_TRACE_ERR(("checkProgress: legacy pairing PIN stored in DB (= \"%50s\") differs from default value = \"%50s\"", legacyPairingPin.c_str(), expectedPairingPin.c_str()));
                  _resetToDefaultResult = BM_RESET_RESULT_ERROR;
               }
            }
            else
            {
               ETG_TRACE_ERR(("checkProgress: could not get legacy pairing PIN from DB (error = %d)", ETG_CENUM(CcErrorInternal, result)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }

         PairingPin pairingPin;
         BmResult bmResult = LocalSpm::getBmCoreMainController().getPairingPin(pairingPin);

         if (BM_RESULT_OK == bmResult)
         {
            PairingPinType expectedPairingPin("");

            if (false == _factorySettingsChanged)
            {
               expectedPairingPin = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultFixedPinLegacyPairing;
            }

            if (expectedPairingPin != pairingPin._pin)
            {
               ETG_TRACE_ERR(("checkProgress: legacy pairing PIN in PairingPin property (= \"%50s\") differs from default value = \"%50s\"", pairingPin._pin.c_str(), expectedPairingPin.c_str()));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkProgress: could not get PairingPin (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
            _resetToDefaultResult = BM_RESET_RESULT_ERROR;
         }

         StandardAutoConnectionType standardAutoConnectionType;
         result = LocalSpm::getDbManager().getStandardAutoConnectionType(standardAutoConnectionType);

         StandardAutoConnectionType expectedStandardAutoConnectionType(BM_STANDARD_AUTO_CONNECTION_UNKNOWN);

         if (false == _factorySettingsChanged)
         {
            expectedStandardAutoConnectionType = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultStandardAutoConnectionType;
         }

         if (CC_ERR_INT_NO_ERROR == result)
         {
            if (expectedStandardAutoConnectionType != standardAutoConnectionType)
            {
               ETG_TRACE_ERR(("checkProgress: standard auto connection type stored in DB (= %d) differs from default value = %d",
                     ETG_CENUM(StandardAutoConnectionType, standardAutoConnectionType), ETG_CENUM(StandardAutoConnectionType, expectedStandardAutoConnectionType)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkProgress: could not get standard auto connection type from DB (error = %d)", ETG_CENUM(CcErrorInternal, result)));
            _resetToDefaultResult = BM_RESET_RESULT_ERROR;
         }

         AutoConnectionType autoConnectionType;
         bmResult = LocalSpm::getBmCoreMainController().getAutoConnectionType(autoConnectionType);

         if (BM_RESULT_OK == bmResult)
         {
            if (expectedStandardAutoConnectionType != autoConnectionType._type)
            {
               ETG_TRACE_ERR(("checkProgress: standard auto connection type stored in AutoConnectionType property  (= %d) differs from default value = %d",
                     ETG_CENUM(StandardAutoConnectionType, standardAutoConnectionType), ETG_CENUM(StandardAutoConnectionType, expectedStandardAutoConnectionType)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkProgress: could not get AutoConnectionType (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
            _resetToDefaultResult = BM_RESET_RESULT_ERROR;
         }

         LocalFriendlyNameType bdName("");
         result = LocalSpm::getDbManager().getLocalBdName(bdName);

         LocalFriendlyNameType expectedBdName("");

         if (false == _factorySettingsChanged)
         {
            expectedBdName = LocalSpm::getDataProvider().getBmCoreConfiguration()._btLocalFriendlyName;
         }

         if (CC_ERR_INT_NO_ERROR == result)
         {
            if(true == isValidLocalBdName(bdName))
            {
               if (expectedBdName != bdName)
               {
                  ETG_TRACE_ERR(("checkProgress: local BD name stored in DB (= \"%50s\") differs from default value = \"%50s\"", bdName.c_str(), expectedBdName.c_str()));
                  _resetToDefaultResult = BM_RESET_RESULT_ERROR;
               }
            }
            else
            {
               ETG_TRACE_USR1(("checkProgress: local BD name stored in DB (= \"%50s\") is internally changed", bdName.c_str()));
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkProgress: could not get local BD name from DB (error = %d)", ETG_CENUM(CcErrorInternal, result)));
            _resetToDefaultResult = BM_RESET_RESULT_ERROR;
         }

         LocalFriendlyName localFriendlyName;
         bmResult = LocalSpm::getBmCoreMainController().getBtLocalFriendlyName(localFriendlyName);

         if (BM_RESULT_OK == bmResult)
         {
            if (expectedBdName != localFriendlyName._localFriendlyName)
            {
               ETG_TRACE_ERR(("checkProgress: local BD name in BtLocalFriendlyName property (= \"%50s\") differs from default value = \"%50s\"", localFriendlyName._localFriendlyName.c_str(), expectedBdName.c_str()));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkProgress: could not get BtLocalFriendlyName (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
            _resetToDefaultResult = BM_RESET_RESULT_ERROR;
         }

         TargetSwitchState defaultTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode;

         SwitchStatus switchStatus;
         bmResult = LocalSpm::getBmCoreMainController().getLocalPairableMode(switchStatus);

         if (BM_RESULT_OK == bmResult)
         {
            if (((TARGET_SWITCH_STATE_SWITCHED_OFF == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_OFF != switchStatus._switchState))
                  || ((TARGET_SWITCH_STATE_SWITCHED_ON == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_ON != switchStatus._switchState)))
            {
               ETG_TRACE_ERR(("checkProgress: switch state in LocalPairableMode property (= %d) differs from default value = %d",
                     ETG_CENUM(SwitchState, switchStatus._switchState), ETG_CENUM(SwitchState, LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkProgress: could not get LocalPairableMode (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
            _resetToDefaultResult = BM_RESET_RESULT_ERROR;
         }

         defaultTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode;
         bmResult = LocalSpm::getBmCoreMainController().getLocalConnectableMode(switchStatus);

         if (BM_RESULT_OK == bmResult)
         {
            if (((TARGET_SWITCH_STATE_SWITCHED_OFF == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_OFF != switchStatus._switchState))
                  || ((TARGET_SWITCH_STATE_SWITCHED_ON == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_ON != switchStatus._switchState)))
            {
               ETG_TRACE_ERR(("checkProgress: switch state in LocalConnectableMode property (= %d) differs from default value = %d",
                     ETG_CENUM(SwitchState, switchStatus._switchState), ETG_CENUM(SwitchState, LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkProgress: could not get LocalConnectableMode (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
            _resetToDefaultResult = BM_RESET_RESULT_ERROR;
         }

         defaultTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultBtStatus;
         bmResult = LocalSpm::getBmCoreMainController().getBtStatus(switchStatus);

         if (BM_RESULT_OK == bmResult)
         {
            if ((((TARGET_SWITCH_STATE_SWITCHED_OFF == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_OFF != switchStatus._switchState))
                  || ((TARGET_SWITCH_STATE_SWITCHED_ON == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_ON != switchStatus._switchState)))
                  && (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._btOnOffSupport))
            {
               ETG_TRACE_ERR(("checkProgress: switch state in BtStatus property (= %d) differs from default value = %d",
                     ETG_CENUM(SwitchState, switchStatus._switchState), ETG_CENUM(SwitchState, LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultBtStatus)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkProgress: could not get BtStatus (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
            _resetToDefaultResult = BM_RESET_RESULT_ERROR;
         }

         defaultTargetSwitchState = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultmultiHFPSupport;
         bmResult = LocalSpm::getBmCoreMainController().getMultiHFPSupportStatus(switchStatus);

         if (BM_RESULT_OK == bmResult)
         {
            if (((TARGET_SWITCH_STATE_SWITCHED_OFF == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_OFF != switchStatus._switchState))
                  || ((TARGET_SWITCH_STATE_SWITCHED_ON == defaultTargetSwitchState) && (SWITCH_STATE_SWITCHED_ON != switchStatus._switchState)))
            {
               ETG_TRACE_ERR(("checkProgress: switch state in MultiHFPSupportStatus property (= %d) differs from default value = %d",
                     ETG_CENUM(SwitchState, switchStatus._switchState), ETG_CENUM(SwitchState, LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultmultiHFPSupport)));
               _resetToDefaultResult = BM_RESET_RESULT_ERROR;
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkProgress: could not get MultiHFPSupportStatus (bmResult = %d) ", ETG_CENUM(BmResult, bmResult)));
            _resetToDefaultResult = BM_RESET_RESULT_ERROR;
         }

         ETG_TRACE_USR4(("checkProgress: sending event RESETTING_FINISHED to ResetToDefaultControllerSm"));
         result = this->SendEventByName("RESETTING_FINISHED", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("checkProgress: could not send event RESETTING_FINISHED (error = %d) ", ETG_CENUM(CcErrorInternal, result)));
            _resetToDefaultResult = BM_RESET_RESULT_ERROR;
         }
      }
   }
}
