/**
 * @defgroup ResetToDefaultControllerModule Reset to Default Controller
 * @ingroup BmCoreModule
 *
 * @brief This module comprises all parts related to
 */

/**
 * @file
 * @ingroup ResetToDefaultControllerModule
 *
 * @brief This file contains the declaration of the class ResetToDefaultController
 *
 * @author Stefan Reis
 */

#ifndef _RESET_TO_DEFAULT_CONTROLLER_H_
#define _RESET_TO_DEFAULT_CONTROLLER_H_

#include "ResetToDefaultControllerSm.h"
#include "ILocalSpm.h"
#include "Lock.h"

namespace bmcore
{
   /**
    * @class ResetToDefaultController
    * @ingroup ResetToDefaultControllerModule
    *
    * @brief The class ResetToDefaultController implements the state machine ResetToDefaultControllerSm
    *
    */
   class ResetToDefaultController : public ResetToDefaultControllerSm, public ILocalSpm
   {

   public:
      /**
       * @brief Constructor of class ResetToDefaultController which creates an instance
       *
       * No further initialization of internal components and resources
       * is performed in the constructor. See the explicit allocation
       * and initialization functions Create() and Init(). Hence, the
       * implementation is trivial.
       *
       * Synchronization: not re-entrant.
       *
       * Performance considerations: none.
       *
       * @return void
       */
      ResetToDefaultController(ComponentId componentID);

      /**
       * @brief Destructor of class PairingController
       *
       * @return void
       */
      virtual ~ResetToDefaultController();

      Result resetToDefault(IN const bool factorySettingsChanged);

      // ************************************
      // * Overridden methods of ResetToDefaultControllerSm *
      // ************************************

      Result initSm();

      Result handleStopSm();

      Result handleDone();

      Result enterIdle();

      Result enterDeletingAllDevices();

      Result enterResettingToDefault();

      Result handlePairableModeChange(IN const SwitchState localPairableModeSwitchState);

      Result handleConnectableModeChange(IN const SwitchState localConnectableModeSwitchState);

      Result handleBtStatusChange(IN const SwitchState btStatusSwitchState);

      Result handleMultiHFPSupportStatusChange(IN const SwitchState multiHFPSupportStatusSwitchState);

      // ************************************
      // * Overridden methods of ILocalSpm *
      // ************************************

      /**
       * @brief Creates the ResetToDefaultController instance
       *
       * Creates the ResetToDefaultController state machine (including creation of message queue).
       * Informs LocalSpm that create is ready -> createDone(0).
       *
       * @attention: running in SPM thread context
       *
       * @return void
       */
      void create();

      /**
       * @brief Performs the initialization of the ResetToDefaultController instance, i.e.:
       * - initialization of member variables
       *
       * Synchronization: re-entrant
       *
       * Performance considerations:
       *
       * @return 0 on success and an error code otherwise.
       */
      Result init(InitReason reason);

      /**
       * @brief Starts the ResetToDefaultController instance, implies that all other
       * components are available
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none.
       *
       * @return 0 on success and an error code otherwise
       */
      Result run();

      /**
       * @brief Stops the ResetToDefaultController instance
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none
       *
       * @return 0 on success and an error code otherwise
       */
      Result stop();

      /**
       * @brief Cleans up the resources used by the ResetToDefaultController instance
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none
       *
       * @return 0 on success and an error code otherwise
       */
      Result done();

      bool isComponentSm() {return true;};

      /**
       * Returns the current state the state machine is in (for debugging of shutdown problems)
       *
       * @param[in,out] stateName buffer for storing the current state name
       * @param[in] size size of the stateName buffer
       *
       * @return pointer to stateName
       */
      char *getSmStateName(OUT tGeneralString stateName, IN size_t size);

      /**
       * @brief Returns statistic data as a string for the over all object statistics
       *
       * @param[out] stat string with statistics information to be printed
       *
       * @return 0 on success and an error code otherwise
       */
   //   int Statistics(OUT Statistics stat);

   private:

      /**
       * @brief Initializes members
       *
       * @return 0: success
       */
      Result initMembers();

      void checkProgress(void);

      bool _firstEntryInStateIdle;

      bool _factorySettingsChanged;

      bool _pairableModeReset;
      bool _connectableModeReset;
      bool _btStatusReset;
      bool _multiHFPSupportReset;

      ResetToDefaultResult _resetToDefaultResult;
   };
}

#endif // _RESET_TO_DEFAULT_CONTROLLER_H_

