#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_PCCONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ProtocolConnectionController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_PCCONTROLLER
#endif
#endif

#include "FunctionTracer.h"
#include "LocalSpm.h"
#include "Dispatcher.h"
#include "ProtocolConnectionController.h"
#include "DeviceConnectionController.h"
#include "RequestResponseSM.h"
#include "BmUtils.h"
#include "IBtStackIfConnectionRequest.h"
#include "BmGlobalLock.h"
#include <algorithm>
#include <sstream>
#include "BmCoreIfMessagesCreator.h"

namespace bmcore
{
   ProtocolConnectionController::ProtocolConnectionController(const ProtocolId protocolId) :
         _protocolId(protocolId), _uuid(""), _deviceId(0u), _rfcommDevicePath(""),
         _disconnectedReason(BM_DISCONNECTED_REASON_UNKNOWN),_btsConnectionStatus(BM_CONNECTION_STATUS_DISCONNECTED),
         _btsDisconnectedReason(BM_DISCONNECTED_REASON_UNKNOWN), _btsRfcommDevicePath(""),_numPendingBtsConnDiscRequests(0u),
         _delayConnectingTimer(), _delayConnectingTimerId(), _disconnectingTimer(), _disconnectingTimerId(),
         _expiredTimerSmEvent(this->GetSMNameFull()), _canBeReleased(true), _pageTimeout(0u)
   {
      ETG_TRACE_USR1(("ProtocolConnectionController(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): being created",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      // threadless SM
      disableAllLocks(true);
   }

   ProtocolConnectionController::~ProtocolConnectionController()
   {
      ETG_TRACE_USR1(("~ProtocolConnectionController(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): being destroyed",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      _delayConnectingTimerId = 0;
      _disconnectingTimerId = 0;
   }

   Result ProtocolConnectionController::assign(IN const DeviceId deviceId, IN const Uuid& uuid, IN const PageTimeout pageTimeout)
   {
      ENTRY

      ETG_TRACE_USR1(("assign(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): assigning PCC instance to deviceId = %d, uuid = \"%50s\", pageTimeout = %d",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), deviceId, uuid.c_str(), pageTimeout));

      Result result(CC_ERR_INT_NO_ERROR);

      if (0u == _deviceId)
      {
         _deviceId = deviceId;
         _uuid = uuid;
         _pageTimeout = pageTimeout;
      }
      else if (deviceId == _deviceId)
      {
         ETG_TRACE_USR1(("assign(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): PCC instance is already assigned to the given device", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

         _canBeReleased = false;
      }
      else
      {
         result = CC_ERR_INT_PCC_ALREADY_ASSIGNED;
         ETG_TRACE_ERR(("assign(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): PCC instance is already assigned to a different device (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result ProtocolConnectionController::release(void)
   {
      ENTRY

      ETG_TRACE_USR1(("release(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);

      if (0u != _deviceId)
      {
         if(true == _canBeReleased)
         {
            (void) initMembers();
         }
         else
         {
            ETG_TRACE_USR1(("release(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): there are still events waiting to be processed, not requesting for releasing this PCC instance",
                  (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));
         }
      }
      else
      {
         result = CC_ERR_INT_PCC_NOT_ASSIGNED;
         ETG_TRACE_ERR(("release(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): PCC instance is currently not assigned to any device (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   void ProtocolConnectionController::updateNumPendingBtsConnDiscRequests(const bool increment)
   {
      ETG_TRACE_USR1(("updateNumPendingBtsConnDiscRequests(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): %s number",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), increment ? "increment" : "decrement"));

      if (true == increment)
      {
         _numPendingBtsConnDiscRequests++;
      }
      else
      {
         if (0u < _numPendingBtsConnDiscRequests)
         {
            _numPendingBtsConnDiscRequests--;
         }
         else
         {
            ETG_TRACE_ERR(("updateNumPendingBtsConnDiscRequests(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): number of pending BTS connection/disconnection requests is already 0",
                  (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));
         }
      }

      this->checkNumPendingBtsConnDiscRequests();
   }

   void ProtocolConnectionController::updateProtocolConnectionStatus(const ConnectionStatus connectionStatus,
         const RfcommDevicePath& rfCommDevicePath, const DisconnectedReason disconnectedReason)
   {
      ETG_TRACE_USR1(("updateProtocolConnectionStatus(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): connectionStatus = %d, rfCommDevicePath = %50s, disconnectedReason = %d",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(),
            ETG_CENUM(ConnectionStatus, connectionStatus), rfCommDevicePath.c_str(), ETG_CENUM(DisconnectedReason, disconnectedReason)));

      _btsConnectionStatus = connectionStatus;
      _btsRfcommDevicePath = rfCommDevicePath;
      _btsDisconnectedReason = disconnectedReason;

      this->checkNumPendingBtsConnDiscRequests();
   }

   void ProtocolConnectionController::create()
   {
      ENTRY

      ETG_TRACE_USR1(("create(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      /* Create the state machine */
      this->Create();
   }

   Result ProtocolConnectionController::init()
   {
      ENTRY

      ETG_TRACE_USR1(("init(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      /* Init the state machine */
      this->Init();
      SetAnswerTimeout(2000);

      /* Register state machine with dispatcher */
      Dispatcher::GetInstance().Register(IN this);

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::run()
   {
      ENTRY

      ETG_TRACE_USR1(("run(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId,
            ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::stop()
   {
      ENTRY

      ETG_TRACE_USR1(("stop(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      ETG_TRACE_USR4(("stop(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): sending event STOP_SM to myself",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result = this->SendForceEvent(ProtocolConnectionController::STOP_SM, (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("stop(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not send event STOP_SM to myself (error = %d)",
               (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(),
               ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::done()
   {
      ENTRY

      ETG_TRACE_USR1(("done(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);

      /* Deregister state machine with dispatcher */
      Dispatcher::GetInstance().DeRegister(IN this);

      /* Send DONE message to own SM */
      result = this->SendForceEvent(ProtocolConnectionController::DONE, (char *) NULL);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("done(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not send event DONE (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result ProtocolConnectionController::initSm()
   {
      ENTRY

      ETG_TRACE_USR1(("initSm(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      (void) initMembers();

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::handleStopSm()
   {
      ENTRY

      ETG_TRACE_USR1(("handleStopSm(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      ETG_TRACE_USR4(("handleStopSm(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): sending event CONN_CTRL_STOPPED to BmControllerOnOffSm",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result = LocalSpm::getBmController().SendEventByName("CONN_CTRL_STOPPED", (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleStopSm(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not send event CONN_CTRL_STOPPED to BmControllerOnOffSm (error = %d)",
               (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(),
               ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::handleDone()
   {
      ENTRY

      ETG_TRACE_USR1(("handleDone(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      ETG_TRACE_USR4(("handleDone(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): sending event CONN_CTRL_TERMINATED to BmControllerOnOffSm",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result = LocalSpm::getBmController().SendEventByName("CONN_CTRL_TERMINATED", (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleDone(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not send event CONN_CTRL_TERMINATED to BmControllerOnOffSm (error = %d)",
               (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(),
               ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::enterDisconnected()
   {
      ENTRY

      ETG_TRACE_USR1(("enterDisconnected(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

     //Result result(CC_ERR_INT_NO_ERROR);

      if (0u != _deviceId)
      {
         if (true == _canBeReleased)
         {
            (void) this->requestReleaseIfPossible();
         }
         else
         {
            ETG_TRACE_USR1(("enterDisconnected(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): not requesting for releasing this PCC instance",
                  (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::sendConnectLocal()
   {
      ENTRY

      ETG_TRACE_USR1(("sendConnectLocal(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      ETG_TRACE_USR4(("sendConnectLocal(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): sending event CONNECT_LOCAL to myself",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result = this->SendEventByName("CONNECT_LOCAL", 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("sendConnectLocal(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not send event CONNECT_LOCAL (error = %d)",
               (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::requestReleaseIfPossible()
   {
      ENTRY

      ETG_TRACE_USR1(("requestReleaseIfPossible(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);

      if (false == this->areAnyMessagesWaiting())
      {
         DeviceConnectionController* dccInstance(0);

         result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, _deviceId);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            //copy member variables before they get cleared by releaseProtocolConnectionController
//            DeviceId tmpDevId(_deviceId);
//            ProtocolId tmpProtocolId(_protocolId);
//            Uuid tmpUuid(_uuid);
            DisconnectedReason tmpDisconnectedReason(_disconnectedReason);

            if (BM_DISCONNECTED_REASON_UNKNOWN == tmpDisconnectedReason)
            {
               tmpDisconnectedReason = BM_DISCONNECTED_REASON_NORMAL_LOSS_LOCAL;
            }
/*
            result = LocalSpm::getDbManager().setProtocolConnectionStatus(tmpDevId, tmpProtocolId, tmpUuid, BM_CONNECTION_STATUS_DISCONNECTED, tmpDisconnectedReason, "");

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("requestReleaseIfPossible(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not set protocol connection status (error = %d)",
                     (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
            }
*/
            result = dccInstance->releaseProtocolConnectionController(_protocolId, _uuid, tmpDisconnectedReason);

            if (CC_ERR_INT_NO_ERROR == result)
            {
/*               result = LocalSpm::getDbManager().setProtocolConnectionStatus(tmpDevId, tmpProtocolId, tmpUuid, BM_CONNECTION_STATUS_DISCONNECTED, tmpDisconnectedReason, "");

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("requestReleaseIfPossible(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not set protocol connection status (error = %d)",
                        (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
               }
*/
            }
            else
//            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("requestReleaseIfPossible(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): releasing the protocol connection controller failed",
                     (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));
            }
         }
         else
         {
            ETG_TRACE_ERR(("requestReleaseIfPossible(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not get DCC instance (error = %d)",
                  (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_USR1(("requestReleaseIfPossible(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): there are still events waiting to be processed, not requesting for releasing this PCC instance",
                     (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

         result = LocalSpm::getDbManager().setProtocolConnectionStatus(_deviceId, _protocolId, _uuid, BM_CONNECTION_STATUS_DISCONNECTED, _disconnectedReason, "");

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("requestReleaseIfPossible(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not set protocol connection status (error = %d)",
                  (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::handleDisconnectedInDisconnected(IN const DisconnectedReason disconnectedReason)
   {
      ENTRY

      ETG_TRACE_USR1(("handleDisconnectedInDisconnected(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): disconnectedReason = %d",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(),
            ETG_CENUM(DisconnectedReason, disconnectedReason)));

      (void) this->saveDisconnectedReason(disconnectedReason);

      (void) this->requestReleaseIfPossible();

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::connectingToBeDelayed(void)
   {
      ENTRY

      ETG_TRACE_USR1(("connectingToBeDelayed(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      DeviceConnectionController* dccInstance(0);

      Result result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, _deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         if (true == dccInstance->areLocalProtocolConnectionsToBeDelayed())
         {
            ETG_TRACE_USR1(("connectingToBeDelayed(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): locally initiated connecting has to be delayed", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

            result = 1;
         }
         else
         {
            ETG_TRACE_USR1(("connectingToBeDelayed(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): locally initiated connecting has not to be delayed", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

            result = 0;
         }
      }
      else
      {
         ETG_TRACE_ERR(("connectingToBeDelayed(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not get DCC instance (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));

         result = 0;
      }

      return result;
   }

   Result ProtocolConnectionController::enterDelayConnecting()
   {
      ENTRY

      ETG_TRACE_USR1(("enterDelayConnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);

      _expiredTimerSmEvent += "::CONTINUE_CONNECTING";

      if (true == _delayConnectingTimer.StartSMTimer(_delayConnectingTimerId,
            1000L * static_cast<long>(LocalSpm::getDataProvider().getBmCoreConfiguration()._localProtocolConnectingDelaySeconds),
            0L, _expiredTimerSmEvent.c_str(), this))
      {
         char milliSecondsStr[200];
         snprintf(milliSecondsStr, 199, "%llu", 1000L * static_cast<long long unsigned>(LocalSpm::getDataProvider().getBmCoreConfiguration()._localProtocolConnectingDelaySeconds));
         ETG_TRACE_USR1(("enterDelayConnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): started delay connecting timer (timer ID = 0x%p, intervalMilliseconds = %50s)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), _delayConnectingTimerId, milliSecondsStr));
      }
      else
      {
         ETG_TRACE_ERR(("enterDelayConnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): starting delay connecting timer failed", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

         ETG_TRACE_USR4(("enterDelayConnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): sending event CONTINUE_CONNECTING to myself", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));
         result = this->SendEventByName("CONTINUE_CONNECTING", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("enterDelayConnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not send event CONTINUE_CONNECTING (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::exitDelayConnecting()
   {
      ENTRY

      ETG_TRACE_USR1(("exitDelayConnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      _delayConnectingTimer.CancelTimer(_delayConnectingTimerId);

      _expiredTimerSmEvent = this->GetSMNameFull();

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::connect()
   {
      ENTRY

      ETG_TRACE_USR1(("connect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);
      BdAddress bdAddress("");

      _canBeReleased = true;

      result = LocalSpm::getDbManager().getBdAddress(bdAddress, _deviceId);

      if(CC_ERR_INT_NO_ERROR == result)
      {
         BTSProtocolId btsProtocolId(BTS_PROTO_LAST);

         result = getBtsProtocolIdFromBmProtocolId(btsProtocolId, _protocolId);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            if (BM_PROTOCOL_ID_SPP == _protocolId)
            {
               UuidList supportedUuids;
               result = LocalSpm::getDbManager().getDeviceSppUuidSupport(OUT supportedUuids, IN _deviceId);

               if(CC_ERR_INT_NO_ERROR == result)
               {
                  if (std::find(supportedUuids.begin(), supportedUuids.end(), _uuid) == supportedUuids.end())
                  {
                     result = CC_ERR_INT_UUID_NOT_SUPPORTED;
                     ETG_TRACE_ERR(("connect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): UUID is not supported by device (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("connect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not get SPP UUIDs supported by device from DB (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
               }
            }

            if (CC_ERR_INT_NO_ERROR == result)
            {
               LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().connectProtocol(bdAddress, btsProtocolId, _uuid, "", _pageTimeout);

               this->updateNumPendingBtsConnDiscRequests(true);

               ProtocolList hmiUserSelectableProtocols;
               (void) LocalSpm::getDataProvider().getHmiUserSelectableProtocols(hmiUserSelectableProtocols);

               Protocol protocolToBeConnected(_protocolId, _uuid);

               if (hmiUserSelectableProtocols.end() != std::find(hmiUserSelectableProtocols.begin(),
                     hmiUserSelectableProtocols.end(), protocolToBeConnected))
               {
                  DeviceConnectionController* dccInstance(0);

                  Result result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, _deviceId);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     dccInstance->setLocalProtocolTargetConnectionStatusInMap(protocolToBeConnected, BM_TARGET_CONNECTION_STATUS_CONNECTED);

                     if (0u < LocalSpm::getDataProvider().getBmCoreConfiguration()._blockDeviceRemoteConnectionsTimeoutSeconds)
                     {
                        // block remote connections for the device currently handled by DCC
                        result = LocalSpm::getBmCoreMainController().blockRemoteConnections(_deviceId);

                        if (CC_ERR_INT_NO_ERROR != result)
                        {
                           ETG_TRACE_ERR(("connect(this = 0x%p, deviceId = %d): could not activate blocking of remote connection requests (error = %d)",
                                 (void *) this, _deviceId, ETG_CENUM(CcErrorInternal, result)));
                        }
                     }
                  }
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(("connect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not map BM's protocolId = %d to a corresponding BtStackIf protocol ID (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(ProtocolId, _protocolId), ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("connect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not get BD address associated with deviceId = %d from DB (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      if (CC_ERR_INT_NO_ERROR != result)
      {
         char eventParams[20];

         if (CC_ERR_INT_UUID_NOT_SUPPORTED == result)
         {
            result = ParameterDISCONNECTED(eventParams, sizeof(eventParams), BM_DISCONNECTED_REASON_CONNECTION_FAILED_UUID_NOT_SUPPORTED);
         }
         else
         {
            result = ParameterDISCONNECTED(eventParams, sizeof(eventParams), BM_DISCONNECTED_REASON_CONNECTION_FAILED);
         }

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("connect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): sending event DISCONNECTED to myself", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));
            result = this->SendEventByName("DISCONNECTED", eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("connect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not send event DISCONNECTED (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("connect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not marshal event parameters for event DISCONNECTED (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::accept()
   {
      ENTRY

      ETG_TRACE_USR1(("accept(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);
      BdAddress bdAddress("");

      _canBeReleased = true;

      result = LocalSpm::getDbManager().getBdAddress(bdAddress, _deviceId);

      if(CC_ERR_INT_NO_ERROR == result)
      {
         BTSProtocolId btsProtocolId(BTS_PROTO_LAST);
         result = getBtsProtocolIdFromBmProtocolId(btsProtocolId, _protocolId);

         if(CC_ERR_INT_NO_ERROR == result)
         {
            LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().acceptRemoteProtocolConnect(bdAddress, btsProtocolId, _uuid);
         }
         else
         {
            ETG_TRACE_ERR(("accept(this = 0x%p, deviceId = %d, protocolId = %d, uuid = \"%50s\"): could not map BM's protocolId = %d to a corresponding BtStackIf protocol ID (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(ProtocolId, _protocolId), ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("accept(this = 0x%p, deviceId = %d, protocolId = %d, uuid = \"%50s\"): could not get bdAddress associated with deviceId = %d from DbManager (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      if (CC_ERR_INT_NO_ERROR != result)
      {
         char eventParams[20];

         result = ParameterDISCONNECTED(eventParams, sizeof(eventParams), BM_DISCONNECTED_REASON_CONNECTION_FAILED);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("accept(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): sending event DISCONNECTED to myself", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));
            result = this->SendEventByName("DISCONNECTED", eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("accept(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not send event DISCONNECTED (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("accept(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not marshal event parameters for event DISCONNECTED (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::resetCanBeReleasedFlag()
   {
      ENTRY

      ETG_TRACE_USR1(("resetCanBeReleasedFlag(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      _canBeReleased = true;

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::enterConnecting()
   {
      ENTRY

      ETG_TRACE_USR1(("enterConnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);

      if (BM_PROTOCOL_ID_SPP == _protocolId)
      {
         // check the requested UUID is already available. If UUID is not available then added the UUID in DB and send the connection request.
         (void) checkAndUpdateUuid();
      }

      result = LocalSpm::getDbManager().setProtocolConnectionStatus(_deviceId, _protocolId, _uuid, BM_CONNECTION_STATUS_CONNECTING, BM_DISCONNECTED_REASON_NOT_APPLICABLE, "");

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("enterConnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not set protocol connection status (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      _disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::checkAndUpdateUuid()
   {
      ENTRY

      ETG_TRACE_USR1(("checkAndUpdateUuid(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);

      UuidList supportedUuids;
      result = LocalSpm::getDbManager().getDeviceSppUuidSupport(OUT supportedUuids, IN _deviceId);

      if(CC_ERR_INT_NO_ERROR == result)
      {
         if (std::find(supportedUuids.begin(), supportedUuids.end(), _uuid) != supportedUuids.end())
         {
            ETG_TRACE_USR4(("checkAndUpdateUuid(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): uuid is avilable",
                  (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));
         }
         else
         {
            supportedUuids.push_back(_uuid);

            BdAddress bdAddress("");
            result = LocalSpm::getDbManager().getBdAddress(bdAddress, _deviceId);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("getRemoteDevicesSppVersion: could not get device Address for given device Id = %d from DB (error = %d)",
                     _deviceId, ETG_CENUM(CcErrorInternal, result)));
            }
            else
            {
               // update supported SPP UUIDs in DB
               LocalSpm::getBmCoreMainController().updateSupportedSppUuid(bdAddress, supportedUuids);
            }
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::saveRfcommDevicePath(IN const RfcommDevicePathChrArr rfcommDevicePath)
   {
      ENTRY

      ETG_TRACE_USR1(("saveRfcommDevicePath(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): rfcommDevicePath = \"%50s\"", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), rfcommDevicePath));

      _rfcommDevicePath = rfcommDevicePath;

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::enterConnected()
   {
      ENTRY

      std::stringstream methodNameStream;
      methodNameStream << "enterConnected(this = 0x" << (void *) this << ", deviceId = " << _deviceId << ", protocol = " << ETG_CENUM(ProtocolId, _protocolId) << ", uuid =\"" << _uuid << "\")";
      std::string methodName = methodNameStream.str();

      ETG_TRACE_USR1(("%s", methodName.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);

      result = LocalSpm::getDbManager().setProtocolConnectionStatus(_deviceId, _protocolId, _uuid, BM_CONNECTION_STATUS_CONNECTED, BM_DISCONNECTED_REASON_NOT_APPLICABLE, _rfcommDevicePath);

      if (CC_ERR_INT_NO_ERROR == result)
      {

         (void) this->sendSmEventToDcc(methodName, "CONNECTED");
      }
      else
      {
         ETG_TRACE_ERR(("enterConnected(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not set protocol connection status (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::handleAlreadyConnected()
   {
      ENTRY

      std::stringstream methodNameStream;
      methodNameStream << "handleAlreadyConnected(this = 0x" << (void *) this << ", deviceId = " << _deviceId << ", protocol = " << ETG_CENUM(ProtocolId, _protocolId) << ", uuid =\"" << _uuid << "\")";
      std::string methodName = methodNameStream.str();

      ETG_TRACE_USR1(("%s", methodName.c_str()));

      _canBeReleased = true;

      (void) this->sendSmEventToDcc(methodName, "CONNECTED");

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::handleDisconnect(DisconnectedReason disconnectedReason)
   {
      ENTRY

      // Set the disconnected reason
      _disconnectedReason = disconnectedReason;

      ETG_TRACE_USR1(("handleDisconnect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): Set the Disconnected Reason = %d",
                  (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(DisconnectedReason, _disconnectedReason)));

      disconnect();

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::disconnect()
   {
      ENTRY

      ETG_TRACE_USR1(("disconnect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);

      BdAddress bdAddress("");
      result = LocalSpm::getDbManager().getBdAddress(bdAddress, _deviceId);

      if(CC_ERR_INT_NO_ERROR == result)
      {
         BTSProtocolId btsProtocolId(BTS_PROTO_LAST);
         result = getBtsProtocolIdFromBmProtocolId(btsProtocolId, _protocolId);

         if(CC_ERR_INT_NO_ERROR == result)
         {
            if(BM_DISCONNECTED_REASON_UNKNOWN == _disconnectedReason)
            {
               _disconnectedReason = BM_DISCONNECTED_REASON_NORMAL_LOSS_LOCAL;
            }

            bool pauseBtStreaming = true;

            if(BM_PROTOCOL_ID_AVP == _protocolId)
            {
               BtLimitationModeInfo btLimitationModeInfo;
               result = LocalSpm::getBmCoreMainController().getBtLimitationModeInfo(btLimitationModeInfo, bdAddress);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  bool isLimitationModeActive = ((BM_LIMITATION_FEATURE_CAR_PLAY == btLimitationModeInfo._limitationMode._limitationFeature) &&
                        ((BM_LIMITATION_STATE_ACTIVATING == btLimitationModeInfo._limitationState) ||
                              (BM_LIMITATION_STATE_ACTIVE == btLimitationModeInfo._limitationState)));

                  if(true == isLimitationModeActive)
                  {
                     pauseBtStreaming = false;
                  }
               }
               else
               {
                  result = CC_ERR_INT_NO_ERROR;
               }
            }

            LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().disconnectProtocol(static_cast<BTSBDAddress>(bdAddress), btsProtocolId, static_cast<BTSUuid>(_uuid), "", pauseBtStreaming);

            this->updateNumPendingBtsConnDiscRequests(true);
         }
         else
         {
            ETG_TRACE_ERR(("disconnect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not map BM's protocolId = %d to a corresponding BtStackIf protocol ID (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(ProtocolId, _protocolId), ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("disconnect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not get bdAddress associated with deviceId = %d from DbManager (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_USR4(("disconnect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): sending event DISCONNECTION_FAILED to myself",
               (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

         result = this->SendEventByName("DISCONNECTION_FAILED", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("disconnect(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not send event DISCONNECTION_FAILED (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::saveDisconnectedReason(IN const DisconnectedReason disconnectedReason)
   {
      ENTRY

      ETG_TRACE_USR1(("saveDisconnectedReason(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): disconnectedReason = %d",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(DisconnectedReason, disconnectedReason)));

      if (BM_DISCONNECTED_REASON_UNKNOWN == _disconnectedReason)
      {
         _disconnectedReason = disconnectedReason;
      }
      else
      {
         ETG_TRACE_USR1(("saveDisconnectedReason(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): ignoring given disconnectedReason since it has been already set internally to %d", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(DisconnectedReason, _disconnectedReason)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::enterDisconnecting()
   {
      ENTRY

      ETG_TRACE_USR1(("enterDisconnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      Result result(CC_ERR_INT_NO_ERROR);

      result = LocalSpm::getDbManager().setProtocolConnectionStatus(_deviceId, _protocolId, _uuid, BM_CONNECTION_STATUS_DISCONNECTING, BM_DISCONNECTED_REASON_NOT_APPLICABLE, _rfcommDevicePath);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("enterDisconnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not set protocol connection status (error = %d)", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
      }
      _expiredTimerSmEvent += "::FORCE_STATE_DISCONNECTED";

      if (true == _disconnectingTimer.StartSMTimer(_disconnectingTimerId,
            1000L * static_cast<long>(LocalSpm::getDataProvider().getBmCoreConfiguration()._protocolDisconnectingTimeoutSeconds),
            0L, _expiredTimerSmEvent.c_str(), this))
      {
         char milliSecondsStr[200];
         snprintf(milliSecondsStr, 199, "%llu", 1000L * static_cast<long long unsigned>(LocalSpm::getDataProvider().getBmCoreConfiguration()._protocolDisconnectingTimeoutSeconds));
         ETG_TRACE_USR1(("enterDisconnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): started disconnecting timer (timer ID = 0x%p, intervalMilliseconds = %50s)",
               (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), _disconnectingTimerId, milliSecondsStr));
      }
      else
      {
         ETG_TRACE_ERR(("enterDisconnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): starting disconnecting timer failed",
               (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

         ETG_TRACE_USR4(("enterDisconnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): sending event FORCE_STATE_DISCONNECTED to myself",
               (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

         result = this->SendEventByName("FORCE_STATE_DISCONNECTED", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("enterDisconnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not send event FORCE_STATE_DISCONNECTED (error = %d)",
                  (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::exitDisconnecting()
   {
      ENTRY

      ETG_TRACE_USR1(("exitDisconnecting(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      _disconnectingTimer.CancelTimer(_disconnectingTimerId);

      _expiredTimerSmEvent = this->GetSMNameFull();

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::handleForceStateDisconnected()
   {
      char milliSecondsStr[200];
      snprintf(milliSecondsStr, 199, "%llu", 1000L * static_cast<long long unsigned>(LocalSpm::getDataProvider().getBmCoreConfiguration()._protocolDisconnectingTimeoutSeconds));

      ETG_TRACE_USR1(("handleForceStateDisconnected(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): WARNING: protocol has not been reported as being disconnected within %50s msec, assuming protocol to be disconnected",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), milliSecondsStr));

      return CC_ERR_INT_NO_ERROR;
   }

   void ProtocolConnectionController::handleSmTimerCallback(const char* message)
   {
      if(0 != message)
      {
         LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(getNewBmCoreIfMessage_BmCoreIfMessage_SmTimeout(message));
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   Result ProtocolConnectionController::initMembers()
   {
      ENTRY

      ETG_TRACE_USR1(("initMembers(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\")", (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str()));

      /* Initialize member variables except of _protocolId */

      _uuid = "";
      _deviceId = 0u;
      _rfcommDevicePath = "";
      _disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;
      _expiredTimerSmEvent = this->GetSMNameFull();
      _canBeReleased = true;
      _pageTimeout = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultConnectionPageTimeoutMilliSeconds;

      _btsConnectionStatus = BM_CONNECTION_STATUS_DISCONNECTED;
      _btsRfcommDevicePath = "";
      _btsDisconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;
      _numPendingBtsConnDiscRequests = 0u;

      return CC_ERR_INT_NO_ERROR;
   }

   Result ProtocolConnectionController::sendSmEventToDcc(IN const std::string& callerMethodName, IN const std::string& eventName)
   {
      DeviceConnectionController* dccInstance(0);

      Result result = LocalSpm::getDbManager().getDeviceConnectionController(&dccInstance, _deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         char eventParams[60] = {0};

         if ("CONNECTED" == eventName)
         {
            result = dccInstance->ParameterCONNECTED(eventParams, sizeof(eventParams), _protocolId, _uuid.c_str());
         }

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("%200s: sending event %50s to DCC SM 0x%p",
                  callerMethodName.c_str(), eventName.c_str(), (void *) dccInstance));

            result = dccInstance->SendEventByName("CONNECTED", eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("%200s: could not send event %50s (error = %d)",
                     callerMethodName.c_str(), eventName.c_str(), ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("%200s: could not marshal event parameters for event %50s (error = %d)",
                  callerMethodName.c_str(), eventName.c_str(), result));
         }
      }
      else
      {
         ETG_TRACE_ERR(("%200s: could not get DCC instance for device with device ID = %d (error = %d)",
               callerMethodName.c_str(), _deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   void ProtocolConnectionController::checkNumPendingBtsConnDiscRequests()
   {
      ETG_TRACE_USR1(("checkNumPendingBtsConnDiscRequests(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): number of pending requests = %d",
            (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), _numPendingBtsConnDiscRequests));

      if (0u == _numPendingBtsConnDiscRequests)
      {
         Result result(CC_ERR_INT_NO_ERROR);
         std::string eventName("");
         char eventParams[280];

         if (BM_CONNECTION_STATUS_CONNECTED == _btsConnectionStatus)
         {
            eventName = "CONNECTED";

            result = this->ParameterCONNECTED(eventParams, sizeof(eventParams), _btsRfcommDevicePath.c_str());
         }
         else
         {
            eventName = "DISCONNECTED";

            if (BM_DISCONNECTED_REASON_NORMAL_LOSS == _btsDisconnectedReason)
            {
               if (0 == strncmp(this->GetCurrentState(), "Connected", strlen(this->GetCurrentState())))
               {
                  _btsDisconnectedReason = BM_DISCONNECTED_REASON_NORMAL_LOSS_REMOTE;
               }
               else
               {
                  _btsDisconnectedReason = BM_DISCONNECTED_REASON_NORMAL_LOSS_LOCAL;
               }
            }

            result = this->ParameterDISCONNECTED(eventParams, sizeof(eventParams), _btsDisconnectedReason);
         }

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("checkNumPendingBtsConnDiscRequests(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): sending event %50s to myself",
                  (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), eventName.c_str()));

            result = this->SendEventByName(eventName.c_str(), eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("checkNumPendingBtsConnDiscRequests(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not send event %50s (error = %d)",
                     (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), eventName.c_str(),
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkNumPendingBtsConnDiscRequests(this = 0x%p, deviceId = %d, protocol = %d, uuid = \"%50s\"): could not marshal event parameters for event %50s (error = %d)",
                  (void *) this, _deviceId, ETG_CENUM(ProtocolId, _protocolId), _uuid.c_str(), eventName.c_str(),
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
   }
}
