/**
 * @defgroup ProtocolConnectionControllerModule Protocol Connection
 * 
 * @brief This module comprises all parts related to protocol connection
 */
 
/**
 * @file
 * @ingroup ProtocolConnectionControllerModule
 * 
 * @brief This file contains the definition of the class ProtocolConnectionController
 * 
 * @author Tino Lippold
 * @author Stefan Reis
 */

#ifndef _PROTOCOL_CONNECTION_CONTROLLER_H_
#define _PROTOCOL_CONNECTION_CONTROLLER_H_

#include "ProtocolConnectionSm.h"
#include "Timer.h"
#include "FwISmTimerCallback.h"

namespace bmcore
{
   /**
    * @class ProtocolConnectionController
    * @ingroup ProtocolConnectionControllerModule
    * 
    * @brief The class ProtocolConnectionController implements the state machine ProtocolConnectionSm and its public interface
    *
    */
   class ProtocolConnectionController : public ProtocolConnectionSm, public ::fw::ISmTimerCallback
   {
   
   public:

      /**
       * @brief Constructor of class ProtocolConnectionController which creates an instance
       *
       * No further initialization of internal components and resources
       * is performed in the constructor. See the explicit allocation
       * and initialization functions Create() and Init(). Hence, the
       * implementation is trivial.
       *
       * Synchronization: not re-entrant.
       *
       * Performance considerations: none.
       *
       * @return void
       */
       ProtocolConnectionController(const ProtocolId protocolId);
   
      /**
       * Destructor of class ProtocolConnectionController
       *
       * @return void
       */
      virtual ~ProtocolConnectionController();

      Result assign(IN const DeviceId deviceId, IN const Uuid& uuid, IN const PageTimeout pageTimeout);

      Result release(void);

      inline ProtocolId getProtocolId() const
      {
         return _protocolId;
      }

      inline Uuid getUuid() const
      {
         return _uuid;
      }

      inline void setUuid(IN const Uuid& uuid)
      {
         _uuid = uuid;
      }

      void updateNumPendingBtsConnDiscRequests(const bool increment);
      void updateProtocolConnectionStatus(const ConnectionStatus connectionStatus,
            const RfcommDevicePath& rfCommDevicePath, const DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_NOT_APPLICABLE);

      void create();

      Result init();

      Result run();

      Result stop();

      Result done();
   
      // ***********************************************
      // * Implemented methods of ProtocolConnectionSm *
      // ***********************************************

      /**
       * old:
       *
       * @brief Stores user context of state machine ProtocolConnectionSm in member variable and sets ProtocolConnectionSm
       * state machine handle in database
       *
       * Gets the user context of the state machine ProtocolConnectionSm by calling @a userContext = ProtocolConnectionSm::GetUserContext()
       * which contains the protocol ID and stores it into member variable mSmContext.\n
       * Sets the ProtocolConnectionSm state machine handle in database by calling
       * DbManager::setProtocolConnectionSm(IN const tSMHandle handle=this, IN const tProtocolId protocolId=mSmContext.protocolId).
       *
       * new:
       * - get protocol ID from database and store it in member _protocolId
       *
       * @return 0 on success
       */
      Result initSm();

      Result handleStopSm();
   
      Result handleDone();
   
      /**
       * @brief Releases ProtocolConnectionSm state machine handle
       *
       * new:
       *
       * If _deviceId != 0
       * Releases ProtocolConnectionSm state machine handle by calling DbManager::releaseProtocolConnectionSm(IN const tSMHandle handle=ProtocolConnectionSm::this).\n
       * Sets BT protocol state to DISCONNECTED by calling
       * DbManager::setProtocolConnectionStatus(IN const DeviceId deviceId=_deviceId, IN const tProtocolId protocolId=mSmContext.protocolId, IN const Uuid uuid=_uuid, IN const ConnectionStatus state=DISCONNECTED).\n
       * Reset member variables.
       *
       * @return 0 on success
       */
      Result enterDisconnected();
   
      Result sendConnectLocal();

      Result requestReleaseIfPossible();

      Result handleDisconnectedInDisconnected(IN const DisconnectedReason disconnectedReason);

      /**
       * @brief Checks if a protocol connection request initiated by the local BT device side is to be delayed
       *
       * Sets members _deviceId=deviceId and _uuid=uuid.\n
       * Checks if a protocol connection request is to be delayed by calling bool isToBeDelayed=BmController::IsInternConnectRequestToBeDelayed(). If isToBeDelayed==true, return 1,
       * otherwise return 0.
       *
       * @param[in] deviceId ID of the remote BT device
       * @param[in] uuid UUID of a specific BT SPP service
       *
       * @return 0: protocol connection request is not to be delayed
       * @return 1: protocol connection request is to be delayed
       */
      Result connectingToBeDelayed(void);

      /**
       * @brief Sets BT protocol state to DELAY_CONNECTING
       *
       * Sets BT protocol state to DELAY_CONNECTING by calling
       * DbManager::setProtocolConnectionStatus(IN const DeviceId deviceId=_deviceId, IN const tProtocolId protocolId=mSmContext.protocolId, IN const Uuid uuid=_uuid, IN const ConnectionStatus state=DELAY_CONNECTING).
       *
       * @return 0 on success
       */
      Result enterDelayConnecting();

      Result exitDelayConnecting();

      /**
       * @brief Starts connection of the BT protocol
       *
       * Gets the BT stack device handle by calling DbManager::getBtsHandleFromDeviceId(OUT tBTSDeviceHandle deviceHandle, IN const DeviceId deviceId=_deviceId).\n
       * Sets member _btStackIfDeviceHandle=deviceHandle.\n
       * Starts connection of the BT protocol by calling
       * @a btsResult = BTStackIF::ConnectProtocol(IN const tBTSDeviceHandle deviceHandle=_btStackIfDeviceHandle, IN const tBTSProtocolId protocolId=mSmContext.protocolId, IN const tBTSUuid sppUuid=_uuid).\n
       * If @a btsResult != BTS_OK the message ProtocolConnectionSm::CONNECTION_FAILED is sent.
       *
       * @param[in] deviceId ID of the remote device
       * @param[in] uuid UUID of a specific BT SPP service
       *
       * @return 0 on success
       */
      Result connect();

      /**
       * @brief Accepts a BT protocol connection request initiated by the remote BT device
       *
       * Gets the BT stack device handle by calling DbManager::getBtsHandleFromDeviceId(OUT tBTSDeviceHandle deviceHandle, IN const DeviceId deviceId=deviceId).\n
       * Sets members _deviceId=deviceId, _uuid=uuid and _btStackIfDeviceHandle=deviceHandle.\n
       * Accepts BT protocol connection request by calling
       * @a btsResult = AcceptRemoteProtocolConnect(IN const tBTSDeviceHandle deviceHandle=_btStackIfDeviceHandle, IN const tBTSProtocolId protocolId=mSmContext.protocolId).\n
       * If @a btsResult != BTS_OK the message ProtocolConnectionSm::CONNECTION_FAILED is sent.
       *
       * @param[in] deviceId ID of the remote device
       * @param[in] uuid UUID of a specific BT SPP service
       *
       * @return 0 on success
       */
      Result accept();

      Result resetCanBeReleasedFlag();

      /**
       * @brief Sets BT protocol state to CONNECTING
       *
       * Sets BT protocol state to CONNECTING by calling
       * DbManager::setProtocolConnectionStatus(IN const DeviceId deviceId=_deviceId, IN const tProtocolId protocolId=mSmContext.protocolId, IN const Uuid uuid=_uuid, IN const ConnectionStatus state=CONNECTING).
       *
       * @return 0 on success
       */
      Result enterConnecting();

      Result saveRfcommDevicePath(IN const RfcommDevicePathChrArr rfcommDevicePath);

      /**
       * @brief Sets BT protocol state to CONNECTED
       *
       * Sets BT protocol state to CONNECTED by calling
       * DbManager::setProtocolConnectionStatus(IN const DeviceId deviceId=_deviceId, IN const tProtocolId protocolId=mSmContext.protocolId, IN const Uuid uuid=_uuid, IN const ConnectionStatus state=CONNECTED).
       *
       * @return 0 on success
       */
      Result enterConnected();

      Result handleAlreadyConnected();
      /**
       * @brief Starts disconnection of the BT protocol
       *
       * Starts disconnection of the BT protocol by calling
       * @a btsResult = DisconnectProtocol(IN const tBTSDeviceHandle deviceHandle=_btStackIfDeviceHandle, IN const tBTSProtocolId protocolId=mSmContext.protocolId, IN const tBTSUuid sppUuid=_uuid).\n
       * If @a btsResult != BTS_OK the message ProtocolConnectionSm::DISCONNECT_FAILED is sent.
       *
       * @return 0 on success
       */
      Result disconnect();

      /**
       * @brief Starts disconnection of the BT protocol
       *
       * Starts disconnection of the BT protocol by calling
       * @a btsResult = DisconnectProtocol(IN const tBTSDeviceHandle deviceHandle=_btStackIfDeviceHandle, IN const tBTSProtocolId protocolId=mSmContext.protocolId, IN const tBTSUuid sppUuid=_uuid).\n
       * If @a btsResult != BTS_OK the message ProtocolConnectionSm::DISCONNECT_FAILED is sent.
       *
       * @return 0 on success
       */
      Result handleDisconnect(DisconnectedReason disconnectedReason);

      /**
       * @brief Removes protocol from last mode table
       *
       * If @a disconnectedReason==BTS_DISCONNECT_REASON_NORMAL_LOSS the protocol with the ID mSmContext.protocolId is removed
       * from last mode table by calling
       * DbManager::RemoveFromLastmode(IN const DeviceId deviceId=_deviceId, IN const tProtocolId protocolId=mSmContext.protocolId, IN const tUUID uuid=_uuid).
       *
       * @param[in] disconnectedReason: reason for a device having been disconnected
       *
       * @return
       */
      Result saveDisconnectedReason(IN const DisconnectedReason disconnectedReason);

      /**
       * @brief Sets BT protocol state to DISCONNECTING and starts disconnecting timer
       *
       * Sets BT protocol state to DISCONNECTING by calling
       * DbManager::setProtocolConnectionStatus(IN const DeviceId deviceId=_deviceId, IN const tProtocolId protocolId=mSmContext.protocolId, IN const Uuid uuid=_uuid, IN const ConnectionStatus state=DISCONNECTING).\n
       * Creates disconnecting timer by calling Timer::Create(..., ProtocolConnectionSm::FORCE_STATE_DISCONNECTED,...).\n
       * When the disconnecting timer expires the message ProtocolConnectionSm::FORCE_STATE_DISCONNECTED is sent.
       * @todo make disconnecting timeout configurable (add configuration parameter)
       *
       * @return 0 on success
       */
      Result enterDisconnecting();

      /**
       * @brief Stops disconnecting timer
       *
       * Stops disconnecting timer by calling Timer::Cancel(...)
       *
       * @return 0 on success
       */
      Result exitDisconnecting();

      Result handleForceStateDisconnected();

      void handleSmTimerCallback(const char* message);

   private:

      Result initMembers();

      Result sendSmEventToDcc(IN const std::string& callerMethodName, IN const std::string& eventName);

      void checkNumPendingBtsConnDiscRequests();

      Result checkAndUpdateUuid();

      ProtocolId _protocolId;

      Uuid       _uuid;        /**< UUID of a BT SPP service (initialized to "")*/
      DeviceId   _deviceId;    /**< Device ID corresponding to the remote BT device (initialized to 0)*/

      RfcommDevicePath _rfcommDevicePath;
      DisconnectedReason _disconnectedReason;

      ConnectionStatus _btsConnectionStatus;
      DisconnectedReason _btsDisconnectedReason;
      RfcommDevicePath _btsRfcommDevicePath;
      uint_least32_t _numPendingBtsConnDiscRequests;

      Timer      _delayConnectingTimer;
      timer_t    _delayConnectingTimerId;
      Timer      _disconnectingTimer;
      timer_t    _disconnectingTimerId;

      std::string _expiredTimerSmEvent;

      bool _canBeReleased;

      PageTimeout _pageTimeout;
   };

   typedef std::vector<ProtocolConnectionController*> ProtocolConnectionControllerList;
}

#endif //_PROTOCOL_CONNECTION_CONTROLLER_H_

