#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_PAIRINGCONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PairingController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_PAIRINGCONTROLLER
#endif
#endif

#include "LocalSpm.h"
#include "FunctionTracer.h"
#include "BmVarTrace.h"
#include "Dispatcher.h"
#include "BmUtils.h"
#include "PairingController.h"
#include <cstring>
#include <algorithm>
#include <functional>
#include "IBtStackIfConnectionRequest.h"
#include "BmGlobalLock.h"
#include "BmCoreIfMessagesCreator.h"

namespace bmcore
{
   static bool isPairingRequestAcceptedByUser = false;

   PairingController::PairingController(const ComponentId componentId) :
         ILocalSpm(componentId),
         _attemptNumber(1u),
         _pairingResult(BM_PAIRING_RESULT_UNKNOWN),
         _sppCapabilitiesReceived(false),
         _waitfor_DID_SPP_InfoTimer(),
         _waitfor_DID_SPP_InfoTimerId(),
         _expiredTimerSmEvent(this->GetSMNameFull())
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("PairingController: is being created"));

      // threadless SM
      disableAllLocks(true);
   }

   PairingController::~PairingController()
   {
      ENTRY_INTERNAL

      _waitfor_DID_SPP_InfoTimerId = 0;
   }

   Result PairingController::initSm()
   {
      ENTRY

      ETG_TRACE_USR1(("initSm"));

      initMembers();

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::handleStopSm()
   {
      ENTRY

      ETG_TRACE_USR1(("handleStopSm"));

      /* Send stop done to SPM in the transition to final state in state machine */
      return stopDone(0);
   }

   Result PairingController::handleDone()
   {
      ENTRY

      ETG_TRACE_USR1(("handleDone"));

      /* Send done done to SPM in the transition to final state in state machine */
      return doneDone(0);

   }

   Result PairingController::savePairingInfoLocal(IN const BdAddressChrArr bdAddress)
   {
      ENTRY

      ETG_TRACE_USR1(("savePairingInfoLocal: bdAddress = \"%50s\"", bdAddress));

      //saving pairing Info
      (void) this->savePairingInfo(bdAddress, "", BM_PAIRING_REQUEST_ORIGIN_LOCAL, BM_PAIRING_TYPE_UNKNOWN, "");

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::savePairingInfoRemote(IN const BdAddressChrArr bdAddress,
         IN const DeviceNameChrArr deviceName, const PairingType pairingType, const SspPinCodeChrArr sspPinCode)
   {
      ENTRY

      ETG_TRACE_USR1(("savePairingInfoRemote: bdAddress = \"%50s\", deviceName = \"%50s\", pairingType = %d, sspPinCode = \"%50s\"",
            bdAddress, deviceName, ETG_CENUM(PairingType, pairingType), sspPinCode));

      //saving pairing Info
      this->savePairingInfo(bdAddress, deviceName, BM_PAIRING_REQUEST_ORIGIN_REMOTE, pairingType, sspPinCode);

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::startRemotePairingSession(void)
   {
      ENTRY

      ETG_TRACE_USR1(("startRemotePairingSession"));

      Result result(CC_ERR_INT_NO_ERROR);

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      if (BM_PAIRING_STATE_IDLE == pairingStatus._state)
      {
         pairingStatus._state = BM_PAIRING_STATE_PAIRING_PREPARED;
         pairingStatus._origin = BM_PAIRING_REQUEST_ORIGIN_REMOTE;

         LocalSpm::getBmCoreMainController().setPairingStatusInt(pairingStatus);
      }
      else
      {
         result = CC_ERR_INT_PAIRING_IN_PROGRESS;
         ETG_TRACE_ERR(("startRemotePairingSession: pairing is already in progress (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::checkDccReservation()
   {
      ENTRY

      ETG_TRACE_USR1(("checkDccReservation"));

      Result result = CC_ERR_INT_NO_ERROR;
      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      if (BM_PAIRING_REQUEST_ORIGIN_LOCAL == pairingStatus._origin)
      {
         ETG_TRACE_USR4(("checkDccReservation: sending event START_PAIRING to myself"));
         result = this->SendEventByName("START_PAIRING", 0);
         //!Fix for CID 117737: Unchecked return value (CHECKED_RETURN)
         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("checkDccReservation: could not send event START_PAIRING (error = %d)", ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else if (BM_PAIRING_REQUEST_ORIGIN_REMOTE == pairingStatus._origin)
      {
         ETG_TRACE_USR4(("checkDccReservation: sending event CHECK_AUTO_CONFIRMATION to myself"));
         result = this->SendEventByName("CHECK_AUTO_CONFIRMATION", 0);
         //!Fix for CID 117737: Unchecked return value (CHECKED_RETURN)
         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("checkDccReservation: could not send event CHECK_AUTO_CONFIRMATION (error = %d)", ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         //this should never occur
         result = CC_ERR_INT_GENERAL_ERROR;
         ETG_TRACE_ERR(("checkDccReservation: pairing request origin = %d is unknown (error = %d)",
               ETG_CENUM(PairingRequestOrigin, pairingStatus._origin), ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::startPairing()
   {
      ENTRY

      ETG_TRACE_USR1(("startPairing"));

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().startPairing(pairingStatus._remoteBdAddress);

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::saveAdditionalPairingInfo(IN const BdAddressChrArr bdAddress,
         IN const DeviceNameChrArr deviceName, IN const PairingType pairingType, IN const SspPinCodeChrArr sspPinCode)
   {
      ENTRY

      ETG_TRACE_USR1(("saveAdditionalPairingInfo: bdAddress = \"%50s\", deviceName = \"%50s\", pairingType = %d, sspPinCode = \"%50s\"",
            bdAddress, deviceName, ETG_CENUM(PairingType, pairingType), sspPinCode));

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      if (0 == strcmp(bdAddress, pairingStatus._remoteBdAddress.c_str()))
      {
         pairingStatus._pairingType = pairingType;
         pairingStatus._remoteBdName = deviceName;

         if (BM_PAIRING_TYPE_LEGACY == pairingType)
         {
            pairingStatus._pin = LocalSpm::getBmCoreMainController().getPairingPinCode();
         }
         else
         {
            pairingStatus._pin = sspPinCode;
         }

         LocalSpm::getBmCoreMainController().setPairingStatusInt(pairingStatus);

         ETG_TRACE_USR4(("saveAdditionalPairingInfo(bdAddress = \"%50s\", deviceName = \"%50s\", pairingType = %d, sspPinCode = \"%50s\"): sending event CHECK_AUTO_CONFIRMATION to myself",
               bdAddress, deviceName, ETG_CENUM(PairingType, pairingType), sspPinCode));

         Result result = this->SendEventByName("CHECK_AUTO_CONFIRMATION", 0);
         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("saveAdditionalPairingInfo: could not send event CHECK_AUTO_CONFIRMATION (error = %d)", ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         // this should never occur

         ETG_TRACE_ERR(("saveAdditionalPairingInfo(bdAddress = \"%50s\", deviceName = \"%50s\", pairingType = %d, sspPinCode = \"%50s\"): a pairing session with a different remote device with BD address = \"%50s\" is already in progress",
               bdAddress, deviceName, ETG_CENUM(PairingType, pairingType), sspPinCode, pairingStatus._remoteBdAddress.c_str()));

         ETG_TRACE_USR1(("saveAdditionalPairingInfo(bdAddress = \"%50s\", deviceName = \"%50s\", pairingType = %d, sspPinCode = \"%50s\"): rejecting pairing with remote device with BD address = \"%50s\"",
               bdAddress, deviceName, ETG_CENUM(PairingType, pairingType), sspPinCode, bdAddress));

         if (BM_PAIRING_TYPE_LEGACY == pairingType)
         {
            //Invoke BTStackIf::setPinCode
            LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().setPinCode(bdAddress, "");
         }
         else if (BM_PAIRING_TYPE_UNKNOWN == pairingType)
         {
            // this should never occur
            ETG_TRACE_ERR(("saveAdditionalPairingInfo(bdAddress = \"%50s\", deviceName = \"%50s\", pairingType = %d, sspPinCode = \"%50s\"): pairing type is unknown",
                  bdAddress, deviceName, ETG_CENUM(PairingType, pairingType), sspPinCode));
         }
         else // SSP
         {
            BTSSspMode btsSspMode;

            (void) getBtsSspModeFromBmPairingType(btsSspMode, pairingType);

            LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().confirmSecureSimplePairing(bdAddress, btsSspMode, "", BTS_CONFIRM_REJECT);
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::shallAutoConfirmPairingRequest()
   {
      ENTRY

      ETG_TRACE_USR1(("shallAutoConfirmPairingRequest"));

      Result result = CC_ERR_INT_NO_ERROR;

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      bool autoConfirm(false);

      if (BM_PAIRING_TYPE_LEGACY == pairingStatus._pairingType)
      {
         //Legacy pairing
         if (BM_PAIRING_REQUEST_ORIGIN_LOCAL == pairingStatus._origin)
         {
            autoConfirm = LocalSpm::getDataProvider().getBmCoreConfiguration()._autoConfirmLocalLegacyPairingRequest;
         }
         else if (BM_PAIRING_REQUEST_ORIGIN_REMOTE == pairingStatus._origin)
         {
            autoConfirm = LocalSpm::getDataProvider().getBmCoreConfiguration()._autoConfirmRemoteLegacyPairingRequest;
         }
         else
         {
            // this should never occur
            ETG_TRACE_ERR(("shallAutoConfirmPairingRequest: pairing request origin = %d is unknown",
                  ETG_CENUM(PairingRequestOrigin, pairingStatus._origin)));
         }
      }
      else if (BM_PAIRING_TYPE_UNKNOWN == pairingStatus._pairingType)
      {
         // this should never occur
         ETG_TRACE_ERR(("shallAutoConfirmPairingRequest: pairing type is unknown"));
      }
      else
      {
         // SSP pairing
         if (BM_PAIRING_REQUEST_ORIGIN_LOCAL == pairingStatus._origin)
         {
            autoConfirm = LocalSpm::getDataProvider().getBmCoreConfiguration()._autoConfirmLocalSecureSimplePairingRequest;
         }
         else if (BM_PAIRING_REQUEST_ORIGIN_REMOTE == pairingStatus._origin)
         {
            autoConfirm = LocalSpm::getDataProvider().getBmCoreConfiguration()._autoConfirmRemoteSecureSimplePairingRequest;

            BtLimitationModeInfo btLimitationModeInfo;
            result = LocalSpm::getBmCoreMainController().getBtLimitationModeInfo(btLimitationModeInfo,  static_cast<BdAddress>(pairingStatus._remoteBdAddress));

            if (CC_ERR_INT_NO_ERROR == result)
            {
               // Bmcore internally autoconfirm the pairing request if the limitation mode(CARLIFE/ONCAR) is set for this device irrespective of the default value.
               if(false == autoConfirm)
               {
                  if(((BM_LIMITATION_FEATURE_CAR_LIFE == btLimitationModeInfo._limitationMode._limitationFeature) || (BM_LIMITATION_FEATURE_ON_CAR == btLimitationModeInfo._limitationMode._limitationFeature))
                        && (BM_LIMITATION_COMMUNICATION_IF_USB == btLimitationModeInfo._limitationMode._limitationCommunicationIf))
                  {
                     ETG_TRACE_USR4(("shallAutoConfirmPairingRequest:  BtLimitionMode - CL/ONCAR is in progress so Set AutoConfirm as TRUE "));
                     autoConfirm = true;
                  }
               }
               else
               {
                  // BmApp waiting(authentication response from phone) for confirm the pairing request
                  // if the limitation mode(AA) is set for this device irrespective of the default value.
                  if((BM_LIMITATION_FEATURE_ANDROID_AUTO == btLimitationModeInfo._limitationMode._limitationFeature)
                        && (BM_LIMITATION_COMMUNICATION_IF_USB == btLimitationModeInfo._limitationMode._limitationCommunicationIf))
                  {
                     ETG_TRACE_USR4(("shallAutoConfirmPairingRequest:  BtLimitionMode - AA is in progress so Set AutoConfirm as FALSE "));
                     autoConfirm = false;
                  }
               }
            }
         }
         else
         {
            // this should never occur
            ETG_TRACE_ERR(("shallAutoConfirmPairingRequest: pairing request origin = %d is unknown",
                  ETG_CENUM(PairingRequestOrigin, pairingStatus._origin)));
         }
      }

      if (true == autoConfirm)
      {
         //User confirmation is NOT required (AUTO CONFIRM)
         pairingStatus._userConfirmationRequired = BM_USER_CONFIRMATION_NOT_REQUIRED;

         ETG_TRACE_USR4(("shallAutoConfirmPairingRequest: sending event AUTO_CONFIRM to myself"));
         result = this->SendEventByName("AUTO_CONFIRM", 0);
         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("shallAutoConfirmPairingRequest: could not send event AUTO_CONFIRM (error = %d)", ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         //User confirmation is required
         pairingStatus._userConfirmationRequired = BM_USER_CONFIRMATION_REQUIRED;

         ETG_TRACE_USR4(("shallAutoConfirmPairingRequest: sending event USER_CONFIRMATION_NEEDED to myself"));
         result = this->SendEventByName("USER_CONFIRMATION_NEEDED", 0);
         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("shallAutoConfirmPairingRequest: could not send event USER_CONFIRMATION_NEEDED (error = %d)", ETG_CENUM(CcErrorInternal, result)));
         }
      }

      LocalSpm::getBmCoreMainController().setPairingStatusInt(pairingStatus);

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::confirmPairing()
   {
      ENTRY

      ETG_TRACE_USR1(("confirmPairing"));

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      if (BM_PAIRING_TYPE_LEGACY == pairingStatus._pairingType)
      {
         PairingPin pairingPin;
         (void) LocalSpm::getBmCoreMainController().getPairingPin(pairingPin);

         //Accept LEGACY pairing request
         LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().setPinCode(pairingStatus._remoteBdAddress,
               pairingPin._pin);
      }
      else if (BM_PAIRING_TYPE_UNKNOWN == pairingStatus._pairingType)
      {
         // this should never occur
         ETG_TRACE_ERR(("shallAutoConfirmPairingRequest: pairing type is unknown"));
      }
      else
      {
         //Accept SSP pairing request
         BTSSspMode btsSspMode;

         (void) getBtsSspModeFromBmPairingType(btsSspMode, pairingStatus._pairingType);

         LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().confirmSecureSimplePairing(pairingStatus._remoteBdAddress, btsSspMode,
               pairingStatus._pin, BTS_CONFIRM_ACCEPT);

         isPairingRequestAcceptedByUser = true;
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::rejectPairing()
   {
      ENTRY

      ETG_TRACE_USR1(("rejectPairing"));

      _pairingResult = BM_PAIRING_RESULT_ERR_PAIRING_REJECTED;

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      if (BM_PAIRING_TYPE_LEGACY == pairingStatus._pairingType)
      {
         ETG_TRACE_USR1(("rejectPairing: rejecting legacy pairing request"));

         LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().setPinCode(pairingStatus._remoteBdAddress, "");
      }

      //!In case of Suzuki, first the pairing request is confirmed on the HU and later can be rejected.
      //!Therefore if the pairing request is already accepted and then rejected, do not call BT Stack Interface.
      //!Instead end the pairing process and SM is transited to IDLE.
      else if ((BM_PAIRING_TYPE_UNKNOWN == pairingStatus._pairingType) || (true == isPairingRequestAcceptedByUser))
      {
         ETG_TRACE_USR1(("rejectPairing: pairing has been canceled"));

         _pairingResult = BM_PAIRING_RESULT_ERR_PAIRING_CANCELED;

         char eventParams[40];

         this->ParameterPAIRING_FINISHED(OUT eventParams, IN sizeof(eventParams),
               IN pairingStatus._remoteBdAddress.c_str() /* empty in this case */, IN BM_PAIRING_RESULT_ERR_PAIRING_CANCELED);

         ETG_TRACE_USR4(("rejectPairing: sending event PAIRING_FINISHED"));
         Result result = this->SendEventByName("PAIRING_FINISHED", eventParams);
         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("rejectPairing: could not send event PAIRING_FINISHED (error = %d)", ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         BTSSspMode btsSspMode;

         (void) getBtsSspModeFromBmPairingType(btsSspMode, pairingStatus._pairingType);

         ETG_TRACE_USR1(("rejectPairing: rejecting SSP pairing request"));

         LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().confirmSecureSimplePairing(pairingStatus._remoteBdAddress, btsSspMode,
               pairingStatus._pin, BTS_CONFIRM_REJECT);
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::handleNotPairable(IN const SwitchedOffReason reason)
   {
      ENTRY

      ETG_TRACE_USR1(("handleNotPairable: reason = %d", ETG_CENUM(SwitchedOffReason, reason)));

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      if (BM_PAIRING_REQUEST_ORIGIN_REMOTE == pairingStatus._origin)
      {
         if (SWITCHED_OFF_REASON_TIMEOUT == reason)
         {
            _pairingResult = BM_PAIRING_RESULT_ERR_NOT_PAIRABLE_AFTER_TIMEOUT;
         }
         else if (SWITCHED_OFF_REASON_CLIENT == reason)
         {
            _pairingResult = BM_PAIRING_RESULT_ERR_PAIRING_CANCELED;
         }
         else
         {
            _pairingResult = BM_PAIRING_RESULT_ERR_NOT_PAIRABLE;
         }

         if (BM_PAIRING_TYPE_LEGACY == pairingStatus._pairingType)
         {
            ETG_TRACE_USR1(("handleNotPairable: rejecting legacy pairing requested by remote device"));

            //Invoke BTStackIf::setPinCode
            LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().setPinCode(pairingStatus._remoteBdAddress, "");
         }
         else if (BM_PAIRING_TYPE_UNKNOWN == pairingStatus._pairingType)
         {
            char eventParams[40];

            this->ParameterPAIRING_FINISHED(OUT eventParams, IN sizeof(eventParams),
                  IN pairingStatus._remoteBdAddress.c_str() /* empty in this case */, IN _pairingResult);

            ETG_TRACE_USR4(("handleNotPairable: sending event PAIRING_FINISHED"));
            Result result = this->SendEventByName("PAIRING_FINISHED", eventParams);
            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleNotPairable: could not send event PAIRING_FINISHED (error = %d)", ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            // SSP
            BTSSspMode btsSspMode;

            (void) getBtsSspModeFromBmPairingType(btsSspMode, pairingStatus._pairingType);

            ETG_TRACE_USR1(("handleNotPairable: rejecting SSP pairing requested by remote device"));

            LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().confirmSecureSimplePairing(pairingStatus._remoteBdAddress, btsSspMode,
                  pairingStatus._pin, BTS_CONFIRM_REJECT);
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::cancelPairing()
   {
      ENTRY

      ETG_TRACE_USR1(("cancelPairing"));

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      if (BM_PAIRING_REQUEST_ORIGIN_LOCAL == pairingStatus._origin)
      {
         _pairingResult = BM_PAIRING_RESULT_ERR_PAIRING_CANCELED;

         if (BM_PAIRING_TYPE_UNKNOWN == pairingStatus._pairingType)
         {
            LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().cancelPairing(pairingStatus._remoteBdAddress);

            //Post event PAIRING_CANCELED
            char pairingResultStr[20];

            this->ParameterPAIRING_CANCELED(OUT pairingResultStr, IN sizeof(pairingResultStr),
                  IN BM_PAIRING_RESULT_ERR_PAIRING_CANCELED);

            ETG_TRACE_USR4(("cancelPairing: sending event PAIRING_CANCELED"));
            Result result = this->SendEventByName("PAIRING_CANCELED", pairingResultStr);
            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("cancelPairing: could not send event PAIRING_CANCELED (error = %d)", ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else if (BM_PAIRING_TYPE_LEGACY == pairingStatus._pairingType)
         {
            //Invoke BTStackIf::setPinCode
            LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().setPinCode(pairingStatus._remoteBdAddress, "");
         }
         else
         {
            // SSP
            BTSSspMode btsSspMode;

            (void) getBtsSspModeFromBmPairingType(btsSspMode, pairingStatus._pairingType);

            LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().confirmSecureSimplePairing(pairingStatus._remoteBdAddress, btsSspMode,
                  pairingStatus._pin, BTS_CONFIRM_REJECT);
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::handleFinishedPairing(IN const BdAddressChrArr bdAddress, IN const PairingResult pairingResult)
   {
      ENTRY

      ETG_TRACE_USR1(("handleFinishedPairing: bdAddress = \"%50s\", pairingResult = %d",
            bdAddress, ETG_CENUM(PairingResult, pairingResult)));

      Result result(CC_ERR_INT_NO_ERROR);
      bool terminatePairingSession(true);
      isPairingRequestAcceptedByUser = false;

      if (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._allowLegacyPairingRetry)
      {
         PairingStatus pairingStatus;
         (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

         if ((BM_PAIRING_TYPE_LEGACY == pairingStatus._pairingType) && (BM_PAIRING_RESULT_OK != pairingResult))
         {
            ETG_TRACE_USR1(("handleFinishedPairing(bdAddress = \"%50s\", pairingResult = %d): %u. legacy pairing attempt failed",
                  bdAddress, ETG_CENUM(PairingResult, pairingResult), _attemptNumber));

            if (2u > _attemptNumber)
            {
               ETG_TRACE_USR1(("handleFinishedPairing(bdAddress = \"%50s\", pairingResult = %d): keeping pairing session, allowing one retry",
                     bdAddress, ETG_CENUM(PairingResult, pairingResult)));

               ETG_TRACE_USR4(("handleFinishedPairing(bdAddress = \"%50s\", pairingResult = %d): sending event KEEP_PAIRING_SESSION to myself",
                     bdAddress, ETG_CENUM(PairingResult, pairingResult)));

               result = this->SendEventByName("KEEP_PAIRING_SESSION", 0);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("handleFinishedPairing(bdAddress = \"%50s\", pairingResult = %d): could not send event KEEP_PAIRING_SESSION (error = %d)",
                        bdAddress, ETG_CENUM(PairingResult, pairingResult), ETG_CENUM(CcErrorInternal, result)));
               }

               _attemptNumber++;
               terminatePairingSession = false;
            }
            else
            {
               ETG_TRACE_ERR(("handleFinishedPairing(bdAddress = \"%50s\", pairingResult = %d): legacy pairing finally failed",
                     bdAddress, ETG_CENUM(PairingResult, pairingResult)));
            }
         }
      }

      if (true == terminatePairingSession)
      {
         char eventParams[40];

         result = this->ParameterEND_PAIRING_SESSION(OUT eventParams, IN sizeof(eventParams),
               IN bdAddress, IN pairingResult);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("handleFinishedPairing(bdAddress = \"%50s\", pairingResult = %d): sending event END_PAIRING_SESSION to myself",
                  bdAddress, ETG_CENUM(PairingResult, pairingResult)));

            result = this->SendEventByName("END_PAIRING_SESSION", eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("handleFinishedPairing(bdAddress = \"%50s\", pairingResult = %d): could not send event END_PAIRING_SESSION (error = %d)",
                     bdAddress, ETG_CENUM(PairingResult, pairingResult), ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("handleFinishedPairing(bdAddress = \"%50s\", pairingResult = %d): could not marshal event parameters for event END_PAIRING_SESSION (error = %d)",
                  bdAddress, ETG_CENUM(PairingResult, pairingResult), ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::keepPairingSession()
   {
      ENTRY

      ETG_TRACE_USR1(("keepPairingSession"));

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::stopPairingSession(IN const BdAddressChrArr bdAddress, IN const PairingResult pairingResult)
   {
      ENTRY

      ETG_TRACE_USR1(("stopPairingSession: bdAddress = \"%50s\", pairingResult = %d",
            bdAddress, ETG_CENUM(PairingResult, pairingResult)));

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      if (0 == strcmp(bdAddress, pairingStatus._remoteBdAddress.c_str()))
      {
         if (BM_PAIRING_RESULT_UNKNOWN == _pairingResult)
         {
            pairingStatus._result = pairingResult;
         }
         else
         {
            pairingStatus._result = _pairingResult;
         }

         LocalSpm::getBmCoreMainController().setPairingStatusInt(pairingStatus);

         //Remove restriction of pairable/connectable mode
         LocalSpm::getBmCoreMainController().setDeviceBdAddrForRestrictedPairingConnecting("");

         //Switching to configured default pairable mode
         (void) LocalSpm::getBmCoreMainController().switchLocalPairableModeInt(LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode, "");

         if (BM_PAIRING_RESULT_OK == pairingStatus._result)
         {
            BmResult bmResult(BM_RESULT_OK);

            //Switching to configured default connectable mode
            bmResult = LocalSpm::getBmCoreMainController().switchLocalConnectableModeInt(LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode, "");

            if (BM_RESULT_OK != bmResult)
            {
               ETG_TRACE_USR1(("stopPairingSession(bdAddress = \"%50s\"): could not switch on local connectable mode", bdAddress));
            }
         }

         //Resetting the PairingStatus property with default constructor value
         LocalSpm::getBmCoreMainController().setPairingStatusInt(PairingStatus());

         //Resetting the class members
         initMembers();
      }
      else
      {
         //todo: add reasonable trace message
      }

      return CC_ERR_INT_NO_ERROR;
   }

   void PairingController::create()
   {
      ENTRY

      ETG_TRACE_USR1(("create"));

      /* Create the state machine */
      PairingSm::Create();

      createDone(0);
   }

   Result PairingController::init(InitReason reason)
   {
      ENTRY

      ETG_TRACE_USR1(("init: reason = %d", reason));

      (void) reason;

      /* Init the state machine */
      PairingSm::Init();
      SetAnswerTimeout(2000);

      /* Register state machine with dispatcher */
      Dispatcher::GetInstance().Register(IN this);

      return initDone(0);
   }

   Result PairingController::run()
   {
      ENTRY

      ETG_TRACE_USR1(("run"));

      return runDone(0);
   }

   Result PairingController::stop()
   {
      ENTRY

      ETG_TRACE_USR1(("stop"));

      Result ret = CC_ERR_INT_NO_ERROR;

      /* Send STOP message to own SM */
      ret = SendForceEvent(STOP_SM, (char *) 0);
      if (CC_ERR_INT_NO_ERROR != ret)
      {
         ETG_TRACE_ERR(("stop: error while sending internal event via SMF (ErrorCode:%d)", ETG_CENUM(CcErrorInternal, ret)));
      }

      return ret;
   }

   Result PairingController::done()
   {
      ENTRY

      ETG_TRACE_USR1(("done"));

      Result ret = CC_ERR_INT_NO_ERROR;

      /* Deregister state machine with dispatcher */
      Dispatcher::GetInstance().DeRegister(IN this);

      /* Send DONE message to own SM */
      ret = SendForceEvent(DONE, (char *) 0);
      if (CC_ERR_INT_NO_ERROR != ret)
      {
         ETG_TRACE_ERR(("done: error while sending internal event via SMF (ErrorCode:%d)", ETG_CENUM(CcErrorInternal, ret)));
      }

      return ret;
   }

   char *PairingController::getSmStateName(OUT tGeneralString stateName, size_t size)
   {
      ENTRY

      ETG_TRACE_USR1(("getSmStateName"));

      GetCurrentState((char *) stateName, size);
      return stateName;
   }

   Result PairingController::initMembers()
   {
      ENTRY

      ETG_TRACE_USR1(("initMembers"));

      _attemptNumber = 1u;
      _pairingResult = BM_PAIRING_RESULT_UNKNOWN;
      _expiredTimerSmEvent = this->GetSMNameFull();
      _sppCapabilitiesReceived = false;
      isPairingRequestAcceptedByUser = false;

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::savePairingInfo(IN const BdAddressChrArr bdAddress,
         IN const DeviceNameChrArr deviceName, IN const PairingRequestOrigin pairingRequestOrigin,
         IN const PairingType pairingType, IN const SspPinCodeChrArr sspPinCode)
   {
      ENTRY

      ETG_TRACE_USR1(("savePairingInfo: bdAddress = \"%50s\", deviceName = \"%50s\", pairingRequestOrigin = %d, pairingType = %d, sspPinCode = \"%50s\"",
            bdAddress, deviceName, pairingRequestOrigin, pairingType, sspPinCode));

      //update PairingStatus property
      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      if ((false == pairingStatus._remoteBdAddress.empty())
            && (0 != strcmp(bdAddress, pairingStatus._remoteBdAddress.c_str())))
      {
         ETG_TRACE_ERR(("savePairingInfo(bdAddress = \"%50s\", deviceName = \"%50s\", pairingRequestOrigin = %d, pairingType = %d, sspPinCode = \"%50s\"): a pairing session with a different remote device with BD address = \"%50s\" is already in progress",
               bdAddress, deviceName, pairingRequestOrigin, pairingType, sspPinCode, pairingStatus._remoteBdAddress.c_str()));

         ETG_TRACE_USR1(("savePairingInfo(bdAddress = \"%50s\", deviceName = \"%50s\", pairingRequestOrigin = %d, pairingType = %d, sspPinCode = \"%50s\"): rejecting the pairing with device with BD address = \"%50s\"",
               bdAddress, deviceName, pairingRequestOrigin, pairingType, sspPinCode, bdAddress));

         if (BM_PAIRING_TYPE_LEGACY == pairingType)
         {
            LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().setPinCode(bdAddress, "");
         }
         else
         {
            BTSSspMode btsSspMode;

            (void) getBtsSspModeFromBmPairingType(btsSspMode, pairingType);

            LocalSpm::getBmCoreMainController().getBtStackIfConnectionRequestIfWrapper().confirmSecureSimplePairing(bdAddress, btsSspMode,
                  sspPinCode, BTS_CONFIRM_REJECT);
         }

         ETG_TRACE_USR1(("savePairingInfo(bdAddress = \"%50s\", deviceName = \"%50s\", pairingRequestOrigin = %d, pairingType = %d, sspPinCode = \"%50s\"): canceling the pairing with device with BD address = \"%50s\"",
               bdAddress, deviceName, pairingRequestOrigin, pairingType, sspPinCode, pairingStatus._remoteBdAddress.c_str()));

         char eventParams[40];

         Result result = this->ParameterPAIRING_FINISHED(OUT eventParams, IN sizeof(eventParams),
               IN pairingStatus._remoteBdAddress.c_str(), IN BM_PAIRING_RESULT_ERR_PAIRING_CANCELED);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("savePairingInfo(bdAddress = \"%50s\", deviceName = \"%50s\", pairingRequestOrigin = %d, pairingType = %d, sspPinCode = \"%50s\"): sending event PAIRING_FINISHED to myself",
                  bdAddress, deviceName, pairingRequestOrigin, pairingType, sspPinCode));

            result = this->SendEventByName("PAIRING_FINISHED", eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("savePairingInfo(bdAddress = \"%50s\", deviceName = \"%50s\", pairingRequestOrigin = %d, pairingType = %d, sspPinCode = \"%50s\"): could not send event PAIRING_FINISHED (error = %d)",
                     bdAddress, deviceName, pairingRequestOrigin, pairingType, sspPinCode, ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("savePairingInfo(bdAddress = \"%50s\", deviceName = \"%50s\", pairingRequestOrigin = %d, pairingType = %d, sspPinCode = \"%50s\"): could not marshal event parameters for event PAIRING_FINISHED (error = %d)",
                  bdAddress, deviceName, pairingRequestOrigin, pairingType, sspPinCode, ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         pairingStatus._state = BM_PAIRING_STATE_PAIRING_IN_PROGRESS;
         pairingStatus._remoteBdAddress = bdAddress;
         pairingStatus._remoteBdName = deviceName;
         pairingStatus._origin = pairingRequestOrigin;
         pairingStatus._pairingType = pairingType;

         if (BM_PAIRING_TYPE_LEGACY == pairingType)
         {
            pairingStatus._pin = LocalSpm::getBmCoreMainController().getPairingPinCode();
         }
         else
         {
            pairingStatus._pin = sspPinCode;
         }

         //Set BtPairingStatus property
         LocalSpm::getBmCoreMainController().setPairingStatusInt(pairingStatus);
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::checkPairingIsSuccessful(IN const BdAddressChrArr bdAddress, IN const PairingResult pairingResult)
   {
      ETG_TRACE_USR1(("checkPairingIsSuccessful: bdAddress = \"%50s\", pairingResult = %d",
                  bdAddress, ETG_CENUM(PairingResult, pairingResult)));

      unsigned short pairingIsSuccessful(0u);
      PairingStatus pairingStatus;

      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      if (0 == strcmp(bdAddress, pairingStatus._remoteBdAddress.c_str()))
      {
         if (BM_PAIRING_RESULT_UNKNOWN == _pairingResult)
         {
            pairingStatus._result = pairingResult;
         }
         else
         {
            pairingStatus._result = _pairingResult;
         }

         if (BM_PAIRING_RESULT_OK == pairingResult)
         {
            pairingIsSuccessful = 1u;
            _pairingResult = BM_PAIRING_RESULT_OK;
         }
      }

      ETG_TRACE_USR1(("checkPairingIsSuccessful: Pairing is %s",
                  (1u == pairingIsSuccessful) ? "successful" : "failure"));

      return pairingIsSuccessful;
   }

   Result PairingController::checkAndStartTimer()
   {
      ETG_TRACE_USR1(("checkAndStartTimer"));

      Result result(CC_ERR_INT_NO_ERROR);

      //Remove restriction of pairable/connectable mode
      LocalSpm::getBmCoreMainController().setDeviceBdAddrForRestrictedPairingConnecting("");

      //Switching to configured default pairable mode
      (void) LocalSpm::getBmCoreMainController().switchLocalPairableModeInt(LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalPairableMode, "");

      BmResult bmResult(BM_RESULT_OK);

      //Switching to configured default connectable mode
      bmResult = LocalSpm::getBmCoreMainController().switchLocalConnectableModeInt(LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode, "");

      if (BM_RESULT_OK != bmResult)
      {
         ETG_TRACE_USR1(("checkAndStartTimer(): could not switch on local connectable mode"));
      }

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      DeviceId deviceId(0u);
      result = LocalSpm::getDbManager().getDeviceId(deviceId, pairingStatus._remoteBdAddress);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         // Check Project is support Carplaywireless or not
         if(true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported)
         {
            bool allInfosAvailable = false;

            result = checkAllInfosAvailable(allInfosAvailable, deviceId);

            if(CC_ERR_INT_NO_ERROR == result)
            {
               if(false == allInfosAvailable)
               {
                  _expiredTimerSmEvent += "::TIMEOUT";

                  // Start the timer for waiting info(DID & SPP Capabilities)
                  if (true == _waitfor_DID_SPP_InfoTimer.StartSMTimer(_waitfor_DID_SPP_InfoTimerId,
                        1000L * static_cast<long>(WAIT_FOR_DID_SPP_INFO_SEC), 0L, _expiredTimerSmEvent.c_str(), this))
                  {
                     char milliSecondsStr[200];
                     snprintf(milliSecondsStr, 199, "%llu", (1000L * static_cast<long long unsigned>(WAIT_FOR_DID_SPP_INFO_SEC)));
                     ETG_TRACE_USR1(("checkAndStartTimer(this = 0x%p, deviceId = %d): started timer for DID and SPP information update (timer ID = 0x%p, intervalMilliseconds = %50s)",
                           (void *) this, deviceId, _waitfor_DID_SPP_InfoTimerId, milliSecondsStr));
                  }
                  else
                  {
                     ETG_TRACE_ERR(("checkAndStartTimer(this = 0x%p, deviceId = %d): starting timer for DID and SPP information is failed",
                           (void *) this, deviceId));
                     result = CC_ERR_INT_GENERAL_ERROR;
                  }
               }
               else
               {
                  result = this->SendEventByName("CONNECT_DEVICE_AS_CPW", 0);
                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("checkAndStartTimer: could not send event CONNECT_DEVICE_AS_CPW (error = %d)", ETG_CENUM(CcErrorInternal, result)));
                  }
               }
            }
         }
         else
         {
            ETG_TRACE_USR1(("checkAndStartTimer(): Carplay wireless is not support"));
            result = CC_ERR_INT_NOT_ALLOWED;
         }
      }
      else
      {
         ETG_TRACE_USR1(("checkAndStartTimer(): could not get deviceId from DB (error = %d)", ETG_CENUM(CcErrorInternal, result)));
      }

      if(CC_ERR_INT_NO_ERROR != result)
      {
         result = this->SendEventByName("CONNECT_DEVICE_AS_CLASSIC_BT", 0);
         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("checkAndStartTimer: could not send event CONNECT_DEVICE_AS_CLASSIC_BT (error = %d)", ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::checkAllInfosAvailable(bool& allInfosAvailable, DeviceId deviceId)
   {
      Result result(CC_ERR_INT_NO_ERROR);

      allInfosAvailable = false;

      DeviceIdentification deviceIdentification;
      result = LocalSpm::getDbManager().getDeviceIdentification(deviceIdentification, deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         //Check VendorID is received or not
         if((0u != deviceIdentification._vendorId) && (0u != deviceIdentification._vendorIdSource))
         {
            //Check device is apple device or not
            if(((0x004C == deviceIdentification._vendorId) && (0x01 == deviceIdentification._vendorIdSource)) ||
                  ((0x05AC == deviceIdentification._vendorId) && (0x02 == deviceIdentification._vendorIdSource)))
            {
               ETG_TRACE_USR1(("checkAllInfosAvailable: Paired device is apple Device"));

               ProtocolInfoMap protocolInfomap;
               result = LocalSpm::getDbManager().getDeviceSupportedProtocols(protocolInfomap, deviceId);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  // todo [20180226, rse2hi]: the following loop seems to be not needed, check if it can be removed
                  for (ProtocolInfoMap::iterator it = protocolInfomap.begin(); it != protocolInfomap.end(); it++)
                  {
                     if(BM_PROTOCOL_ID_SPP == it->first)
                     {
                        ETG_TRACE_USR1(("checkAllInfosAvailable: device is spp supported"));
                        break;
                     }
                  }

                  UuidList supportedSppUuids;
                  result = LocalSpm::getDbManager().getDeviceSppUuidSupport(supportedSppUuids, deviceId);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     if(supportedSppUuids.size() == 0)
                     {
                        if(true == _sppCapabilitiesReceived)
                        {
                           result = CC_ERR_INT_UUID_NOT_SUPPORTED;

                           ETG_TRACE_USR1(("checkAllInfosAvailable: uuids is not available"));
                        }
                        else
                        {
                           ETG_TRACE_USR1(("checkAllInfosAvailable: uuids is not received"));
                        }
                     }
                     else
                     {
                        for(size_t index = 0; index < supportedSppUuids.size(); index++)
                        {
                           if(0 == (strcmp((const char *)IAP2BT_SPP_UUID, supportedSppUuids[index].c_str())))
                           {
                              ETG_TRACE_USR1(("checkAllInfosAvailable: IAP2BT SPP UUID is available"));
                              allInfosAvailable = true;

                              // IAP over BT supported
                              (void) LocalSpm::getBmCoreMainController().setDeviceiAPoverBTSupported(deviceId, true);
                              break;
                           }
                           else
                           {
                              ETG_TRACE_USR1(("checkAllInfosAvailable: uuid is not matched with IAP2BT SPP UUID"));
                           }
                        }

                        if((true == _sppCapabilitiesReceived) && (allInfosAvailable == false))
                        {
                           ETG_TRACE_USR1(("checkAllInfosAvailable: IAP2BT uuids is not supported"));
                           result = CC_ERR_INT_UUID_NOT_SUPPORTED;
                        }
                     }
                  }
                  else
                  {
                     ETG_TRACE_USR1(("checkAllInfosAvailable(): could not get devicesppUUID support from DB (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                  }
               }
               else
               {
                  ETG_TRACE_USR1(("checkAllInfosAvailable(): could not get device supported protocols from DB (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               ETG_TRACE_USR1(("checkAllInfosAvailable: Device is not apple Device"));

               result = CC_ERR_INT_UUID_NOT_SUPPORTED;
            }
         }
         else
         {
            ETG_TRACE_USR1(("checkAllInfosAvailable(): device identification is not yet received"));
         }
      }
      else
      {
         ETG_TRACE_USR1(("checkAllInfosAvailable(): could not get device identification from DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result PairingController::stopTimerAndSendPairingResult()
   {
      ETG_TRACE_USR1(("stopTimerAndSendPairingResult : entered"));

      // cancel the Timer if running
      if(_waitfor_DID_SPP_InfoTimerId != 0)
      {
         _waitfor_DID_SPP_InfoTimer.CancelTimer(_waitfor_DID_SPP_InfoTimerId);

         _waitfor_DID_SPP_InfoTimerId = 0;

         _expiredTimerSmEvent = this->GetSMNameFull();
      }

      //Update the result as BM_PAIRING_RESULT_OK and send the pairing status to clients
      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      pairingStatus._result = BM_PAIRING_RESULT_OK;

      LocalSpm::getBmCoreMainController().setPairingStatusInt(pairingStatus);

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::connectDeviceAsClassicBT()
   {
      ETG_TRACE_USR1(("connectDeviceAsClassicBT : entered"));

      Result result(CC_ERR_INT_NO_ERROR);
      bool pairingInitatedbyAADevice = false;

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      //Check pairing is initiated by HMI or Device
      BtLimitationModeInfo btLimitationModeInfo;
      result = LocalSpm::getBmCoreMainController().getBtLimitationModeInfo(btLimitationModeInfo,  static_cast<BdAddress>(pairingStatus._remoteBdAddress));

      if (CC_ERR_INT_NO_ERROR == result)
      {
         if(true == IS_LIMIMODE_FOR_HANDSFREE_PROFILE(btLimitationModeInfo._limitationMode._limitationFeature, btLimitationModeInfo._limitationMode._limitationCommunicationIf))
         {
            pairingInitatedbyAADevice = true;
         }
      }

      DeviceId deviceId(0u);
      result = LocalSpm::getDbManager().getDeviceId(deviceId, pairingStatus._remoteBdAddress);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         // Check Project is support Carplaywireless or not. If not supported then set the usage preference to CLASSIC_BT
         // else HMI will set the Usage preference.
         if(false == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported)
         {
            (void) LocalSpm::getDbManager().setDeviceUsagePreference(deviceId, BM_UP_CLASSIC_BT);
         }
         else
         {
            // Exception for AA if the Project is support Carplaywireless.
            //For Andriod Auto, UsagePreference need to be set internally because pairing is not triggered by HMI
            if(true == pairingInitatedbyAADevice)
            {
               (void) LocalSpm::getDbManager().setDeviceUsagePreference(deviceId, BM_UP_CLASSIC_BT);
            }
         }

         // Internally connect the device if the _connectAfterPairing flag is true.
         if((true == LocalSpm::getDataProvider().getBmCoreConfiguration()._connectAfterPairing) &&
               (false == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported))
         {
            //check pairing is initiated by AA device.
            // connect the device only if pairing is not initiated by AA device.
            if(false == pairingInitatedbyAADevice)
            {
               ETG_TRACE_USR1(("connectDeviceAsClassicBT: deviceId = %d", deviceId));

               BmResult bmResult = LocalSpm::getBmCoreMainController().connectDeviceInt(deviceId, true);

               if (BM_RESULT_OK != bmResult)
               {
                  ETG_TRACE_USR1(("connectDevice(): could not initiate connection to paired device - %d", deviceId));
               }
            }
            else
            {
               ETG_TRACE_USR1(("connectDevice(): Pairing is initiated by Remote Deivce so waiting for remote connection request"));
            }
         }
         else
         {
            ETG_TRACE_USR1(("connectDeviceAsClassicBT: Connection will be initated for CLASSIC_BT by HMI"));
         }
      }

      //Resetting the PairingStatus property with default constructor value
      LocalSpm::getBmCoreMainController().setPairingStatusInt(PairingStatus());

      //Resetting the class members
      initMembers();

      return result;
   }

   Result PairingController::connectDeviceAsCPW()
   {
      ETG_TRACE_USR1(("connectDeviceAsCPW : entered"));

      ETG_TRACE_USR1(("connectDeviceAsCPW: Connection will be initiated for CPW by HMI"));

      //Resetting the PairingStatus property with default constructor value
      LocalSpm::getBmCoreMainController().setPairingStatusInt(PairingStatus());

      //Resetting the class members
      (void) initMembers();

      return CC_ERR_INT_NO_ERROR;
   }

   Result PairingController::handleTimeout()
   {
      ETG_TRACE_USR1(("handleTimeout"));

      Result result(CC_ERR_INT_NO_ERROR);

      result = this->SendEventByName("CONNECT_DEVICE_AS_CLASSIC_BT", 0);
      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleTimeout: could not send event CONNECT_DEVICE_AS_CLASSIC_BT (error = %d)", ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result PairingController::updateInfo(DeviceId deviceId, DeviceInfoUpdateType deviceInfoUpdateType)
   {
      ETG_TRACE_USR1(("updateInfo: deviceId - %d", deviceId));

      Result result(CC_ERR_INT_NO_ERROR);

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      //Get the deviceId for newly paired device from dB.
      DeviceId deviceID(0u);
      result = LocalSpm::getDbManager().getDeviceId(deviceID, pairingStatus._remoteBdAddress);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_USR1(("updateInfo(): could not get device handle from DB (error = %d)", ETG_CENUM(CcErrorInternal, result)));

      }
      else
      {
         //Check info is received for newly paired device.
         if(deviceId == deviceID)
         {
            bool allInfosAvailable = false;

            if(BM_DEVICEINFO_UPDATE_SPP_CAPABILITIES == deviceInfoUpdateType)
            {
               _sppCapabilitiesReceived = true;

               // Check iAPoverBTSupportedflag is already set for the device in the dB
               // If already set then send the event CONNECT_DEVICE_AS_CPW to myself.
               bool iAPoverBTSupported = false;
               result = LocalSpm::getBmCoreMainController().getDeviceiAPoverBTSupported(iAPoverBTSupported, deviceId);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  if(true == iAPoverBTSupported)
                  {
                     allInfosAvailable = true;
                  }
                  else
                  {
                     // Check all the info's(UUID, DID, etc.,) are available. If all info's are available then check
                     // the paired device is an Apple device and IAP2BT UUID is supported. If supported then
                     // send the event CONNECT_DEVICE_AS_CPW to myself else send the event CONNECT_DEVICE_AS_CLASSIC_BT to myself
                     result = checkAllInfosAvailable(allInfosAvailable, deviceId);
                  }
               }
            } // if(BM_DEVICEINFO_UPDATE_SPP_CAPABILITIES == deviceInfoUpdateType)
            else
            {
               // Check all the info's(UUID, DID, etc.,) are available. If all info's are available then check
               // the paired device is an Apple device and IAP2BT UUID is supported. If supported then
               // send the event CONNECT_DEVICE_AS_CPW to myself else send the event CONNECT_DEVICE_AS_CLASSIC_BT to myself
               result = checkAllInfosAvailable(allInfosAvailable, deviceId);
            }

            if(CC_ERR_INT_NO_ERROR != result)
            {
               result = this->SendEventByName("CONNECT_DEVICE_AS_CLASSIC_BT", 0);
               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("updateInfo: could not send event CONNECT_DEVICE_AS_CLASSIC_BT (error = %d)", ETG_CENUM(CcErrorInternal, result)));
               }
            }
            else
            {
               if(allInfosAvailable == true)
               {
            	  result = this->SendEventByName("CONNECT_DEVICE_AS_CPW", 0);
                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("updateInfo: could not send event CONNECT_DEVICE_AS_CPW (error = %d)", ETG_CENUM(CcErrorInternal, result)));
                  }
               }
            }
         }
         else
         {
            ETG_TRACE_USR1(("updateInfo(): Info is not updated for paired device (deviceId = %d)", deviceId));
         }
      }

      return result;
   }

   Result PairingController::handlePairingFinished()
   {
      Result result(CC_ERR_INT_NO_ERROR);

      ETG_TRACE_USR1(("handlePairingFinished: entered"));

      PairingStatus pairingStatus;
      (void) LocalSpm::getBmCoreMainController().getPairingStatus(pairingStatus);

      DeviceId deviceId(0u);
      result = LocalSpm::getDbManager().getDeviceId(deviceId, pairingStatus._remoteBdAddress);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_USR1(("handlePairingFinished(): could not get deviceId from DB (error = %d)", ETG_CENUM(CcErrorInternal, result)));
      }
      else
      {
         LocalSpm::getDbManager().setDeviceUsagePreference(deviceId, BM_UP_CLASSIC_BT);
      }

      //Resetting the PairingStatus property with default constructor value
      LocalSpm::getBmCoreMainController().setPairingStatusInt(PairingStatus());

      //Resetting the class members
      initMembers();

      return result;
   }

   void PairingController::handleSmTimerCallback(const char* message)
   {
      if(0 != message)
      {
         LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(getNewBmCoreIfMessage_BmCoreIfMessage_SmTimeout(message));
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   void PairingController::checkPairingResult(OUT PairingResult& pairingResult)
   {
      ETG_TRACE_USR1(("checkPairingResult"));

      pairingResult = _pairingResult;

      ETG_TRACE_USR1(("checkPairingResult: pairingResult - %d", ETG_CENUM(PairingResult, pairingResult)));

      return;
   }
}
