/* ETG definitions */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_LOCALSPM
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/LocalSpm.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_LOCALSPM
#endif
#endif

#include "FunctionTracer.h"

/* others */
#include "LocalSpm.h"
#include "Dispatcher.h"

namespace bmcore
{
   #define STRINGIZE(x) #x
   #define STRINGIZE_VALUE_OF(x) STRINGIZE(x)

   LocalSpm::LocalSpm() :
      _components(),
      _threadFactory(1, "CONN"),
      _dbManager(COMPONENT_ID_DB_MANAGER),
      _dataProvider(COMPONENT_ID_DATA_PROVIDER),
      _autoConnectionController(COMPONENT_ID_AUTOCONNECTION_CONTROLLER),
      _resetToDefaultController(COMPONENT_ID_RESET_TO_DEFAULT_CONTROLLER),
      _pairingController(COMPONENT_ID_PAIRING_CONTROLLER),
      _btLimitationController(COMPONENT_ID_BT_LIMITATION_CONTROLLER),
      _bmCoreMainController(COMPONENT_ID_BM_CORE_MAIN_CONTROLLER),
      _bmController(COMPONENT_ID_BM_CONTROLLER),
      _conflictManager(COMPONENT_ID_CONFLICTMANAGER),
      _bmCoreCallbackIfWrapper(0),
      _waitCondition(),
      _waitMutex(),
      _timer(),
      _timerID(),
      _statisticsTimer(),
      _statisticsTimerID(),
      _spmState(SPM_STATE_OFF)
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("LocalSpm: is being created"));

      //cc::initErrorStrings();

      /* sets timer class for SPM */
      _timer.SetTimerClass(TIMER_CLASS_SPM);
   }

   LocalSpm& LocalSpm::getInstance(void)
   {
      static LocalSpm spm;
      return spm;
   }

   ThreadFactory& LocalSpm::getThreadFactory(void)
   {
      return getInstance()._threadFactory;
   }

   DataProvider& LocalSpm::getDataProvider(void)
   {
      return getInstance()._dataProvider;
   }

   DbManager& LocalSpm::getDbManager(void)
   {
      return getInstance()._dbManager;
   }

   AutoConnectionController& LocalSpm::getAutoConnectionController(void)
   {
      return getInstance()._autoConnectionController;
   }

   ResetToDefaultController& LocalSpm::getResetToDefaultController(void)
   {
      return getInstance()._resetToDefaultController;
   }

   BtLimitationController& LocalSpm::getBtLimitationController(void)
   {
      return getInstance()._btLimitationController;
   }

   PairingController& LocalSpm::getPairingController(void)
   {
      return getInstance()._pairingController;
   }

   BmCoreMainController& LocalSpm::getBmCoreMainController(void)
   {
      return getInstance()._bmCoreMainController;
   }

   BmController& LocalSpm::getBmController(void)
   {
      return getInstance()._bmController;
   }

   ConflictManager& LocalSpm::getConflictManager(void)
   {
      return getInstance()._conflictManager;
   }

   BmCoreCallbackIfWrapper& LocalSpm::getBmCoreCallbackIfWrapper(void)
   {
      return *(getInstance()._bmCoreCallbackIfWrapper);
   }

   Result LocalSpm::Register(ILocalSpm* component)
   {
      ETG_TRACE_USR1(("Register(component = 0x%p)", (void *) component));

      _components.push_back(component);

      return 0;
   }

   Result LocalSpm::Register(IBmCoreCallbackIf* bmCoreCallbackIf)
   {
      ENTRY

      ETG_TRACE_USR1(("Register: bmCoreCallbackIf = 0x%p", (void *) bmCoreCallbackIf));

      _bmCoreCallbackIfWrapper = new BmCoreCallbackIfWrapper(COMPONENT_ID_BM_CORE_CALLBACK_IF_WRAPPER, bmCoreCallbackIf);

      ETG_TRACE_USR1(("registering BmCoreCallbackIfWrapper: _bmCoreCallbackIfWrapper = 0x%p", (void *) _bmCoreCallbackIfWrapper));
      _components.push_back(_bmCoreCallbackIfWrapper);

      return 0;
   }

   Result LocalSpm::createBtConnectionMgr(void)
   {
      ENTRY

      ETG_TRACE_USR1(("createBtConnectionMgr"));

      ETG_TRACE_USR1(("registering BmCoreMainController: &_bmCoreMainController = 0x%p", (void *) &_bmCoreMainController));
      Register(&_bmCoreMainController);

      ETG_TRACE_USR1(("registering DbManager: &_dbManager = 0x%p", (void *) &_dbManager));
      Register(&_dbManager);

      ETG_TRACE_USR1(("registering DataProvider: &_dataProvider = 0x%p", (void *) &_dataProvider));
      Register(&_dataProvider);

      ETG_TRACE_USR1(("registering AutoConnectionController: &_autoConnectionController = 0x%p", (void *) &_autoConnectionController));
      Register(&_autoConnectionController);

      ETG_TRACE_USR1(("registering ResetToDefaultController: &_resetToDefaultController = 0x%p", (void *) &_resetToDefaultController));
      Register(&_resetToDefaultController);

      ETG_TRACE_USR1(("registering BtLimitationController: &_btLimitationController = 0x%p", (void *) &_btLimitationController));
      Register(&_btLimitationController);

      ETG_TRACE_USR1(("registering PairingController: &_pairingController = 0x%p", (void *) &_pairingController));
      Register(&_pairingController);

      ETG_TRACE_USR1(("registering BmController: &_bmController = 0x%p", (void *) &_bmController));
      Register(&_bmController);

      ETG_TRACE_USR1(("registering ConflictManager: &_conflictManager = 0x%p", (void *) &_conflictManager));
      Register(&_conflictManager);

      /*Create thread factory*/
      _threadFactory.Create();

      ETG_TRACE_USR1(("createBtConnectionMgr: loading configuration values"));

      // get the project specific configuration
      getDataProvider().updateConfiguration();

      ETG_TRACE_USR1(("createBtConnectionMgr: calling create() for all registered components"));
      /*Navigate Create to all the registered ILocalSpm based internal MP components*/
      unsigned int iter;
      for(iter=0; iter < _components.size(); iter++)
      {
         _components[iter]->create();
      }

      /*Create SPM internal State machines -> Start and Stop*/
      LocalSpmStartSm::Create();
      LocalSpmStopSm::Create();

      return 0;
   }

   Result LocalSpm::stateChangeNormal(void)
   {
      ENTRY

      ETG_TRACE_USR1(("stateChangeNormal"));

      // if in state OFF then switch to NORMAL
      if (SPM_STATE_OFF == _spmState)
      {
         /*register START State machine with dispatcher*/
         Dispatcher::GetInstance().Register(LocalSpmStartSm::GetInstance());

   #if 1 // switch off if you want to test only the smf test cases
         /*initialize LocalSpm Start State machine*/
         LocalSpmStartSm::Reset(); //LocalSpmStartSm::Init();

         /* enable SPM timers */
         _timer.EnableTimerClass(TIMER_CLASS_SPM);

         /* send all components to state normal by using the state machine */
         while(LocalSpmStartSm::STATE_MACHINE_FINISHED != LocalSpmStartSm::StateMachine_Main()) {}
         ETG_TRACE_USR1(("stateChangeNormal: end of LocalSpm Start state machine "));

         Dispatcher::GetInstance().DeRegister(LocalSpmStartSm::GetInstance());

         ETG_TRACE_USR1(("stateChangeNormal: canceling startup supervision timer"));
         _timer.CancelTimer(_timerID);

         /* enable all timers */
         _timer.EnableTimerClass(TIMER_CLASS_ALL);

         // set new state to Normal
         _spmState = SPM_STATE_NORMAL;

         _bmCoreMainController.onLocalSpmStateChanged(_spmState);

         return (isAllRun())?0:-1;
   #else
         return 0;
   #endif
      }

      // all other states
      return CC_ERR_INT_NO_ERROR;
   }

   Result LocalSpm::stateChangeOff(void)
   {
      ENTRY

      ETG_TRACE_USR1(("stateChangeOff"));

      // if in state NORMAL then switch to OFF
      if (SPM_STATE_NORMAL == _spmState)
      {
         // set new state to OFF
         _spmState = SPM_STATE_OFF;

         _bmCoreMainController.onLocalSpmStateChanged(_spmState);

         /*register STOP State machine with dispatcher*/
         Dispatcher::GetInstance().Register(LocalSpmStopSm::GetInstance());

         /* Initialize LocalSpm Stop State machine */
         LocalSpmStopSm::Reset();    //LocalSpmStopSm::Init();

         /* enable SPM timers only (disables all other timers) */
         _timer.EnableTimerClass(TIMER_CLASS_SPM);

         /* send all components to state off */
         while(LocalSpmStopSm::STATE_MACHINE_FINISHED != LocalSpmStopSm::StateMachine_Main()) {}
         ETG_TRACE_USR1(("stateChangeOff: end of LocalSpm Stop state machine "));

         ETG_TRACE_USR1(("stateChangeOff: destroying connection controller instances "));
         getBmController().destroyConnectionControllers();

         /* disables all timers */
         _timer.EnableTimerClass(TIMER_CLASS_NONE);

         // Dispatcher::GetInstance().DeRegister(LocalSpmStopSm::GetInstance());
         Dispatcher::GetInstance().DeRegisterAll();

         ETG_TRACE_USR1(("stateChangeOff: canceling shutdown supervision timer"));
         /* cancel the shutdown timer */
         _timer.CancelTimer(_timerID);

         return CC_ERR_INT_NO_ERROR;
      }
      // if in state UNDERVOLTAGE then switch to OFF. A state change of all components is not necessary.
      else if (SPM_STATE_UNDERVOLTAGE == _spmState)
      {
         // set new state to OFF
         _spmState = SPM_STATE_OFF;

         _bmCoreMainController.onLocalSpmStateChanged(_spmState);

         return CC_ERR_INT_NO_ERROR;
      }

      // all other states
      return CC_ERR_INT_NO_ERROR;
   }

   Result LocalSpm::stateChangeUndervoltage(const Undervoltage undervoltage)
   {
      ENTRY

      ETG_TRACE_USR1(("stateChangeUndervoltage: undervoltage = %d", undervoltage));

      // if an under voltage was detected and current state is NORMAL, then switch all components in OFF state
      if (undervoltage && SPM_STATE_NORMAL == _spmState)
      {
         /*register STOP State machine with dispatcher*/
         Dispatcher::GetInstance().Register(LocalSpmStopSm::GetInstance());

         /* Initialize LocalSpm Stop State machine */
         LocalSpmStopSm::Reset();    //LocalSpmStopSm::Init();

         /* send all components to state off */
         while(LocalSpmStopSm::STATE_MACHINE_FINISHED != LocalSpmStopSm::StateMachine_Main()) {}
         ETG_TRACE_USR1(("stateChangeUndervoltage: end of LocalSpm Stop state machine "));

         Dispatcher::GetInstance().DeRegister(LocalSpmStopSm::GetInstance());

         _timer.CancelTimer(_timerID);

         // set new state to UNDERVOLTAGE
         _spmState = SPM_STATE_UNDERVOLTAGE;

         _bmCoreMainController.onLocalSpmStateChanged(_spmState);

         return (isAllInDone())?0:-1;
      }
      // if under voltage is finished and and current state is UNDERVOLTAGE, then switch all components in NORMAL state
      else if (false == undervoltage && SPM_STATE_UNDERVOLTAGE == _spmState)
      {

         /*register START State machine with dispatcher*/
         Dispatcher::GetInstance().Register(LocalSpmStartSm::GetInstance());

         /*initialize LocalSpm Start State machine*/
         LocalSpmStartSm::Reset(); //LocalSpmStartSm::Init();

         /* send all components to state normal by using the state machine */
         while(LocalSpmStartSm::STATE_MACHINE_FINISHED != LocalSpmStartSm::StateMachine_Main()) {}
         ETG_TRACE_USR1(("stateChangeUndervoltage: end of LocalSpm Start state machine "));

         Dispatcher::GetInstance().DeRegister(LocalSpmStartSm::GetInstance());

         ETG_TRACE_USR1(("stateChangeUndervoltage: canceling shutdown supervision timer"));
         _timer.CancelTimer(_timerID);

         // set new state to Normal
         _spmState = SPM_STATE_NORMAL;

         _bmCoreMainController.onLocalSpmStateChanged(_spmState);

         return (isAllRun())?0:-1;
      }

      // all other states
      return CC_ERR_INT_NO_ERROR;
   }

   SpmState LocalSpm::getSpmState(void) const
   {
      return _spmState;
   }

   void LocalSpm::printStatistics()
   {
      /* ask for statics data and print them */

      FILE *fpStat = fopen("/tmp/BMCoreStat.log", "w");
      if (!fpStat) return;

      /* header */
      time_t t = time(NULL);
      struct tm tm = *localtime(&t);
      fprintf(fpStat, "BM Core Statistics:\n");;
      fprintf(fpStat, "time: %d-%02d-%02d %02d:%02d:%02d\n", tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);

      /* entries from the components */
      size_t iter;
      int ret = 0;
      for(iter=_components.size(); iter > 0; iter--)
      {
         Statistics stat;

         ret = _components[iter-1]->statistics(OUT stat);
         if (ret) break;

         if (!strlen(stat)) continue;

         fprintf(fpStat, "%s: %s", _components[iter-1]->getComponentName(), stat);
         fprintf(fpStat, "\n");
      }

      /* end of log */
      fclose(fpStat);

      return;
   }

   LocalSpm::~LocalSpm()
   {
      ENTRY_INTERNAL

      /* work around for undefined destruction sequence between static and dynamic objects,
       * by this call the Dispatcher is not beeing destroyed before the local spm is destroyed
       * must be solved by a clean solution */
      Dispatcher::GetInstance().ForceNoDestruction();

      delete _bmCoreCallbackIfWrapper;
      _bmCoreCallbackIfWrapper = 0;

      _timerID = 0;
      _statisticsTimerID = 0;

      // GetThreadFactory().KillAllThreads();
   }

   int LocalSpm::initAll(void)
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("initAll"));

      // trace current BM Core version
      ETG_TRACE_FATAL(("BM Core version: ID: \"%50s\", date: %50s", STRINGIZE_VALUE_OF(VERSION_ID_OF_BM_CORE), STRINGIZE_VALUE_OF(VERSION_DATE_OF_BM_CORE)));

      (void) _timer.StartSMTimer(OUT _timerID, getDataProvider().SPMStartupTimeout(), 0L, "LocalSpmStartSm::TIMEOUT");

      unsigned int iter;
      int ret = 0;
      InitReason initReason;

      // if the state is UNDERVOLTAGE then the Init function must be called with reason IR_UNDERVOLTAGE
      if (SPM_STATE_UNDERVOLTAGE == _spmState)
         initReason = IR_UNDERVOLTAGE;
      else
         initReason = IR_BOOT;

      for(iter=0; iter < _components.size(); iter++)
      {
         ret = _components[iter]->init(IN initReason);
         if (ret) break;
      }
      return ret;
   }

   int LocalSpm::runAll()
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("runAll"));

      unsigned int iter;
      int ret = 0;
      for(iter=0; iter < _components.size(); iter++)
      {
         ret = _components[iter]->run();
         if (ret) break;
      }

      ETG_TRACE_USR1(("runAll: starting statistics reporting timer (interval = 2000 ms)"));

      _statisticsTimer.StartTimer(OUT _statisticsTimerID, 2000L /*ms*/, 2000L /* interval */,
            this, LocalSpm::statisticsTimer, this);

      return ret;
   }

   int LocalSpm::stopAll(void)
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("stopAll"));

      ETG_TRACE_USR1(("stopAll: canceling statistics reporting timer"));

      /* cancel the statistics timer */
      _statisticsTimer.CancelTimer(IN _statisticsTimerID);

      ETG_TRACE_USR1(("stopAll: starting shutdown supervision timer (timeout = %d)", this->getDataProvider().SPMShutdownTimeout()));

      /* start shutdown supervision timer */
      (void) _timer.StartSMTimer(OUT _timerID, getDataProvider().SPMShutdownTimeout(), 0L, "LocalSpmStopSm::TIMEOUT");

      size_t iter;
      int ret = 0;

      for(iter=_components.size(); iter > 0; iter--)
      {
         ret = _components[iter-1]->stop();
         if (ret) break;
      }

      return ret;
   }

   int LocalSpm::doneAllSm(void)
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("doneAllSm"));

      size_t iter;
      int ret = 0;

      for(iter=_components.size(); iter > 0; iter--)
      {
         if (_components[iter-1]->isComponentSm())
         {
            ret = _components[iter-1]->done();
            if (ret) break;
         }
      }
      return ret;
   }

   int LocalSpm::doneAll(void)
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("doneAll"));

      size_t iter;
      int ret = 0;

      for(iter=_components.size(); iter > 0; iter--)
      {
         if (false == _components[iter-1]->isComponentSm())
         {
            ret = _components[iter-1]->done();
            if (ret) break;
         }
      }
   #ifdef ENABLE_MEDIA_ENGINE
      sleep(1);
   #endif
      return ret;
   }

   int LocalSpm::isAllInit()
   {
      ENTRY_INTERNAL

      return checkComponentsState(COMPONENT_STATE_INIT);
   }

   int LocalSpm::isAllRun()
   {
      ENTRY_INTERNAL

      return checkComponentsState(COMPONENT_STATE_RUNNING);
   }

   int LocalSpm::isAllInStop()
   {
      ENTRY_INTERNAL

      return checkComponentsState(COMPONENT_STATE_STOP);
   }

   int LocalSpm::isAllSmInDone()
   {
      ENTRY_INTERNAL

      return checkComponentsState(COMPONENT_STATE_DONE, COMPONENT_GROUP_SM);
   }

   int LocalSpm::isAllInDone()
   {
      ENTRY_INTERNAL

      return checkComponentsState(COMPONENT_STATE_DONE, COMPONENT_GROUP_NON_SM);
   }

   int LocalSpm::checkComponentsState(ComponentState state, ComponentGroup group /*= COMPONENT_GROUP_ALL*/)
   {
      ETG_TRACE_USR1(("checkComponentsState: state = %d, group = %d", ETG_CENUM(ComponentState, state), ETG_CENUM(ComponentGroup, group)));

      unsigned int iter;

      for(iter=0; iter < _components.size(); iter++)
      {
         if((COMPONENT_GROUP_SM == group) && (false == _components[iter]->isComponentSm()))
            continue;

         if((COMPONENT_GROUP_NON_SM == group) && (_components[iter]->isComponentSm()))
            continue;

         if(state != _components[iter]->getState())
            return 0;
      }

      /*all the components are in the requested state*/
      return 1 ;
   }

   int LocalSpm::setState(ComponentId componentID, ComponentState componentState)
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("setState: componentID = %d, componentState = %d", ETG_CENUM(ComponentId, componentID), ETG_CENUM(ComponentState, componentState)));

      unsigned int iter;
      for(iter=0; iter < _components.size(); iter++)
      {
         if(componentID == _components[iter]->getComponentId())
         {
            /*Set the state for component with ID:componentID */
            _components[iter]->setState(componentState);
            /*Inform the statemachine with  the state set response*/
            if((componentState == COMPONENT_STATE_INIT ) || (componentState == COMPONENT_STATE_RUNNING ))
            {
            	//!Fix for CID 20235: Unchecked return value (CHECKED_RETURN)
               (void)LocalSpmStartSm::SendEvent(LocalSpmStartSm::SET_STATE_DONE, NULL);
            }
            else if((componentState == COMPONENT_STATE_STOP ) || (componentState == COMPONENT_STATE_DONE ))
            {
            	//!Fix for CID 20235: Unchecked return value (CHECKED_RETURN)
               (void)LocalSpmStopSm::SendEvent(LocalSpmStopSm::SET_STATE_DONE, NULL);
            }
            return 0;
         }
      }

      /*component is not registered with the LocalSpm*/
      return -1 ;
   }

   int LocalSpm::setInitState(ComponentId componentID)
   {
      return setState(componentID, COMPONENT_STATE_INIT);
   }

   int LocalSpm::setRunState(ComponentId componentID)
   {
      return setState(componentID, COMPONENT_STATE_RUNNING);
   }

   int LocalSpm::setStopState(ComponentId componentID)
   {
      return setState(componentID, COMPONENT_STATE_STOP);
   }

   int LocalSpm::setDoneState(ComponentId componentID)
   {
      return setState(componentID, COMPONENT_STATE_DONE);
   }

   int LocalSpm::startupTimedout(void)
   {
      ENTRY_INTERNAL

      ETG_TRACE_FATAL(("BM Core startup timed out and hence aborted !! "));

      logComponentStates();

      return 0;
   }

   int LocalSpm::shutdownTimedout(void)
   {
      ENTRY_INTERNAL

      ETG_TRACE_FATAL(("BM Core shutdown timed out and hence aborted !!  "));
      Dispatcher::GetInstance().DeRegisterAll();

      logComponentStates();

      return 0;
   }

   void LocalSpm::logComponentStates(void)
   {
      ENTRY_INTERNAL

      unsigned int iter;

      for(iter=0; iter < _components.size(); iter++)
      {
         tGeneralString smStateName;
         ETG_TRACE_ERRMEM(("logComponentStates: Component (%32s) State is -> %32s[%32s]",
               _components[iter]->getComponentName(), _components[iter]->getStateName(),
               _components[iter]->getSmStateName(smStateName, sizeof(smStateName))));
      }

      DeviceConnectionStatusList deviceConnectionStatusList;
      (void) this->getBmCoreMainController().getDeviceConnectionStatusList(deviceConnectionStatusList);

      if (false == deviceConnectionStatusList._deviceConnectionInfoList.empty())
      {
         ETG_TRACE_ERRMEM(("logComponentStates: connected devices:"));

         BdAddress bdAddress("");
         ProtocolConnectionInfoMap::iterator it;
         SppConnectionInfoMap::iterator itSpp;

         for (size_t idx = 0u; idx < deviceConnectionStatusList._deviceConnectionInfoList.size(); ++idx)
         {
            Result result = this->getDbManager().getBdAddressByDeviceHandle(bdAddress,
                  deviceConnectionStatusList._deviceConnectionInfoList[idx]._deviceHandle);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               bdAddress = "";
            }

            ETG_TRACE_ERRMEM(("logComponentStates: device handle = %u, BD address = \"%50s\", connection status = %d, disconnected reason = %d",
                  deviceConnectionStatusList._deviceConnectionInfoList[idx]._deviceHandle, bdAddress.c_str(),
                  ETG_CENUM(ConnectionStatus, deviceConnectionStatusList._deviceConnectionInfoList[idx]._connectionStatus),
                  ETG_CENUM(DisconnectedReason, deviceConnectionStatusList._deviceConnectionInfoList[idx]._disconnectedReason)));

            for (it = deviceConnectionStatusList._deviceConnectionInfoList[idx]._protocolConnectionInfo.begin(); it != deviceConnectionStatusList._deviceConnectionInfoList[idx]._protocolConnectionInfo.end(); ++it)
            {
               if (BM_CONNECTION_STATUS_DISCONNECTED != it->second._connectionStatus)
               {
                  ETG_TRACE_ERRMEM(("logComponentStates: protocol = %d, uuid = \"\", connection status = %d, disconnected reason = %d",
                        ETG_CENUM(ProtocolId, it->first), ETG_CENUM(ConnectionStatus, it->second._connectionStatus),
                        ETG_CENUM(DisconnectedReason, it->second._disconnectedReason)));
               }
            }

            for (itSpp = deviceConnectionStatusList._deviceConnectionInfoList[idx]._sppConnectionInfo.begin(); itSpp != deviceConnectionStatusList._deviceConnectionInfoList[idx]._sppConnectionInfo.end(); ++itSpp)
            {
               if (BM_CONNECTION_STATUS_DISCONNECTED != itSpp->second._connectionStatus)
               {
                  ETG_TRACE_ERRMEM(("logComponentStates: protocol = %d, uuid = \"%50s\", connection status = %d, disconnected reason = %d",
                        ETG_CENUM(ProtocolId, BM_PROTOCOL_ID_SPP), itSpp->first.c_str(),
                        ETG_CENUM(ConnectionStatus, itSpp->second._connectionStatus),
                        ETG_CENUM(DisconnectedReason, itSpp->second._disconnectedReason)));
               }
            }
         }
      }
      else
      {
         ETG_TRACE_ERRMEM(("logComponentStates: no connected devices"));
      }

      /* send myself a SIGUSR2 to trace out all callstacks of all threads */
      raise(34);
   }

   bool LocalSpm::statisticsTimer(timer_t timerID , void *pObject, const void *userData)
   {
      (void) timerID;
      (void) userData;

      static_cast<LocalSpm*>(pObject)->printStatistics();

      return 0;
   }
}
