/**
 * @addtogroup LocalSpm
 * 
 * @author Matthias Thömel , Dinesh D
 *
 * Public Interface for all Local SPM Clients
 * @{
 */

#ifndef _I_LOCAL_SPM_H_
#define _I_LOCAL_SPM_H_

#include "BmAllTypes.h"
#include "Timer.h"
#include <string.h>

namespace bmcore
{
   /**
    * Class ILocalSpm defines the LocalSpm interface which all the components are expected to implement
    */
   class ILocalSpm : public Timer
   {

   private:

      ComponentId _componentID; /**< component Id assigned by the LocalSpm 		*/
      ComponentState _componentState; /**< current Local SPM state of the component 	*/

   public:

      ILocalSpm(ComponentId componentID) :
            _componentID(componentID), _componentState(COMPONENT_STATE_UNKNOWN)
      {
      }

      ComponentState getState(void)
      {
         return _componentState;
      }

      ComponentId getComponentId(void)
      {
         return _componentID;
      }

      const char* getStateName(void) const;
      const char* getComponentName(void) const;

      void setState(ComponentState componentState);

      /**
       * allocate memory, create components, call constructors, create threads but don't let them run
       * no access to other components allowed
       *
       * @return < 0: error, = 0: OK
       */
      virtual void create()
      {
         createDone(0);
      }

      /**
       * initialize variables, open databases creates queues, don't use foreign components
       *
       * @return < 0: error, = 0: OK
       */
      virtual int init(InitReason reason)
      {
         (void) reason;

         return initDone(0);
      }

      /**
       * start threads, start state machines, other components are now usable
       *
       * @return < 0: error, = 0: OK
       */
      virtual int run()
      {
         return runDone(0);
      }

      /**
       * stop state machines, store last mode , stop the running threads
       *
       * @return < 0: error, = 0: OK
       */
      virtual int stop()
      {
         return stopDone(0);
      }

      /**
       *  delete message queues, close databases, don't access foreign components
       *
       * @return < 0: error, = 0: OK
       */
      virtual int done()
      {
         return doneDone(0);
      }

      /**
       * Inform Start State machine with create answer (uses Dispatcher interface)
       *
       * @return < 0: error, = 0: OK
       */
      int createDone(int error);

      /**
       * Inform Start State machine with Init answer (uses Dispatcher interface internally)
       *
       * @return < 0: error, = 0: OK
       */
      int initDone(int error);

      /**
       * Answer from the component when it reaches the state RUNNING
       *
       * @return < 0: error, = 0: OK
       */
      int runDone(int error);

      /**
       * Inform Stop State machine with Stop answer (uses Dispatcher interface internally)
       *
       * @return < 0: error, = 0: OK
       */
      int stopDone(int error);

      /**
       * Inform Stop StateMachine with Done answer (uses Dispatcher interface internally)
       *
       * @return < 0: error, = 0: OK
       */
      int doneDone(int error);

      /**
       * return a statistic string dependent on the component
       *
       * @return < 0: error, = 0: OK
       */
      virtual int statistics(OUT Statistics stat)
      {
         stat[0] = 0;
         return 0;
      }

      /**
       * return the name of the curent state the state machine is in or "" if no state machine class
       *
       * @return < 0: error, = 0: OK
       */
      virtual char *getSmStateName(OUT tGeneralString stateName, IN size_t size);

      /**
       * return answer if the component is a state machine
       *
       * @return true or false
       */
      virtual bool isComponentSm();

   };
}

#endif // _I_LOCAL_SPM_H_
/** @} */
