/**
 * @file BtStackIfConnectionCallbackIf.cpp
 *
 * @swcomponent BluetoothConnectionManagerCore
 *
 * @brief This file contains the definition of the class BtStackIfConnectionCallbackIf
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BmControllerModule
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BtStackIfConnectionCallbackIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#endif
#endif


#include "BtStackIfConnectionCallbackIf.h"
#include "BmCoreIfMessagesCreator.h"
#include "BmCoreMainController.h"
#include "LocalSpm.h"
#include "BmUtils.h"
#include "FunctionTracer.h"
#include "BmGlobalLock.h"

namespace bmcore
{
   BtStackIfConnectionCallbackIf::BtStackIfConnectionCallbackIf(BmCoreMainController& bmCoreMainController) :
         _bmCoreMainController(bmCoreMainController),
         _blockCallback(false)
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("BtStackIfConnectionCallbackIf: is being created"));
   }

   BtStackIfConnectionCallbackIf::~BtStackIfConnectionCallbackIf()
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("~BtStackIfConnectionCallbackIf"));
   }

   void BtStackIfConnectionCallbackIf::fbConnectionInitializedCB(IN const BTSRequestResult btsResult)
   {
      ETG_TRACE_USR1(("fbConnectionInitializedCB: btsResult = %d", ETG_CENUM(BTSRequestResult, btsResult)));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("fbConnectionInitializedCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsFbConnectionInitialized* bmCoreIfMessage = getNewBmCoreIfMessage_BtsFbConnectionInitialized(btsResult,
            0u);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("fbConnectionInitializedCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::localVersionInfoCB(IN const BTSLocalVersionInfo& btsVersionInfo)
   {
      ETG_TRACE_USR1(("localVersionInfoCB: btsStackType = %d, btsStackVersion = \"%50s\", btsModuleName = \"%50s\", btsBtVersion = %d, btsFwVersion = \"%50s\"",
            ETG_CENUM(BTSInterfaceType, btsVersionInfo.stackType), btsVersionInfo.stackVersion.c_str(),
            btsVersionInfo.moduleName.c_str(), ETG_CENUM(BTSBtCoreVersion, btsVersionInfo.btVersion),
            btsVersionInfo.fwVersion.c_str()));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("localVersionInfoCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsLocalVersionInfo* bmCoreIfMessage = getNewBmCoreIfMessage_BtsLocalVersionInfo(btsVersionInfo,
            0u);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("localVersionInfoCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::switchBluetoothOnOffResultCB(IN const BTSRequestResult btsResult)
   {
      ETG_TRACE_USR1(("switchBluetoothOnOffResultCB: btsResult = %d", ETG_CENUM(BTSRequestResult, btsResult)));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("switchBluetoothOnOffResultCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsSwitchBluetoothOnOffResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsSwitchBluetoothOnOffResult(btsResult,
            0u);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("switchBluetoothOnOffResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::currentBluetoothOnOffStatusCB(IN const BTSBluetoothMode btsBluetoothMode,
         IN const BTSBluetoothOffReason btsBluetoothOffReason, IN const BTSAdapterMode btsAdapterMode,
         IN const BTSBDAddress& btsLocalDeviceAddress)
   {
      ETG_TRACE_USR1(("currentBluetoothOnOffStatusCB: btsBluetoothMode = %d, btsBluetoothOffReason = %d, btsAdapterMode = %d, btsLocalDeviceAddress = \"%50s\"",
            ETG_CENUM(BTSBluetoothMode, btsBluetoothMode), ETG_CENUM(BTSBluetoothOffReason, btsBluetoothOffReason),
            ETG_CENUM(BTSAdapterMode, btsAdapterMode), btsLocalDeviceAddress.c_str()));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("currentBluetoothOnOffStatusCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsCurrentBluetoothOnOffStatus* bmCoreIfMessage = getNewBmCoreIfMessage_BtsCurrentBluetoothOnOffStatus(btsBluetoothMode,
            btsBluetoothOffReason, btsAdapterMode, btsLocalDeviceAddress, 0u);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("currentBluetoothOnOffStatusCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::switchBluetoothHardwareOnOffResultCB(IN const BTSRequestResult btsResult)
   {
      ETG_TRACE_USR1(("switchBluetoothHardwareOnOffResultCB: btsResult = %d", ETG_CENUM(BTSRequestResult, btsResult)));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("switchBluetoothHardwareOnOffResultCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsBluetoothHardwareOnOffResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsBluetoothHardwareOnOffResult(btsResult,
            0u);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("switchBluetoothHardwareOnOffResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::setLocalBtNameResultCB(IN const BTSRequestResult btsResult)
   {
      ETG_TRACE_USR1(("setLocalBtNameResultCB: btsResult = %d", ETG_CENUM(BTSRequestResult, btsResult)));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("setLocalBtNameResultCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsSetLocalBtNameResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsSetLocalBtNameResult(btsResult,
            0u);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("setLocalBtNameResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::currentLocalBtNameCB(IN const BTSDeviceName& btsAdapterName)
   {
      ETG_TRACE_USR1(("currentLocalBtNameCB: btsAdapterName = \"%50s\"", btsAdapterName.c_str()));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("currentLocalBtNameCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsCurrentLocalBtName* bmCoreIfMessage = getNewBmCoreIfMessage_BtsCurrentLocalBtName(btsAdapterName);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("currentLocalBtNameCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::currentDeviceForRestrictedPairingConnectingCB(IN const BTSBDAddress& btsExpectedDeviceAddress,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("currentDeviceForRestrictedPairingConnectingCB: btsExpectedDeviceAddress = \"%50s\", btsSessionHandle = %u",
            btsExpectedDeviceAddress.c_str(), btsSessionHandle));

      if (_blockCallback)
      {
         ETG_TRACE_USR1(("currentDeviceForRestrictedPairingConnectingCB: block"));
         return;
      }
   }

   void BtStackIfConnectionCallbackIf::setLocalAdapterModesResultCB(IN const BTSRequestResult btsResult, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("setLocalAdapterModesResultCB: btsResult = %d, btsSessionHandle = %u",
            ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("setLocalAdapterModesResultCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsSetLocalAdapterModesResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsSetLocalAdapterModesResult(btsResult,
            btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("setLocalAdapterModesResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::currentDiscoverableModeCB(IN const BTSLocalMode btsDiscoverableMode,
         IN const BTSLocalModeOffReason btsOffReason, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("currentDiscoverableModeCB: btsDiscoverableMode = %d, btsOffReason = %d, btsSessionHandle = %u",
            ETG_CENUM(BTSLocalMode, btsDiscoverableMode), ETG_CENUM(BTSLocalModeOffReason, btsOffReason),
            btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("currentDiscoverableModeCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsCurrentDiscoverableMode* bmCoreIfMessage = getNewBmCoreIfMessage_BtsCurrentDiscoverableMode(btsDiscoverableMode,
            btsOffReason, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("currentDiscoverableModeCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::currentConnectableModeCB(IN const BTSLocalMode btsConnectableMode,
         IN const BTSLocalModeOffReason btsOffReason, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("currentConnectableModeCB: btsConnectableMode = %d, btsOffReason = %d, btsSessionHandle = %u",
            ETG_CENUM(BTSLocalMode, btsConnectableMode), ETG_CENUM(BTSLocalModeOffReason, btsOffReason),
            btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("currentConnectableModeCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsCurrentConnectableMode* bmCoreIfMessage = getNewBmCoreIfMessage_BtsCurrentConnectableMode(btsConnectableMode,
            btsOffReason, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("currentConnectableModeCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::startDiscoveryResultCB(IN const BTSRequestResult btsResult,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("startDiscoveryResultCB: btsResult = %d, btsSessionHandle = %u",
            ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("startDiscoveryResultCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsStartDiscoveryResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsStartDiscoveryResult(btsResult,
            btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("startDiscoveryResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::stopDiscoveryResultCB(IN const BTSRequestResult btsResult,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("stopDiscoveryResultCB: btsResult = %d, btsSessionHandle = %u",
            ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("stopDiscoveryResultCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsStopDiscoveryResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsStopDiscoveryResult(btsResult,
            btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("stopDiscoveryResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::discoveringStatusCB(IN const BTSDiscoveringStatus btsDiscoveringStatus,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("discoveringStatusCB: btsDiscoveringStatus = %d, btsSessionHandle = %u",
            ETG_CENUM(BTSDiscoveringStatus, btsDiscoveringStatus), btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("discoveringStatusCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsDiscoveringStatus* bmCoreIfMessage = getNewBmCoreIfMessage_BtsDiscoveringStatus(btsDiscoveringStatus,
            btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("discoveringStatusCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::discoveredDeviceFoundCB(IN const BTSDiscoveredDeviceInfo& btsDiscoveredDevice,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("discoveredDeviceFoundCB: btsDiscoveredDevice.address = \"%50s\", btsDiscoveredDevice.name = \"%50s\", btsSessionHandle = %u",
            btsDiscoveredDevice.address.c_str(), btsDiscoveredDevice.name.c_str(), btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("discoveredDeviceFoundCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsDiscoveredDeviceFound* bmCoreIfMessage = getNewBmCoreIfMessage_BtsDiscoveredDeviceFound(btsDiscoveredDevice,
            btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("discoveredDeviceFoundCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::startPairingResultCB(IN const BTSBDAddress& btsDeviceAddress,
         IN const BTSRequestResult btsResult, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("startPairingResultCB: btsDeviceAddress = \"%50s\", btsResult = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      // no global lock because internal message is created

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("startPairingResultCB: block"));
         return;
      }

      BmCoreIfMessage_BtsStartPairingResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsStartPairingResult(btsDeviceAddress,
            btsResult, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("startPairingResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);

      return;
   }

   void BtStackIfConnectionCallbackIf::pinCodeRequestedCB(IN const BTSBDAddress& btsDeviceAddress, IN const BTSDeviceName& btsDeviceName,
         IN const BTSPairConnectOriginator btsPairingOriginator, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("pinCodeRequestedCB: btsDeviceAddress = \"%50s\", btsDeviceName = \"%50s\", btsPairingOriginator = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), btsDeviceName.c_str(), btsPairingOriginator, btsSessionHandle));

      // no global lock because internal message is created

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("pinCodeRequestedCB: block"));
         // reject request
         _bmCoreMainController.getBtStackIfConnectionRequestIfWrapper().setPinCode(btsDeviceAddress, "");
         return;
      }

      BmCoreIfMessage_BtsPinCodeRequested* bmCoreIfMessage = getNewBmCoreIfMessage_BtsPinCodeRequested(btsDeviceAddress,
            btsDeviceName, btsPairingOriginator, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("pinCodeRequestedCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);

      return;
   }

   void BtStackIfConnectionCallbackIf::secureSimplePairingRequestedCB(IN const BTSBDAddress& btsDeviceAddress,
         IN const BTSDeviceName& btsDeviceName, IN const BTSSspMode btsSspMode, IN const BTSNumericValue& btsNumericValue,
         IN const BTSPairConnectOriginator btsPairingOriginator, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("secureSimplePairingRequestedCB: btsDeviceAddress = \"%50s\", btsDeviceName = \"%50s\", btsSspMode = %d, btsNumericValue = \"%50s\", btsPairingOriginator = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), btsDeviceName.c_str(), btsSspMode, btsNumericValue.c_str(), btsPairingOriginator, btsSessionHandle));

      // no global lock because internal message is created

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("secureSimplePairingRequestedCB: block"));
         // reject request
         _bmCoreMainController.getBtStackIfConnectionRequestIfWrapper().confirmSecureSimplePairing(btsDeviceAddress,
               btsSspMode, btsNumericValue, BTS_CONFIRM_REJECT);
         return;
      }

      BmCoreIfMessage_BtsSecureSimplePairingRequested* bmCoreIfMessage = getNewBmCoreIfMessage_BtsSecureSimplePairingRequested(btsDeviceAddress,
            btsDeviceName, btsSspMode, btsNumericValue, btsPairingOriginator, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("secureSimplePairingRequestedCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);

      return;
   }

   void BtStackIfConnectionCallbackIf::pairingFinishedCB(IN const BTSBDAddress& btsDeviceAddress, IN const BTSRequestResult btsResult,
         IN const BTSLinkKeyType btsLinkKeyType, IN const BTSLinkKey& btsLinkKey, IN const BTSDLinkKey& btsDLinkKey,
         IN const BTSMajorServiceClass& btsMajorServiceClass, IN const BTSMajorDeviceClass btsMajorDeviceClass,
         IN const BTSMinorDeviceClass btsMinorDeviceClass, IN const BTSDeviceName& btsDeviceName,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("pairingFinishedCB: btsDeviceAddress = \"%50s\", btsDeviceName = \"%50s\", btsResult = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), btsDeviceName.c_str(), ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("pairingFinishedCB: block"));
         return;
      }

      BmCoreIfMessage_BtsPairingFinished* bmCoreIfMessage = getNewBmCoreIfMessage_BtsPairingFinished(btsDeviceAddress,
            btsResult, btsLinkKeyType, btsLinkKey, btsDLinkKey, btsMajorServiceClass, btsMajorDeviceClass,
            btsMinorDeviceClass, btsDeviceName, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("pairingFinishedCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::linkKeyRequestedCB(IN const BTSBDAddress& btsDeviceAddress,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("linkKeyRequestedCB: btsDeviceAddress = \"%50s\", btsSessionHandle = %u",
            btsDeviceAddress.c_str(), btsSessionHandle));

      if (_blockCallback)
      {
         ETG_TRACE_USR1(("linkKeyRequestedCB: block"));
         // reject request => wait for stack timeout
         return;
      }

      BmCoreIfMessage_BtsLinkKeyRequested* bmCoreIfMessage = getNewBmCoreIfMessage_BtsLinkKeyRequested(btsDeviceAddress,
            btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("linkKeyRequestedCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::connectProtocolResultCB(IN const BTSBDAddress& btsDeviceAddress,
         IN const BTSRequestResult btsResult, IN const BTSProtocolId btsProtocolId, IN const BTSUuid& btsSppUuid,
         IN const BTSMasInstanceName& btsMasInstance, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("connectProtocolResultCB: btsDeviceAddress = \"%50s\", btsProtocolId = %d, btsSppUuid = \"%50s\", btsMasInstance = \"%50s\", btsResult = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), ETG_CENUM(BTSProtocolId, btsProtocolId), btsSppUuid.c_str(), btsMasInstance.c_str(),
            ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      // no global lock because internal message is created

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("connectProtocolResultCB: block"));
         return;
      }

      BmCoreIfMessage_BtsConnectProtocolResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsConnectProtocolResult(btsDeviceAddress,
            btsResult, btsProtocolId, btsSppUuid, btsMasInstance, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("connectProtocolResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);

      return;
   }

   void BtStackIfConnectionCallbackIf::remoteProtocolConnectRequestCB(IN const BTSBDAddress& btsDeviceAddress,
         IN const BTSProtocolId btsProtocolId, IN const BTSUuid& btsSppUuid, IN const BTSSessionHandle btsSessionHandle)
   {
      // TODO: @Stefan+Muthu: btsSppUuid added, needed for remote SPP connect
      ETG_TRACE_USR1(("remoteProtocolConnectRequestCB: btsDeviceAddress = \"%50s\", btsProtocolId = %d, btsSppUuid =\"%50s\", btsSessionHandle = %u",
            btsDeviceAddress.c_str(), ETG_CENUM(BTSProtocolId, btsProtocolId), btsSppUuid.c_str(), btsSessionHandle));

      // no global lock because internal message is created

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("remoteProtocolConnectRequestCB: block"));
         return;
      }

      BmCoreIfMessage_BtsRemoteProtocolConnectRequest* bmCoreIfMessage = getNewBmCoreIfMessage_BtsRemoteProtocolConnectRequest(btsDeviceAddress,
            btsProtocolId, btsSppUuid, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("remoteProtocolConnectRequestCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);

      return;
   }

   void BtStackIfConnectionCallbackIf::disconnectProtocolResultCB(IN const BTSBDAddress& btsDeviceAddress, IN const BTSRequestResult btsResult,
         IN const BTSProtocolId btsProtocolId, IN const BTSUuid& btsSppUuid, IN const BTSMasInstanceName& btsMasInstance,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("disconnectProtocolResultCB: btsDeviceAddress = \"%50s\", btsProtocolId = %d, btsSppUuid = \"%50s\", btsMasInstance = \"%50s\", btsResult = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), ETG_CENUM(BTSProtocolId, btsProtocolId), btsSppUuid.c_str(), btsMasInstance.c_str(),
            ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      // no global lock because internal message is created

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("disconnectProtocolResultCB: block"));
         return;
      }

      BmCoreIfMessage_BtsDisconnectProtocolResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsDisconnectProtocolResult(btsDeviceAddress,
            btsResult, btsProtocolId, btsSppUuid, btsMasInstance, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("disconnectProtocolResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);

      return;
   }

   void BtStackIfConnectionCallbackIf::disconnectDeviceResultCB(IN const BTSBDAddress& btsDeviceAddress,
         IN const BTSRequestResult btsResult, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("disconnectDeviceResultCB: btsDeviceAddress = \"%50s\", btsResult = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      // no global lock because internal message is created

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("disconnectDeviceResultCB: block"));
         return;
      }

      BmCoreIfMessage_BtsDisconnectDeviceResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsDisconnectDeviceResult(btsDeviceAddress,
            btsResult, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("disconnectDeviceResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);

      return;
   }

   void BtStackIfConnectionCallbackIf::protocolConnectionStatusCB(IN const BTSBDAddress& btsDeviceAddress,
         IN const BTSConnectionStatus btsConnectionStatus, IN const BTSDisconnectReason btsDisconnectReason,
         IN const BTSProtocolId btsProtocolId, IN const BTSUuid& btsSppUuid,
         IN const BTSMasInstanceName& btsMasInstance, IN const BTSDeviceName& btsRfCommDevice,
         IN const BTSSessionHandle btsSessionHandle)
   {
      std::string helperString("protocolConnectionStatusCB: btsSppUuid = \"");
      helperString += btsSppUuid;
      helperString += "\"";
      helperString += ", btsMasInstance = \"";
      helperString += btsMasInstance;
      helperString += "\"";
      helperString += ", btsRfCommDevice = \"";
      helperString += btsRfCommDevice;
      helperString += "\"";

      ETG_TRACE_USR1(("protocolConnectionStatusCB: btsDeviceAddress = \"%50s\", btsProtocolId = %d, %100s, btsConnectionStatus = %d, btsDisconnectReason = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), ETG_CENUM(BTSProtocolId, btsProtocolId), helperString.c_str(),
            ETG_CENUM(BTSConnectionStatus, btsConnectionStatus), ETG_CENUM(BTSDisconnectReason, btsDisconnectReason),
            btsSessionHandle));

      // no global lock because internal message is created

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("protocolConnectionStatusCB: block"));
         return;
      }

      BmCoreIfMessage_BtsProtocolConnectionStatus* bmCoreIfMessage = getNewBmCoreIfMessage_BtsProtocolConnectionStatus(btsDeviceAddress,
            btsConnectionStatus, btsDisconnectReason, btsProtocolId, btsSppUuid, btsMasInstance, btsRfCommDevice,
            btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("protocolConnectionStatusCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);

      return;
   }

   void BtStackIfConnectionCallbackIf::deviceConnectionStatusCB(IN const BTSBDAddress& btsDeviceAddress,
         IN const BTSConnectionStatus btsConnectionStatus, IN const BTSDisconnectReason btsDisconnectReason,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("deviceConnectionStatusCB: btsDeviceAddress = \"%50s\", btsConnectionStatus = %d, btsDisconnectReason = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), ETG_CENUM(BTSConnectionStatus, btsConnectionStatus),
            ETG_CENUM(BTSDisconnectReason, btsDisconnectReason), btsSessionHandle));

      // no global lock because internal message is created

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("deviceConnectionStatusCB: block"));
         return;
      }

      BmCoreIfMessage_BtsDeviceConnectionStatus* bmCoreIfMessage = getNewBmCoreIfMessage_BtsDeviceConnectionStatus(btsDeviceAddress,
            btsConnectionStatus, btsDisconnectReason, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("deviceConnectionStatusCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);

      return;
   }

   void BtStackIfConnectionCallbackIf::deviceCapabilitiesCB(IN const BTSBDAddress& btsDeviceAddress, IN const BTSDeviceCapabilities& btsDeviceCapabilities,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("deviceCapabilitiesCB: btsDeviceAddress = \"%50s\", btsDeviceCapabilities.inbandRingtoneSupported = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), btsDeviceCapabilities.inbandRingtoneSupported, btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("deviceCapabilitiesCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsDeviceCapabilities* bmCoreIfMessage = getNewBmCoreIfMessage_BtsDeviceCapabilities(btsDeviceAddress,
            btsDeviceCapabilities, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("deviceCapabilitiesCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::remoteSupportedServicesCB(IN const BTSBDAddress& btsDeviceAddress, IN const BTSRequestResult btsResult,
         IN const BTSSupportedServices& btsSupportedServices, IN const BTSServiceInfoList& btsServiceInfoList,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("remoteSupportedServicesCB: btsDeviceAddress = \"%50s\", btsResult = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("remoteSupportedServicesCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsSupportedServices* bmCoreIfMessage = getNewBmCoreIfMessage_BtsSupportedServices(btsDeviceAddress,
            btsResult, btsSupportedServices, btsServiceInfoList, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("remoteSupportedServicesCB(btsDeviceAddress = \"%50s\"): bmCoreIfMessage is 0",
               btsDeviceAddress.c_str()));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::remoteSppCapabilitiesCB(IN const BTSBDAddress& btsDeviceAddress, IN const BTSRequestResult btsResult,
         IN const BTSSppCapabilityList& btsSppCapabilities, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("remoteSppCapabilitiesCB: btsDeviceAddress = \"%50s\", btsResult = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("remoteSppCapabilitiesCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsSppCapabilities* bmCoreIfMessage = getNewBmCoreIfMessage_BtsSppCapabilities(btsDeviceAddress,
            btsResult, btsSppCapabilities, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("remoteSppCapabilitiesCB(btsDeviceAddress = \"%50s\"): bmCoreIfMessage is 0",
               btsDeviceAddress.c_str()));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::remoteDeviceIdServiceRecordsCB(IN const BTSBDAddress& btsDeviceAddress, IN const BTSRequestResult btsResult,
         IN const BTSDeviceIdServiceRecordList& btsDeviceIdServiceRecords, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("remoteDeviceIdServiceRecordsCB: btsDeviceAddress = \"%50s\", btsResult = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("remoteDeviceIdServiceRecordsCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsRemoteDeviceIdServiceRecords* bmCoreIfMessage = getNewBmCoreIfMessage_BtsRemoteDeviceIdServiceRecords(btsDeviceAddress,
            btsResult, btsDeviceIdServiceRecords, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("remoteDeviceIdServiceRecordsCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::remoteServiceSearchResultCB(IN const BTSBDAddress& btsBdAddress,
         IN const BTSRequestResult btsSupportedServicesResult, IN const BTSRequestResult btsSppCapabilitiesResult,
         IN const BTSRequestResult btsDeviceIdServiceRecordsResult, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("remoteServiceSearchResultCB: btsBdAddress = \"%50s\", btsSupportedServicesResult = %d, btsSppCapabilitiesResult = %d, btsDeviceIdServiceRecordsResult = %d, btsSessionHandle = %u",
            btsBdAddress.c_str(), ETG_CENUM(BTSRequestResult, btsSupportedServicesResult),
            ETG_CENUM(BTSRequestResult, btsSppCapabilitiesResult),
            ETG_CENUM(BTSRequestResult, btsDeviceIdServiceRecordsResult), btsSessionHandle));

      if (_blockCallback)
      {
         ETG_TRACE_USR1(("remoteServiceSearchResultCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsRemoteServiceSearchResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsRemoteServiceSearchResult(btsBdAddress,
            btsSupportedServicesResult, btsSppCapabilitiesResult, btsDeviceIdServiceRecordsResult, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("remoteServiceSearchResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::remoteNameResultCB(IN const BTSBDAddress& btsBdAddress,
         IN const BTSRequestResult btsResult, IN const BTSDeviceName& btsBdName,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("remoteNameResultCB: btsBdAddress = \"%50s\", btsBdName = \"%50s\", btsResult = %d, btsSessionHandle = %u",
            btsBdAddress.c_str(), btsBdName.c_str(), ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      if (_blockCallback)
      {
         ETG_TRACE_USR1(("remoteNameResultCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsRemoteNameResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsRemoteNameResult(btsBdAddress,
            btsBdName, btsResult, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("remoteNameResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::remoteNameStatusCB(IN const BTSBDAddress& btsBdAddress, IN const BTSDeviceName& btsDeviceName,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("remoteNameStatusCB: btsBdAddress = \"%50s\", btsDeviceName = \"%50s\", btsSessionHandle = %u",
            btsBdAddress.c_str(), btsDeviceName.c_str(), btsSessionHandle));

      if (_blockCallback)
      {
         ETG_TRACE_USR4(("remoteNameStatusCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsRemoteNameStatusUpdate* bmCoreIfMessage = getNewBmCoreIfMessage_BtsRemoteNameStatusUpdate(btsBdAddress,
    		  btsDeviceName, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("remoteNameStatusCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::linkQualityResultCB(IN const BTSBDAddress& btsDeviceAddress, IN const BTSRequestResult btsResult,
         IN const BTSRssi btsRssi, IN const BTSLinkQuality btsLinkQuality, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("linkQualityResultCB: btsDeviceAddress = \"%50s\", btsResult = %d, btsRssi = %d, btsLinkQuality = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), ETG_CENUM(BTSRequestResult, btsResult), btsRssi, btsLinkQuality, btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("linkQualityResultCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsLinkQualityResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsLinkQualityResult(btsDeviceAddress,
            btsResult, btsRssi, btsLinkQuality, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("linkQualityResultCB(btsDeviceAddress = \"%50s\"): bmCoreIfMessage is 0", btsDeviceAddress.c_str()));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::startTestModeResultCB(IN const BTSRequestResult btsResult,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("startTestModeResultCB: btsResult = %d, btsSessionHandle = %u",
            ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("startTestModeResultCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsStartTestModeResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsStartTestModeResult(btsResult,
            btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("startTestModeResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::stopTestModeResultCB(IN const BTSRequestResult btsResult,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("stopTestModeResultCB: btsResult = %d, btsSessionHandle = %u",
            ETG_CENUM(BTSRequestResult, btsResult), btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("stopTestModeResultCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsStopTestModeResult* bmCoreIfMessage = getNewBmCoreIfMessage_BtsStopTestModeResult(btsResult,
            btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("stopTestModeResultCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::testModeStatusCB(IN const BTSBDAddress& btsDeviceAddress,
         IN const BTSConnectionStatus btsConnectionStatus,
         IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("testModeStatusCB: btsDeviceAddress = \"%50s\", btsConnectionStatus = %d, btsSessionHandle = %u",
            btsDeviceAddress.c_str(), ETG_CENUM(BTSConnectionStatus, btsConnectionStatus), btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("testModeStatusCB: block"));
         return;
      }

      // no global lock because internal message is created

      BmCoreIfMessage_BtsTestModeStatus* bmCoreIfMessage = getNewBmCoreIfMessage_BtsTestModeStatus(btsDeviceAddress,
            btsConnectionStatus, btsSessionHandle);

      if (0 == bmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("testModeStatusCB: bmCoreIfMessage is 0"));
         return;
      }

      _bmCoreMainController.pushBmCoreIfMessage(bmCoreIfMessage);
   }

   void BtStackIfConnectionCallbackIf::testModeLinkQualityResultCB(IN const BTSRequestResult btsResult,
         IN const BTSRssi btsRssi, IN const BTSLinkQuality btsLinkQuality, IN const BTSSessionHandle btsSessionHandle)
   {
      ETG_TRACE_USR1(("testModeLinkQualityResultCB: btsResult = %d, btsRssi = %d, btsLinkQuality = %d, btsSessionHandle = %u",
            ETG_CENUM(BTSRequestResult, btsResult), btsRssi, btsLinkQuality, btsSessionHandle));

      if(_blockCallback)
      {
         ETG_TRACE_USR1(("testModeLinkQualityResultCB: block"));
         return;
      }

      // CCA layer: in requestTestModeLinkQualityResponse() Diaglib handler is called; in Diaglib handler no further access to BmCore => global lock can be used
      BM_GLOBAL_LOCKER_BTSTACKIF

      BmResult bmResult(BM_RESULT_OK);

      LinkQualityInfo testModeLinkQualityInfo(btsRssi, btsLinkQuality);

      if (BTS_REQ_SUCCESS != btsResult)
      {
         testModeLinkQualityInfo._rssi = 127;
         testModeLinkQualityInfo._linkQuality = 0xFFu;

         ETG_TRACE_ERR(("testModeLinkQualityResultCB: requesting link quality in test mode failed (btsResult = %d)",
               ETG_CENUM(BTSRequestResult, btsResult)));

         bmResult = BM_RESULT_ERR_GENERAL;
      }

      LocalSpm::getBmCoreCallbackIfWrapper().doRequestTestModeLinkQualityResponse(bmResult, testModeLinkQualityInfo);
   }

}
