/**
 * @defgroup BtConnectionManagerInterfaceModule BT Connection Manager Interface
 * @ingroup BluetoothConnectionManagerCore
 * 
 * @brief This module comprises all parts related to BM Interface
 *
 * @details A detailed description is not yet available
 */

/**
 * @file IBmCoreRequestIf.h
 * 
 * @swcomponent BluetoothConnectionManagerCore
 *
 * @brief This file contains the declaration of the class IBmCoreRequestIf
 * 
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BtConnectionManagerInterfaceModule
 *
 */

#ifndef _I_BM_CORE_REQUEST_IF_H_
#define _I_BM_CORE_REQUEST_IF_H_

#include "BmCoreIfTypes.h"

namespace bmcore
{
   //using namespace cc;

   class IBmCoreCallbackIf;

   /**
    * @class IBmCoreRequestIf
    * @ingroup BtConnectionManagerInterfaceModule
    * 
    * @brief The abstract class IBmCoreRequestIf declares the BT Connection Manager interface.
    * 
    */
   class IBmCoreRequestIf
   {
   public:

      /**
       * @brief Destructor of class IBmCoreRequestIf
       *
       * @return void
       */
      virtual ~IBmCoreRequestIf() {};

      /**
       * Function to register custom output wrapper component with BtConnectionMgr
       * This function must be called before the function CreateBtConnectionMgr is called
       *
       * @param[in] outputWrapper pointer to custom output wrapper object which derives from the OutputWrapper class
       *
       * @return int = 0: ok, !=0: failed
       */
      virtual BmResult registerBmCoreCallbackIf(IBmCoreCallbackIf* bmCoreCallbackIf) = 0;


      /**
       * Function to create all internal Mediaplayer components
       * This function must be called before the function StateChangeNormal is called
       *
       * @return int = 0: ok, !=0: failed
       */
      virtual BmResult initializeBmCore(void) = 0;

      /**
       * Function to request Mediaplayer transition to NORMAL state
       * This function must be called before the BtConnectionMgr can be used
       *
       * @return int = 0: ok, !=0: failed
       */
      virtual BmResult stateChangeNormal(IN const ActType act = 0u) = 0;

      /**
       * Function to request Mediaplayer transition to OFF state
       * After calling of this function the BtConnectionMgr functions cannot be used anymore.
       * If afterwards the BtConnectionMgr is required again, function StateChangeNormal must be called first again.
       *
       * @return int = 0: ok, !=0: failed
       */
      virtual BmResult stateChangeOff(IN const ActType act = 0u) = 0;

      virtual BmResult setBtSystemState(IN const BtSystemState& btSystemState, IN const ActType act = 0u) = 0;

      virtual BmResult setProtocolUserAvailability(IN const Protocol& protocol, IN const bool available, IN const ActType act = 0u) = 0;

      virtual BmResult getBtSystemState(OUT BtSystemState& btSystemState) = 0;

      virtual BmResult switchBtStatus(IN const TargetSwitchState targetSwitchState, IN const bool suppressAutoConnectionAfterSwitchedOn = false,
            IN const bool delayBtOnRequest = false, IN const ActType act = 0u) = 0;

      virtual BmResult getBtStatus(OUT SwitchStatus& bluetoothStatus) = 0;

      virtual BmResult switchLocalPairableMode(IN const TargetSwitchState targetSwitchState,
            IN const BdAddress& bdAddress, IN const ActType act = 0u) = 0;

      virtual BmResult getLocalPairableMode(OUT SwitchStatus& localPairableMode) = 0;

      virtual BmResult switchLocalConnectableMode(IN const TargetSwitchState targetSwitchState,
            IN const BdAddress& bdAddress, IN const ActType act = 0u) = 0;

      virtual BmResult getLocalConnectableMode(OUT SwitchStatus& localConnectableMode) = 0;

      virtual BmResult switchDiscoveryStatus(IN const TargetSwitchState targetSwitchState,
            IN const ActType act = 0u) = 0;

      virtual BmResult getDiscoveryStatus(OUT SwitchStatus& discoveryStatus) = 0;

      virtual BmResult getDiscoveredDeviceList(OUT DiscoveredDeviceList& discoveredDeviceList) = 0;

      virtual BmResult setBtLocalFriendlyName(IN const LocalFriendlyName& localFriendlyName,
            IN const ActType act = 0u) = 0;

      virtual BmResult getBtLocalFriendlyName(OUT LocalFriendlyName& localFriendlyName) = 0;

      virtual BmResult getBtLocalInfo(OUT LocalInfo& localInfo) = 0;

      virtual BmResult startPairing(IN const BdAddress& bdAddress, IN const ActType act = 0u) = 0;

      virtual BmResult cancelPairing(IN const ActType act = 0u) = 0;

      virtual BmResult acceptPairingRequest(IN const ActType act = 0u) = 0;

      virtual BmResult rejectPairingRequest(IN const ActType act = 0u) = 0;

      virtual BmResult getPairingStatus(OUT PairingStatus& pairingStatus) = 0;

      virtual BmResult setPairingPin(IN const PairingPin& pairingPin, IN const ActType act = 0u) = 0;

      virtual BmResult getPairingPin(OUT PairingPin& pairingPin) = 0;

      virtual BmResult getPairedDeviceList(OUT PairedDeviceList& pairedDeviceList) = 0;

      virtual BmResult addOobPairedDevice(IN const BdAddress& bdAddress, IN const BdName& bdName,
            IN const LinkKey& linkKey, IN const OobType oobType, IN const ActType act = 0u) = 0;

      virtual BmResult deleteDevice(IN const DeviceId deviceHandle, IN const DeviceHandleType deviceHandleType,
            IN const ActType act = 0u) = 0;

      virtual BmResult connectDevice(IN const DeviceId deviceHandle, IN const ActType act = 0u) = 0;

      virtual BmResult disconnectDevice(IN const DeviceId deviceHandle, IN const DeviceHandleType deviceHandleType,
            IN const ActType act = 0u) = 0;

      virtual BmResult connectProfiles(IN const DeviceId deviceHandle, IN const ProtocolList& protocolList,
            IN const ActType act = 0u) = 0;

      virtual BmResult disconnectProfiles(IN const DeviceId deviceHandle, IN const ProtocolList& protocolList,
            IN const ActType act = 0u) = 0;

      virtual BmResult storeDeviceRingtoneIdInDb(IN const DeviceId deviceHandle, IN const RingtoneId ringtoneId,
            IN const ActType act = 0u) = 0;

      virtual BmResult readDeviceRingtoneIdFromDb(OUT RingtoneId& ringtoneId, IN const DeviceId deviceHandle) = 0;

      virtual BmResult storeSystemWideRingtoneIdInDb(IN const RingtoneId ringtoneId,
            IN const ActType act = 0u) = 0;

      virtual BmResult readSystemWideRingtoneIdFromDb(OUT RingtoneId& ringtoneId) = 0;

      virtual BmResult getRemoteDeviceInfo(OUT RemoteDeviceInfo& remoteDeviceInfo, IN const DeviceId deviceHandle) = 0;

      virtual BmResult getDeviceConnectionStatusList(OUT DeviceConnectionStatusList& deviceConnectionStatusList) = 0;

      virtual BmResult blockDevice(IN const DeviceId deviceHandle, IN const DeviceHandleType deviceHandleType,
            IN const ActType act = 0u) = 0;

      virtual BmResult unblockDevice(IN const DeviceId deviceHandle, IN const DeviceHandleType deviceHandleType,
            IN const ActType act = 0u) = 0;

      virtual BmResult blockProfiles(IN const DeviceId deviceHandle, IN const ProtocolList& protocolList,
            IN const ActType act = 0u) = 0;

      virtual BmResult unblockProfiles(IN const DeviceId deviceHandle, IN const ProtocolList& protocolList,
            IN const ActType act = 0u) = 0;

      virtual BmResult getBlockStatus(OUT BlockStatus& blockStatus) = 0;

      virtual BmResult setFavorite(IN const DeviceId deviceHandle, IN const FavoriteType favoriteType,
            IN const FavoriteIndex favoriteIndex, IN const ActType act = 0u) = 0;

      virtual BmResult clearFavorite(IN const DeviceId deviceHandle, IN const FavoriteType favoriteType,
            IN const ActType act = 0u) = 0;

      virtual BmResult startServiceSearch (IN const DeviceId deviceHandle, IN const ActType act = 0u) = 0;

      virtual BmResult stopServiceSearch (IN const DeviceId deviceHandle, IN const ActType act = 0u) = 0;

      virtual BmResult switchMultiHFPSupport (IN const TargetSwitchState targetSwitchState, IN const ActType act = 0u) = 0;

      virtual BmResult setAutoConnectionType(IN const AutoConnectionType autoConnectionType,
            IN const ActType act = 0u) = 0;

      virtual BmResult getAutoConnectionType(OUT AutoConnectionType& autoConnectionType) = 0;

      virtual BmResult startAutoConnectSequence(IN const TemporaryAutoConnectionType autoConnectionType,
            IN const ActType act = 0u) = 0;

      virtual BmResult stopAutoConnectSequence(IN const ActType act = 0u) = 0;

      virtual BmResult getBmConfigData(OUT BmConfigData& bmConfigData) = 0;

      virtual void updateLinkQuality(IN const ActType act = 0u) = 0;

      virtual BmResult getLinkQuality(OUT LinkQuality& linkQuality) = 0;

      virtual void setBtLimitationMode(IN const BdAddress& bdAddress, IN const BdName& bdName, IN const LimitationMode& limitationMode,
            IN const LimitationAction limitationAction, IN const ActType act = 0u) = 0;

      virtual BmResult getBtLimitationMode(OUT BtLimitationMode& btLimitationMode) = 0;

      virtual void replaceBtLimitationMode(IN const BtLimitationModeInfo& limitationModeToBeReplaced, IN const BdAddress& bdAddress,
            IN const BdName& bdName, IN const LimitationMode& limitationMode, IN const LimitationAction limitationAction, IN const ActType act = 0u) = 0;

      virtual void notifyUserDecision(IN const UserDecisionInfo userDecisionInfo, IN const ActType act = 0u) = 0;

      virtual BmResult resetToDefault(IN const bool factorySettingsChanged = true, IN const ActType act = 0u) = 0;

      virtual BmResult getResetToDefaultStatus(OUT ResetToDefaultStatus& status) = 0;

      virtual BmResult switchTestMode(IN const TargetSwitchState& targetSwitchState, IN const BdAddress& bdAddress,
            IN const ActType act = 0u) = 0;

      virtual BmResult getTestMode(OUT SwitchStatus& testMode) = 0;

      virtual BmResult requestTestModeLinkQuality(IN const ActType act = 0u) = 0;

      virtual BmResult getLinkQualityTestMode(OUT TestModeLinkQuality& linkQuality) = 0;

      virtual BmResult setPrimaryHfpDevice(IN const DeviceId deviceHandle) = 0;

      virtual void setBtProfileUsageInfo(IN const ProfileUsageList& profileUsageInfoList, IN const ActType act = 0u) = 0;
	  
      virtual BmResult getMultiHFPSupport(OUT SwitchStatus& multiHFPSupport) = 0;

      virtual BmResult setDeviceUsagePreference(IN const DeviceId deviceHandle, IN const UsagePreference usagePreference, IN const ActType act = 0u) = 0;

      virtual BmResult getDeviceUsagePreference(OUT UsagePreference& usagePreference, IN const DeviceId deviceHandle) = 0;

   protected:

   private:

   };
}

#endif //_I_BM_CORE_REQUEST_IF_H_

