/**
 * @file ServiceSearchController.cpp
 *
 * @par SW-Component
 * Bluetooth Connection Manager Core
 *
 * @brief This file contains the definition of the class ServiceSearchController
 *
 * @copyright (c) 2017 Robert Bosch GmbH
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BmCoreModule
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ServiceSearchController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#endif
#endif

#include "ServiceSearchController.h"
#include "BmMeetsDeviceHandle.h"
#include "BmVarTrace.h"
#include <algorithm> // required for using find_if

namespace bmcore
{
   ServiceSearchController::ServiceSearchController(BmCoreMainController& bmCoreMainController) :
         _bmCoreMainController(bmCoreMainController), _deviceServiceSearchControlList()
   {
   }

   ServiceSearchController::~ServiceSearchController()
   {
   }

   Result ServiceSearchController::addDeviceForServiceSearch(const DeviceId deviceHandle)
   {
      bool deviceIsAvailable = false;
      ETG_TRACE_USR1(("addDeviceForServiceSearch: deviceHandle = %d", deviceHandle));

      // Check whether device is already available in the list
      for (size_t idx = 0u; idx < _deviceServiceSearchControlList.size(); ++idx)
      {
         if(_deviceServiceSearchControlList[idx]->getDeviceHandle() == deviceHandle)
         {
            deviceIsAvailable = true;
            break;
         }
      }

      if(false == deviceIsAvailable)
      {
         DeviceServiceSearchControl *deviceServiceSearchControl = new DeviceServiceSearchControl(deviceHandle);

         if(deviceServiceSearchControl)
         {
            ETG_TRACE_USR4(("addDeviceForServiceSearch: added in the deviceServiceSearchControlList"));
            _deviceServiceSearchControlList.push_back(deviceServiceSearchControl);
         }
         else
         {
            ETG_TRACE_ERR(("addDeviceForServiceSearch: not able to create memory for deviceServiceSearchControl"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("addDeviceForServiceSearch: device service search control with given device handle is already in device service search control list"));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ServiceSearchController::removeDeviceFromServiceSearch(const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("removeDeviceFromServiceSearch: deviceHandle = %d", deviceHandle));

      for (size_t idx = 0u; idx < _deviceServiceSearchControlList.size(); ++idx)
      {
         if(_deviceServiceSearchControlList[idx]->getDeviceHandle() == deviceHandle)
         {
            DeviceServiceSearchControl *deviceServiceSearchControl =
                  static_cast<DeviceServiceSearchControl*>(_deviceServiceSearchControlList[idx]);

            _deviceServiceSearchControlList.erase(_deviceServiceSearchControlList.begin() + idx);

            ETG_TRACE_USR4(("removeDeviceFromServiceSearch: erased the device from list"));

            delete deviceServiceSearchControl;

            break;
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ServiceSearchController::startServiceSearchTimer(const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("startServiceSearchTimer: deviceHandle = %d", deviceHandle));

      if (0u == deviceHandle)
      {
         ETG_TRACE_USR4(("startServiceSearchTimer: starting service search timer for ALL paired devices"));

         for (size_t idx = 0u; idx < _deviceServiceSearchControlList.size(); ++idx)
         {
            (void) _deviceServiceSearchControlList[idx]->startServiceSearchTimer();
         }
      }
      else
      {
         for (size_t idx = 0u; idx < _deviceServiceSearchControlList.size(); ++idx)
         {
            if(_deviceServiceSearchControlList[idx]->getDeviceHandle() == deviceHandle)
            {
               ETG_TRACE_USR4(("startServiceSearchTimer: starting service search timer for device with given device handle"));
               (void) _deviceServiceSearchControlList[idx]->startServiceSearchTimer();
               break;
            }
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ServiceSearchController::stopServiceSearchTimer(const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("stopServiceSearchTimer: deviceHandle = %d", deviceHandle));

      if (0u == deviceHandle)
      {
         ETG_TRACE_USR4(("stopServiceSearchTimer: stopping service search timer for ALL paired devices"));

         for (size_t idx = 0u; idx < _deviceServiceSearchControlList.size(); ++idx)
         {
            (void) _deviceServiceSearchControlList[idx]->stopServiceSearchTimer();
         }
      }
      else
      {
         for (size_t idx = 0u; idx < _deviceServiceSearchControlList.size(); ++idx)
         {
            if(_deviceServiceSearchControlList[idx]->getDeviceHandle() == deviceHandle)
            {
               ETG_TRACE_USR4(("stopServiceSearchTimer: stopping service search timer for device with given device handle"));
               (void) _deviceServiceSearchControlList[idx]->stopServiceSearchTimer();
               break;
            }
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result ServiceSearchController::serviceSearchFinished(const DeviceId deviceHandle)
   {
      ETG_TRACE_USR1(("serviceSearchFinished: deviceHandle = %d", deviceHandle));

      for (size_t idx = 0u; idx < _deviceServiceSearchControlList.size(); ++idx)
      {
         if(_deviceServiceSearchControlList[idx]->getDeviceHandle() == deviceHandle)
         {
            ETG_TRACE_USR4(("serviceSearchFinished: starting service search timer for device with given device handle"));
            (void) _deviceServiceSearchControlList[idx]->startServiceSearchTimer();
            break;
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }
}
