/**
 * @file DeviceServiceSearchControl.cpp
 *
 * @par SW-Component
 * Bluetooth Connection Manager Core
 *
 * @brief This file contains the definition of the class DeviceServiceSearchControl
 *
 * @copyright (c) 2017 Robert Bosch GmbH
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BmCoreModule
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DeviceServiceSearchControl.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#endif
#endif

#include "DeviceServiceSearchControl.h"
#include "BmCoreIfMessagesCreator.h"
#include "LocalSpm.h"

namespace bmcore
{
   DeviceServiceSearchControl::DeviceServiceSearchControl(const DeviceId deviceHandle) :
         _deviceHandle(deviceHandle), _serviceSearchTimer(), _serviceSearchTimerId(0),
         _serviceSearchTimerLock()
   {
   }

   DeviceServiceSearchControl::DeviceServiceSearchControl(IN const DeviceServiceSearchControl& other) :
         _deviceHandle(other._deviceHandle),
         _serviceSearchTimer(other._serviceSearchTimer),
         _serviceSearchTimerId(other._serviceSearchTimerId),
         _serviceSearchTimerLock(other._serviceSearchTimerLock)
   {
   }

   DeviceServiceSearchControl::~DeviceServiceSearchControl()
   {
   }

   void swap(DeviceServiceSearchControl& first, DeviceServiceSearchControl& second)
   {
      using std::swap;

      swap(first._deviceHandle, second._deviceHandle);
      swap(first._serviceSearchTimer, second._serviceSearchTimer);
      swap(first._serviceSearchTimerId, second._serviceSearchTimerId);
      swap(first._serviceSearchTimerLock, second._serviceSearchTimerLock);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   DeviceServiceSearchControl& DeviceServiceSearchControl::operator=(DeviceServiceSearchControl other)
   {
      swap(*this, other);

      return *this;
   }

   Result DeviceServiceSearchControl::startServiceSearchTimer()
   {
      _serviceSearchTimerLock.lock();

      if (0 != _serviceSearchTimerId)
      {
         ETG_TRACE_USR4(("startServiceSearchTimer: stopping the already running service search timer (_serviceSearchTimerId = 0x%p)",
               _serviceSearchTimerId));

         _serviceSearchTimer.CancelTimer(_serviceSearchTimerId);
         _serviceSearchTimerId = 0;
      }

      long intervalMilliseconds = 1000L * static_cast<long>(LocalSpm::getDataProvider().getBmCoreConfiguration()._serviceSearchTimeoutSeconds);

      if (true == _serviceSearchTimer.StartTimer(_serviceSearchTimerId, intervalMilliseconds, 0L,
            this, DeviceServiceSearchControl::serviceSearchTimerCb, 0))
      {
         ETG_TRACE_USR4(("startServiceSearchTimer: started the service search timer (_serviceSearchTimerId = 0x%p, intervalMilliseconds = %d)",
               _serviceSearchTimerId, intervalMilliseconds));
      }
      else
      {
         ETG_TRACE_ERR(("startServiceSearchTimer: starting the service search timer failed"));

         _serviceSearchTimerId = 0;
      }

      _serviceSearchTimerLock.unlock();

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceServiceSearchControl::stopServiceSearchTimer()
   {
      _serviceSearchTimerLock.lock();

      if (0 != _serviceSearchTimerId)
      {
         ETG_TRACE_USR4(("stopServiceSearchTimer: stopping the service search timer (_serviceSearchTimerId = 0x%p)",
               _serviceSearchTimerId));

         _serviceSearchTimer.CancelTimer(_serviceSearchTimerId);
         _serviceSearchTimerId = 0;
      }
      else
      {
         ETG_TRACE_USR4(("stopServiceSearchTimer: the service search timer is already stopped"));
      }

      _serviceSearchTimerLock.unlock();

      return CC_ERR_INT_NO_ERROR;
   }

   bool DeviceServiceSearchControl::serviceSearchTimerCb(timer_t timerId, void *object, const void *userData)
   {
      (void) timerId;
      (void) userData;

      DeviceServiceSearchControl* deviceServiceSearchControl =
            static_cast<DeviceServiceSearchControl*>(object);

      if (0 != deviceServiceSearchControl)
      {
         ETG_TRACE_USR1(("serviceSearchTimerCb: service search timer expired for device with device handle = %d",
               deviceServiceSearchControl->getDeviceHandle()));

         (void) deviceServiceSearchControl->startServiceSearch();
      }
      else
      {
         ETG_TRACE_FATAL(("serviceSearchTimerCb: deviceServiceSearchControl is 0"));
      }

      return true;
   }

   Result DeviceServiceSearchControl::startServiceSearch()
   {
      _serviceSearchTimerLock.lock();

      Result result(CC_ERR_INT_NO_ERROR);

      if (0 != _serviceSearchTimerId)
      {
         ETG_TRACE_USR4(("startServiceSearch: entered"));

         BmCoreIfMessage_StartServiceSearchRequest* bmCoreIfMessage = getNewBmCoreIfMessage_StartServiceSearchRequest(_deviceHandle,
               0u, BM_CORE_IF_MSG_ORIGIN_INTERNAL);

         if (0 == bmCoreIfMessage)
         {
            ETG_TRACE_FATAL(("startServiceSearch: bmCoreIfMessage is 0"));

            result = CC_ERR_INT_MEMORY_ALLOCATION_ERROR;
         }
         else
         {
            LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);
         }

         _serviceSearchTimerId = 0;
      }
      else
      {
         ETG_TRACE_USR4(("startServiceSearch: service search timer has been stopped already -> will not send service search"));
      }

      _serviceSearchTimerLock.unlock();

      return result;
   }
}
