/**
 * @file DeviceBlockRemoteConnectionsControl.cpp
 *
 * @par SW-Component
 * Bluetooth Connection Manager Core
 *
 * @brief This file contains the definition of the class DeviceBlockRemoteConnectionsControl
 *
 * @copyright (c) 2017 Robert Bosch GmbH
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BmCoreModule
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DeviceBlockRemoteConnectionsControl.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_BMCONTROLLER
#endif
#endif

#include "DeviceBlockRemoteConnectionsControl.h"
#include "BmCoreIfMessagesCreator.h"
#include "LocalSpm.h"

namespace bmcore
{
   DeviceBlockRemoteConnectionsControl::DeviceBlockRemoteConnectionsControl(const DeviceId deviceHandle) :
         _deviceHandle(deviceHandle), _blockRemoteConnectionsTimer(), _blockRemoteConnectionsTimerId(0),
         _blockRemoteConnectionsTimerLock()
   {
   }

   DeviceBlockRemoteConnectionsControl::DeviceBlockRemoteConnectionsControl(IN const DeviceBlockRemoteConnectionsControl& other) :
         _deviceHandle(other._deviceHandle),
         _blockRemoteConnectionsTimer(other._blockRemoteConnectionsTimer),
         _blockRemoteConnectionsTimerId(other._blockRemoteConnectionsTimerId),
         _blockRemoteConnectionsTimerLock(other._blockRemoteConnectionsTimerLock)
   {
   }

   DeviceBlockRemoteConnectionsControl::~DeviceBlockRemoteConnectionsControl()
   {
   }

   void swap(DeviceBlockRemoteConnectionsControl& first, DeviceBlockRemoteConnectionsControl& second)
   {
      using std::swap;

      swap(first._deviceHandle, second._deviceHandle);
      swap(first._blockRemoteConnectionsTimer, second._blockRemoteConnectionsTimer);
      swap(first._blockRemoteConnectionsTimerId, second._blockRemoteConnectionsTimerId);
      swap(first._blockRemoteConnectionsTimerLock, second._blockRemoteConnectionsTimerLock);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   DeviceBlockRemoteConnectionsControl& DeviceBlockRemoteConnectionsControl::operator=(DeviceBlockRemoteConnectionsControl other)
   {
      swap(*this, other);

      return *this;
   }
/*
   bool DeviceBlockRemoteConnectionsControl::operator==(const DeviceBlockRemoteConnectionsControl& rhs) const
   {
      return ((this->_deviceHandle == rhs._deviceHandle)
            && (this->_blockRemoteConnectionsTimer == rhs._blockRemoteConnectionsTimer)
            && (this->_blockRemoteConnectionsTimerId == rhs._blockRemoteConnectionsTimerId));
   }
*/
   Result DeviceBlockRemoteConnectionsControl::startBlockRemoteConnectionsTimer()
   {
      _blockRemoteConnectionsTimerLock.lock();

      if (0 != _blockRemoteConnectionsTimerId)
      {
         ETG_TRACE_USR1(("startBlockRemoteConnectionsTimer: stopping the already running block remote connections timer (_blockRemoteConnectionsTimerId = 0x%p)",
               _blockRemoteConnectionsTimerId));

         _blockRemoteConnectionsTimer.CancelTimer(_blockRemoteConnectionsTimerId);
         _blockRemoteConnectionsTimerId = 0;
      }

      long intervalMilliseconds = 1000L * static_cast<long>(LocalSpm::getDataProvider().getBmCoreConfiguration()._blockDeviceRemoteConnectionsTimeoutSeconds);

      if (true == _blockRemoteConnectionsTimer.StartTimer(_blockRemoteConnectionsTimerId, intervalMilliseconds, 0L,
            this, DeviceBlockRemoteConnectionsControl::blockRemoteConnectionsTimerCb, 0))
      {
         ETG_TRACE_USR1(("startBlockRemoteConnectionsTimer: started the block remote connections timer (_blockRemoteConnectionsTimerId = 0x%p, intervalMilliseconds = %d)",
               _blockRemoteConnectionsTimerId, intervalMilliseconds));
      }
      else
      {
         ETG_TRACE_ERR(("startBlockRemoteConnectionsTimer: starting the block remote connections timer failed"));

         _blockRemoteConnectionsTimerId = 0;
      }

      _blockRemoteConnectionsTimerLock.unlock();

      return CC_ERR_INT_NO_ERROR;
   }

   Result DeviceBlockRemoteConnectionsControl::stopBlockRemoteConnectionsTimer()
   {
      _blockRemoteConnectionsTimerLock.lock();

      if (0 != _blockRemoteConnectionsTimerId)
      {
         ETG_TRACE_USR1(("stopBlockRemoteConnectionsTimer: stopping the block remote connections timer (_blockRemoteConnectionsTimerId = 0x%p)",
               _blockRemoteConnectionsTimerId));

         _blockRemoteConnectionsTimer.CancelTimer(_blockRemoteConnectionsTimerId);
         _blockRemoteConnectionsTimerId = 0;
      }
      else
      {
         ETG_TRACE_USR1(("stopBlockRemoteConnectionsTimer: the block remote connections timer is already stopped"));
      }

      _blockRemoteConnectionsTimerLock.unlock();

      return CC_ERR_INT_NO_ERROR;
   }

   bool DeviceBlockRemoteConnectionsControl::blockRemoteConnectionsTimerCb(timer_t timerId, void *object, const void *userData)
   {
      (void) timerId;
      (void) userData;

      DeviceBlockRemoteConnectionsControl* deviceBlockRemoteConnectionsControl =
            static_cast<DeviceBlockRemoteConnectionsControl*>(object);

      if (0 != deviceBlockRemoteConnectionsControl)
      {
         ETG_TRACE_USR1(("blockRemoteConnectionsTimerCb: block remote connections timer expired for device with device handle = %d",
               deviceBlockRemoteConnectionsControl->getDeviceHandle()));

         deviceBlockRemoteConnectionsControl->setDeviceRemoteConnectable();
      }
      else
      {
         ETG_TRACE_FATAL(("blockRemoteConnectionsTimerCb: deviceBlockRemoteConnectionsControl is 0"));
      }

      return true;
   }

   Result DeviceBlockRemoteConnectionsControl::setDeviceRemoteConnectable()
   {
      _blockRemoteConnectionsTimerLock.lock();

      Result result(CC_ERR_INT_NO_ERROR);

      if (0 != _blockRemoteConnectionsTimerId)
      {
         RemoteConnectable remoteConnectable = (TARGET_SWITCH_STATE_SWITCHED_ON == LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultLocalConnectableMode) ? true : false;

         ETG_TRACE_USR1(("setDeviceRemoteConnectable: setting device remote connectable"));

         BmCoreIfMessage_SetDeviceRemoteConnectable* bmCoreIfMessage = getNewBmCoreIfMessage_SetDeviceRemoteConnectable(_deviceHandle,
               remoteConnectable, 0u, BM_CORE_IF_MSG_ORIGIN_INTERNAL);

         if (0 == bmCoreIfMessage)
         {
            ETG_TRACE_FATAL(("setDeviceRemoteConnectable: bmCoreIfMessage is 0"));

            result = CC_ERR_INT_MEMORY_ALLOCATION_ERROR;
         }
         else
         {
            LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(bmCoreIfMessage);
         }

         _blockRemoteConnectionsTimerId = 0;
      }
      else
      {
         ETG_TRACE_USR1(("setDeviceRemoteConnectable: block remote connections timer has been stopped already -> will not set device remote connectable"));
      }

      _blockRemoteConnectionsTimerLock.unlock();

      return result;
   }
}
