/**
 * @defgroup DeviceConnectionControllerModule Device Connection Controller
 * 
 * @brief This module comprises all parts related to device connection
 */

/**
 * @file
 * @ingroup DeviceConnectionControllerModule
 * 
 * @brief This file contains the definition of the class DeviceConnectionController
 * 
 * @author Matthias Thömel
 * @author Stefan Reis
 */

#ifndef _DEVICE_CONNECTION_CONTROLLER_H_
#define _DEVICE_CONNECTION_CONTROLLER_H_

#include "DeviceConnectionSm.h"
#include "ILocalSpm.h"
#include "DbManager.h"
#include "Timer.h"
#include "Lock.h"
#include "FwISmTimerCallback.h"
#include "BmCoreIfMessages.h"

namespace bmcore
{
   /**
    * @class DeviceConnectionController
    * @ingroup DeviceConnectionControllerModule
    * 
    * @brief The class DeviceConnectionController implements the state machine DeviceConnectionSm and its public interface
    * 
    */
   class DeviceConnectionController : public DeviceConnectionSm, public ::fw::ISmTimerCallback
   {

   public:

      /**
       * @brief Constructor of class DeviceConnectionController which creates an instance
       *
       * No further initialization of internal components and resources
       * is performed in the constructor. See the explicit allocation
       * and initialization functions Create() and Init(). Hence, the
       * implementation is trivial.
       *
       * Synchronization: not re-entrant.
       *
       * Performance considerations: none.
       *
       * @return void
       */
      DeviceConnectionController();

      /**
       * Destructor of class DeviceConnectionController
       *
       * @return void
       */
      virtual ~DeviceConnectionController();

      Result assign(const DeviceId deviceId);

      Result release();

      inline bool isInUse(void) const
      {
         return (0u != _deviceId);
      }

      inline bool isHandlingLostDevice(void) const
      {
         return _deviceLost;
      }

      inline DeviceId getDeviceId(void) const
      {
         return _deviceId;
      }

      inline bool areLocalProtocolConnectionsToBeDelayed(void)
      {
         return _delayLocalProtocolConnections;
      }

      inline bool isReservedforCPW(void)
      {
         return _reservedForCpw;
      }

      inline void setLocalProtocolConnectionsToBeDelayed(void)
      {
         _delayLocalProtocolConnections = true;
      }

      Result checkAndReleasePccForSlaveProtocols(bool disconnectOnRequest = false, ProtocolId protocolId = BM_PROTOCOL_ID_UNKNOWN);

      void create();

      Result init();

      Result run();

      Result stop();

      Result done();

      /**
       * @brief Initiates the connection of BT protocols
       *
       * @param[in] protocolList list of protocols (protocol ID, UUID) to be connected
       * @param[in] connectionRequestOrigin origin of the connection request (either remote or local BT device)
       *
       * @return cc::CC_ERR_INT_DCC_NOT_ASSIGNED: Device Connection Controller instance is not assigned,
       * @return cc::CC_ERR_INT_PCC_NO_RESOURCES: no Protocol Connection Controller instances available,
       * @return : ,
       * @return cc::CC_ERR_INT_NO_ERROR: all requested BT protocol connections could be initiated
       */
      Result requestProtocolConnections(IN const ProtocolList& protocolList, IN const PageTimeout pageTimeout, IN const ConnectionRequestOrigin connectionRequestOrigin);

      /**
       * @brief Initiates a BT protocol disconnection
       *
       * Gets the ProtocolConnectionSM state machine handle by calling
       * DbManager::getProtocolConnectionSm(OUT tSMHandle smHandle, IN const DeviceId deviceId=deviceId, IN const ProtocolId protocolId=protocolId, IN const Uuid uuid=uuid).\n
       * Initiates a BT protocol disconnection by sending the message ProtocolConnectionSM::DISCONNECT()
       * to the ProtocolConnectionSM instance with state machine handle @a smHandle.\n
       * Removes BT protocol entry from last mode table in database by calling
       * DbManager::removeFromLastMode(IN const DeviceId deviceId=deviceId, IN const ProtocolId protocolId=protocolId, IN const tUUID uuid=uuid).
       *
       * @param[in] deviceId ID of the remote device
       * @param[in] protocolId ID of the BT protocol
       * @param[in] uuid UUID of a specific BT SPP service
       *
       * @return 0
       */
      Result requestProtocolDisconnections(IN const ProtocolList& protocolList, IN const DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN);

      /**
       * @brief Initiates disconnection of all protocol connections to a remote device
       *
       * Gets state machine handles of all ProtocolConnectionSM instances corresponding to a remote BT device with given
       * BT device ID @a deviceId by calling getAllProtocolConnectionSm(OUT tProcolSMList smHandleList, IN const DeviceId deviceId=deviceId).\n
       * Iterates through the list of state machine handles @a smHandleList and initiates BT protocol disconnection by sending
       * the message ProtocolConnectionSM::DISCONNECT() to each respective ProtocolConnectionSM instance.
       *
       * @param[in] deviceId ID of the remote device
       *
       * @return 0
       */
      Result requestDeviceDisconnection(IN const DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN, IN const bool deleteDevice = false);

      Result releaseProtocolConnectionController(IN const ProtocolId protocolId, IN const Uuid uuid, IN const DisconnectedReason disconnectedReason);

      void setConnectionRequestOrigin(IN const ConnectionRequestOrigin origin);

      void setLocalProtocolTargetConnectionStatusInMap(IN Protocol protocol,IN TargetConnectionStatus targetConnectionStatus);

      // *********************************************
      // * Overriden methods of DeviceConnectionSm   *
      // *********************************************

      Result initSm();

      Result handleStopSm();

      Result handleDone();

      /**
       * old:
       *
       * @brief Releases device connection state machine
       *
       * If @a mDevice != 0:
       * - Sets device state to DEVICE_STATE_DISCONNECTED in database by calling
       * DbManager::setState(IN const DeviceId deviceId=_deviceId, IN const tDeviceState state=DEVICE_STATE_DISCONNECTED).\n
       * - Sets the "disconnected state" to NORMAL in the database by calling
       * DbManager::setDisconnectedReason(IN const DeviceId deviceId=_deviceId, IN const DisconnectedReason state=NORMAL).\n
       * - Sets member @a _deviceId=0.
       *
       * Always:
       * Releases device connection state machine by calling
       * DbManager::ReleaseDeviceSm(IN const tDeviceSmHandle deviceSmHandle=this).
       *
       * new:
       *
       * Only if _deviceId != 0:
       * - Sets device state to DEVICE_STATE_DISCONNECTED in database
       * - Sets the "disconnected reason" to NORMAL in the database
       * - set deviceSmHandle to 0 in database
       * - reset members
       *
       * @return 0 on success
       */
      Result enterIdle();

      /**
       * @brief Sets device state to DEVICE_STATE_DISCONNECTED in database
       *
       * Sets device state to DEVICE_STATE_DISCONNECTED in database by calling
       * DbManager::setState(IN const DeviceId deviceId=_deviceId, IN const tDeviceState state=DEVICE_STATE_DISCONNECTED).\n
       * Sets the "disconnected state" to NORMAL in the database by calling
       * DbManager::setDisconnectedReason(IN const DeviceId deviceId=_deviceId, IN const DisconnectedReason state=NORMAL).
       *
       * @return 0 on success
       */
      Result enterInUse();

      Result reserveDcc();

      Result unreserveDcc();

      /**
       * old:
       *
       * @brief Initiates a BT protocol connection requested on the local BT device side
       *
       * Initiates a BT protocol connection requested on the local BT device side by calling
       * ProtocolDispatcher::ConnectProtocol(IN const DeviceId deviceId=_deviceId, IN const ProtocolId protocolId=protocolId, IN const Uuid uuid=uuid, IN const tExternalFlag externalFlag=0).
       *
       * new:
       *
       * -
       * @param[in] protocolId BT protocol ID
       * @param[in] uuid UUID of a specific BT SPP service
       *
       * @return 0 on success
       */
      Result connectProtocolLocal(IN const ProtocolId protocolId, IN const UuidChrArr uuid);

      /**
       * @brief Initiates a BT protocol connection requested by a remote BT device
       *
       * Initiates a BT protocol connection requested on the local BT device side by calling
       * ProtocolDispatcher::ConnectProtocol(IN const DeviceId deviceId=_deviceId, IN const ProtocolId protocolId=protocolId, IN const Uuid uuid=uuid, IN const tExternalFlag externalFlag=1).
       *
       * Assumptions:
       * - A protocol connection request for the same device and protocol is not initiated by the same remote BT device
       * - A protocol connection request for the protocol SPP is not initiated by a remote BT device
       *
       * @param[in] protocolId BT protocol ID
       * @param[in] uuid UUID of a specific BT SPP service
       *
       * @return 0 on success
       */
      Result connectProtocolRemote(IN const ProtocolId protocolId, IN const UuidChrArr uuid);

      /**
       * @brief Sets the device state to DEVICE_STATE_CONNECTING in database
       *
       * Sets the device state to DEVICE_STATE_CONNECTING in database by calling
       * DbManager::setState(IN DeviceId deviceId=_deviceId, IN const tDeviceState state=DEVICE_STATE_CONNECTING).
       *
       * @return 0 on success
       */
      Result enterConnecting();

      Result releaseDcc();

      /**
       * @brief Checks if "disconnected state" is LOST
       *
       * Checks if "disconnected state" is LOST by calling
       * DbManager::getDisconnectedReason(OUT DisconnectedReason state, IN const DeviceId deviceId=_deviceId).
       * If state==LOST return 1, otherwise return 0.
       *
       * @return 0: "disconnected reason" is not LOST
       * @return 1: "disconnected reason" is LOST
       */
      Result isDeviceLost();

      Result handleLostDevice();

      /**
       * @brief Handles a lost connection to the remote BT device
       *
       * Sets "disconnected state" to LOST in database by calling
       * DbManager::setDisconnectedReason(IN const DeviceId deviceId=_deviceId, IN const DisconnectedReason state=LOST).\n
       * Creates the reconnection timer by calling Timer::CreateTimer(..., DeviceConnectionSm::CONNECTION_RETRY,...).
       * When the reconnection timer expires the message DeviceConnectionSm::CONNECTION_RETRY is sent.
       *
       * @return 0 on success
       */
      Result enterLost();

      /**
       * @brief Stops the reconnection timer
       *
       * Stops the reconnection timer by calling Timer::CancelTimer(...)
       *
       * @return 0 on success
       */
      Result exitLost();

      /**
       * @brief Initiates restoring of previously connected BT protocol connections
       *
       * Gets previously connected BT protocols by calling
       * DbManager::getLastModeProtocols(OUT ProtocolIdList protocols, IN const DeviceId deviceId).\n
       * Initiates restoring of previously connected BT protocol connections by calling
       * ProtocolDispatcher::ConnectProtocols(IN const DeviceId deviceId=_deviceId, IN const ProtocolIdList protocols=protocols).
       *
       * @return 0 on success
       */
      Result reconnectDevice();

      /**
       * @brief callback called as a notification when a protocol with protocolId gets connected
       * for this device
       *
       * @param[in] protocolId BT protocol ID
       *
       * @return 0 on success
       */
      Result handleProtocolConnected(IN const ProtocolId protocolId, IN const UuidChrArr uuid);

      /**
       * @brief Sets the device state to DEVICE_STATE_CONNECTED in database
       *
       * Sets the device state to DEVICE_STATE_CONNECTED in database by calling
       * DbManager::setState(IN DeviceId deviceId=_deviceId, IN const tDeviceState state=DEVICE_STATE_CONNECTED).
       *
       * @return 0 on success
       */
      Result enterConnected();

      /**
       * @brief Initiates a BT protocol disconnection
       *
       * Removes entry from last mode database table by calling
       * DbManager::removeFromLastMode(IN const DeviceId deviceId=_deviceId, IN const ProtocolId protocolId=protocolId, IN const Uuid uuid=uuid).\n
       * Initiates a BT protocol disconnection by calling
       * ProtocolDispatcher::DisconnectProtocol(IN const DeviceId deviceId=_deviceId, IN const ProtocolId protocolId=protocolId, IN const Uuid uuid=uuid).
       *
       * @param[in] protocolId BT protocol ID
       * @param[in] uuid UUID of a specific BT SPP service
       *
       * @return 0 on success
       */
      Result disconnectProtocol(IN const ProtocolId protocolId, IN const UuidChrArr uuid);

      Result handleDisconnect(void);
      /**
       * @brief Initiates device disconnection by initiating disconnection of all BT protocols
       *
       * Sets "disconnected state" to NORMAL in database by calling
       * DbManager::setDisconnectedReason(IN const DeviceId _deviceId, IN const DisconnectedReason state=NORMAL).\n
       * Sets device state to DEVICE_STATE_DISCONNECTING in database by calling DbManager::setState(IN const DeviceId deviceId=_deviceId, IN const tDeviceState state=DEVICE_STATE_DISCONNECTING).\n
       * Initiates disconnection of all BT protocols by calling DisconnectAllProtocols(IN const DeviceId deviceId=_deviceId).
       *
       * @return 0 on success
       */
      Result enterDisconnecting();

      Result handleAllProtocolsDisconnected();

      Result handleDeviceDisconnected();

      Result rejectProtocolConnectionRemote(IN const ProtocolId protocolId, IN const UuidChrArr uuid);

      Result enterDisconnectionFailed();

      Result exitDisconnectionFailed();

      Result disconnectDevice();

      Result disconnectInIdle();

      void setOOBPairedDeviceRequest(IN const BmCoreIfMessage_AddOobPairedDeviceRequest OobPairedDeviceRequest);

      void getOOBPairedDeviceRequest(OUT BmCoreIfMessage_AddOobPairedDeviceRequest& OobPairedDeviceRequest);

      void handleSmTimerCallback(const char* message);

   private:

      typedef std::map<Protocol, TargetConnectionStatus> ProtocolTargetConnectionStatusMap;

      void traceProtocolTargetConnectionStatusMap();

      Result initMembers();

      Result assignProtocolConnectionController(OUT ProtocolConnectionController** pccInstance, IN const ProtocolId protocolId, IN const Uuid& uuid, IN const PageTimeout pageTimeout);

      Result initiateProtocolConnections(IN const ProtocolConnectionControllerList& pccInstances, IN const ConnectionRequestOrigin connectionRequestOrigin);

      Result initiateProtocolDisconnections(IN const ProtocolConnectionControllerList& pccInstances,
            IN const DisconnectedReason disconnectedReason);

      Result connectProtocol(IN const ProtocolId protocolId, IN const Uuid& uuid, IN const ConnectionRequestOrigin connectionRequestOrigin);

      Result doConnectProtocols(IN const ProtocolConnectionControllerList& pccInstances, IN const ConnectionRequestOrigin connectionRequestOrigin);

      Result removeProtocolPendingToConnect(IN const ProtocolId protocolId, IN const Uuid& uuid, IN const bool sendFinishedEvent = true);

      Result isDisconnectionFinished();

      void checkAndSendAutoConnectedFinished();

      DeviceId _deviceId; /**< initialized to 0 */
      bool     _deviceToBeDeleted;
      bool     _reservedForCpw;

      bool _delayLocalProtocolConnections;

      bool _disconnectionLocallyInitiated;
      bool _deviceLost;
      bool _allowDisconnectedReasonUpdate;
      bool _allProtocolsDisconnected;
      bool _deviceDisconnected;
      DisconnectedReason _disconnectedReason;

      //This is used for to start the BTLimitation mode-CPW if the origin is Remote request
      ConnectionRequestOrigin _origin;

      Timer      _delayConnectionRetryTimer;
      timer_t    _delayConnectionRetryTimerId;

      Timer      _delayDisconnectionRetryTimer;
      timer_t    _delayDisconnectionRetryTimerId;

      std::string _expiredTimerSmEvent;

      ProtocolList _protocolsPendingToConnect;

      ProtocolTargetConnectionStatusMap _localProtocolTargetConnectionStatusMap;

      LockForever _protocolConnectionsLock;

      BmCoreIfMessage_AddOobPairedDeviceRequest _oobPairedDeviceRequest;
      bool _oobPairedDeviceRequestReceived;

      PageTimeout _pageTimeout;
   };
}

#endif // _DEVICE_CONNECTION_CONTROLLER_H_

