#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_DBMANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/Database.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_DBMANAGER
#endif
#endif

#include <TimeTrace.h>
#include "FunctionTracer.h"

#include <stdarg.h>
#include <cstring>
#include <sys/stat.h>
#include <string>
#include <vector>
#include <errno.h>
#include <openssl/md5.h>

#include "BmUtils.h"
#include "Database.h"
#include "BmAllTypes.h"
#include "LocalSpm.h"
#include "DataProvider.h"
//#include "DbTriggerManager.h"
#include <fcntl.h>
namespace bmcore
{
   /*
    * define the collation for all ordered lists indizes (is used by the schema-2.0 include files
    */
#define NO_COLLATE " "
#define SORTLIB_COLLATE " COLLATE SORT_LIB "
#define NOCASE_COLLATE " COLLATE NOCASE "

   /* this switches the collation for all queries underneath */
#define COLLATE SORTLIB_COLLATE

   // Static constants initialization
   const char* Database::mDatabaseFile = "/var/opt/bosch/dynamic/connectivity/db/BM.db";
   const char* Database::mDatabaseRecreateFile = "/var/opt/bosch/dynamic/connectivity/db/BM.db.recreate";
   //const char* Database::mDatabaseSchemaHashFile = "/var/opt/bosch/dynamic/connectivity/db/BM.db.schema.md5";

#define QUICK_CHECK 1

#define DEBUG_OPEN_DB_HANDLES 0

   static std::vector<sqlite3 *> mFreeDBConnections;
   static Lock mFreeDBConnectionsLock;
   static unsigned int mOpenedConnections = 0;
   static int mShutDown = 0;
   static int mStopUpdate = 0;
   static timeval mProgressCallbackCallTime;
   static FILE *testFile = NULL;
   int mSlowDownUpdateMyMedia;

   Result Database::CheckSqlite3Version(sqlite3 *dbHandle) const
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("CheckSqlite3Version"));

      const char *sqlCommand = "select sqlite_version();";
      sqlite3_stmt *statement;
      int sqlError;
      static int print = 1;

      sqlError = sqlite3_prepare_v2(dbHandle, sqlCommand, -1, &statement, NULL);

      /* error on sqlite3_prepare */
      if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE && sqlError != SQLITE_ROW) {
         ETG_TRACE_ERR(("sqlite3_prepare_v2: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
         ETG_TRACE_ERR(("sqlite3_step: sql=%s", sqlCommand));
         return CC_ERR_INT_DB_UNEXPECTED;
      }

      /* process it */
      sqlError = sqlite3_step(statement);
      if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE && sqlError != SQLITE_ROW) {
         ETG_TRACE_ERR(("sqlite3_step: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
         ETG_TRACE_ERR(("sqlite3_step: sql=%s", sqlCommand));
         return CC_ERR_INT_DB_UNEXPECTED;
      }

      /* get the version string */
      const char *version = (const char *)sqlite3_column_text(statement, 0);
      if (!version) {
         ETG_TRACE_ERR(("sqlite3_column_text: %s", sqlite3_errmsg(dbHandle)));
         ETG_TRACE_ERR(("sqlite3_column_text: sql=%s", sqlCommand));
         return CC_ERR_INT_DB_UNEXPECTED;
      }

      sqlite3_finalize(statement);

      if (print) {
         ETG_TRACE_USR1(("used sqlite version: %s", version));
         print = 0;
      }

      /* read out the version numbers */
      int major;
      int minor;
      int subminor;
      sscanf(version, "%d.%d.%d", &major, &minor, &subminor);
      if (major < 3 && minor < 7 && subminor < 15) {
         ETG_TRACE_FATAL(("Wrong sqlite3 version: %s", version));
         return CC_ERR_INT_DB_UNEXPECTED;
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result Database::Refresh()
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("Refresh"));

      Refresh(Handle());

      return CC_ERR_INT_NO_ERROR;
   }

   Result Database::Refresh(sqlite3 *dbHandle)
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("Refresh"));

      /* re-register the busy callback with the current calculated instruction count */
      sqlite3_progress_handler(dbHandle, LocalSpm::getDataProvider().DBUpdateProgressInstructions(), ProgressCallbackCaller, this);

      return CC_ERR_INT_NO_ERROR;
   }

   std::string Database::recreateDBSchemaHash()
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("recreateDBSchemaHash"));

      char computedSchemaMD5[MD5_DIGEST_LENGTH+1];

      char* buf = new char[32];
      std::string str;

         std::string schema =
#include "schema-2.0.h"
               ;
        MD5(reinterpret_cast<const unsigned char*>(schema.c_str()), schema.size(), reinterpret_cast<unsigned char*>(computedSchemaMD5));
         computedSchemaMD5[MD5_DIGEST_LENGTH] = '\0';

         for(int i=0;i<MD5_DIGEST_LENGTH;i++)
         {
            sprintf(buf,"%02x",computedSchemaMD5[i]);
            str.append(buf);

         }

         delete [] buf;



      return str;
   }

   Result Database::Sync(sqlite3 *dbHandle) const
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("Sync"));

      int nPackets;
      int error;

      error = sqlite3_wal_checkpoint_v2(dbHandle, "Main", SQLITE_CHECKPOINT_RESTART, NULL, OUT &nPackets);
      ETG_TRACE_USR1(("sqlite3_wal_checkpoint_v2(Main): error=%d packets=%d", error, nPackets));

      return CC_ERR_INT_NO_ERROR;
   }
   Result Database::fSync() const
   {
      ENTRY_INTERNAL
      int rc =0;
      int fileNumber; //file descriptor
      ETG_TRACE_USR1(("fSync"));
      fileNumber = open (Database::mDatabaseFile, O_RDONLY);
      if(fileNumber <0)
      {
         ETG_TRACE_USR1(("Failed to open the DataBase File"));
      }
      else
      {
         rc = fsync(fileNumber);

         if(0 != rc)
         {
            ETG_TRACE_USR1(("Failed to sync database to FFS "));
         }

         (void) close(fileNumber);
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Database::Database() : mDatabase(0) // finished: 100%
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("Database"));

      Result res;
      static int first = 1;

      /* very first inits */
      if (first)
      {
         first = 0;
         mFreeDBConnectionsLock.setNotReantrant();
         mShutDown = 0;
         memset(&mProgressCallbackCallTime, 0, sizeof(mProgressCallbackCallTime));

         /* accept URI's as filenames */
         sqlite3_config(SQLITE_CONFIG_URI, 1);
      }

      mSlowDownUpdateMyMedia = 0;

      // check if a previous run wants us to recreate the database because of a malformed error
      FILE *fp;
      fp = fopen(Database::mDatabaseRecreateFile, "r");

      if (!fp)
      {
         // recreate file does not exist => open the existing database

         ETG_TRACE_FATAL(("Database: file \"%100s\" does not exist => DB is not requested to be recreated",
               Database::mDatabaseRecreateFile));

         //Check for DB Schema Changes
         //compute the DB Schema Hash and check if the stored DB Hash is the same, if not recreate the database
         mDatabase = OpenDatabase();

         if (mDatabase != NULL)
         {
            res = CheckSchemaHash(mDatabase);

            if (CC_ERR_INT_NO_ERROR == res)
            {
               ETG_TRACE_FATAL(("Database: calculated and stored DB schema hashes match"));
            }
            else
            {
               //Hash values are different, store the new hash and recreate the database
               ETG_TRACE_FATAL(("Database: calculated DB schema hash value differs from the one stored in DB"));
               ETG_TRACE_ERRMEM(("Database: calculated DB schema hash value differs from the one stored in DB"));

               CloseDatabase(mDatabase);

               CloseAllHandles();

               mDatabase = NULL;
            }
         }
         else
         {
            ETG_TRACE_FATAL(("Database: could not open the DB"));
         }
      }
      else
      {
         // recreate file exists => DB file has to be recreated

         ETG_TRACE_FATAL(("Database: file \"%100s\" exists, DB is requested to be recreated", Database::mDatabaseRecreateFile));
         ETG_TRACE_ERRMEM(("Database: file \"%100s\" exists, DB is requested to be recreated", Database::mDatabaseRecreateFile));

         // close and remove the recreation trigger file
         fclose(fp);

         ETG_TRACE_FATAL(("Database: removing file \"%100s\"", Database::mDatabaseRecreateFile));

         int ret = remove(Database::mDatabaseRecreateFile);

         if (0 == ret)
         {
            ETG_TRACE_FATAL(("Database: removed file \"%100s\"", Database::mDatabaseRecreateFile));
         }
         else
         {
            ETG_TRACE_FATAL(("Database: could not remove file \"%100s\"", Database::mDatabaseRecreateFile));
         }

         CloseDatabase(mDatabase);

         CloseAllHandles();
         mDatabase = NULL;
      }

      if (!mDatabase)
      {
         // database open error or recreation needed

         //recreateDBSchemaHash(computedSchemaMD5, recomputeHash);

         /* create a new database */
         ETG_TRACE_ERRMEM(("Database: recreating the DB"));

         res = RecreateDatabase();
         if (res != CC_ERR_INT_NO_ERROR)
         {
            ETG_TRACE_FATAL(("Database: recreating the DB failed"));
            ETG_TRACE_ERRMEM(("Database: recreating the DB failed"));
         }

         /* try to open this */
         ETG_TRACE_FATAL(("Database: opening recreated DB"));
         mDatabase = OpenDatabase();

         if (!mDatabase)
         {
            ETG_TRACE_ERRMEM(("Database: opening recreated DB failed"));
         }
         else
         {
            InsertSchemaHasch(mDatabase);
         }
         FW_FATAL_ASSERT(mDatabase);

         // database is open:
      }
      else
      {
         /* check if the database is up to date */
         res = CheckDatabaseIntegrity(mDatabase);
         if (res != CC_ERR_INT_NO_ERROR) { // if not ok:

            // for sure, close the database
            CloseDatabase(mDatabase);

            // recreate the database
            CloseAllHandles();

            ETG_TRACE_ERRMEM(("Database: recreating the DB"));
            res = RecreateDatabase();
            if (res != CC_ERR_INT_NO_ERROR)
            {
               ETG_TRACE_FATAL(("Database: recreating the DB failed"));
               ETG_TRACE_ERRMEM(("Database: recreating the DB failed"));
            }

            ETG_TRACE_FATAL(("Database: opening recreated DB"));

            mDatabase = OpenDatabase();

            if (!mDatabase)
            {
               ETG_TRACE_ERRMEM(("Database: opening recreated DB failed"));
            }
            else
            {
               InsertSchemaHasch(mDatabase);
            }
            FW_FATAL_ASSERT(mDatabase);
         }
      }

      /* check the sqlite version */
      CheckSqlite3Version(mDatabase);

      /* attach trigger callbacks to this connection */
      //DbTriggerManager::GetInstance().AttachDB(mDatabase);
   }
   void Database::Close() // finished: 100%
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("Close"));

      if (mDatabase)
      {
         if (testFile)
         {
            fprintf(testFile, "%p\t%p\t%s\n", mDatabase, this, "Close");
         }

         //DbTriggerManager::GetInstance().DetachDB(mDatabase);

         CloseDatabase(mDatabase);

         mDatabase = NULL;
      }
   }

   Database::~Database() // finished: 100%
   {
      ENTRY_INTERNAL
      Close();
   }




   Result Database::DeInit() // finished: 100%
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("DeInit"));

      /* set the shut down flag to break the busy callbacks and the running update thread */
      mShutDown = 1;

      /* remove all triggers */
      if (mDatabase)
      {
    	  /*DbTriggerManager::GetInstance().DeRegisterAllTriggers();

         // detach from db
         DbTriggerManager::GetInstance().DetachDB(mDatabase);*/

    	  if (testFile)
    	  {
    		  fprintf(testFile, "%p\t%p\t%s\n", mDatabase, this, "DeInit");
    	  }

    	  // close this handle (goes back into cache)
    	  CloseDatabase(mDatabase);

    	  /* close all open handles in cache */
    	  CloseAllHandles();

    	  mDatabase = NULL;
      }

      mShutDown = 0;

      return CC_ERR_INT_NO_ERROR;
   }

   sqlite3* Database::Handle() // finished: 100%
   {
#if 0 // debug
      if (!testFile) {
         testFile = fopen("/tmp/testfile.txt", "w");
      }
#endif

      if (testFile) fprintf(testFile, "%p\t%p\n", mDatabase, this);
      if (testFile) fflush(testFile);

      return mDatabase;
   }

   Result Database::ConfigureConnection(sqlite3 *dbHandle) const // finished: 100%
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("ConfigureConnection"));

      Result result = CC_ERR_INT_DB_UNEXPECTED;
      std::vector<std::string> connectionConfigurationScript;
      connectionConfigurationScript.push_back(std::string(
            "PRAGMA FOREIGN_KEYS = OFF;"));
      connectionConfigurationScript.push_back(std::string(
            "PRAGMA JOURNAL_MODE = OFF;"));
      connectionConfigurationScript.push_back(std::string(
            "PRAGMA LOCKING_MODE = NORMAL;"));
      connectionConfigurationScript.push_back(std::string(
            "PRAGMA SYNCHRONOUS = OFF;")); // NORMAL
      connectionConfigurationScript.push_back(std::string(
            "PRAGMA AUTOMATIC_INDEX = OFF;"));
      connectionConfigurationScript.push_back(std::string(
            "PRAGMA SECURE_DELETE = OFF;"));
      connectionConfigurationScript.push_back(std::string(
            "PRAGMA TEMP_STORE = MEMORY;"));
      connectionConfigurationScript.push_back(std::string(
            "PRAGMA ENCODING = \"UTF-8\";"));
      connectionConfigurationScript.push_back(std::string(
            "PRAGMA CACHE_SIZE = 10000;"));

      typedef std::vector<std::string>::const_iterator tStringVectorIterator;
      tStringVectorIterator connectionConfigurationScriptIterator =
            connectionConfigurationScript.begin();
      tInteger sqliteResult = SQLITE_ERROR;

      /* configure the db connection to be in serialized mode */
      sqliteResult = sqlite3_config(SQLITE_CONFIG_SERIALIZED);
      if (SQLITE_OK == sqliteResult) {
         ETG_TRACE_ERR(("sqlite3_config(SQLITE_CONFIG_SERIALIZED): error=%d", sqliteResult));
      }

      // Execute the database initialization script
      while (connectionConfigurationScript.end() !=
            connectionConfigurationScriptIterator)
      {
         char* sqlError = NULL;

         // Execute SQL query
         sqliteResult = sqlite3_exec(
               dbHandle,
               connectionConfigurationScriptIterator->c_str(),
               NULL, // No callback, as no data selected
               NULL, // No callback arguments
               &sqlError);

         // Check for success
         if (SQLITE_OK == sqliteResult)
         {
            result = CC_ERR_INT_NO_ERROR;
            connectionConfigurationScriptIterator++;
         }
         else
         {
            // Failed to execute SQL query.
            ETG_TRACE_ERR(("Error: sqlite3_exec(%64s): %64s",
                  connectionConfigurationScriptIterator->c_str(),
                  sqlite3_errmsg(dbHandle)));

            // Error handling
            if (NULL == sqlError)
            {
               ETG_TRACE_FATAL(("Undefined SQLite3 error."));
            }
            else
            {
               ETG_TRACE_FATAL(("SQLite3 error: %s.", sqlError));
               sqlite3_free(sqlError);
            }

            result = CC_ERR_INT_DB_DATABASE_CONNECTION_FAILURE;
            break;
         }
      }

      return result;
   }

   Result Database::CloseDatabase(sqlite3 *dbHandle) const // finished: 100%
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("CloseDatabase"));

      /* instead of closing the db, the used handle goes into a free list to be reused by next OpenDatabase() */
      mFreeDBConnectionsLock.lock();
      mFreeDBConnections.push_back(dbHandle);
      mFreeDBConnectionsLock.unlock();

      return CC_ERR_INT_NO_ERROR;
   }

   void Database::CloseAllHandles() const // finished: 100%
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("CloseAllHandles"));

      tInteger sqliteResult = SQLITE_ERROR;
      Result res;

      mFreeDBConnectionsLock.lock();

      // check the free cache count
      if (mFreeDBConnections.size() != mOpenedConnections) {
         ETG_TRACE_ERR(("connection cache leak: mOpenedConnections=%d, mFreeDBConnections.size()=%d", mOpenedConnections, mFreeDBConnections.size()));
      }

      for(unsigned int i=0; i<mFreeDBConnections.size(); i++) {

         /* detach the my media data base */
         const char *detach = "DETACH DATABASE MyMe;";

         sqliteResult = sqlite3_exec(mFreeDBConnections[i], detach, NULL, NULL, NULL);
         if (sqliteResult != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_exec: failed"));
         }

         /* detach the index helper data base */
         const char *detach2 = "DETACH DATABASE IndexHelper;";

         sqliteResult = sqlite3_exec(mFreeDBConnections[i], detach2, NULL, NULL, NULL);
         if (sqliteResult != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_exec: failed"));
         }

         /* finalize all cached queries */
         res = Query::Finalize(mFreeDBConnections[i]);
         if (res) {
            ETG_TRACE_ERR(("query.Finalize: failed"));
         }

         sqliteResult = sqlite3_close_v2(mFreeDBConnections[i]); // the v2 variant closes the file but leaves a zombie connection open.

         if (sqliteResult != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_close: failed: error=%d", sqliteResult));
         }

         ETG_TRACE_USR2(("closed handle: 0x%x", mFreeDBConnections[i]));
      }

      mFreeDBConnections.clear();
      mOpenedConnections = 0;

      mFreeDBConnectionsLock.unlock();
   }

   Result Database::ResetDatabaseIntegrity(sqlite3 *dbHandle) const // finished: 100%
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("ResetDatabaseIntegrity"));

      char* sqlError = NULL;
      tInteger sqliteResult = SQLITE_ERROR;
      std::string resetIntegrityQuery =
            "UPDATE"
            "  [MAIN].[SchemaVersion] "
            "SET"
            "  [Integrity] = 1"
            ";";

      // Execute SQL query for integrity reset.
      sqliteResult = sqlite3_exec(
            dbHandle,
            resetIntegrityQuery.c_str(),
            NULL, // No callback, as no data selected
            NULL, // No callback arguments
            &sqlError);

      // Check for success
      if (SQLITE_OK != sqliteResult)
      {
         // Failed to execute SQL query.
         ETG_TRACE_FATAL(("Failed to execute: %s.", resetIntegrityQuery.c_str()));

         // Error handling
         if (NULL == sqlError)
         {
            ETG_TRACE_FATAL(("Undefined SQLite3 error."));
         }
         else
         {
            ETG_TRACE_FATAL(("SQLite3 error: %s.", sqlError));
            sqlite3_free(sqlError);
         }

         return CC_ERR_INT_DB_DATABASE_SCHEMA_FAILURE;
      }

      return CC_ERR_INT_NO_ERROR;
   }

  Result Database::CheckDatabaseIntegrity(sqlite3 *dbHandle) const // finished: 100%
  {
     ENTRY_INTERNAL

     ETG_TRACE_USR1(("CheckDatabaseIntegrity"));

     Result result = CC_ERR_INT_DB_UNEXPECTED;

     sqlite3_stmt* sqliteStatement = NULL;
     tInteger sqliteResult = SQLITE_ERROR;
     char queryString[128];
     sqlite3_stmt *statement;

     /* create the explain query string */
#if QUICK_CHECK
     strncpy_r(OUT queryString, IN "PRAGMA quick_check;", IN sizeof(queryString));
#else
     strncpy_r(OUT queryString, IN "PRAGMA integrity_check;", IN sizeof(queryString));
#endif

     /* prepare the update statement */
     sqliteResult = sqlite3_prepare_v2(dbHandle, queryString, -1, &statement, NULL);
     if (sqliteResult != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2(): %d/%s", sqliteResult, sqlite3_errmsg(dbHandle)));
        return CC_ERR_INT_DB_INTEGRITY_FAILURE;
     }

     /* get the resulting lines */
     int integrityOk = 0;
     int lineNo = 1;
     TimeTrace ticks("IntegrityCheck");

     while(1) {

        /* do the explain */
        sqliteResult = sqlite3_step(statement);
        if (sqliteResult == SQLITE_DONE) break;
        if (sqliteResult != SQLITE_OK && sqliteResult != SQLITE_ROW) {
           ETG_TRACE_ERR(("sqlite3_step(): %d/%s", sqliteResult, sqlite3_errmsg(dbHandle)));
           break;
        }

        /* get one text line */
        const char *cptr;
        cptr = (const char *)sqlite3_column_text(statement,  0);
        if (cptr == NULL) break;

        /* check if it is OK */
        if (lineNo == 1) {
           if (!strcmp(cptr, "ok")) {
              integrityOk = 1;
              ETG_TRACE_USR2(("integrityCheck: %s", cptr));
           } else {
              ETG_TRACE_ERR(("integrityError: %s", cptr));
           }
        } else {
           ETG_TRACE_ERR(("integrityError: %s", cptr));
        }

        lineNo++;
     }

     /* finalize pragma query */
     sqlite3_finalize(statement);

     ticks.elapsed();

     if (!integrityOk) {
        ETG_TRACE_ERR(("DB integrity damaged"));
        return CC_ERR_INT_DB_INTEGRITY_FAILURE;
     }

     char query[] =
           "SELECT"
           "  [Version],"
           "  [Integrity] "

           "FROM"
           "  [MAIN].[SchemaVersion];";

     // Prepare the database integrity check statement.
     sqliteResult = sqlite3_prepare_v2(dbHandle,
           query,
           sizeof(query),
           &sqliteStatement,
           NULL);

     // Check for success
     if (SQLITE_OK == sqliteResult)
     {
        // Execute the integrity check statement.
        sqliteResult = sqlite3_step(sqliteStatement);

        // Check for success
        // Retrieve the selected data if available.
        if (SQLITE_ROW == sqliteResult)
        {
           // Check whether the schema version is valid.
           if (mSchemaVersion == sqlite3_column_int(sqliteStatement, 0))
           {
              // Check the database integrity status.
              if (false == sqlite3_column_int(sqliteStatement, 1))
              {
                 // The database integrity is invalid.
                 ETG_TRACE_ERR(("The database integrity is invalid."));
                 result = CC_ERR_INT_DB_INTEGRITY_FAILURE;
              }
              else
              {
                 result = CC_ERR_INT_NO_ERROR;
              }
           }
           else
           {
              ETG_TRACE_ERR(("Invalid database schema version."));
              result = CC_ERR_INT_DB_INVALID_SCHEMA_VERSION;
           }
        }
        else
        {
           // Failed to execute the integrity check statement.
           ETG_TRACE_ERR(("Failed to execute the integrity check statement: %s", sqlite3_errmsg(dbHandle)));

           // Handle the error
           result = CC_ERR_INT_DB_STEP_FAILURE;
        }

        // Destroy the integrity check statement.
        sqliteResult = sqlite3_finalize(sqliteStatement);

        // Check for success
        if (SQLITE_OK != sqliteResult)
        {
           // Failed to finalize the integrity check statement.
           ETG_TRACE_ERR(("Failed to finalize the integrity check statement."));
        }
     }
     else
     {
        // Failed to prepare the integrity check statement.
        ETG_TRACE_ERR(("Failed to prepare the integrity check statement: %d/%s", sqliteResult, sqlite3_errmsg(dbHandle)));

        // Handle the error
        result = CC_ERR_INT_DB_PREPARE_FAILURE;
     }

     return result;
  }

   Result Database::CreateDatabaseSchema(sqlite3 *dbHandle) const // finished: 100%
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("CreateDatabaseSchema"));

      /* include the schema as header file for the standard case sensitive behavior */
#define UNIFY_COLLATE NO_COLLATE
#define UNIFY_TAG1 "[NEW].[FilterTag1]"
#define UNIFY_TAG2 "[NEW].[FilterTag2]"
#define UNIFY_TAG3 "[NEW].[FilterTag3]"
#define UNIFY_TAG4 "[NEW].[FilterTag4]"
      const char *sqlCreate =
#include "schema-2.0.h"
            ;

      /* include the same schema as header file for the unified behavior */
#undef UNIFY_COLLATE
#undef UNIFY_TAG1
#undef UNIFY_TAG2
#undef UNIFY_TAG3
#undef UNIFY_TAG4
#define UNIFY_COLLATE NOCASE_COLLATE
#define UNIFY_TAG1 "LOWER([NEW].[FilterTag1])"
#define UNIFY_TAG2 "LOWER([NEW].[FilterTag2])"
#define UNIFY_TAG3 "LOWER([NEW].[FilterTag3])"
#define UNIFY_TAG4 "LOWER([NEW].[FilterTag4])"
      const char *sqlCreateUnified =
#include "schema-2.0.h"
            ;

      const char *restOfCommand = sqlCreate;
      const char *lastCommand;
      const char *endOfCommand = sqlCreate + strlen(sqlCreate);
      sqlite3_stmt *statement;
      int sqlError = SQLITE_OK;

      /* switch to data base schema defined in data provider */
      if (LocalSpm::getDataProvider().DBUnifiedSchema()) {
         restOfCommand = sqlCreateUnified;
         endOfCommand = sqlCreateUnified + strlen(sqlCreateUnified);
         ETG_TRACE_USR1(("Using UNIFIED Schema"));
      } else {
         restOfCommand = sqlCreate;
         endOfCommand = sqlCreate + strlen(sqlCreate);
         ETG_TRACE_USR1(("Using STANDARD Schema"));
      }

      /* start DB creation monitoring
       * write a file onto the disk to issue a data base file delete on next restart of system */
      FILE *fp;
      fp = fopen(Database::mDatabaseRecreateFile, "w");

      if (0 != fp)
      {
         ETG_TRACE_ERRMEM(("CreateDatabaseSchema: created file \"%100s\"", Database::mDatabaseRecreateFile));
         fclose(fp);
      }
      else
      {
         ETG_TRACE_ERRMEM(("CreateDatabaseSchema: could not open/create file \"%100s\"", Database::mDatabaseRecreateFile));
      }

      /* process the commands in the included command list */
      while(restOfCommand != endOfCommand) {

         /* interpret one command */
         lastCommand = restOfCommand;
         printf("sqlite3_prepare: %60s\n", restOfCommand); fflush(stdout);
         sqlError = sqlite3_prepare_v2(dbHandle, restOfCommand, -1, &statement, &restOfCommand);

         /* error on sqlite3_prepare */
         if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE && sqlError != SQLITE_ROW) {
            ETG_TRACE_ERR(("sqlite3_prepare_v2: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
            const char *cSemicolon = strstr(lastCommand, ";");
            if (cSemicolon) {
               char *singleCommand = strndup(lastCommand, (size_t)(cSemicolon-lastCommand)+1);
               ETG_TRACE_ERR(("sqlite3_step: sql=%s", singleCommand));
               free(singleCommand);
            }
            break;
         }

         /* process it */
         sqlError = sqlite3_step(statement);
         if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE && sqlError != SQLITE_ROW) {
            ETG_TRACE_ERR(("sqlite3_step: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
            const char *cSemicolon = strstr(lastCommand, ";");
            if (cSemicolon) {
               char *singleCommand = strndup(lastCommand, (size_t)(cSemicolon-lastCommand)+1);
               ETG_TRACE_ERR(("sqlite3_step: sql=%s", singleCommand));
               free(singleCommand);
            }
            break;
         }

         sqlite3_finalize(statement);
      }

      /* end DB creation monitoring
       * remove the recreation trigger file in case of success */
      if (sqlError == SQLITE_OK || sqlError == SQLITE_DONE || sqlError == SQLITE_ROW) {
         int ret = remove(Database::mDatabaseRecreateFile);
         if(0 == ret)
         {
            ETG_TRACE_FATAL(("CreateDatabaseSchema: removed file \"%100s\"", Database::mDatabaseRecreateFile));
            ETG_TRACE_ERRMEM(("CreateDatabaseSchema: removed file \"%100s\"", Database::mDatabaseRecreateFile));
         }
         else
         {
            ETG_TRACE_FATAL(("CreateDatabaseSchema:Eroor while deleting  \"%100s\"", Database::mDatabaseRecreateFile));
         }

      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result Database::RecreateDatabase() const // finished: 100%
   {
      ENTRY_INTERNAL
      Result result = CC_ERR_INT_DB_UNEXPECTED;
      tInteger sqliteResult = SQLITE_ERROR;
      sqlite3 *dbHandle;

      /* remove the data base file */
      ETG_TRACE_FATAL(("RecreateDatabase: Removing DB file \"%100s\"", mDatabaseFile));
      int ret =remove(mDatabaseFile); // if not exist, dont care
      if(0 == ret)
      {
         ETG_TRACE_FATAL(("RecreateDatabase: removed file \"%100s\"", mDatabaseFile));
      }
      else
      {
         ETG_TRACE_FATAL(("Error while deleting  \"%100s\"", mDatabaseFile ));
      }
      // open the data base
      // this method is only called on startup so nobody else will have access to the database
      sqliteResult = sqlite3_open_v2(
            mDatabaseFile,
            &dbHandle,
            SQLITE_OPEN_READWRITE |
            SQLITE_OPEN_CREATE,
            NULL);

      if (sqliteResult != SQLITE_OK)
      {
         ETG_TRACE_FATAL(("RecreateDatabase: sqlite3_open_v2 failed: %d, errno=%d/%s", sqliteResult, errno, strerror(errno)));
         ETG_TRACE_ERRMEM(("RecreateDatabase: sqlite3_open_v2 failed: %d, errno=%d/%s", sqliteResult, errno, strerror(errno)));

         // -> Handle the error
         FW_FATAL_ASSERT(sqliteResult == SQLITE_OK);
      }
      else
      {
         //Set the right permissions("-rw-r-----") for the recreated database file
         ETG_TRACE_USR4(( " RecreateDatabase : Correcting the database 'permissions' to '-rw-r-----' " ));
         result = chmod(mDatabaseFile, S_IRUSR | S_IWUSR | S_IRGRP);
         if( CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_FATAL (( " RecreateDatabase : 'chmod(file, -rw-r-----)' has FAILED with error=%d, [file='%s'] ",
                     result, mDatabaseFile ));
            ETG_TRACE_ERRMEM(( " RecreateDatabase : 'chmod(file, -rw-r-----)' has FAILED with error=%d, [file='%s'] ",
                     result, mDatabaseFile ));
         }
      }

      /* Enable loadable modules (DSOs) */
      result = AddModules(IN dbHandle);

      if (result != CC_ERR_INT_NO_ERROR)
      {
         ETG_TRACE_FATAL(("RecreateDatabase: error on adding modules"));
      }

      result = ConfigureConnection(IN dbHandle);
      if (result != CC_ERR_INT_NO_ERROR)
      {
         ETG_TRACE_FATAL(("RecreateDatabase: error on ConfigureConnection"));
      }



      if (result != CC_ERR_INT_NO_ERROR)
      {
         ETG_TRACE_FATAL(("RecreateDatabase: error on sortlib_register_sort_function"));
      }

      // Create the database schema
      result = CreateDatabaseSchema(IN dbHandle);

      if (result != CC_ERR_INT_NO_ERROR)
      {
         ETG_TRACE_FATAL(("CreateDatabaseSchema(..): failed"));
         ETG_TRACE_ERRMEM(("CreateDatabaseSchema(..): failed"));

         FW_FATAL_ASSERT(sqliteResult == SQLITE_OK);
      }

      /* insert the db schema version number */
      result = ResetDatabaseIntegrity(IN dbHandle);

      if (result != CC_ERR_INT_NO_ERROR)
      {
         ETG_TRACE_FATAL(("ResetDatabaseIntegrity(..): failed"));
      }

      // close the database
      sqlite3_close(dbHandle);

      return CC_ERR_INT_NO_ERROR;
   }

   Result Database::AddModules(sqlite3 *dbHandle) const // finished: 100%
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("AddModules"));

      Result result = CC_ERR_INT_NO_ERROR;

      /* Enable loadable modules (DSOs) */
      result = sqlite3_enable_load_extension(dbHandle, 1);
      if (result != SQLITE_OK) {
         ETG_TRACE_ERR(("AddModules: error on sqlite3_enable_load_extension"));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   static Lock mLockRegister;
   static std::vector<void *> mPtrRegister;

   Result Database::Register(const void *ptr) const
   {
      ETG_TRACE_USR4(("Register"));

#if DEBUG_OPEN_DB_HANDLES
      ENTRY_INTERNAL
      mLockRegister.lock();
      mPtrRegister.push_back((void *)ptr);
      mLockRegister.unlock();
#else
      (void)ptr;
#endif
      return CC_ERR_INT_NO_ERROR;
   }

   Result Database::UnRegister(const void *ptr) const
   {
      ETG_TRACE_USR4(("UnRegister"));

#if DEBUG_OPEN_DB_HANDLES
      ENTRY_INTERNAL
      mLockRegister.lock();
      for(unsigned int i=0; i<mPtrRegister.size(); i++) {
         if (mPtrRegister[i] == ptr) {
            mPtrRegister.erase(mPtrRegister.begin()+i);
            i--;
         }
      }
      mLockRegister.unlock();
#else
      (void)ptr;
#endif
      return CC_ERR_INT_NO_ERROR;
   }

   int Database::Shutdown(const int waitForAllThreadsFinished) const
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("Shutdown"));

      /* set the shutdown flag */
      mShutDown = 1;

      /* wait for the shutdown start (max 3 seconds) (threads increasing this value) */
      for(unsigned int retry=0; retry < 6 && mShutDown == 1; retry++) {
         usleep(500);
      }

      /* now, at least one thread is running the shutdown now, but we do  not now if every thread is doing so. */

      if (waitForAllThreadsFinished) {

         /* wait for some threads stopping their work in the thread factory */
         for(unsigned int retry=0; retry < 6; retry++) {
            ETG_TRACE_USR2(("DB Threads running=%d", LocalSpm::getThreadFactory().GetThreadsRunning()));
            usleep(500);
            if (LocalSpm::getThreadFactory().GetThreadsRunning()+1 >= LocalSpm::getThreadFactory().GetMaxThreads()) {
               ETG_TRACE_USR2(("all threads stopped"));
               break;
            }
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   int Database::BusyCallback(void *_dbHandle, int times)
   {
      ETG_TRACE_USR4(("BusyCallback"));

      ENTRY_INTERNAL
      long sleepTimeUS = times * 10000;
      sqlite3 *dbHandle = (sqlite3 *)_dbHandle;

      /* ask a global variable if this busy should end because of shutdown */
      if (mShutDown) {
         ETG_TRACE_USR2(("DB busy, but asked for shutdown"));
         mShutDown++;
         return 0;
      }

      /* limit the sleep time to max shutdown delay time */
      long maxShutdownDelayUS = LocalSpm::getDataProvider().SPMMaxShutdownDelay() * 1000L /* us */;
      if (sleepTimeUS > maxShutdownDelayUS) sleepTimeUS = maxShutdownDelayUS;
      if (!(times%10)) { // print only every 10th time
         ETG_TRACE_USR2(("DB busy, waiting for %d us: %s", (int)sleepTimeUS, sqlite3_errmsg(dbHandle)));
#if DEBUG_OPEN_DB_HANDLES
         for(unsigned int i=0; i<mPtrRegister.size(); i++) {
            ETG_TRACE_USR2(("DB busy, open ptr[%d]: 0x%x", i, mPtrRegister[i]));
         }
#endif
      }

      /* sleep a while */
      usleep((useconds_t)sleepTimeUS);

      return 1; // = 1: continue trying
   }

   int Database::ProgressCallbackCaller(void *_ptr)
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR4(("ProgressCallbackCaller"));

      Database *_this = (Database *)_ptr;
      return _this->ProgressCallback();
   }

   int Database::ProgressCallback() const
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR4(("ProgressCallback"));

      /* ask a global variable if this progress should end because of shutdown */
      if (mShutDown) {
         ETG_TRACE_USR2(("DB in progress, but asked for shutdown"));
         mShutDown++;
         return 1; /* end this long runner */
      }

      /* ask a global variable if this progress should end because of user wish */
      if (mStopUpdate) {
         ETG_TRACE_USR2(("DB in progress, but asked for stop"));
         mStopUpdate = 0;
         return 1; /* end this long runner */
      }

      return 0; // = 0: continue trying
   }

   sqlite3 *Database::OpenDatabase()  // finished: 100%
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("OpenDatabase"));

      sqlite3 *dbHandle = NULL;
      Result result = CC_ERR_INT_NO_ERROR;
      tInteger sqliteResult = SQLITE_ERROR;

      /* is a free handle in the list? */
      mFreeDBConnectionsLock.lock();
      if (mFreeDBConnections.size()) {
         dbHandle = mFreeDBConnections.back();
         mFreeDBConnections.pop_back();
         mFreeDBConnectionsLock.unlock();
         Refresh(dbHandle);
         ETG_TRACE_USR2(("reusing handle: 0x%x", dbHandle));
         return dbHandle;
      }
      mFreeDBConnectionsLock.unlock();

      // open the data base
      sqliteResult = sqlite3_open_v2(
            mDatabaseFile,
            &dbHandle,
            SQLITE_OPEN_READWRITE
            ,
            NULL);
      if (sqliteResult != SQLITE_OK) {
         ETG_TRACE_ERR(("sqlite3_open_v2: failed: %d, errno=%d/%s", sqliteResult, errno, strerror(errno)));
         return NULL;
      }

      /* register busy handler */
      sqliteResult = sqlite3_busy_handler(dbHandle, BusyCallback, dbHandle);
      if (sqliteResult != SQLITE_OK) {
         ETG_TRACE_ERR(("sqlite3_busy_handler: failed: %d", sqliteResult));
         return NULL;
      }

      /* add sorting lib collation */
      /* if(LocalSpm::getDataProvider().DBSortUseUCAOn())
       {
           result = UcaSort_RegisterSortFunction(dbHandle,LocalSpm::getDataProvider().DBSortLanguage().c_str());
       }
       else
       {
           result = sortlib_register_sort_function(dbHandle);
       }

       if (result != CC_ERR_INT_NO_ERROR) {
           ETG_TRACE_FATAL(("OpenDatabase: error on sortlib_register_sort_function"));
           sqlite3_close(dbHandle);
           return NULL;
       }
       */

      // Configure the connection
      result = ConfigureConnection(IN dbHandle);

      // Check for success
      if (result != CC_ERR_INT_NO_ERROR)
      {
         ETG_TRACE_FATAL(("Failed to configure the database connection."));
         sqlite3_close(dbHandle);
         return NULL;
      }

      /* Enable loadable modules (DSOs) */
      result = AddModules(IN dbHandle);
      if (result != CC_ERR_INT_NO_ERROR) {
         ETG_TRACE_FATAL(("OpenDatabase: error on adding modules"));
         sqlite3_close(dbHandle);
         return NULL;
      }

      Refresh(dbHandle);

      // count this connection
      mOpenedConnections++;

      ETG_TRACE_USR2(("created handle: 0x%x, count=%d", dbHandle, mOpenedConnections));

      return dbHandle;
   }

   Result Database::CheckSchemaHash(sqlite3 *dbHandle)
   {
      ETG_TRACE_USR1(("CheckSchemaHash"));

      sqlite3_stmt* sqliteStatement = NULL;
      tInteger sqliteResult = SQLITE_ERROR;
      Result result;
      //char* sqlError = NULL;
      std::string DefaultSchemaHash="0";

      char query[] =
            "SELECT"
            "  [Version],"
            "  [Integrity],"
            "  [SchemaHash] "
            "FROM"
            "  [MAIN].[SchemaVersion];";

      sqliteResult = sqlite3_prepare_v2(dbHandle,
            query,
            sizeof(query),
            &sqliteStatement,
            NULL);

      // Check for success
      if (SQLITE_OK == sqliteResult)
      {
         sqliteResult = sqlite3_step(sqliteStatement);

         // Check for success
         // Retrieve the selected data if available.
         if (SQLITE_ROW == sqliteResult)
         {
            if(NULL != sqlite3_column_text(sqliteStatement, 2))
            {
               // Check whether the schema version is valid.
               /*if (0 == DefaultSchemaHash.compare(std::string(reinterpret_cast<const char*>(sqlite3_column_text(sqliteStatement, 2)))))
               {
                  // DB contains the default DB schema hash, DB schema hash has to be calculated and stored into DB

                  ETG_TRACE_FATAL(("CheckSchemaHash: DB contains the default DB schema hash, calculating schema hash value..."));

                  //computedSchemaMD5 = test;
                  char* sqlError = NULL;
                  std::string RecomputedSchemaHash = recreateDBSchemaHash();

                  std::string UpQuery =
                        "UPDATE"
                        "  [MAIN].[SchemaVersion] "
                        "SET"
                        "  [SchemaHash] = '" + RecomputedSchemaHash +
                        "'WHERE"
                        "  [SchemaHash] = '0'"
                        ";";

                  sqliteResult = sqlite3_exec(
                        dbHandle,
                        UpQuery.c_str(),
                        NULL, // No callback, as no data selected
                        NULL, // No callback arguments
                        &sqlError);

                  // Check for success
                  if (SQLITE_OK != sqliteResult)
                  {
                     // Failed to execute SQL query.
                     ETG_TRACE_FATAL(("CheckSchemaHash: failed to execute: %s.", UpQuery.c_str()));

                     // Error handling
                     if (NULL == sqlError)
                     {
                        ETG_TRACE_FATAL(("CheckSchemaHash: undefined SQLite3 error."));
                     }
                     else
                     {
                        ETG_TRACE_FATAL(("CheckSchemaHash: SQLite3 error: %s.", sqlError));
                        sqlite3_free(sqlError);
                     }

                     ETG_TRACE_FATAL(("CheckSchemaHash: could not store calculated DB schema hash value in DB"));

                     return CC_ERR_INT_DB_DATABASE_SCHEMA_FAILURE;
                  }

                  ETG_TRACE_FATAL(("CheckSchemaHash: stored the calculated DB schema hash value in DB => OK"));

                  result = CC_ERR_INT_NO_ERROR;
               }  */

               if (0 == std::string(reinterpret_cast<const char*>(sqlite3_column_text(sqliteStatement, 2))).compare(recreateDBSchemaHash()) )
               {
                  ETG_TRACE_FATAL(("CheckSchemaHash: calculated and stored DB schema hashes match => OK"));
                  result = CC_ERR_INT_NO_ERROR;
               }
               else
               {
                  ETG_TRACE_FATAL(("CheckSchemaHash: calculated DB schema hash differs from the one stored in file => not OK  "));
                  result = CC_ERR_INT_GENERAL_ERROR;
               }
            }
            else
            {
               // DB does not have the column containing the DB schema hash value
               ETG_TRACE_FATAL(("CheckSchemaHash: DB does not provide the DB schema hash value column => not OK"));
               result = CC_ERR_INT_GENERAL_ERROR;
            }
         }
         else
         {
            ETG_TRACE_ERR(("CheckSchemaHash: failed to step into the schema hash check statement: %s", sqlite3_errmsg(dbHandle)));

            // Handle the error
            result = CC_ERR_INT_DB_STEP_FAILURE;
         }
      }
      else
      {
         ETG_TRACE_ERR(("CheckSchemaHash: failed to execute the schema hash check statement: %s", sqlite3_errmsg(dbHandle)));

         // Handle the error
         result = CC_ERR_INT_DB_PREPARE_FAILURE;
      }

      // Destroy the schema hash statement
      sqliteResult = sqlite3_finalize(sqliteStatement);

      // Check for success
      if (SQLITE_OK != sqliteResult)
      {
         // Failed to finalize the schema hash statement
         ETG_TRACE_ERR(("CheckSchemaHash: failed to finalize the schema hash statement"));
         result = CC_ERR_INT_DB_FINALIZE_FAILURE;
      }

      return result;
   }

   void Database::InsertSchemaHasch(sqlite3 *dbHandle)
   {
      sqlite3_stmt* sqliteStatement = NULL;
      tInteger sqliteResult = SQLITE_ERROR;
      std::string DefaultSchemaHash="0";
      char query[] =
            "SELECT"
            "  [Version],"
            "  [Integrity],"
            "  [SchemaHash] "
            "FROM"
            "  [MAIN].[SchemaVersion];";

      sqliteResult = sqlite3_prepare_v2(dbHandle,
            query,
            sizeof(query),
            &sqliteStatement,
            NULL);

      // Check for success
      if (SQLITE_OK == sqliteResult)
      {
         sqliteResult = sqlite3_step(sqliteStatement);

         // Check for success
         // Retrieve the selected data if available.
         if (SQLITE_ROW == sqliteResult)
         {
            if(NULL != sqlite3_column_text(sqliteStatement, 2))
            {
               if (0 == DefaultSchemaHash.compare(std::string(reinterpret_cast<const char*>(sqlite3_column_text(sqliteStatement, 2)))))
               {
                  // DB contains the default DB schema hash, DB schema hash has to be calculated and stored into DB

                  ETG_TRACE_FATAL(("InsertSchemaHasch: DB contains the default DB schema hash, calculating schema hash value..."));

                  //computedSchemaMD5 = test;
                  char* sqlError = NULL;
                  std::string RecomputedSchemaHash = recreateDBSchemaHash();

                  std::string UpQuery =
                        "UPDATE"
                        "  [MAIN].[SchemaVersion] "
                        "SET"
                        "  [SchemaHash] = '" + RecomputedSchemaHash +
                        "'WHERE"
                        "  [SchemaHash] = '0'"
                        ";";

                  sqliteResult = sqlite3_exec(
                        dbHandle,
                        UpQuery.c_str(),
                        NULL, // No callback, as no data selected
                        NULL, // No callback arguments
                        &sqlError);

                  // Check for success
                  if (SQLITE_OK != sqliteResult)
                  {
                     // Failed to execute SQL query.
                     ETG_TRACE_FATAL(("InsertSchemaHasch: failed to execute: %s.", UpQuery.c_str()));

                     // Error handling
                     if (NULL == sqlError)
                     {
                        ETG_TRACE_FATAL(("InsertSchemaHasch: undefined SQLite3 error."));
                     }
                     else
                     {
                        ETG_TRACE_FATAL(("InsertSchemaHasch: SQLite3 error: %s.", sqlError));
                        sqlite3_free(sqlError);
                     }

                     ETG_TRACE_FATAL(("InsertSchemaHasch: could not store calculated DB schema hash value in DB"));


                  }

                  ETG_TRACE_FATAL(("InsertSchemaHasch: stored the calculated DB schema hash value in DB => OK"));


               }
            }
         }
      }
   }
}
