/**
 * @defgroup DbManagerModule DbManager
 * @ingroup BmCoreModule
 *
 * @brief This module comprises all parts related to DB Manager
 */

/**
 * @file
 * @ingroup DbManagerModule
 *
 * @brief This file contains the declaration of the class DbManager
 *
 * @copyright 2015 Robert Bosch Car Multimedia GmbH
 */

#ifndef _DB_MANAGER_H_
#define _DB_MANAGER_H_

#include "BmAllTypes.h"
#include "BtStackIfTypes.h"
#include "ILocalSpm.h"
#include "Lock.h"
#include "../sqlite/sqlite3.h"
#include "Database.h"
#include "Query.h"
#include "Timer.h"
#include <ProtocolConnectionController.h>
#include <DeviceConnectionController.h>

#include <map>
#include <vector>
#include <string>

namespace bmcore
{
   /**
    * @class DbManager
    * @ingroup DbManagerModule
    *
    * @brief The class DbManager provides the interface to access the database
    *
    */
   class DbManager : public ILocalSpm
   {

   public:

      /**
       * Brief: Constructor of DbManager component which creates an instance.
       *        No further initialization of internal components and resources
       *        is performed in the constructor. See the explicit allocation
       *        and initialization functions Create() and Init(). Hence, the
       *        implementation is trivial.
       *
       * Synchronization: not re-entrant.
       *
       * Performance considerations: none.
       *
       * @return void
       */
      DbManager(ComponentId componentID);

      /**
       * Destructor of component.
       *
       * @return void
       */
      virtual ~DbManager();

      // ***********************
      // * SPM related methods *
      // ***********************

      /**
       * Brief: performs an explicit allocation of the resources used by
       * the DbManager internally.
       *
       * Synchronization: not re-entrant.
       *
       * Performance considerations: none.
       *
       * @return void
       */
      void create();

      /**
       * Brief: performs the initialization of the DbManager and its
       *        subcomponents, i.e.:
       *        - initialization of member variables;
       *        - establishment of a database connection;
       *        - integrity check of the database;
       *        - initialization of temporary and virtual tables;
       *        - create internal flash devices; //Roadmap 13005
       *
       * Synchronization: re-entrant.
       *
       * Performance considerations: can take up to 1 second if the database
       *                             recreation is required.
       *
       * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
       */
      Result init(InitReason reason); //Roadmap 13005: 100%

      /**
       * Brief: starts the component, implies that all other mediaplayer
       *        components are available.
       *        Since the DbManager does not require a state machine, the
       *        implementation of this function is trivial.
       *
       * Synchronization: re-entrant.
       *
       * Performance considerations: none.
       *
       * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
       */
      Result run();

      /**
       * Brief: stops the component.
       *        Since the DbManager does not require a state machine, the
       *        implementation of this function is trivial.
       *
       * Synchronization: re-entrant.
       *
       * Performance considerations: none.
       *
       * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
       */
      Result stop();

      /**
       * Brief: cleans up the resources used by the component, i.e:
       *        - release the database resources (e.g. prepared statements);
       *        - closing the database connection(s).
       *
       * Synchronization: re-entrant.
       *
       * Performance considerations: none.
       *
       * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
       */
      Result done();

      /**
       * Brief: returns statistic data as a string for the over all component object statistics
       *
       * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
       */
      int statistics(OUT Statistics stat);

      // ******************************
      // * End of SPM related methods *
      // ******************************

      /* --- paste here --- */

      void switchOnDatabaseAccess();

      void switchOffDatabaseAccess();

      bool isDatabaseAccessSwitchedOn();

      Result setDeviceInbandRingtoneSupport(IN const DeviceId deviceId, IN const InbandRingtoneSupport inbandRingtoneSupport);

      Result setDeviceInbandRingtoneSupport(IN const BdAddress& bdAddress, IN const InbandRingtoneSupport inbandRingtoneSupport);

      Result getDeviceInbandRingtoneSupport(OUT InbandRingtoneSupport& inbandRingtoneSupport, IN const DeviceId deviceId);

      Result setDeviceRingtoneId(IN const DeviceId deviceId, IN const RingtoneId ringtoneId);

      Result getDeviceRingtoneId(OUT RingtoneId& ringtoneId, IN const DeviceId deviceId);

      Result setSystemWideRingtoneId(IN const RingtoneId ringtoneId);

      Result getSystemWideRingtoneId(OUT RingtoneId& ringtoneId);

      /**
       * @brief saves the disconnected state for a phone into the device table
       *
       * @param[in] deviceID of the device (phone)
       * @param[in] state state to be saved to database
       *
       * @return 0 on success and an error code otherwise.
       */
      Result setDeviceDisconnectedReason(IN const DeviceId deviceID, IN const DisconnectedReason reason);

      /**
       * @brief gets the disconnected state for a phone from the device table
       *
       * @param[out] state last disconnection state of a phone in the device table
       * @param[in] deviceID of the device (phone)
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getDeviceDisconnectedReason(OUT DisconnectedReason& reason, IN const DeviceId deviceID);

      /**
       * @brief gets information about Last Connection of the given Device ID
       *
       * @param[out] connectionOrderInfo   LastConnectedTimeStamp and LastConnectedIndicator will be written to it
       * @param[in]  deviceId              device Id
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getDeviceLastConnectedInfo(OUT ConnectionOrderInfo& connectionOrderInfo, IN const DeviceId deviceId);

      /**
       * @brief sets the current state for a phone in the device table
       *
       * @param[in] deviceID of the device (phone)
       * @param[in] state current state of the phone
       * @param[in] reason reason for device disconnection
       *
       * @return 0 on success and an error code otherwise.
       */
      Result setDeviceConnectionStatus(IN const DeviceId deviceId, IN const ConnectionStatus connectionStatus, IN const DisconnectedReason disconnectedReason);

      /**
       * @brief get the stored state for a phone from the device table
       *
       * @param[out] state stored state of the phone
       * @param[out] reason reason for disconnection
       * @param[in] deviceID of the device (phone)
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getDeviceConnectionStatus(OUT ConnectionStatus& connectionStatus, OUT DisconnectedReason& reason, IN const DeviceId deviceID);

      Result getDeviceConnectionStatus(OUT ConnectionStatus& connectionStatus,
            OUT DisconnectedReason& disconnectedReason, IN const BdAddress& bdAddress);

      Result isUnusedProtocolConnectionControllerAvailable(OUT bool& available, IN const ProtocolId protocolId, IN const Uuid& uuid);

      Result isUsedProtocolConnectionControllerAvailable(OUT bool& available, OUT DeviceId& deviceid, IN const ProtocolId protocolId);

      /**
       * @brief: assigns an unused / free protocol connection controller instance (uses mutex).
       * If found one, it is marked as in use by writing the deviceId to column inUseBy
       * and sets the UUID if the protocol is SPP. If the protocol connection controller instance
       * is already assigned to the device with the given device ID this instance is given back.
       *
       * @param[out] handle handle to a protocol SM (set SMInUse=1 during this call)
       * @param[in] deviceId ID of the device this protocol will be assigned to
       * @param[in] protocolId ID of the protocol type for which the SM should be searched
       * @param[in] uuid UUID for SPP protocol
       *
       * @return 0 on success and an error code otherwise.
       *         if handle returned == 0: there was no free state machine
       */
      Result assignProtocolConnectionController(OUT ProtocolConnectionController** pccHandle, IN const DeviceId deviceId, IN const ProtocolId protocolId, IN const Uuid& uuid);

      /**
       * @brief assigns an unused / free protocol state machine (uses mutex).
       * If found one, marks it as in use by writing the deviceId to column inUseBy
       * and sets the UUID if the protocol is a SPP
       *
       * new: if protocol state machine is already assigned the already assigned one is given back
       *
       * @param[out] pccHandle handle to a protocol SM (set SMInUse=1 during this call)
       * @param[in] bdAddress bdAddress of the device this protocol will be assigned to
       * @param[in] protocolId ID of the protocol type for which the SM should be searched
       * @param[in] uuid UUID for SPP protocol
       *
       * @return 0 on success and an error code otherwise.
       *         if handle returned == 0: there was no free state machine
       */
      //Result assignProtocolConnectionController(OUT ProtocolConnectionController** pccHandle, IN const BdAddress& bdAddress, IN const ProtocolId protocolId, IN const Uuid& uuid);

      Result getProtocolConnectionController(OUT ProtocolConnectionController** pccInstance, IN const DeviceId deviceId, IN const ProtocolId protocolId, IN const Uuid& uuid);

      /**
       * @brief get the assigned protocol state machine
       *
       * @param[out] pccInstance handle to a protocol SM
       * @param[in] bdAddress bluetooth address of device handled by pccInstance
       * @param[in] protocolId ID of the protocol type for which the SM should be searched
       * @param[in] uuid UUID for SPP protocol
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getProtocolConnectionController(OUT ProtocolConnectionController** pccInstance, IN const BdAddress& bdAddress, IN const ProtocolId protocolId, IN const Uuid& uuid);

      /**
       * @brief sets the device connection state machine handle to a device id (in devices table)
       *
       * @param[in] deviceID device id
       * @param[in] dccInstance pointer to device connection state machine
       *
       * @return 0 on success and an error code otherwise.
       */

      Result setDeviceConnectionController(IN const DeviceId deviceId, IN const DeviceConnectionController* dccInstance);

      /**
       * @brief gets the device connection state machine handle from a deviceId. Uses devices table
       *
       * @param[out] dccInstance pointer to device connection controller instance pointer
       * @param[in] bdAddress BD address of the remote device
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getDeviceConnectionController(OUT DeviceConnectionController** dccInstance, IN const BdAddress& bdAddress);

      /**
       * @brief gets the device connection state machine handle from a deviceId. Uses devices table
       *
       * @param[out] dccInstance handle to device connection handler SM
       * @param[in] deviceID device id
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getDeviceConnectionController(OUT DeviceConnectionController** dccInstance, IN const DeviceId deviceId);

      /**
       * @brief gets BD Address from deviceId Uses devices table
       *
       * @param[out] bdAddress bluetooth address
       * @param[in] deviceID ID of phone device
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getBdAddress(OUT BdAddress& bdAddress, IN const DeviceId deviceID);

      /**
       * @brief gets BD Address from device Handle Uses devices table
       *
       * @param[out] bdAddress bluetooth address
       * @param[in] deviceHandle handle of phone device
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getBdAddressByDeviceHandle(OUT BdAddress& bdAddress, IN const DeviceId deviceHandle);

      /**
       * @brief gets Device Handle from bdAddress Uses devices table
       *
       * @param[out] devHandle device Handle
       * @param[in] bdAddress bdAddress of phone device
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getDeviceHandle(OUT DeviceId& devHandle, IN const BdAddress bdAddress);

      /**
       * @brief gets Device Handle from deviceId Uses devices table
       *
       * @param[out] devHandle device Handle
       * @param[in] deviceID ID of phone device
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getDeviceHandle(OUT DeviceId& devHandle, IN const DeviceId deviceId);

      /**
       * @brief get a list of all device IDs
       *
       * @param[out] deviceIdList list of all device IDs
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getAllDeviceIds(OUT DeviceIdList& deviceIdList);
      /**
       * @brief get a list of all device Addresses
       *
       * @param[out] bdAddressList list of all BD addresses of paired devices
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getAllBdAddresses(OUT BdAddressList& bdAddressList);

      /**
       * @brief releases protocol state machine handle InUse flag (uses mutex)
       *
       * @param[in] handle handle to protocol SM (set SMInUse=0 during this call)
       *
       * @return 0 on success and an error code otherwise.
       */
      Result releaseProtocolConnectionController(IN const ProtocolConnectionController* pccInstance,
            IN const DisconnectedReason disconnectedReason);

      Result getProtocolConnectionStatus(OUT ConnectionStatus& connectionStatus, OUT DisconnectedReason& disconnectedReason,
               IN const DeviceId deviceId, IN const ProtocolId protocolId, IN const Uuid& uuid);

      Result getProtocolConnectionStatus(OUT ConnectionStatus& connectionStatus, OUT DisconnectedReason& disconnectedReason,
            IN const BdAddress& bdAddress, IN const ProtocolId protocolId, IN const Uuid& uuid);

      /**
       * @brief sets the protocol state for a device / protocol / uuid combination in DeviceProtocols table
       *
       * if deviceID == 0: does nothing
       *
       * if HFP or AVP AND CONNECTING/CONNCTED:
       *   set LastMode for device/protocol also in LastMode table
       *      if state was CONNECTING: enter CONNECTED in lastmode table
       *
       * @param[in] deviceId ID of the device
       * @param[in] protcolId ID of the protocol
       * @param[in] uuid uuid for SPP
       * @param[in] state state, to be set
       * @param[in] reason reason for protocol disconnection
       *
       * @return 0 on success and an error code otherwise.
       */
      Result setProtocolConnectionStatus(IN const DeviceId deviceId, IN const ProtocolId protocolId,
            IN const Uuid& uuid, IN const ConnectionStatus connectionStatus, IN const DisconnectedReason reason, IN const RfcommDevicePath& rfcommDevicePath);

      /**
       * @brief sets LastExpectedConnectionStatus for a device/protocol combination
       *
       * @param[in] deviceId ID of the device
       * @param[in] protocolId ID of the protocol
       * @param[in] lastExpectedConnectionStatus lastExpectedConnectionStatus
       *
       * @return 0 on success and an error code otherwise.
       */
      Result setLastExpectedProtocolConnectionStatus(IN const DeviceId deviceId, IN const ProtocolId protocolId,
            IN const Uuid& uuid, IN const LastExpectedConnectionStatus lastExpectedConnectionStatus);

      Result resetLastExpectedProtocolConnectionStatus(IN const DeviceId deviceId);

      /**
       * @brief sets Protocol Version for a device/protocol combination
       *
       * @param[in] deviceId ID of the device
       * @param[in] protocolId ID of the protocol
       * @param[in] protocolVersion protocolVersion as Hex Value
       *
       * @return 0 on success and an error code otherwise.
       */
      Result setProtocolVersion(IN const DeviceId deviceId, IN const ProtocolId protocolId, IN const ProtocolVersion protocolVersion);
      /**
       * @brief gets LastExpectedConnectionStatus for a device/protocol combination
       *
       * @param[out] lastExpectedConnectionStatus lastExpectedConnectionStatus
       * @param[in] deviceId ID of the device
       * @param[in] protocolId ID of the protocol
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getLastExpectedProtocolConnectionStatus(OUT LastExpectedConnectionStatus& lastExpectedConnectionStatus,
            IN const DeviceId deviceId, IN const ProtocolId protocolId, IN const Uuid& uuid);

      /**
       * @brief Creates a device row in the Devices table and creates corresponding row(s) in DeviceProtocols table
       *
       * - add a new row in the Devices table
       *    - in case of string parameters store corresponding cstrings (c_str())
       *    - in column MajorServiceClass store the value of majorServiceClass.getData()
       *
       * - for each protocol ID given in supportedProtocols (except SPP) create a new row in the
       *   DeviceProtocols table
       *
       * - for each UUID given in supportedSppUuids create a new row in the DeviceProtocols table with ProtocolID
       *   representing SPP
       */
      Result createDevice(OUT DeviceId& deviceId, IN const BdName& bdName, IN const BdAddress& bdAddress,
            IN const BTSLinkKeyType linkKeyType,IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey,
            IN const BTSMajorServiceClass& majorServiceClass, IN const BTSMajorDeviceClass majorDeviceClass,
            IN const BTSMinorDeviceClass minorDeviceClass, IN const ProtocolIdVersionList& supportedProtocols,
            IN const UuidList& supportedSppUuids, IN const DeviceIdentification& deviceIdentification);

      Result isDeviceInDb(OUT bool& inDb, IN const BdAddress& bdAddress);

      Result updateOobPairingInfo(IN const BdAddress& bdAddress, IN const BdName& bdName, IN const LinkKey& linkKey);

      /**
       * @brief Updates the DeviceProtocols table corresponding to the protocols supported by the given device
       *
       * - get the device ID from the Devices table for the given BD address
       * - get a list of protocol IDs from DeviceProtocols table for this device ID (oldProtocolIdList)
       * - for each protocol ID in oldProtocolIdList that does not exist in newSupportedProtocols delete the
       *   corresponding row from DeviceProtocols table (in case of SPP all rows corresponding to SPP
       *   have to be deleted)
       * - for protocol IDs that exist in newSupportedProtocols but not in oldProtocolIdList add a new row
       *   to DeviceProtocols table
       */
      Result updateProtocolSupport(IN const BdAddress& bdAddress, IN const ProtocolIdVersionList& supportedProtocols, IN const bool partialUpdate);

      /**
       * @brief Updates the DeviceProtocols table corresponding to the SPP UUIDs supported by the given device
       *
       * - get the device ID from the Devices table for the given BD address
       * - get a list of SPP UUIDs from DeviceProtocols table for this device ID (oldSppUuidList)
       * - for each UUID in oldSppUuidList that does not exist in supportedSppUuids delete the
       *   corresponding row from DeviceProtocols table
       * - for SPP UUIDs that exist in supportedSppUuids but not in oldSppUuidList add a new row
       *   to DeviceProtocols table with ProtocolID representing SPP
       */
      Result updateSppUuidSupport(IN const BdAddress& bdAddress, IN const UuidList& supportedSppUuids, IN const ProtocolVersion sppVersion);

      /**
       * @brief Updates the Device Identification info in Devices table
       */
      Result updateDeviceIdentification(IN const BdAddress& bdAddress, IN const DeviceIdentification& deviceIdentification);

      /**
       * @brief get the device Id associated with a bdAddress
       * @param[out] deviceId deviceId
       * @param[in] bdAddress bdAddress
       * @return 0 on success and an error code otherwise.
       */
      Result getDeviceId(OUT DeviceId & deviceId, IN const BdAddress& bdAddress);

      /**
       * @brief get the device Id associated with a device Handle
       * @param[out] deviceId deviceId
       * @param[in] deviceHandle device Handle
       * @return 0 on success and an error code otherwise.
       */
      Result getDeviceId(OUT DeviceId& deviceId, IN const DeviceId deviceHandle);

      /**
       * @brief get list of supported protocols by the device with deviceId
       * @param[out] protocolIdList list of ids/versions of supported protocols
       * @param[in] deviceId deviceId
       * @return 0 on success and an error code otherwise.
       */
      Result getDeviceSupportedProtocols(OUT ProtocolInfoMap& protocolInfomap, IN const DeviceId deviceId);
      /**
       * @brief get list of SppUuid Support by the device with deviceId
       * @param[out] uuidList list of uuids supported for the device.
       * @param[in] deviceId deviceId
       * @return 0 on success and an error code otherwise.
       */
      Result getDeviceSppUuidSupport(OUT UuidList& uuidList, IN const DeviceId deviceId);

      Result getDeviceSppUuidSupport(OUT UuidList& uuidList, IN const BdAddress& bdAddress);

      Result getDeviceBlockStatusList(OUT DeviceBlockStatusList& deviceBlockStatusList);

      Result setDeviceBlockState(IN const DeviceId deviceId, IN const BlockState blockState);

      Result getDeviceBlockState(OUT BlockState& blockState, IN const DeviceId deviceId);

      Result getDeviceBlockState(OUT BlockState& blockState, IN const BdAddress& bdAddress);

      Result getDeviceBlockStatus(OUT DeviceBlockStatus& devBlockStatus, IN const DeviceId deviceId);

      Result setProtocolBlockState(IN const DeviceId deviceId, IN const ProtocolId protocolId, IN const Uuid& uuid, IN const BlockState blockState);

      Result getProtocolBlockState(OUT BlockState& blockState, IN const DeviceId deviceId, IN const Protocol& protocol);

      Result getProtocolBlockState(OUT BlockState& blockState, IN const BdAddress& bdAddress, IN const Protocol& protocol);

      /**
       * @brief Gets the device base info from the Devices and the DeviceProtocols table for a given device
       *
       * - fill given deviceBaseInfo object with corresponding information from Devices/DeviceProtocols table
       *   (for deviceBaseInfo._connectionStatus: use the ConnectionStatus from Devices table;
       *   for deviceBaseInfo._protocolInfo: only set the _isSupported member for each protocol ID found
       *   in the DeviceProtocols table for the given device ID)
       */
      Result getDeviceBaseInfo(OUT DeviceBaseInfo& deviceBaseInfo, IN const DeviceId deviceId);

      Result getDeviceBaseInfo(OUT DeviceBaseInfo& deviceBaseInfo, IN const BdAddress& bdAddress);

      /**
       * @brief Gets the device base info from the Devices and the DeviceProtocols table for all devices
       *
       * - do the same as described for getDeviceBaseInfo(...) but for all devices in Devices table
       */
      Result getAllDeviceBaseInfo(OUT DeviceBaseInfoList& deviceBaseInfoList);

      /**
       * @brief get info about the specified remote device with deviceId
       * @param[out] remoteDeviceInfo deviceId
       * @param[in] deviceId deviceId
       * @return 0 on success and an error code otherwise.
       */
      Result getRemoteDeviceInfo(OUT RemoteDeviceInfo& remoteDeviceInfo, IN const DeviceId deviceId);

      /**
       * @brief Creates one protocol entry in the protocols table
       *
       * @param[in] protocolId id of protocol (from enum)
       * @param[in] protocolName user friendly name of protocol
       * @param[in] pccInstance pointer to Protocol Connection Controller instance
       * @return 0 on success and an error code otherwise.
       */
      Result createProtocol(IN const ProtocolId protocolId, IN const ProtocolName protocolName);

      /**
       * @brief gets a link key for a given device id
       *
       * @param[out] linkKey link key of given device id
       * @param[in] deviceID ID of phone device
       * @return 0 on success and an error code otherwise.
       */
      Result getLinkKey(OUT LinkKey& linkKey, IN const DeviceId deviceId);

      /**
       * @brief gets  the link key information for a specific device address.
       *
       * @param[out] linkKeyType link key type of a given address
       * @param[out] linkKey link key  of a given address
       * @param[out] dLinkKey dlink key  of a given address
       * @param[in] bdAddress address of phone device
       * @return 0 on success and an error code otherwise.
       */
      Result getLinkKeyInformation(OUT BTSLinkKeyType& linkKeyType, OUT BTSLinkKey& linkKey, OUT BTSDLinkKey& dLinkKey, IN const BdAddress& bdAddress);

      /**
       * @brief gets a list of all protocol connection state machine handle which are related to given device ID
       *
       * @param[out] pccInstances list of state machine handles
       * @param[in] deviceId ID of device to search for
       *
       * @return 0 on success and an error code otherwise.
       */

      Result getProtocolConnectionControllers(OUT ProtocolConnectionControllerList& pccInstances, IN const DeviceId deviceId);

      Result getProtocolConnectionControllers(OUT ProtocolConnectionControllerList& pccInstances, IN const BdAddress& bdAddress);

      /**
       * @brief gets information about Last Connection of ProtocolId/UUID of the given Device ID
       *
       * @param[out] connectionOrderInfo   LastConnectedTimeStamp and LastConnectedIndicator will be written to it
       * @param[in]  deviceId              device Id
       * @param[in]  protocolId            protocol Id
       * @param[in]  uuid                  uuid will be used only for SPP protocol
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getProtocolLastConnectedInfo(OUT ConnectionOrderInfo& connectionOrderInfo,
               IN const DeviceId deviceId, IN const ProtocolId protocolId, IN const UuidChrArr uuid);
      /**
       * @brief gets max(LastConnectedIndicator) from all the rows in DeviceProtocols table
       *
       * @param[out] maxlastConnectedIndicator max(LastConnectedIndicator)
       * @param[in]  useDevicesTable true(read from Devices table) false(read from DeviceProtcols Table)
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getMaxLastConnectedIndicator(OUT unsigned int & maxlastConnectedIndicator, IN bool useDevicesTable);

      Result getMaxLastPairedIndicator(OUT unsigned int & maxlastConnectedIndicator);

      Result getNumberOfDevices(OUT uint32_t& numberOfDevices);

      /**
       * @brief find the least unassigned value for device Handle in Devices table
       *
       * @param[out] deviceHandle least unassigned value for a Device Handle
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getUnassignedDeviceHandle(OUT DeviceId& deviceHandle);

      /**
       * @brief gets a list of all protocol connection state machine handles
       *
       * @param[out] pccInstances list of state machine handles
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getProtocolConnectionControllers(OUT ProtocolConnectionControllerList& pccInstances);



      /**
       * @brief gets connection info for all protocols for a specific device
       *
       * @param[out] protocolConnectionInfo connection info for all protocols except SPP
       * @param[out] sppConnectionInfo connection info for each SPP connection identified by uuid
       * @param[in] deviceId deviceId
       *
       * @return 0 on success and an error code otherwise.
       */
      Result getProtocolConnectionInfos(OUT ProtocolConnectionInfoMap& protocolConnectionInfo,
                                                      OUT SppConnectionInfoMap& sppConnectionInfo, IN const DeviceId deviceId);

      Result getProtocolsExpectedToBeConnected(OUT ProtocolList& protocolList, IN const DeviceId deviceId);

      Result getProtocolsLastConnected(OUT ProtocolList& protocolList, IN const DeviceId deviceId);

      Result getDevicesLastConnected(OUT DeviceIdList& deviceList);

      Result getConnectedDevices(OUT DeviceIdList& deviceList);

      Result getDeviceProtocolsConnectedStatus(OUT ProtocolList& protocolList, IN const DeviceId deviceId);

      /**
       * @brief deletes a device from the database completely
       *
       * @param[in] deviceId ID of device (if deviceId==0 all devices are deleted)
       *
       * @return 0 on success and an error code otherwise.
       */
      Result deleteDevice(IN const DeviceId deviceId);

      Result getBtStatusTargetState(OUT TargetSwitchState& btStatusTargetState);

      Result setBtStatusTargetState(IN const TargetSwitchState btStatusTargetState);

      Result getLocalBdAddress(OUT BdAddress& bdAddress);

      Result setLocalBdAddress(IN const BdAddress& bdAddress);

      Result getLocalBdName(OUT BdName& bdName);

      Result setLocalBdName(IN const BdName& bdName);

      Result getLegacyPairingPin(OUT PairingPinType& legacyPairingPin);

      Result setLegacyPairingPin(IN const PairingPinType& legacyPairingPin);

      Result getStandardAutoConnectionType(OUT StandardAutoConnectionType& standardAutoConnectionType);

      Result setStandardAutoConnectionType(IN const StandardAutoConnectionType standardAutoConnectionType);

      Result getMultiHFPSupportStatus(OUT TargetSwitchState& multiHFPSupportTargetState);

      Result setMultiHFPSupportStatus(IN const int multiHFPSupportTargetState);

      Result getPrimaryHfpDevice(OUT DeviceId& deviceId);

      Result getPrimaryStatus(IN const DeviceId deviceId, OUT unsigned int &primaryStatus);

      Result getSecondaryHfpDevice(OUT DeviceId& deviceId);

      Result setDeviceUsagePreference(IN const DeviceId deviceId, IN const UsagePreference usagePreference);

      Result getDeviceUsagePreference(OUT UsagePreference& usagePreference, IN const DeviceId deviceId);

      Result getDeviceUsagePreference(OUT UsagePreference& usagePreference, IN const BdAddress& bdAddress);

      Result setMasterHFPDevice(IN const DeviceId deviceId);

      Result getMasterHFPDevice(OUT DeviceId& deviceId);

      Result setCPWDevice(IN const DeviceId deviceId, IN const bool reset = false);

      Result getCPWDevice(OUT DeviceId& deviceId);

      // Added for NCG3D-61585 - PAN is not automatically reconnected after cold restart of HU
      Result setLastConnectedPANDevice(IN const DeviceId deviceId, IN const bool reset = false);

      Result getLastConnectedPANDevice(OUT DeviceId& deviceId);

      Result getDevicePairingOrderInfo(OUT ConnectionOrderInfo& pairingOrderInfo, IN const DeviceId deviceId);

      Result getDeviceIdentification(OUT const DeviceIdentification& deviceIdentification, IN const DeviceId deviceId);

      Result getOrderedDeviceConnectionControllers(OUT DeviceConnectionControllerList& orderedDccInstances, IN DeviceConnectionControllerList& dccInstances);

      Result getOrderedDeviceProtocolListMap(OUT DeviceProtocolListMap& orderedDeviceProtocolListMap, IN DeviceProtocolListMap& deviceProtocolListMap);

      Result getOldestConnectedDevice(OUT DeviceId& deviceId, OUT ConnectionStatus &connectionStatus, OUT DisconnectedReason& disconnectedReason);

      Result resetProtocolsTable(void);

      Result resetDevicesTable(void);

      Result resetDeviceProtocolsTable(void);

      Result setBdDeviceName(IN const DeviceId& deviceId, IN const BdName& updatedBdDeviceName);

      Result setBdDeviceName(IN const BdAddress& bdAddress, IN const BdName& updatedBdDeviceName);

      Result getBdDeviceName(OUT BdName& bdDeviceName, IN const DeviceId deviceId);

      Result getBdDeviceName(OUT BdName& bdDeviceName, IN const BdAddress& bdAddress);

      /**
       * @brief resets all DB tables
       *
       * @return 0 on success and an error code otherwise.
       */
      Result resetAllTables(void);
      /**
             * @brief delete the Map containing the PCC instances
             *
             * @return 0 on success and an error code otherwise.
             */
      Result destroyProtocolConnectionController(void);

      /**
       * @brief resets protocol connection controller instances
       */
      void resetProtocolConnectionController(void);

      /**
             * @brief insert data into Protocol list
             *
             * @return 0 on success and an error code otherwise.
              */
      Result insertProtocolList(IN const ProtocolId protocolId,IN const ProtocolName protocolName,IN const Uuid UUID,IN  ProtocolConnectionController* ProtocolConnectionSMHandle,IN const int SMInUseByDeviceID);
      /**
       * @brief get ProtocolConnectionController instance
       *
       */
      ProtocolConnectionController* getNewProtocolConnectionController(IN const ProtocolId protocolId);
      /**
                   * @brief Set primary HFP device
                   *
                   * @return 0 on success and an error code otherwise.
                    */
      Result SetPrimaryHfpDevice(IN const DeviceId deviceId);

      /**
                         * @brief Set primary device
                         *
                         * @return 0 on success and an error code otherwise.
                          */

      Result SetPrimaryDevice(IN const DeviceId deviceId,IN const ConnectionStatus connectionStatus);
      /**
                               * @brief Get favorite Index
                               *
                               * @return 0 on success and an error code otherwise.
                                */

      Result getFavoriteIndex(OUT FavoriteIndex& favoriteIndex, IN const DeviceId deviceId, IN const FavoriteType favoriteType);

      Result getFavoriteIndex(OUT FavoriteIndex &favoriteIndex, IN const BdAddress& bdAddress, IN const FavoriteType favoriteType);
      /**
                                          * @brief Insert favorite Index
                                          *
                                          * @return 0 on success and an error code otherwise.
                                           */
      Result InsertFavoriteindex(IN const DeviceId deviceId, IN const FavoriteType favoriteType, IN  FavoriteIndex newfavoriteIndex);
      /**
                                    * @brief Set favorite Index
                                    *
                                    * @return 0 on success and an error code otherwise.
                                     */
      Result setFavoriteIndex(IN const DeviceId deviceId, IN const FavoriteType favoriteType, IN const FavoriteIndex favoriteIndex);
      /**
                                          * @brief Get Maximum favorite Index
                                          *
                                          * @return 0 on success and an error code otherwise.
                                           */
      Result getMaximumFavoriteIndex(OUT FavoriteIndex& maxFavoriteIndex, IN const FavoriteType favoriteType);
      /**
                                                * @brief get Devices Sorted By Favorite
                                                *
                                                * @return 0 on success and an error code otherwise.
                                                 */
      Result getDevicesSortedByFavorite(OUT DeviceIdList& favoriteDeviceIds, IN const FavoriteType favoriteType);
      /**
                                                      * @brief Set Device as Favorite
                                                      *
                                                      * @return 0 on success and an error code otherwise.
                                                       */
      Result SetDeviceFavorite(IN const DeviceId deviceId,IN const FavoriteType favoriteType,IN const FavoriteIndex favoriteIndex);
	  
      static bool compareProtocolIds(const Protocol& a, const Protocol& b);	  
      /* --- end of paste --- */

      // *********************************
      // * further core internal methods *
      // *********************************

      Database* getDatabase()
      {
         return _db;
      }

   private:

      /**
       * @brief fill in a list of DeviceBaseInfo for a single device or for all devices
       *
       * @param[out] deviceBaseInfoList list of DeviceBaseInfo to be read from db
       * @param[in] deviceId ID of device to retrieve (if deviceId==0 all devices are retrieved)
       *
       * @return 0 on success and an error code otherwise.
       */
      Result privGetDeviceBaseInfo(OUT DeviceBaseInfoList& deviceBaseInfoList, IN const DeviceId deviceId);

      Result privGetDeviceBaseInfo(OUT DeviceBaseInfoList& deviceBaseInfoList, IN const BdAddress& bdAddress);

      template<class T>
      static bool compareLastConnectedIndicator(const std::pair<unsigned int, T>& a, const std::pair<unsigned int, T>& b);

      static bool compareDeviceIndicator(const std::pair<DeviceIndicator, DeviceId>& a, const std::pair<DeviceIndicator, DeviceId>& b);

      Result setDeviceConnectionStatus(IN DeviceId deviceId, IN const ConnectionStatus connectionStatus, IN const DisconnectedReason disconnectedReason, IN const bool ignoreDisconnectedReason);

      Result getProtocolBlockStatusTemplateMap(OUT ProtocolBlockStatusMap& protocolBlockStatusMap, IN const BdAddress& bdAddress, IN BdName bdName = "");

      // the data base handling object
      Database *_db;

      /* data base manager list management */
      bool _databaseAccessSwitchedOn;
      BtLimitationDeviceTypeProtocolBlockStatusMap _newDeviceProtocolBlockStatusTemplate;
      ProtocolConnectionSMHandleMap _protocolconnectionsmhandlemapp;
      ProtocolConnectionControllerInfoList _protocollist;

      LockForever _lockAssignProtocol;
   };
}

#endif // _DB_MANAGER_H_
