/**
 * @defgroup DataProviderModule Data Provider
 * @ingroup BmCoreModule
 *
 * @brief This module comprises all parts related to data provider
 */

/**
 * @file
 * @ingroup DataProviderModule
 *
 * @brief This file contains the declaration of the class DataProvider
 *
 * @author Stefan Reis
 */

#ifndef _DATA_PROVIDER_H_
#define _DATA_PROVIDER_H_

#include <vector>
#include <string>
#include "BmAllTypes.h"
#include "DbManager.h"
#include "ILocalSpm.h"
#include "FastUTF8.h"
#include "BmCoreConfiguration.h"



namespace bmcore
{
   /**
    * @class DataProvider
    * @ingroup DataProviderModule
    *
    * @brief The class DataProvider can be used by components which must retrieve single data items like configuration values
    *
    */
   class DataProvider : public ILocalSpm
   {

   public:
      /**
       * Constructor of component.
       * This function is used by LocalSpm to store the DataProvider componentID in mComponentID.
       *
       * @param[in] componentID component ID assigned by SPM
       * @return void
       */
      DataProvider(const ComponentId componentID);

      /**
       * Destructor of component.
       *
       * @return void
       */
      virtual ~DataProvider();

      void updateConfiguration(void);

      inline const BmCoreConfiguration& getBmCoreConfiguration(void) const
      {
         return _bmCoreConfiguration;
      }

      Result getBtLocalInfo(OUT LocalInfo& btLocalInfo);

      Result getProtocolInfos(OUT ProtocolInfoMap& protocolInfoMap, IN const bool includeOnlySupportedProtocols,
            IN const bool includeVersionInfo = true, IN const bool includeSupportedInfo = true);

      Result getMaxNumProtocolInstances(OUT uint8_t& maxNumInstances, IN const ProtocolId protocolId);

      Result getMasterProtocolIds(OUT ProtocolIdList& masterProtocolIds, IN const ProtocolId protocolId);

      Result getSlaveProtocolIds(OUT ProtocolIdList& slaveProtocolIds);

      Result isSlaveProtocolId(OUT bool& isSlaveProtocolId, IN const ProtocolId protocolId);

      Result getDeviceConnectionProtocols(OUT ProtocolIdList& deviceConnectionProtocols);

      Result getHmiUserSelectableProtocols(OUT ProtocolList& hmiUserSelectableProtocols);

      Result getVehicleIdentification(OUT DeviceIdentification& vehicleIdentification);
      /**********************************************************************************************
       * The following definitions are for the on-the-fly definitions for single
       * and simple configuration values by all developers
       */

      /**
       * logs config value to the ETG trace by using name and value
       *
       * @param[in] dbName Name of the value to be used
       * @param[in] data   pointer to the slot where data is available
       *
       * @return < 0: error, = 0: OK
       */
      static void traceConfigValue(const bool data, const char *dbName);
      static void traceConfigValue(const int data, const char *dbName);
      static void traceConfigValue(const long data, const char *dbName);
      static void traceConfigValue(const float data, const char *dbName);
      static void traceConfigValue(std::string &data, const char *dbName);
      static void traceConfigValue(const uint8_t data, const char *dbName);
      static void traceConfigValue(const uint32_t data, const char *dbName);

      class ConfigItemBase
      {
      public:
         virtual void trace()=0;
      };

      static std::vector <ConfigItemBase*> configurationItems; // pointer to list of configuration items

      // template for all simple config values in the data base
      template<typename T, typename dbName> class DatabaseItem : public ConfigItemBase
      {
      public:
         T _value;
         const char *_dbName;
         int _loaded;

         DatabaseItem()
         {
            dbName name;            // needed to get access to name and default value of item
            _loaded = 0;            // value not yet loaded from db
            _dbName = name.get();      // set the name in db for this value
            _value = name.defaultValue; // set the default value
            //traceConfigValue(_value, _dbName); // trace configuration value resolving its type

            configurationItems.push_back(this); // add item into master list of configurations
         }

         virtual T operator ()()
         {
            return _value;
         }

         virtual T operator = (T Value)
         {
            _value = Value;
            //traceConfigValue(_value, _dbName);   // trace configuration value resolving its type
            return _value;
         }

         void trace()
         {
            traceConfigValue(_value, _dbName); // trace configuration value resolving its type
         }
      };

      // macro to define every data base config item in one line
   #define CREATE_DB_VALUE(name,type,def) \
         class name {\
         public: type defaultValue; \
         name() {defaultValue = def;} \
         const char *get() {return ""#name"";}\
   }; \
   DatabaseItem<type,name> name;

      /**
       * definition of all config values which are loaded out of the
       * database without any additional handling
       * (incl. default value if db has no apropriate value)
       */

      /**
       * This is only a demo creation of two different values:
       * if one wants to create additional values, the signature of this macro is:
       *
       * CREATE_DB_VALUE(dBValueName, realType, defaultValue)
       */
      CREATE_DB_VALUE(IntValue, int, 0); // only for demo // finished: 100%
      CREATE_DB_VALUE(FloatValue, float, 0); // only for demo // finished: 100%

      /**
       * definition of higher aggregated functions
       */
      float IntTimesFloat()   //((CustomControlSM &)LocalSpm::GetCustomControlRef()).ExpectSetString("on");
      {
         // accesses own data base values by local member value definition
         return (float)(IntValue()) * FloatValue();
      }

      /**
       * example: returns the flag if customer wants autoplay
       *
       * this is the way how a data provider value my defined by the developer:
       *
       * <B>AutoPlayInPause</B> : name of database item, will be used as method name too
       *
       * <B>int</B>     : base type of database value for the item AutoPlayInPause
       *
       * <B>0</B>    : default value for item if no database value was given
       *
       * usage in user code:
      \code{.cpp}
       DataProvider dp;
       int autoPlay = dp.AutoPlayInPause()
      \endcode
       */

      CREATE_DB_VALUE(SPMStartupTimeout, long, 40000 /**< default value */);
      CREATE_DB_VALUE(SPMShutdownTimeout, long, 40000 /**< default value */);
      CREATE_DB_VALUE(SPMMaxShutdownDelay, long, 5000 /**< default value */);
      CREATE_DB_VALUE(DBUnifiedSchema, int, 1 /**< default value */);
      CREATE_DB_VALUE(DBSortUseUCAOn, int, 0 /**< default value */);
      CREATE_DB_VALUE(DBSortLanguage, std::string, "en_US" /**< default value */);     /*Ref:http://en.wikipedia.org/wiki/List_of_ISO_639-1_codes#Partial_ISO_639_table*/
      CREATE_DB_VALUE(DBUpdateProgressInstructions, int, 5000 /**< default value */);

     

      /**
       * This function is used by LocalSpm to cleanup the DataProvider.
       * Free and delete old album art object -> SetAlbumArtObjectToCache(ptrToAlbumArtObject=NULL)
       * Inform LocalSpm that Done is ready -> DoneDone(0)
       *
       * @return < 0: error, = 0: OK
       */
      Result done();

      /**
       * Brief: returns statistic data as a string for the over all component object statistics
       *
       * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
       */
      int statistics(OUT Statistics stat);

   private:

      void traceBmConfiguration(void) const;

      /**
       * Read BM core configuration from file.
       *
       * @param[out] configuration: BM core configuration
       * @param[in] configurationFile: configuration filename
       *
       * @return true: configuration data was successfully read from file,
       * @return false: error occurred
       */
      bool readConfiguration(OUT BmCoreConfiguration& configuration, IN const ::std::string& configurationFile) const;

      BmCoreConfiguration _bmCoreConfiguration;
   };
}

#endif // _DATA_PROVIDER_H_
