/**
 * @file BmUtils.cpp
 *
 * @par SW-Component
 * Bluetooth Connection Manager Core
 *
 * @brief This file contains the definition of utility functions used by Bluetooth Connection Manager Core
 *
 * @copyright (c) 2016 Robert Bosch GmbH
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BmCoreModule
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_UTILS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BmUtils.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_UTILS
#endif
#endif

#include "FunctionTracer.h"
#include "Utils.h"
#include "BmUtils.h"


namespace bmcore
{
   static const BdAddress bdAddressExample = "aabbccddeeff";
   static const Uuid uuidExample = "0000110100001000800000805F9B34FB"; // 128-bit UUID of SPP: 00001101-0000-1000-8000-00805F9B34FB
   static const LegacyPinCode legacyPinCodeExample = "0123456789012345";
   static const SspPinCode sspPinCodeExample = "012345";

   bool isValidBluetoothSystemState(IN const BluetoothSystemState bluetoothSystemState)
   {
      ENTRY

      if ((BM_BT_SYSTEM_STATE_OFF == bluetoothSystemState)
            || (BM_BT_SYSTEM_STATE_BLOCK == bluetoothSystemState)
            || (BM_BT_SYSTEM_STATE_DISCONNECT == bluetoothSystemState)
            || (BM_BT_SYSTEM_STATE_NORMAL_SWITCH_OFF_BT == bluetoothSystemState)
            || (BM_BT_SYSTEM_STATE_NO_ACTION == bluetoothSystemState)
            || (BM_BT_SYSTEM_STATE_NORMAL == bluetoothSystemState))
      {
         return true;
      }

      return false;
   }

   bool isValidLimitationMode(IN const LimitationMode& limitationMode)
   {
      ENTRY

      if (((BM_LIMITATION_FEATURE_CAR_PLAY == limitationMode._limitationFeature)
               && ((BM_LIMITATION_COMMUNICATION_IF_USB == limitationMode._limitationCommunicationIf)
                  || (BM_LIMITATION_COMMUNICATION_IF_WIFI   == limitationMode._limitationCommunicationIf)))
            || ((BM_LIMITATION_FEATURE_ANDROID_AUTO == limitationMode._limitationFeature)
                  && ((BM_LIMITATION_COMMUNICATION_IF_USB == limitationMode._limitationCommunicationIf)
                  || (BM_LIMITATION_COMMUNICATION_IF_WIFI   == limitationMode._limitationCommunicationIf)))
            || ((BM_LIMITATION_FEATURE_CAR_LIFE == limitationMode._limitationFeature)
                  && (BM_LIMITATION_COMMUNICATION_IF_USB == limitationMode._limitationCommunicationIf))
            || ((BM_LIMITATION_FEATURE_ON_CAR == limitationMode._limitationFeature)
                  &&(BM_LIMITATION_COMMUNICATION_IF_USB == limitationMode._limitationCommunicationIf))
            || ((BM_LIMITATION_FEATURE_MIRROR_LINK == limitationMode._limitationFeature)
                  &&(BM_LIMITATION_COMMUNICATION_IF_USB == limitationMode._limitationCommunicationIf))
            || ((BM_LIMITATION_FEATURE_MY_SPIN_IOS == limitationMode._limitationFeature)
                  &&(BM_LIMITATION_COMMUNICATION_IF_USB == limitationMode._limitationCommunicationIf))
            || ((BM_LIMITATION_FEATURE_CAR_LIFE_IOS == limitationMode._limitationFeature)
                  &&(BM_LIMITATION_COMMUNICATION_IF_USB == limitationMode._limitationCommunicationIf)))
      {
         return true;
      }

      return false;
   }

   bool isValidLimitationAction(IN const LimitationAction limitationAction)
   {
      ENTRY

      return (BM_LIMITATION_ACTION_LAST > limitationAction)?(true):(false);
   }

   bool isValidTargetSwitchState(IN const TargetSwitchState targetSwitchState)
   {
      ENTRY

      if ((TARGET_SWITCH_STATE_SWITCHED_OFF == targetSwitchState)
            || (TARGET_SWITCH_STATE_SWITCHED_ON == targetSwitchState))
      {
         return true;
      }

      return false;
   }

   bool isValidBdAddress(IN const BdAddress& bdAddress, IN const bool allowEmpty /*= false*/)
   {
      ENTRY

      if ((true == allowEmpty) && (0 == bdAddress.size()))
      {
         return true;
      }

      if (bdAddressExample.size() != bdAddress.size())
      {
         return false;
      }

      for (size_t i = 0; i < bdAddress.size(); i++)
      {
         if (('0' <= bdAddress[i]) && (bdAddress[i] <= '9'))
         {
            // numeric character
         }
         else if (('a' <= bdAddress[i]) && (bdAddress[i] <= 'f'))
         {
            // alphabetic character
         }
         else if (('A' <= bdAddress[i]) && (bdAddress[i] <= 'F'))
         {
            // alphabetic character
         }
         else
         {
            // invalid character
            return false;
         }
      }

      return true;
   }

   bool isValidUuid(IN const Uuid& uuid, IN const bool allowEmpty /*= false*/)
   {
      ENTRY

      if ((true == allowEmpty) && (0 == uuid.size()))
      {
         return true;
      }

      if (uuidExample.size() != uuid.size())
      {
         return false;
      }

      for (size_t i = 0; i < uuid.size(); i++)
      {
         if (('0' <= uuid[i]) && (uuid[i] <= '9'))
         {
            // numeric character
         }
         else if (('a' <= uuid[i]) && (uuid[i] <= 'f'))
         {
            // alphabetic character
         }
         else if (('A' <= uuid[i]) && (uuid[i] <= 'F'))
         {
            // alphabetic character
         }
         else
         {
            // invalid character
            return false;
         }
      }

      return true;
   }

   bool isValidLocalBdName(IN const LocalFriendlyNameType& localBdName)
   {
      ENTRY

      if (0 == localBdName.size())
      {
         return false;
      }

      return true;
   }

   bool isValidLegacyPairingPin(IN const PairingPinType& legacyPairingPin, IN const bool allowEmpty /*= false*/)
   {
      ENTRY

      if ((false == allowEmpty) && (0u == legacyPairingPin.size()))
      {
         return false;
      }

      if (legacyPinCodeExample.size() < legacyPairingPin.size())
      {
         return false;
      }

      return true;
   }

   bool isValidSspPin(IN const SspPinCode& sspPin, IN const bool allowEmpty /*= false*/)
   {
      ENTRY

      if ((false == allowEmpty) && (0u == sspPin.size()))
      {
         return false;
      }

      if (sspPinCodeExample.size() < sspPin.size())
      {
         return false;
      }

      return true;
   }

   bool isValidRequestType(IN const RequestType& requestType)
   {
      if (BM_REQUEST_TYPE_CONNECTION == requestType)
      {
         return true;
      }

      return false;
   }

   bool isValidDeviceHandleType(IN const DeviceHandleType& deviceHandleType, IN const bool isBlockDeviceRequest)
   {
      if (true == isBlockDeviceRequest)
      {
         if ((BM_DEVICE_HANDLE_TYPE_SINGLE == deviceHandleType)
               || (BM_DEVICE_HANDLE_TYPE_ALL == deviceHandleType)
               || (BM_DEVICE_HANDLE_TYPE_ALL_EXCEPT == deviceHandleType))
         {
            return true;
         }
      }
      else
      {
         if ((BM_DEVICE_HANDLE_TYPE_SINGLE == deviceHandleType)
               || (BM_DEVICE_HANDLE_TYPE_ALL == deviceHandleType))
         {
            return true;
         }
      }

      return false;
   }

   bool isValidStandardAutoConnectionType(IN const StandardAutoConnectionType standardAutoConnectionType)
   {
      if ((BM_STANDARD_AUTO_CONNECTION_OFF == standardAutoConnectionType)
            || (BM_STANDARD_AUTO_CONNECTION_LAST_CONNECTED_DEVICES == standardAutoConnectionType))
      {
         return true;
      }

      return false;
   }

   bool isValidTemporaryAutoConnectionType(IN const TemporaryAutoConnectionType temporaryAutoConnectionType)
   {
      if ((BM_TEMPORARY_AUTO_CONNECTION_DEFAULT == temporaryAutoConnectionType)
            || (BM_TEMPORARY_AUTO_CONNECTION_LAST_CONNECTED_DEVICES == temporaryAutoConnectionType))
      {
         return true;
      }

      return false;
   }

   bool isValidAutoConnectionStartMode(IN const AutoConnectionStartMode autoConnectionStartMode)
   {
      if ((BM_AUTOCONNECTION_START_MODE_TRY_ONCE_ONLY_IF_BOOKED_BEFORE == autoConnectionStartMode)
            || (BM_AUTOCONNECTION_START_MODE_TRY_ONCE == autoConnectionStartMode)
            || (BM_AUTOCONNECTION_START_MODE_BOOK_AND_TRY_ONCE == autoConnectionStartMode))
      {
         return true;
      }

      return false;
   }

   bool isValidProtocolId(IN const ProtocolId protocolId)
   {
      if ((BM_PROTOCOL_ID_HFP == protocolId)
            || (BM_PROTOCOL_ID_AVP == protocolId)
            || (BM_PROTOCOL_ID_PBDL == protocolId)
            || (BM_PROTOCOL_ID_MSG == protocolId)
            || (BM_PROTOCOL_ID_PAN == protocolId)
            || (BM_PROTOCOL_ID_SPP == protocolId))
      {
         return true;
      }

      return false;
   }

   bool isValidOobType(IN const OobType oobType)
   {
      if ((BM_OOB_TYPE_GENERIC == oobType)
            || (BM_OOB_TYPE_APPLE == oobType))
      {
         return true;
      }

      return false;
   }

   bool isValidFavoriteType(IN const FavoriteType favoriteType)
   {
      if ((BM_FAVORITE_TYPE_DEVICE == favoriteType)
            || (BM_FAVORITE_TYPE_PROTOCOL_HFP == favoriteType)
            || (BM_FAVORITE_TYPE_PROTOCOL_AVP == favoriteType))
      {
         return true;
      }

      return false;
   }

   bool isValidUsagePreference(IN const UsagePreference usagePreference)
   {
      if ((BM_UP_CLASSIC_BT == usagePreference)
            || (BM_UP_CPW == usagePreference)
            || (BM_UP_TEMP_CLASSIC_BT == usagePreference))
      {
         return true;
      }

      return false;
   }

   void convertBdAddress2LowerCase(INOUT BdAddress& bdAddress)
   {
      // TODO: framework contains such a function
      for(size_t i = 0; i < bdAddress.size(); i++)
      {
         if(('A' <= bdAddress[i]) && (bdAddress[i] <= 'Z'))
         {
            bdAddress[i] = (char)(bdAddress[i] + 32);
         }
      }
   }

   Result getBmSwitchStateFromBtsBluetoothMode(OUT SwitchState& switchState, IN const BTSBluetoothMode btsBluetoothMode)
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      switch (btsBluetoothMode)
      {
         case BTS_BT_MODE_OFF:
            switchState = SWITCH_STATE_SWITCHED_OFF;
            break;
         case BTS_BT_MODE_ON:
            switchState = SWITCH_STATE_SWITCHED_ON;
            break;
         default:
            result = CC_ERR_INT_GENERAL_ERROR; //No match was found
            break;
      }

      return result;
   }

   Result getBmSwitchStateFromBtsLocalMode(OUT SwitchState& switchState, IN const BTSLocalMode btsLocalMode)
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      switch (btsLocalMode)
      {
         case BTS_MODE_DISABLED:
            switchState = SWITCH_STATE_SWITCHED_OFF;
            break;
         case BTS_MODE_ENABLED:
            switchState = SWITCH_STATE_SWITCHED_ON;
            break;
         default:
            result = CC_ERR_INT_GENERAL_ERROR; //No match was found
            break;
      }

      return result;
   }

   Result getBmSwitchedOffReasonFromBtsBluetoothOffReason(OUT SwitchedOffReason& switchedOffReason,
         IN const BTSBluetoothOffReason btsBluetoothOffReason)
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      switch (btsBluetoothOffReason)
      {
         case BTS_BT_OFF_REASON_NOT_VALID:
            switchedOffReason = SWITCHED_OFF_REASON_NOT_VALID;
            break;
         case BTS_BT_OFF_REASON_APP_REQUESTED:
            switchedOffReason = SWITCHED_OFF_REASON_CLIENT;
            break;
         case BTS_BT_OFF_REASON_STARTUP:
            switchedOffReason = SWITCHED_OFF_REASON_STARTUP;
            break;
         case BTS_BT_OFF_REASON_STACK_NOT_RUNNING:
            switchedOffReason = SWITCHED_OFF_REASON_STACK_NOT_RUNNING;
            break;
         case BTS_BT_OFF_REASON_DBUS_ERROR:
            switchedOffReason = SWITCHED_OFF_REASON_ERROR;
            break;
         default:
            result = CC_ERR_INT_GENERAL_ERROR; //No match was found
            break;
      }

      return result;
   }

   Result getBmSwitchedOffReasonFromBtsLocalModeOffReason(OUT SwitchedOffReason& switchedOffReason, IN const BTSLocalModeOffReason btsOffReason)
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      switch (btsOffReason)
      {
         case BTS_MODE_OFF_REASON_NOT_VALID:
            switchedOffReason = SWITCHED_OFF_REASON_NOT_VALID;
            break;
         case BTS_MODE_OFF_REASON_APP_REQUESTED:
            switchedOffReason = SWITCHED_OFF_REASON_CLIENT;
            break;
         case BTS_MODE_OFF_REASON_TIMEOUT:
            switchedOffReason = SWITCHED_OFF_REASON_TIMEOUT;
            break;
         case BTS_MODE_OFF_REASON_INTERNAL:
            switchedOffReason = SWITCHED_OFF_REASON_INTERNAL;
            break;
         default:
            result = CC_ERR_INT_GENERAL_ERROR; //No match was found
            break;
      }

      return result;
   }

   Result getBmSwitchStateFromBtsDiscoveringStatus(OUT SwitchState& switchState,
         IN const BTSDiscoveringStatus btsDiscoveringStatus)
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      switch (btsDiscoveringStatus)
      {
         case BTS_DISCOVERING_OFF:
            switchState = SWITCH_STATE_SWITCHED_OFF;
            break;
         case BTS_DISCOVERING_ON:
            switchState = SWITCH_STATE_SWITCHED_ON;
            break;
         default:
            result = CC_ERR_INT_GENERAL_ERROR; // no match was found
            break;
      }

      return result;
   }

   Result getBtsSspModeFromBmPairingType(OUT BTSSspMode& btsSspMode, IN const PairingType pairingType)
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      switch(pairingType)
      {
      case BM_PAIRING_TYPE_SSP_NUMERIC_COMPARISON:
         btsSspMode = BTS_SSP_WITH_NUMERIC_COMPARISON;
         break;
      case BM_PAIRING_TYPE_SSP_JUST_WORK:
         btsSspMode = BTS_SSP_WITH_JUST_WORK_MODE;
         break;
      case BM_PAIRING_TYPE_SSP_PASSKEY_ENTRY_MODE_INPUT_BY_REMOTE_DEVICE:
         btsSspMode = BTS_SSP_WITH_PASSKEY_ENTRY_MODE_INPUT_BY_REMOTE_DEVICE;
         break;
      case BM_PAIRING_TYPE_SSP_PASSKEY_ENTRY_MODE_INPUT_BY_LOCAL_DEVICE:
         btsSspMode = BTS_SSP_WITH_PASSKEY_ENTRY_MODE_INPUT_BY_LOCAL_DEVICE;
         break;
      default:
         result = CC_ERR_INT_GENERAL_ERROR; //No match was found
         break;
      }

      return result;
   }

   Result getBmPairingTypeFromBtsSspMode(OUT PairingType& pairingType, IN const BTSSspMode btsSspMode)
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      switch(btsSspMode)
      {
      case BTS_SSP_WITH_NUMERIC_COMPARISON:
         pairingType = BM_PAIRING_TYPE_SSP_NUMERIC_COMPARISON;
         break;
      case BTS_SSP_WITH_JUST_WORK_MODE:
         pairingType = BM_PAIRING_TYPE_SSP_JUST_WORK;
         break;
      case BTS_SSP_WITH_PASSKEY_ENTRY_MODE_INPUT_BY_REMOTE_DEVICE:
         pairingType = BM_PAIRING_TYPE_SSP_PASSKEY_ENTRY_MODE_INPUT_BY_REMOTE_DEVICE;
         break;
      case BTS_SSP_WITH_PASSKEY_ENTRY_MODE_INPUT_BY_LOCAL_DEVICE:
         pairingType = BM_PAIRING_TYPE_SSP_PASSKEY_ENTRY_MODE_INPUT_BY_LOCAL_DEVICE;
         break;
      default:
         pairingType = BM_PAIRING_TYPE_UNKNOWN;
         result = CC_ERR_INT_GENERAL_ERROR; //No match was found
         break;
      }

      return result;
   }

   Result getBtsSupportedServicesBitFromBmProtocolId(OUT BTSSupportedServicesBit& btsSupportedServicesBit, IN const ProtocolId protocolId)
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      switch(protocolId)
      {
         case BM_PROTOCOL_ID_HFP:
            btsSupportedServicesBit = BTS_SUPP_SRV_HFP;
            break;
         case BM_PROTOCOL_ID_AVP:
            btsSupportedServicesBit = BTS_SUPP_SRV_A2DP;
            break;
         case BM_PROTOCOL_ID_PBDL:
            btsSupportedServicesBit = BTS_SUPP_SRV_PBAP;
            break;
         case BM_PROTOCOL_ID_MSG:
            btsSupportedServicesBit = BTS_SUPP_SRV_MAP;
            break;
         case BM_PROTOCOL_ID_PAN:
            btsSupportedServicesBit = BTS_SUPP_SRV_PAN;
            break;
         case BM_PROTOCOL_ID_SPP:
            btsSupportedServicesBit = BTS_SUPP_SRV_SPP;
            break;
         default:
            result = CC_ERR_INT_GENERAL_ERROR; //No match was found
            break;
      }

      return result;
   }

   Result getBmProtocolIdFromBtsSupportedServicesBit(OUT ProtocolId& protocolId, IN const BTSSupportedServicesBit btsSupportedServicesBit)
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      switch(btsSupportedServicesBit)
      {
         case BTS_SUPP_SRV_HFP:
            protocolId = BM_PROTOCOL_ID_HFP;
            break;
         case BTS_SUPP_SRV_A2DP:
            protocolId = BM_PROTOCOL_ID_AVP;
            break;
         case BTS_SUPP_SRV_PBAP:
            protocolId = BM_PROTOCOL_ID_PBDL;
            break;
         case BTS_SUPP_SRV_MAP:
            protocolId = BM_PROTOCOL_ID_MSG;
            break;
         case BTS_SUPP_SRV_PAN:
            protocolId = BM_PROTOCOL_ID_PAN;
            break;
         case BTS_SUPP_SRV_SPP:
            protocolId = BM_PROTOCOL_ID_SPP;
            break;
         case BTS_SUPP_SRV_AVRCP:
         case BTS_SUPP_SRV_DUN:
         case BTS_SUPP_SRV_PNP:
         case BTS_SUPP_SRV_DID:
            protocolId = BM_PROTOCOL_ID_UNKNOWN;
            break;
         default:
            protocolId = BM_PROTOCOL_ID_UNKNOWN;
            result = CC_ERR_INT_GENERAL_ERROR; //No match was found
            break;
      }

      return result;
   }

   Result getBtsProtocolIdFromBmProtocolId(OUT BTSProtocolId& btsProtocolId, IN const ProtocolId protocolId)
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      switch(protocolId)
      {
         case BM_PROTOCOL_ID_HFP:
            btsProtocolId = BTS_PROTO_HFP;
            break;
         case BM_PROTOCOL_ID_AVP:
            btsProtocolId = BTS_PROTO_AVP;
            break;
         case BM_PROTOCOL_ID_PBDL:
            btsProtocolId = BTS_PROTO_PIM;
            break;
         case BM_PROTOCOL_ID_MSG:
            btsProtocolId = BTS_PROTO_MSG;
            break;
         case BM_PROTOCOL_ID_PAN:
            btsProtocolId = BTS_PROTO_PAN;
            break;
         case BM_PROTOCOL_ID_SPP:
            btsProtocolId = BTS_PROTO_SPP;
            break;
         default:
            result = CC_ERR_INT_GENERAL_ERROR; //No match was found
            break;
      }

      return result;
   }

   Result getBmProtocolIdFromBtsProtocolId(OUT ProtocolId& protocolId, IN const BTSProtocolId btsProtocolId)
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      switch(btsProtocolId)
      {
         case BTS_PROTO_HFP:
            protocolId = BM_PROTOCOL_ID_HFP;
            break;
         case BTS_PROTO_AVP:
            protocolId = BM_PROTOCOL_ID_AVP;
            break;
         case BTS_PROTO_PIM:
            protocolId = BM_PROTOCOL_ID_PBDL;
            break;
         case BTS_PROTO_MSG:
            protocolId = BM_PROTOCOL_ID_MSG;
            break;
         case BTS_PROTO_PAN:
            protocolId = BM_PROTOCOL_ID_PAN;
            break;
         case BTS_PROTO_SPP:
            protocolId = BM_PROTOCOL_ID_SPP;
            break;
         default:
            result = CC_ERR_INT_GENERAL_ERROR; //No match was found
            break;
      }

      return result;
   }

   Result getBmDisconnectedReasonFromBtsDisconnectReason(OUT DisconnectedReason& disconnectedReason, IN const BTSDisconnectReason btsDisconnectReason)
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      switch (btsDisconnectReason)
      {
         case BTS_DISCONNECT_REASON_NOT_VALID:
            disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;
            break;
         case BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL_OR_REMOTE:
            disconnectedReason = BM_DISCONNECTED_REASON_NORMAL_LOSS;
            break;
         case BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL:
            disconnectedReason = BM_DISCONNECTED_REASON_NORMAL_LOSS_LOCAL;
            break;
         case BTS_DISCONNECT_REASON_NORMAL_LOSS_REMOTE:
            disconnectedReason = BM_DISCONNECTED_REASON_NORMAL_LOSS_REMOTE;
            break;
         case BTS_DISCONNECT_REASON_ABNORMAL_LOSS:
            disconnectedReason = BM_DISCONNECTED_REASON_ABNORMAL_LOSS;
            break;
         case BTS_DISCONNECT_REASON_ACL_CONNECT_FAILED:
         case BTS_DISCONNECT_REASON_PROTO_CONNECT_FAILED:
         case BTS_DISCONNECT_REASON_PROTO_CONNECT_ABORTED:
            disconnectedReason = BM_DISCONNECTED_REASON_CONNECTION_FAILED;
            break;
         case BTS_DISCONNECT_REASON_PROTO_CONNECT_TIMEOUT:
            disconnectedReason = BM_DISCONNECTED_REASON_CONNECTION_FAILED_TIMEOUT;
            break;
         case BTS_DISCONNECT_REASON_PROTO_CONNECT_REJECTED:
            disconnectedReason = BM_DISCONNECTED_REASON_CONNECTION_FAILED_REJECTED;
            break;
         case BTS_DISCONNECT_REASON_PROTO_NOT_SUPPORTED:
            disconnectedReason = BM_DISCONNECTED_REASON_CONNECTION_FAILED_NOT_SUPPORTED;
            break;
         case BTS_DISCONNECT_REASON_SPP_UUID_NOT_FOUND:
            disconnectedReason = BM_DISCONNECTED_REASON_CONNECTION_FAILED_UUID_NOT_SUPPORTED;
            break;
         case BTS_DISCONNECT_REASON_SPP_LIMIT_REACHED:
            disconnectedReason = BM_DISCONNECTED_REASON_CONNECTION_FAILED_LIMIT_REACHED;
            break;
         case BTS_DISCONNECT_REASON_MISSING_LINK_KEY:
            disconnectedReason = BM_DISCONNECTED_REASON_MISSING_LINK_KEY;
            break;
         case BTS_DISCONNECT_REASON_OUT_OF_RANGE:
            disconnectedReason = BM_DISCONNECTED_REASON_OUT_OF_RANGE;
            break;
         default:
            disconnectedReason = BM_DISCONNECTED_REASON_NORMAL_LOSS;
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }

      return result;
   }
}
