/**
 * @file BmAllTypesInternal.h
 *
 * @par SW-Component
 * Bluetooth Connection Manager Core
 *
 * @brief This file contains the declaration of all types used internally by Bluetooth Connection Manager Core
 *
 * @copyright (c) 2016 Robert Bosch GmbH
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BmCoreModule
 */

#ifndef _BM_ALL_TYPES_INTERNAL_H_
#define _BM_ALL_TYPES_INTERNAL_H_

#include "BaseTypes.h"
#include "BmInterfaceTypesInternal.h"
#include <bitset>

namespace bmcore
{
   class DeviceConnectionController;
   class ProtocolConnectionController;

   /**
    * Size of Bluetooth device address string including NULL termination
    * (hexadecimal characters without colons)
    */
   #define BM_BDADDRESS_STRING_SIZE 13

   #define BM_DEVICE_NAME_STRING_SIZE 128

   /**
    * Size of link key string including NULL termination
    * (hexadecimal characters)
    */
   #define BM_LINKKEY_STRING_SIZE 33

   #define BM_DLINKKEY_STRING_SIZE 65

   /**
    * Size of legacy pin code string including NULL termination
    * (including NULL termination)
    */
   #define BM_LEGACY_PIN_CODE_STRING_SIZE 17

   /**
    * Size of secure simple pairing pin code string including NULL termination
    */
   #define BM_SSP_PIN_CODE_STRING_SIZE 7

   /**
    * Size of a Bluetooth service UUID string including NULL termination
    * (e.g. "00000000-0000-1000-8000-00805F9B34FB")
    */
   #define BM_UUID_STRING_SIZE 37

   #define BM_RFCOMM_DEVICE_PATH_STRING_SIZE 256

   /**
    * Size of the Database Column Name string
    */
   #define DB_COLUMN_NAME_SIZE 25

   #define DEVICE_ID_NOT_SET ((DeviceId)-1)

   #define DEVICE_HANDLE_NOT_SET 5000

   #define DEVICE_HANDLE_TMP_VAL 6000

   #define IAP2BT_SPP_UUID "00000000decafadedecadeafdecacafe"

   #define AAW_SPP_UUID "4de17a0052cb11e6bdf40800200c9a66"

   // 6 sec
   #define WAIT_FOR_DID_SPP_INFO_SEC 8  //6

   #define WAIT_FOR_WBL_SERVICE_AVAILABLE_INFO_SEC 20

   #define WAIT_FOR_WBL_LAST_INTENDED_MODE_SEC 40

   #define DELAY_BT_ON_REQ_IN_SEC 5

   /**
    * Component identifiers for all internal components
    */
   typedef enum
   {
      COMPONENT_ID_UNKNOWN = 0UL,
      COMPONENT_ID_THREAD_FACTORY = 1UL,
      COMPONENT_ID_DISPATCHER = 2UL,
      COMPONENT_ID_DB_MANAGER = 3UL,
      COMPONENT_ID_DATA_PROVIDER = 4UL,
      COMPONENT_ID_AUTOCONNECTION_CONTROLLER = 5UL,
      COMPONENT_ID_RESET_TO_DEFAULT_CONTROLLER = 6UL,
      COMPONENT_ID_PAIRING_CONTROLLER = 7UL,
      COMPONENT_ID_BT_LIMITATION_CONTROLLER = 8UL,
      COMPONENT_ID_BM_CORE_MAIN_CONTROLLER = 10UL,
      COMPONENT_ID_BM_CONTROLLER = 11UL,
      COMPONENT_ID_OUTPUTWRAPPER = 12UL,
      COMPONENT_ID_BM_CORE_CALLBACK_IF_WRAPPER = 13UL,
//      COMPONENT_ID_ALL_DEVICES_DISCONNECTION = 14UL,
      COMPONENT_ID_CONFIGURATION = 15UL,
      COMPONENT_ID_CUSTOM_CONTROL = 16UL,
	  COMPONENT_ID_CONFLICTMANAGER = 17UL
   } ComponentId;

   #define COMPONENTID_FORMAT "i"
   #define COMPONENTID_INIT COMPONENT_ID_UNKNOWN

   #define STANDARDAUTOCONNECTIONTYPE_FORMAT "i"
   #define STANDARDAUTOCONNECTIONTYPE_INIT BM_STANDARD_AUTO_CONNECTION_UNKNOWN
   /**
    * Bluetooth device address string
    * (hexadecimal characters without colons)
    */
   typedef char BdAddressChrArr[BM_BDADDRESS_STRING_SIZE];
   #define BDADDRESSCHRARR_FORMAT "t"
   #define BDADDRESSCHRARR_INIT ""

   typedef char DeviceNameChrArr[BM_DEVICE_NAME_STRING_SIZE];
   #define DEVICENAMECHRARR_FORMAT "t"
   #define DEVICENAMECHRARR_INIT ""

   /**
    * Legacy pin code string
    */
   typedef std::string LegacyPinCode;

   typedef char LegacyPinCodeChrArr[BM_LEGACY_PIN_CODE_STRING_SIZE];
   #define LEGACYPINCODECHRARR_FORMAT "t"
   #define LEGACYPINCODECHRARR_INIT ""

   typedef std::string SspPinCode;

   /**
    * Secure simple pairing pin code string
    */
   typedef char SspPinCodeChrArr[BM_SSP_PIN_CODE_STRING_SIZE];
   #define SSPPINCODECHRARR_FORMAT "t"
   #define SSPPINCODECHRARR_INIT ""

   /**
    * Link key string
    * (hexadecimal characters)
    */
   typedef char LinkKeyChrArr[BM_LINKKEY_STRING_SIZE];
   #define LINKKEYCHRARR_FORMAT "t"
   #define LINKKEYCHRARR_INIT ""

   typedef char DLinkKeyChrArr[BM_DLINKKEY_STRING_SIZE];
   #define DLINKKEYCHRARR_FORMAT "t"
   #define DLINKKEYCHRARR_INIT ""

   /**
    * Bluetooth service UUID string
    * (e.g. "00000000-0000-1000-8000-00805F9B34FB")
    */
   typedef char UuidChrArr[BM_UUID_STRING_SIZE];
   #define UUIDCHRARR_FORMAT "t"
   #define UUIDCHRARR_INIT ""

   #define SPPMAGICUUID "0xDEADC0DE"

   typedef char RfcommDevicePathChrArr[BM_RFCOMM_DEVICE_PATH_STRING_SIZE];
   #define RFCOMMDEVICEPATHCHRARR_FORMAT "t"
   #define RFCOMMDEVICEPATHCHRARR_INIT ""

   /**
    * Device ID representing a BT remote device within Bluetooth connection manager
    */
   //typedef unsigned int DeviceId;
   #define DEVICEID_FORMAT "i"
   #define DEVICEID_INIT 0
   //#define tDeviceID tDeviceId /* this is for the old mediaplayer interfaces like Query */

   typedef std::vector<DeviceId> DeviceIdList;

   typedef std::vector<BdAddress> BdAddressList;
   typedef std::map<int,const ProtocolConnectionController*> ProtocolConnectionSMHandleMap;
   typedef enum
   {
      BM_CONNECTION_REQUEST_ORIGIN_UNKNOWN = 0,
      BM_CONNECTION_REQUEST_ORIGIN_LOCAL,
      BM_CONNECTION_REQUEST_ORIGIN_REMOTE
   } ConnectionRequestOrigin;

   typedef enum
   {
      BM_CORE_IF_MSG_ID_UNKNOWN = 0,

      BM_CORE_IF_MSG_ID_SET_PROFILE_USER_AVAILABILITY_REQUEST,

      BM_CORE_IF_MSG_ID_SET_AUTO_CONNECTION_TYPE_REQUEST,

      BM_CORE_IF_MSG_ID_SWITCH_BT_STATUS_REQUEST,

      BM_CORE_IF_MSG_ID_SET_BT_LOCAL_FRIENDLY_NAME_REQUEST,

      BM_CORE_IF_MSG_ID_SWITCH_LOCAL_PAIRABLE_MODE_REQUEST,
      BM_CORE_IF_MSG_ID_SWITCH_LOCAL_CONNECTABLE_MODE_REQUEST,
      BM_CORE_IF_MSG_ID_SWITCH_DISCOVERY_STATUS_REQUEST,
      BM_CORE_IF_MSG_ID_SWITCH_DISCOVERY_STATUS_RESPONSE,
      BM_CORE_IF_MSG_ID_DISCOVERY_STATUS_UPDATE,

      BM_CORE_IF_MSG_ID_PAIRING_STATUS_UPDATE,

      BM_CORE_IF_MSG_ID_START_PAIRING_REQUEST,
      BM_CORE_IF_MSG_ID_CANCEL_PAIRING_REQUEST,
      BM_CORE_IF_MSG_ID_ACCEPT_PAIRING_REQUEST_REQUEST,
      BM_CORE_IF_MSG_ID_REJECT_PAIRING_REQUEST_REQUEST,
      BM_CORE_IF_MSG_ID_SET_PAIRING_PIN_REQUEST,
      BM_CORE_IF_MSG_ID_ADD_OOB_PAIRED_DEVICE_REQUEST,

      BM_CORE_IF_MSG_ID_DELETE_DEVICE_REQUEST,
      BM_CORE_IF_MSG_ID_START_AUTO_CONNECTION_SEQUENCE_REQUEST,
      BM_CORE_IF_MSG_ID_STOP_AUTO_CONNECTION_SEQUENCE_REQUEST,
      BM_CORE_IF_MSG_ID_CONNECT_DEVICE_REQUEST,
      BM_CORE_IF_MSG_ID_DISCONNECT_DEVICE_REQUEST,
      BM_CORE_IF_MSG_ID_CONNECT_PROFILES_REQUEST,
      BM_CORE_IF_MSG_ID_DISCONNECT_PROFILES_REQUEST,
      BM_CORE_IF_MSG_ID_BLOCK_DEVICE_REQUEST,
      BM_CORE_IF_MSG_ID_UNBLOCK_DEVICE_REQUEST,
      BM_CORE_IF_MSG_ID_BLOCK_PROFILES_REQUEST,
      BM_CORE_IF_MSG_ID_UNBLOCK_PROFILES_REQUEST,

      BM_CORE_IF_MSG_ID_PAIRED_DEVICE_LIST_UPDATE,

      BM_CORE_IF_MSG_ID_DEVICE_CONNECTION_STATUS_LIST_UPDATE,

      BM_CORE_IF_MSG_ID_SET_FAVORITE_REQUEST,
      BM_CORE_IF_MSG_ID_CLEAR_FAVORITE_REQUEST,

      BM_CORE_IF_MSG_START_SERVICE_SEARCH_REQUEST,
      BM_CORE_IF_MSG_STOP_SERVICE_SEARCH_REQUEST,

      BM_CORE_IF_MSG_ID_SET_PRIMARY_HFP_DEVICE_REQUEST,

      BM_CORE_IF_MSG_ID_UPDATE_LINK_QUALITY_REQUEST,

      BM_CORE_IF_MSG_ID_BLOCK_STATUS_UPDATE,

      BM_CORE_IF_MSG_ID_SET_BT_LIMITATION_MODE_REQUEST,
      BM_CORE_IF_MSG_ID_REPLACE_BT_LIMITATION_MODE_REQUEST,
      BM_CORE_IF_MSG_ID_BT_LIMITATION_MODE_UPDATE,

      BM_CORE_IF_MSG_ID_NOTIFY_USER_DECISION_REQUEST,

      BM_CORE_IF_MSG_ID_SET_BT_PROFILEUSAGE_INFO_REQUEST,

      BM_CORE_IF_MSG_ID_SWITCH_TEST_MODE_REQUEST,

      BM_CORE_IF_MSG_ID_BTS_FB_CONNECTION_INITIALIZED,
      BM_CORE_IF_MSG_ID_BTS_LOCAL_VERSION_INFO,
      BM_CORE_IF_MSG_ID_BTS_SWITCH_BLUETOOTH_ON_OFF_RESULT,
      BM_CORE_IF_MSG_ID_BTS_CURRENT_BLUETOOTH_ON_OFF_STATUS,
      BM_CORE_IF_MSG_ID_BTS_BLUETOOTH_HARDWARE_ON_OFF_RESULT,
      BM_CORE_IF_MSG_ID_BTS_SET_LOCAL_BT_NAME_RESULT,
      BM_CORE_IF_MSG_ID_BTS_CURRENT_LOCAL_BT_NAME,
      BM_CORE_IF_MSG_ID_BTS_SET_LOCAL_ADAPTER_MODES_RESULT,
      BM_CORE_IF_MSG_ID_BTS_CURRENT_DISCOVERABLE_MODE,
      BM_CORE_IF_MSG_ID_BTS_CURRENT_CONNECTABLE_MODE,

      BM_CORE_IF_MSG_ID_BTS_START_DISCOVERY_RESULT,
      BM_CORE_IF_MSG_ID_BTS_STOP_DISCOVERY_RESULT,
      BM_CORE_IF_MSG_ID_BTS_DISCOVERING_STATUS,

      BM_CORE_IF_MSG_ID_BTS_DISCOVERED_DEVICE_FOUND,

      BM_CORE_IF_MSG_ID_BTS_PAIRING_FINISHED,
      BM_CORE_IF_MSG_ID_BTS_LINK_KEY_REQUESTED,

      BM_CORE_IF_MSG_ID_BTS_SUPPORTED_SERVICES,
      BM_CORE_IF_MSG_ID_BTS_SPP_CAPABILITIES,
      BM_CORE_IF_MSG_ID_BTS_REMOTE_DEVICE_ID_SERVICE_RECORDS,
      BM_CORE_IF_MSG_ID_BTS_DEVICE_CAPABILITIES,
      BM_CORE_IF_MSG_ID_BTS_REMOTE_SERVICE_SEARCH_RESULT,
      BM_CORE_IF_MSG_ID_BTS_REMOTE_NAME_RESULT,
      BM_CORE_IF_MSG_ID_BTS_REMOTE_NAME_STATUS_UPDATE,
      BM_CORE_IF_MSG_ID_BTS_START_PAIRING_RESULT,
      BM_CORE_IF_MSG_ID_BTS_PIN_CODE_REQUESTED,
      BM_CORE_IF_MSG_ID_BTS_SECURE_SIMPLE_PAIRING_REQUESTED,
      BM_CORE_IF_MSG_ID_BTS_CONNECT_PROTOCOL_RESULT,
      BM_CORE_IF_MSG_ID_BTS_REMOTE_PROTOCOL_CONNECT_REQUEST,
      BM_CORE_IF_MSG_ID_BTS_DISCONNECT_PROTOCOL_RESULT,
      BM_CORE_IF_MSG_ID_BTS_DISCONNECT_DEVICE_RESULT,
      BM_CORE_IF_MSG_ID_BTS_PROTOCOL_CONNECTION_STATUS,
      BM_CORE_IF_MSG_ID_BTS_DEVICE_CONNECTION_STATUS,
      BM_CORE_IF_MSG_ID_BTS_LINK_QUALITY_RESULT,

      BM_CORE_IF_MSG_ID_BTS_START_TEST_MODE_RESULT,
      BM_CORE_IF_MSG_ID_BTS_STOP_TEST_MODE_RESULT,
      BM_CORE_IF_MSG_ID_BTS_TEST_MODE_STATUS,

      BM_CORE_IF_MSG_ID_BTS_FB_WBL_INITIALIZED,
      BM_CORE_IF_MSG_ID_BTS_WBL_SERVICE_AVAILABILITY,
      BM_CORE_IF_MSG_ID_BTS_WBL_LAST_INTENDED_MODE,
      BM_CORE_IF_MSG_ID_BTS_WBL_SUPPORTED_FREQUENCIES,
      BM_CORE_IF_MSG_ID_BTS_WBL_HEALTHINESS_INDICATOR,
      BM_CORE_IF_MSG_ID_BT_SYSTEM_STATE_UPDATE,

      BM_CORE_IF_MSG_ID_TERMINATE_MESSAGE_HANDLER_THREAD,
      BM_CORE_IF_MSG_ID_SET_DEVICE_REMOTE_CONNECTABLE,
      BM_CORE_IF_MSG_ID_INIT_DISCOVERED_DEVICE_LIST,
      BM_CORE_IF_MSG_ID_DEVICE_CONNECTION_STATUS_UPDATE,
      BM_CORE_IF_MSG_ID_PROTOCOL_CONNECTION_STATUS_UPDATE,
      BM_CORE_IF_MSG_ID_DEVICE_DELETED,

      BM_CORE_IF_MSG_ID_SM_TIMEOUT,

      BM_CORE_IF_MSG_ID_SWITCH_MULTI_HFP_SUPPORT_REQUEST,
      BM_CORE_IF_MSG_ID_SET_DEVICE_USAGEPREFERENCE_REQUEST,
      BM_CORE_IF_MSG_ID_RESET_TO_DEFAULT_REQUEST,

      BM_CORE_IF_MSG_ID_LAST
   } BmCoreIfMsgId;

   typedef enum
   {
      BM_CORE_IF_MSG_ORIGIN_INTERNAL = 0,
      BM_CORE_IF_MSG_ORIGIN_CLIENT,
      BM_CORE_IF_MSG_ORIGIN_BT_STACK_IF
   } BmCoreIfMsgOrigin;

   /**
    * Type declaration of an enum containing the list type
    */
   typedef enum
   {
      LTY_NONE = 0UL,

      LTY_GENERAL = 1UL,
      LTY_GENERAL_UPDATE,

      LTY_DEVICES_COUNT,                          /**< Count of known devices */
      LTY_DEVICES_INSERT,                         /**< insert a new device */
      LTY_DEVICES_UPDATE,                         /**< update a device */
      LTY_DEVICES,                                /**< get a device */
      LTY_DEVICES_DELETE,                         /**< delete a device */
      LTY_DEVICES_SELECT_MAX_CONNECTED_INDICATOR, /**< get max(LastConnectedIndicator) from devices */
      LTY_DEVICES_SELECT_MAX_FAVORITE,            /**< get max(FavoriteDevcie) from devices */
      LTY_DEVICES_SELECT_MAX_PAIRED_INDICATOR,    /**< get max(LastPairedIndicator) from devices */

      LTY_DEVICEPROTOCOLS_INSERT,                 /**< insert a new line in deviceprotocols*/
      LTY_DEVICEPROTOCOLS_UPDATE,                 /**< update deviceprotocols */
      LTY_DEVICEPROTOCOLS,                        /**< read deviceprotocols */
      LTY_DEVICEPROTOCOLS_DELETE,                 /**< delete device from deviceprotocols */
      LTY_DEVICEPROTOCOLS_SELECT_MAX_INDICATOR,   /**< get max(LastConnectedIndicator) from deviceprotocols */
      LTY_DEVICEPROTOCOLS_SELECT_MAX_FAVORITE,    /**< get max(FavoriteProtocol) from deviceprotocols */

      LTY_END_OF_EXTERNAL_LIST_TYPES,             // this list type marks the end of the public list types
      LTY_SQLITE_MASTER,                          // to remove all triggers from database on startup

      LTY_END_OF_LTY,
      LTY_END_OF_LTY_COUNT
   } ListType;
   #define LISTTYPE_FORMAT "i"
   #define LISTTYPE_INIT LTY_END_OF_EXTERNAL_LIST_TYPES

   /**
    * Type Enum declaration for the ProtocolState (PS) of one protocol
    */
   typedef enum {
      BM_PS_NOT_SET = 0,
      BM_PS_CONNECTED,
      BM_PS_DISCONNECTED
   } ProtocolState;
   #define PROTOCOLSTATE_FORMAT "i"
   #define PROTOCOLSTATE_INIT 0

   /**
    * Type Enum declaration for the protocol type (PT)
    */
   typedef enum {
      BM_PT_NOT_SET = 0,
      BM_PT_NEVER,
      BM_PT_OPTIONAL,
      BM_PT_MANDATORY
   } ProtocolType;
   #define PROTOCOLTYPE_FORMAT "i"
   #define PROTOCOLTYPE_INIT 0

   /**
    * Error codes representing the result of starting discovery session
    */
   typedef enum {
      BM_START_DISCOVERY_OK = 0,  /**< OK, must be the first code */
      BM_START_DISCOVERY_REJECTED,
      BM_START_DISCOVERY_ERROR,   /**< general error */
      BM_START_DISCOVERY_LAST     /**< end marker, must be the last one */
   } StartDiscoveryResult;
   #define STARTDISCOVERYRESULT_FORMAT "i"
   #define STARTDISCOVERYRESULT_INIT BM_START_DISCOVERY_OK

   /**
    * Error codes representing the result of stopping discovery session
    */
   typedef enum {
      BM_STOP_DISCOVERY_OK = 0,  /**< OK, must be the first code */
      BM_STOP_DISCOVERY_ERROR,   /**< general error */
      BM_STOP_DISCOVERY_LAST     /**< end marker, must be the last one */
   } StopDiscoveryResult;
   #define STOPDISCOVERYRESULT_FORMAT "i"
   #define STOPDISCOVERYRESULT_INIT BM_STOP_DISCOVERY_OK

   /**
    * Error codes representing the result of switching the local BT device pairable and discoverable
    */
   typedef enum {
      BM_ENABLE_PAIRABLE_OK = 0,  /**< OK, must be the first code */
      BM_ENABLE_PAIRABLE_ERROR,   /**< general error */
      BM_ENABLE_PAIRABLE_LAST     /**< end marker, must be the last one */
   } EnablePairableResult;
   #define ENABLEPAIRABLERESULT_FORMAT "i"
   #define ENABLEPAIRABLERESULT_INIT BM_ENABLE_PAIRABLE_OK

   /**
    * Error codes representing the result of switching the local BT device non-pairable and non-discoverable
    */
   typedef enum {
      BM_DISABLE_PAIRABLE_OK = 0,  /**< OK, must be the first code */
      BM_DISABLE_PAIRABLE_ERROR,   /**< general error */
      BM_DISABLE_PAIRABLE_LAST     /**< end marker, must be the last one */
   } DisablePairableResult;
   #define DISABLEPAIRABLERESULT_FORMAT "i"
   #define DISABLEPAIRABLERESULT_INIT BM_DISABLE_PAIRABLE_OK

   #define DISCONNECTEDREASON_FORMAT "i"
   #define DISCONNECTEDREASON_INIT BM_DISCONNECTED_REASON_UNKNOWN

   typedef enum
   {
      BM_LAST_EXPECTED_CONNECTION_STATUS_UNKNOWN = 0,
      BM_LAST_EXPECTED_CONNECTION_STATUS_DISCONNECTED,
      BM_LAST_EXPECTED_CONNECTION_STATUS_CONNECTED
   } LastExpectedConnectionStatus;

   typedef unsigned int Count;
   #define COUNT_FORMAT "i"
   #define COUNT_INIT (0)

   typedef unsigned int RowId;
   #define ROWID_FORMAT "i"
   #define ROWID_INIT (0)

   typedef char ColumnName[DB_COLUMN_NAME_SIZE];
   #define COLUMNNAME_FORMAT "t"
   #define COLUMNNAME_INIT ""

   typedef enum
   {
      BM_DEVICE_UPDATE_TYPE_UNKNOWN = 0,
      BM_DEVICE_UPDATE_TYPE_DEVICE_IDENTIFICAION,
      BM_DEVICE_UPDATE_TYPE_INBAND_RINGTONE_SUPPORT,
      BM_DEVICE_UPDATE_TYPE_CONNECTION_STATUS,
      BM_DEVICE_UPDATE_TYPE_PROTOCOL_CONNECTION_STATUS,
      BM_DEVICE_UPDATE_TYPE_SPP_UUID_CONNECTION_STATUS,
      BM_DEVICE_UPDATE_TYPE_DELETE_DEVICE,
      BM_DEVICE_UPDATE_TYPE_DELETE_ALL_DEVICES
   } DeviceUpdateType;

   typedef enum
   {
      BM_BLOCK_STATUS_UPDATE_TYPE_UNKNOWN = 0,
      BM_BLOCK_STATUS_UPDATE_TYPE_OVERALL,
      BM_BLOCK_STATUS_UPDATE_TYPE_DEVICE,
      BM_BLOCK_STATUS_UPDATE_TYPE_DEVICE_PROTOCOL_BLOCK_STATE,
      BM_BLOCK_STATUS_UPDATE_TYPE_DEVICE_PROTOCOL_SUPPORT,
      BM_BLOCK_STATUS_UPDATE_TYPE_DELETE_DEVICE,
      BM_BLOCK_STATUS_UPDATE_TYPE_INSERT_DEVICE
   } BlockStatusUpdateType;


   typedef std::vector<DeviceConnectionController*> DeviceConnectionControllerList;

   typedef struct
   {
      ProtocolConnectionController* _protocolconnectionsmhandle;
      ProtocolId _id;
      Uuid _uuid;
      DeviceId _sminusebydeviceid;
      ProtocolName _name;
   } ProtocolConnectionControllerInfo;

   typedef std::vector<ProtocolConnectionControllerInfo> ProtocolConnectionControllerInfoList;

   typedef enum
   {
      BM_REQUEST_TYPE_UNKNOWN = 0,
      BM_REQUEST_TYPE_CONNECTION
   } RequestType;

   typedef enum
   {
      BM_RESTRICTION_BIT_LOCAL_SPM_STATE_OFF = 0,
      BM_RESTRICTION_BIT_LOCAL_SPM_STATE_UNDERVOLTAGE,
      BM_RESTRICTION_BIT_BT_SYSTEM_STATE_BLOCKED,
      BM_RESTRICTION_BIT_BT_SYSTEM_STATE_OFF,
      BM_RESTRICTION_BIT_BT_STATUS_OFF,
      BM_RESTRICTION_BIT_OVERALL_DEVICE_BLOCK_STATE_BLOCKED,
      BM_RESTRICTION_BIT_RESET_TO_DEFAULT_IN_PROGRESS,
      BM_RESTRICTION_BIT_MAX_EXCEEDED // must be always the last enum value
   } RestrictionBit;

   const size_t RestrictionBitfieldSize = static_cast<size_t>(BM_RESTRICTION_BIT_MAX_EXCEEDED);

   typedef std::bitset<RestrictionBitfieldSize> FunctionalityRestrictionInfo;

   const FunctionalityRestrictionInfo BM_RESTRICTION_GROUP_00(0u);   // 0000000
   const FunctionalityRestrictionInfo BM_RESTRICTION_GROUP_01(3u);   // 0000011
   const FunctionalityRestrictionInfo BM_RESTRICTION_GROUP_02(11u);  // 0001011
   const FunctionalityRestrictionInfo BM_RESTRICTION_GROUP_03(79u);  // 1001111
   const FunctionalityRestrictionInfo BM_RESTRICTION_GROUP_04(95u);  // 1011111
   const FunctionalityRestrictionInfo BM_RESTRICTION_GROUP_05(111u); // 1101111
   const FunctionalityRestrictionInfo BM_RESTRICTION_GROUP_06(127u); // 1111111

   typedef enum
   {
      BM_AUTOCONNECTION_START_MODE_TRY_ONCE_ONLY_IF_BOOKED_BEFORE = 0,
      BM_AUTOCONNECTION_START_MODE_TRY_ONCE,
      BM_AUTOCONNECTION_START_MODE_BOOK_AND_TRY_ONCE
   } AutoConnectionStartMode;

   #define AUTOCONNECTIONSTARTMODE_FORMAT "i"
   #define AUTOCONNECTIONSTARTMODE_INIT BM_AUTOCONNECTION_START_MODE_TRY_ONCE_ONLY_IF_BOOKED_BEFORE

   #define BMCORECONFIGPATH "/var/opt/bosch/static/connectivity/config/BM_config.json"

   typedef enum
   {
      BM_DEVICEINFO_UPDATE_UNKNOWN = 0,
      BM_DEVICEINFO_UPDATE_DEVICE_IDENTIFICAION,
      BM_DEVICEINFO_UPDATE_SPP_CAPABILITIES,
      BM_DEVICEINFO_UPDATE_SUPPORTED_PROTOCOLS
   } DeviceInfoUpdateType;

   #define DEVICEINFOUPDATETYPE_FORMAT "i"
   #define DEVICEINFOUPDATETYPE_INIT BM_DEVICEINFO_UPDATE_UNKNOWN

   typedef struct
   {
      unsigned int _connectedIndicator;
      unsigned int _pairedIndicator;
   } DeviceIndicator;


   #define IS_LIMIMODE_FOR_HANDSFREE_PROFILE(curLimiFeature, curLimiCommIf) (((BM_LIMITATION_FEATURE_ANDROID_AUTO == curLimiFeature) \
                                                         || (BM_LIMITATION_FEATURE_CAR_LIFE == curLimiFeature) \
                                                         || (BM_LIMITATION_FEATURE_ON_CAR == curLimiFeature)) \
                                                         && (BM_LIMITATION_COMMUNICATION_IF_USB == curLimiCommIf))

   #define IS_LIMIFEATURE_FOR_HANDSFREE_PROFILE(curLimiFeature) ((BM_LIMITATION_FEATURE_ANDROID_AUTO == curLimiFeature) \
                                                      || (BM_LIMITATION_FEATURE_CAR_LIFE == curLimiFeature) \
                                                      || (BM_LIMITATION_FEATURE_ON_CAR == curLimiFeature))

   #define IS_LIMIFEATURE_FOR_MANUAL_HFP(curLimiFeature) ((BM_LIMITATION_FEATURE_MIRROR_LINK == curLimiFeature) \
                                                   || (BM_LIMITATION_FEATURE_MY_SPIN == curLimiFeature) \
                                                   || (BM_LIMITATION_FEATURE_MY_SPIN_IOS == curLimiFeature) \
                                                   || (BM_LIMITATION_FEATURE_CARLIFE_IOS == curLimiFeature))

   /**
    * Type declaration for a state machine handle
    */
   typedef long long int SmHandle;

   /**
    * Type for a row number in a database list
    */
   typedef uint32_t RowNumber;
   #define ROW_NUMBER_NONE ((RowNumber)-1)

   /**
    * Type declaration of an index
    */
   typedef uint32_t Index;
   #define INDEX_FORMAT "i"
   #define INDEX_LAST_ACTIVE_OBJECT ((Index)-1)
   #define INDEX_NO_LIMIT ((Index)-1)
   #define INDEX_NONE ((Index)-2)
   #define INDEX_INIT 0

   /**
    * Type declaration of a element position in a list
    */
   typedef uint32_t Position;
   #define POSITION_FORMAT "i"
   #define POSITION_INIT 0
   #define POSITION_NOT_SET (Position)(-1)

   /**
    * Type declaration of a database object ID
    */
   typedef uint32_t ObjectId;
   #define OBJECTID_FORMAT "i"
   #define OBJECT_ID_NONE ((ObjectId)-1)
   #define OBJECTID_INIT OBJECT_ID_NONE

   #define BM_REQUEST_TYPE_PAIRABLE_MODE 1
   #define BM_REQUEST_TYPE_CONNECTABLE_MODE 2

   /**
    * Type for a trigger id for a database trigger
    */
   //typedef uint32_t TriggerId;

   //#define MY_MEDIA (0)
}

#endif // _BM_ALL_TYPES_INTERNAL_H_
