#ifndef _BT_LIMITATION_CONTROLLER_H_
#define _BT_LIMITATION_CONTROLLER_H_

#include "BtLimitationSm.h"
#include "ILocalSpm.h"
#include "BmAllTypesInternal.h"
#include "Lock.h"
#include "Timer.h"

namespace bmcore
{
//using namespace cc;
using namespace std;

//!Forward Declarations
class BmCoreIfMessage_SetBtLimitationModeRequest;
class BmCoreIfMessage_ReplaceBtLimitationModeRequest;

class BtLimitationController : public BtLimitationSm, public ILocalSpm
{
public:
   /***************************************************************************
    *********************************PUBLIC*************************************
    ***************************************************************************/

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::BtLimitationController(IN const ComponentId componentID)
    ***************************************************************************/
   /*!
    * \fn      BtLimitationController()
    * \brief   Default Constructor
    * \param   [IN] componentID: Component ID of Limitation Controller
    ***************************************************************************/
   BtLimitationController(IN const ComponentId componentID);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::~BtLimitationController()
    ***************************************************************************/
   /*!
    * \fn      ~BtLimitationController()
    * \brief   Virtual Destructor
    ***************************************************************************/
   virtual ~BtLimitationController();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::validateBTLimitationModeRequest(...)
    ***************************************************************************/
   /*!
    * \fn      validateBTLimitationModeRequest()
    * \brief   Method to validate the incoming BT Limitation mode request from the client.
    * \param   [IN] bmCoreIfMessage: Pointer to class BmCoreIfMessage_SetBtLimitationModeRequest
    * \retval  BmResult: BM_RESULT_OK if success else Non-zero error code
    ***************************************************************************/
   BmResult validateBTLimitationModeRequest(BmCoreIfMessage_SetBtLimitationModeRequest* bmCoreIfMessage);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::validateReplaceBTLimitationModeRequest(...)
    ***************************************************************************/
   /*!
    * \fn      validateReplaceBTLimitationModeRequest()
    * \brief   Method to validate the incoming BT Replace Limitation mode request from the client.
    * \param   [IN] bmCoreIfMessage: Pointer to class BmCoreIfMessage_ReplaceBtLimitationModeRequest
    * \retval  BmResult: BM_RESULT_OK if success else Non-zero error code
    ***************************************************************************/
   BmResult validateReplaceBTLimitationModeRequest(BmCoreIfMessage_ReplaceBtLimitationModeRequest* bmCoreIfMessage);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::sendLimitationActionSmEvent(...)
    ***************************************************************************/
   /*!
    * \fn      sendLimitationActionSmEvent(IN const string& callingMethodName,
    *          IN const LimitationAction& limitationAction)
    * \brief   Method to send Limitation Mode Action from the client to the state machine.
    * \param   [IN] callingMethodName: Reference to the string of calling Method Name
    * \param   [IN] limitationAction: Limitation Action enum value
    * \retval  Result: 0 is Success else Non-zero
    ***************************************************************************/
   Result sendLimitationActionSmEvent(IN const string& callingMethodName,
         IN const LimitationAction& limitationAction);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::onBtStatusChanged(IN const SwitchStatus& bluetoothStatus)
    ***************************************************************************/
   /*!
    * \fn      onBtStatusChanged(IN const SwitchStatus& bluetoothStatus)
    * \brief   Method to inform about the BT ON/OFF status change.
    * \param   [IN] bluetoothStatus: Reference to SwitchStatus class object
    * \retval  None
    ***************************************************************************/
   void onBtStatusChanged(IN const SwitchStatus& bluetoothStatus);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::onLocalPairableModeChanged(IN const SwitchStatus& localPairableMode)
    ***************************************************************************/
   /*!
    * \fn      onLocalPairableModeChanged(IN const SwitchStatus& localPairableMode)
    * \brief   Method to inform about the Local Pairable Mode status change.
    * \param   [IN] localPairableMode: Reference to SwitchStatus class object
    * \retval  None
    ***************************************************************************/
   void onLocalPairableModeChanged(IN const SwitchStatus& localPairableMode);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::onLocalConnectableModeChanged(IN const SwitchStatus& localConnectableMode)
    ***************************************************************************/
   /*!
    * \fn      onLocalConnectableModeChanged(IN const SwitchStatus& localConnectableMode)
    * \brief   Method to inform about the Local Connectable Mode status change.
    * \param   [IN] localConnectableMode: Reference to SwitchStatus class object
    * \retval  None
    ***************************************************************************/
   void onLocalConnectableModeChanged(IN const SwitchStatus& localConnectableMode);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::onProtocolConnectionStatusChange(...)
    ***************************************************************************/
   /*!
    * \fn      onProtocolConnectionStatusChange(IN const DeviceId deviceId,
    *          IN const Protocol& protocol, IN const ConnectionStatus connectionStatus)
    * \brief   Method to inform about the Device Protocol connection status change.
    * \param   [IN] deviceId: BT Device ID.
    * \param   [IN] protocol: BT Protocol Info.
    * \param   [IN] connectionStatus: Connection status of the protocol.
    * \retval  None
    ***************************************************************************/
   void onProtocolConnectionStatusChange(IN const DeviceId deviceId, IN const Protocol& protocol,
         IN const ConnectionStatus connectionStatus);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::onDeviceConnectionStatusChange(...)
    ***************************************************************************/
   /*!
    * \fn      onDeviceConnectionStatusChange(IN const DeviceId deviceId,
    * 				IN const ConnectionStatus connectionStatus)
    * \brief   Method to update about device connection status.
    * \param   [IN] deviceId: Unique BT device Identifier
    * \param   [IN] connectionStatus: Device connection status.
    * \retval  None
    ***************************************************************************/
   void onDeviceConnectionStatusChange(IN const DeviceId deviceId, IN const ConnectionStatus connectionStatus);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::UpdateAvpBlockedStateForDevice(...)
    ***************************************************************************/
   /*!
    * \fn      UpdateAvpBlockedStateForDevice(BdAddress& bdAddress, bool AvpBlockedState)
    * \brief   Method to update the AVP block status of the projection device.
    * \param   [IN] bdAddress:BT Address of the device for which AVP should be blocked
    * \param   [IN] AvpBlockedState: Blocked state of AVP profile
    * \retval  None
    ***************************************************************************/
   void UpdateAvpBlockedStateForDevice(IN const BdAddress& bdAddress, IN bool AvpBlockedState);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::onConflictsResolved(IN const UserDecisionInfo&
    ** 			userDecision)
    ***************************************************************************/
   /*!
    * \fn      onConflictsResolved(IN const UserDecisionInfo& userDecision)
    * \brief   Method called by ConflictManager after the conflicts are resolved.
    * \param   [IN] userDecision: Reference to UserDecision type class
    * \retval  None
    ***************************************************************************/
   void onConflictsResolved(IN const UserDecisionInfo& userDecision);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::updateWblSupportedFrequencies()
    ***************************************************************************/
   /*!
    * \fn      updateWblSupportedFrequencies(BTSWblSupportedRestrictionEntryList&
    * 		 	supportedFrequencies)
    * \brief   Method to update the supported frequencies received from WBL
    * \param   [IN] supportedFrequencies - Reference to BTSWblSupportedRestrictionEntryList
    * \retval  None
    ***************************************************************************/
   void updateWblSupportedFrequencies(IN BTSWblSupportedRestrictionEntryList& supportedFrequencies);

   // ************************************
   // * Overridden methods of BtLimitationSm *
   // ************************************

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::messageNotConsumed()
    ***************************************************************************/
   /*!
    * \fn      messageNotConsumed()
    * \brief   Method called when DEFAULT state is triggered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result messageNotConsumed();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::initSm()
    ***************************************************************************/
   /*!
    * \fn      initSm()
    * \brief   Method called when INIT state is triggered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result initSm();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::handleStopSm()
    ***************************************************************************/
   /*!
    * \fn      handleStopSm()
    * \brief   Method called when STOP_SM state is triggered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result handleStopSm();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::handleDoneSm()
    ***************************************************************************/
   /*!
    * \fn      handleDoneSm()
    * \brief   Method called when DONE_SM state is triggered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result handleDoneSm();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterActivating()
    ***************************************************************************/
   /*!
    * \fn      enterActivating()
    * \brief   Method called when state ACTIVATING is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterActivating();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterActive()
    ***************************************************************************/
   /*!
    * \fn      enterActive()
    * \brief   Method called when state ACTIVE is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterActive();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterDeactivateWaiting()
    ***************************************************************************/
   /*!
    * \fn      enterDeactivateWaiting()
    * \brief   Method called when state DEACTIVATE_WAITING is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterDeactivateWaiting();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterDeactivating()
    ***************************************************************************/
   /*!
    * \fn      enterDeactivating()
    * \brief   Method called when state DEACTIVATING is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterDeactivating();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterDeactivatingHandsFreeProfile()
    ***************************************************************************/
   /*!
    * \fn      enterDeactivatingHandsFreeProfile()
    * \brief   Method called when state DEACTIVATING_HANDSFREEPROFILE is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterDeactivatingHandsFreeProfile();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterDeactivatingManualHFP()
    ***************************************************************************/
   /*!
    * \fn      enterDeactivatingManualHFP()
    * \brief   Method called when state DEACTIVATING_MANUALHFP is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterDeactivatingManualHFP();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterDeactivatingCP()
    ***************************************************************************/
   /*!
    * \fn      enterDeactivatingCP()
    * \brief   Method called when state DEACTIVATING_CP is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterDeactivatingCP();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterDeactivatingCPW()
    ***************************************************************************/
   /*!
    * \fn      enterDeactivatingCPW()
    * \brief   Method called when state DEACTIVATING_CPW is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterDeactivatingCPW();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterDeactivatingInternal()
    ***************************************************************************/
   /*!
    * \fn      enterDeactivatingInternal()
    * \brief   Method called when state DEACTIVATING_INTERNAL is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterDeactivatingInternal();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterDeactivatingSpmStateOff()
    ***************************************************************************/
   /*!
    * \fn      enterDeactivatingSpmStateOff()
    * \brief   Method called when state DEACTIVATING_SPMSTATEOFF is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterDeactivatingSpmStateOff();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterError()
    ***************************************************************************/
   /*!
    * \fn      enterError()
    * \brief   Method called when state ERROR is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterError();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterIdle()
    ***************************************************************************/
   /*!
    * \fn      enterIdle()
    * \brief   Method called when state IDLE is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterIdle();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::checkLimitationFeatureToBePrepared()
    ***************************************************************************/
   /*!
    * \fn      checkLimitationFeatureToBePrepared()
    * \brief   Method called to check which limitation mode to be prepared.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result checkLimitationFeatureToBePrepared();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterPreparing()
    ***************************************************************************/
   /*!
    * \fn      enterPreparing()
    * \brief   Method called when state PREPARING is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterPreparing();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterPrepared()
    ***************************************************************************/
   /*!
    * \fn      enterPrepared()
    * \brief   Method called when state PREPARED is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterPrepared();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterPreactivating()
    ***************************************************************************/
   /*!
    * \fn      enterPreactivating()
    * \brief   Method called when state PREACTIVATING is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterPreactivating();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::enterPreactivated()
    ***************************************************************************/
   /*!
    * \fn      enterPreactivated()
    * \brief   Method called when state PREACTIVATED is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result enterPreactivated();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::ConnectProfilesCPW()
    ***************************************************************************/
   /*!
    * \fn      ConnectProfilesCPW()
    * \brief   Method called when state PREPARINGCPW_CONNECTPROFILES is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result ConnectProfilesCPW();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::DisconnectAndBlockProfilesCPW()
    ***************************************************************************/
   /*!
    * \fn      DisconnectAndBlockProfilesCPW()
    * \brief   Method called when state PREPARINGCPW_BLOCKPROFILES is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result DisconnectAndBlockProfilesCPW();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::DisconnectAndBlockProfilesOnActivatingHandsFreeProfile()
    ***************************************************************************/
   /*!
    * \fn      DisconnectAndBlockProfilesOnActivatingHandsFreeProfile()
    * \brief   Method called when state ACTIVATINGHANDSFREEPROFILE_BLOCKPROFILES is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result DisconnectAndBlockProfilesOnActivatingHandsFreeProfile();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::DisconnectAndBlockProfilesOnActivatingCP()
    ***************************************************************************/
   /*!
    * \fn      DisconnectAndBlockProfilesOnActivatingCP()
    * \brief   Method called when state ACTIVATINGCP_BLOCKPROFILES is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result DisconnectAndBlockProfilesOnActivatingCP();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::DisconnectAndBlockProfilesOnActivatingManualHFP()
    ***************************************************************************/
   /*!
    * \fn      DisconnectAndBlockProfilesOnActivatingManualHFP()
    * \brief   Method called when state ACTIVATINGMANULHFP_BLOCKPROFILES is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result DisconnectAndBlockProfilesOnActivatingManualHFP();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::disconnectSppIap()
    ***************************************************************************/
   /*!
    * \fn      disconnectSpp()
    * \brief   Method called to disconnect SPP iAP on CPW device
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result disconnectSppIap();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::unblockSppIap()
    ***************************************************************************/
   /*!
    * \fn      unblockSppIap()
    * \brief   Method called to unblock SPP iAP on CPW device
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result unblockSppIap();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::IsConflictResAndProfileBlockReqdOnPreparingCPW()
    ***************************************************************************/
   /*!
    * \fn      IsConflictResAndProfileBlockReqdOnPreparingCPW()
    * \brief   Method called to validate if Conflict Resolution and Blocking Profiles is required
    *        while entering PREPARINGCPW_RESOLVECONFLICTS state.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result IsConflictResAndProfileBlockReqdOnPreparingCPW();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::IsConflictResAndProfileBlockReqdOnPreactivatingCPW()
    ***************************************************************************/
   /*!
    * \fn      IsConflictResAndProfileBlockReqdOnPreactivatingCPW()
    * \brief   Method called to validate if Conflict Resolution and Blocking Profiles is required
    *        while entering PREACTIVATINGCPW_RESOLVECONFLICTS state.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result IsConflictResAndProfileBlockReqdOnPreactivatingCPW();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::IsConflictResReqdOnActivatingCPW()
    ***************************************************************************/
   /*!
    * \fn      IsConflictResReqdOnActivatingCPW()
    * \brief   Method called to validate if Conflict Resolution and Blocking Profiles is required
    *        while entering ACTIVATINGCPW_RESOLVECONFLICTS state.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result IsConflictResReqdOnActivatingCPW();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::ReconnectCPW()
    ***************************************************************************/
   /*!
    * \fn      ReconnectCPW()
    * \brief   Method called when PREPARE event is received while the SM is in ACTIVE state.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result ReconnectCPW();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::ResolveConflicts()
    ***************************************************************************/
   /*!
    * \fn      ResolveConflicts()
    * \brief   Method called when PREPARINGCPW/PREACTIVATINGCPW/ACTIVATINGCPW ResolveCOnflicts
    *          state is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result ResolveConflicts();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::SwitchLocalPairableMode()
    ***************************************************************************/
   /*!
    * \fn      SwitchLocalPairableMode()
    * \brief   Method called while entering ACTIVATINGAAP_SwitchPairable
    *          state is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result SwitchLocalPairableMode();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::SwitchLocalConnectableMode()
    ***************************************************************************/
   /*!
    * \fn      SwitchLocalConnectableMode()
    * \brief   Method called while entering ACTIVATINGAAP_SwitchConnectable
    *          state is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result SwitchLocalConnectableMode();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::HandlePrepareEvent()
    ***************************************************************************/
   /*!
    * \fn      HandlePrepareEvent()
    * \brief   Method called when Prepare event is received while CPW
    *          is in Activating state waiting for conflicts to be resolved
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result HandlePrepareEvent();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::SwitchOnBT()
    ***************************************************************************/
   /*!
    * \fn      SwitchOnBT()
    * \brief   Method called while entering ACTIVATINGAAP_SwitchBTOnOff
    *        state is entered.
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result SwitchOnBT();

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::ValidateConflictsStatus()
    ***************************************************************************/
   /*!
    * \fn      ValidateConflictsStatus()
    * \brief   Method called when CONFLICTS_RESOLVED event is triggered
    * \param   None
    * \retval  None
    ***************************************************************************/
   Result ValidateConflictsStatus(const UserDecision ConflictResolvedStatus);

   // wifi lost device
   inline bool isHandlingLostDevice(void) const
   {
      return _deviceLost;
   }

   inline DeviceId getProjectionDeviceId(void) const
   {
      return _ProjectionDeviceBTDeviceId;
   }

   // ************************************
   // * Overridden methods of ILocalSpm *
   // ************************************

   /**
    * @brief Creates the BtLimitationController instance
    *
    * Creates the Pairing state machine (including creation of message queue).
    * Informs LocalSpm that create is ready -> createDone(0).
    *
    * @attention: running in SPM thread context
    *
    * @return void
    */
   void create();

   /**
    * @brief Performs the initialization of the Pairing Controller instance, i.e.:
    * - initialization of member variables
    *
    * Synchronization: re-entrant
    *
    * Performance considerations:
    *
    * @return 0 on success and an error code otherwise.
    */
   Result init(InitReason reason);

   /**
    * @brief Starts the Pairing Controller instance, implies that all other
    * components are available
    *
    * Synchronization: re-entrant
    *
    * Performance considerations: none.
    *
    * @return 0 on success and an error code otherwise
    */
   Result run();

   /**
    * @brief Stops the Pairing Controller instance
    *
    * Synchronization: re-entrant
    *
    * Performance considerations: none
    *
    * @return 0 on success and an error code otherwise
    */
   Result stop();

   /**
    * @brief Cleans up the resources used by the Pairing Controller instance
    *
    * Synchronization: re-entrant
    *
    * Performance considerations: none
    *
    * @return 0 on success and an error code otherwise
    */
   Result done();

   bool isComponentSm() {return true;};

   /**
    * Returns the current state the state machine is in (for debugging of shutdown problems)
    *
    * @param[in,out] stateName buffer for storing the current state name
    * @param[in] size size of the stateName buffer
    *
    * @return pointer to stateName
    */
   char * getSmStateName(OUT tGeneralString stateName, IN size_t size);

   /**
    * @brief Returns statistic data as a string for the over all object statistics
    *
    * @param[out] stat string with statistics information to be printed
    *
    * @return 0 on success and an error code otherwise
    */
   //   int Statistics(OUT Statistics stat);

private:

   /***************************************************************************
    *********************************PRIVATE************************************
    ***************************************************************************/
   class DeviceProtocol
   {
   public:
      DeviceProtocol(IN const DeviceId deviceId, IN const Protocol& protocol) :
         _deviceId(deviceId),
         _protocol(protocol)
      {
      }

      DeviceProtocol(const DeviceProtocol& other) :
         _deviceId(other._deviceId), _protocol(other._protocol)
      {
      }

      virtual ~DeviceProtocol()
      {
      }

      friend void swap(DeviceProtocol& first, DeviceProtocol& second)
      {
         using std::swap;

         swap(first._deviceId, second._deviceId);
         swap(first._protocol, second._protocol);
      }

      DeviceProtocol& operator=(DeviceProtocol other)
      {
         swap(*this, other);

         return *this;
      }

      bool operator==(const DeviceProtocol& rhs) const
        		                               {
         return ((this->_deviceId == rhs._deviceId)
               && (this->_protocol == rhs._protocol));
        		                               }

      DeviceId _deviceId;
      Protocol _protocol;
   };

   typedef vector<DeviceProtocol> DeviceProtocolList;

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::getBTDeviceIdForProjectionDevice()
    ***************************************************************************/
   /*!
    * \fn      getBTDeviceIdForProjectionDevice(void)
    * \brief   Method to get BT Device ID for projection device.
    * \param   None
    * \retval  None
    ***************************************************************************/
   void getBTDeviceIdForProjectionDevice(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::sendSmEvent(...)
    ***************************************************************************/
   /*!
    * \fn      sendSmEvent(IN const string& callingMethodName,
    *        IN const string& eventName)
    * \brief   Method to send events to the state machine.
    * \param   [IN] callingMethodName: Reference to the string of calling Method Name
    * \param   [IN] limitationAction: Limitation Action enum value
    * \retval  Result: 0 if Success else Non-zero
    ***************************************************************************/
   Result sendSmEvent(IN const string& callingMethodName, IN const string& eventName);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::getSmEventNameByLimitationAction(...)
    ***************************************************************************/
   /*!
    * \fn      getSmEventNameByLimitationAction(IN const LimitationAction& limitationAction) const
    * \brief   Method to get event name based on the Limitation action .
    * \param   [IN] limitationAction: Limitation Action enum value
    * \retval  string : Event Name if Success else NULL string
    ***************************************************************************/
   const string getSmEventNameByLimitationAction(IN const LimitationAction limitationAction) const;

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::BlockProfiles(void)
    ***************************************************************************/
   /*!
    * \fn      BlockProfiles(void)
    * \brief   Method to block set of profiles for projection and non-projection devices .
    * \param   None
    * \retval  Result: 0 if Success else Non-zero
    ***************************************************************************/
   Result BlockProfiles(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::UnblockProfiles(void)
    ***************************************************************************/
   /*!
    * \fn      UnblockProfiles(void)
    * \brief   Method to unblock set of profiles for projection and non-projection devices
    *        which are already blocked.
    * \param   None
    * \retval  Result: 0 if Success else Non-zero
    ***************************************************************************/
   Result UnblockProfiles(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::SwitchPairableMode(void)
    ***************************************************************************/
   /*!
    * \fn      SwitchPairableMode(void)
    * \brief   Method to switch Local Pairable mode .
    * \param   None
    * \retval  Result: 0 if Success else Non-zero
    ***************************************************************************/
   Result SwitchPairableMode(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::SwitchConnectableMode(void)
    ***************************************************************************/
   /*!
    * \fn      SwitchConnectableMode(void)
    * \brief   Method to switch Local Connectable mode .
    * \param   None
    * \retval  Result: 0 if Success else Non-zero
    ***************************************************************************/
   Result SwitchConnectableMode(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::InitiateAutoConnection(void)
    ***************************************************************************/
   /*!
    * \fn      InitiateAutoConnection(void)
    * \brief   Method to trigger event to AutoConnectionController to initiate
    *        AutoConnection sequence.
    * \param   None
    * \retval  Result: 0 if Success else Non-zero
    ***************************************************************************/
   Result InitiateAutoConnection(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::SetProtocolsToBeBlockedForHandsFreeProfile(void)
    ***************************************************************************/
   /*!
    * \fn      SetProtocolsToBeBlockedForHandsFreeProfile(void)
    * \brief   Method to set the list of profiles to be blocked while
    *          activating OOB HandFree for technologies like Android Auto or CarLife
    * \param   None
    * \retval  None
    ***************************************************************************/
   void SetProtocolsToBeBlockedForHandsFreeProfile(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::SetProtocolsToBeBlockedForManualHFP(void)
    ***************************************************************************/
   /*!
    * \fn      SetProtocolsToBeBlockedForManualHFP(void)
    * \brief   Method to set the list of profiles to be blocked while
    *          activating Manual HandFree for technologies like ML or mySPIN
    * \param   None
    * \retval  None
    ***************************************************************************/
   void SetProtocolsToBeBlockedForManualHFP(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::SetProtocolsToBeBlockedForCarPlay(void)
    ***************************************************************************/
   /*!
    * \fn      SetProtocolsToBeBlockedForCarPlay(void)
    * \brief   Method to set the list of profiles to be blocked while
    *          preparing/pre-activating CarPlay
    * \param   None
    * \retval  None
    ***************************************************************************/
   void SetProtocolsToBeBlockedForCarPlay(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::SetProtocolsToBeBlockedForCarPlayOnActivating(void)
    ***************************************************************************/
   /*!
    * \fn      SetProtocolsToBeBlockedForCarPlayOnActivating(void)
    * \brief   Method to set the list of profiles to be blocked while
    *          activating CarPlay
    * \param   None
    * \retval  None
    ***************************************************************************/
   void SetProtocolsToBeBlockedForCarPlayOnActivating(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::IsIncomingLimitationActionAllowed(...)
    ***************************************************************************/
   /*!
    * \fn      IsIncomingLimitationActionAllowed(const LimitationState CurrentLimitationState,
    *          const LimitationAction NewLimitationActionReq)
    * \brief   Method to check if the incoming Limitation Action is allowed in the
    *        current limitation state.
    * \param   CurrentLimitationState - [IN]Enum value of type LimitationState
    * \param   NewLimitationActionReq - [IN]Enum value of type LimitationAction
    * \retval  Result: 0 if Success else Non-zero
    ***************************************************************************/
   BmResult IsLimitationActionRequestAllowed(IN const LimitationState CurrentLimitationState,
         IN const LimitationAction NewLimitationActionReq);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::traceDeviceProtocolList(...)
    ***************************************************************************/
   /*!
    * \fn      traceDeviceProtocolList(IN const std::string& callingMethodName,
    *          IN const DeviceProtocolList& deviceProtocolList) const
    * \brief   Method to trace out the device protocol list.
    * \param   callingMethodName - [IN]String : Name of the calling method
    * \param   deviceProtocolList -[IN]List of elements of type DeviceProtocol class.
    * \retval  None
    ***************************************************************************/
   void traceDeviceProtocolList(IN const std::string& callingMethodName,
         IN const DeviceProtocolList& deviceProtocolList) const;

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::checkAvpBlockStateOfProjectionDevice(void)
    ***************************************************************************/
   /*!
    * \fn      checkAvpBlockStateOfProjectionDevice(void)
    * \brief   Method to check and store AVP profile block status of
    *        projection device before limitation mode is activated.
    * \param   None
    * \retval  None
    ***************************************************************************/
   void checkAvpBlockStateOfProjectionDevice(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::CheckIfDccIsAvailableForConnection(void)
    ***************************************************************************/
   /*!
    * \fn      CheckIfDccIsAvailableForConnection(void)
    * \brief   Method to check if DCC is already in use for projection device.
    *        If not in use, make one of the used DCC's available for device connection of HFP profile
    *        by disconnecting the device.
    * \param   None
    * \retval  Result: 0 if Success else Non-zero
    ***************************************************************************/
   Result CheckIfDccIsAvailableForConnection(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::getConflictTriggerFromCurrLimiMode(void)
    ***************************************************************************/
   /*!
    * \fn      getConflictTriggerFromCurrLimiMode(void)
    * \brief   Method to get Conflict Trigger from the current limitation mode
    * \param   None
    * \retval  ConflictTrigger: Enum value of type ConflictTrigger
    ***************************************************************************/
   ConflictTrigger getConflictTriggerFromCurrLimiMode(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::CPWReconnectTimerCb
    ***************************************************************************/
   /*!
    * \fn     CPWReconnectTimerCb
    * \brief  called on expiry of CPW Reconnect Timer
    * \param  rTimerID: ID of the timer which has expired
    * \param  pvObject: pointer to object passed while starting the timer
    * \param  pvUserData: data passed during start of the timer
    **************************************************************************/
   static bool CPWReconnectTimerCb(timer_t rTimerID, void *pvObject,
         const void *pvUserData);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::deactivateActionforCPW(void)
    ***************************************************************************/
   /*!
    * \fn      defaultActionforDeactivateCPW(void)
    * \brief   Method to handle default action for deactivate CPW
    * \param   None
    * \retval  None
    ***************************************************************************/
   void defaultActionforDeactivateCPW(const DeviceId& projectionDeviceBTDeviceId, const BdAddress& projectionDeviceBDAddress);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::reserveDccforCPW(void)
    ***************************************************************************/
   /*!
    * \fn      reserveDccforCPW(void)
    * \brief   Method to reserve the Dcc for CPW. Dcc will be released once the
    * CPW session is deactivated
    * \param   None
    * \retval  None
    ***************************************************************************/
   void reserveDccforCPW(void);

   /***************************************************************************
    ** FUNCTION:  BtLimitationController::getListOfProtocolsToBeBlocked(...)
    ***************************************************************************/
   /*!
    * \fn      getListOfProtocolsToBeBlocked(OUT DeviceProtocolListMap& protocolsToBeBlockedForDevices)
    * \brief   Method to get the list of protocols to be blocked for projection and
    * 		  non-projection devices from the list of all paired devices.
    * \param   protocolsToBeBlockedForDevices : Reference to std map of type DeviceProtocolListMap
    * \retval  None
    ***************************************************************************/
   void getListOfProtocolsToBeBlocked(OUT DeviceProtocolListMap& protocolsToBeBlockedForDevices);

   /***************************************************************************
    * ! Data members
    ***************************************************************************/

   /***************************************************************************
    ** Stores the BT address of the Projection Device
    ***************************************************************************/
   BdAddress _ProjectionDeviceBDAddress;

   /***************************************************************************
    ** Stores the BT Name of the Projection Device
    ***************************************************************************/
   BdName _ProjectionDeviceBDName;

   /***************************************************************************
    ** Current Active Limitation Mode(Communication Interface and Feature)
    ***************************************************************************/
   LimitationMode _CurrentLimitationMode;

   /***************************************************************************
    ** Current Active Limitation Action
    ***************************************************************************/
   LimitationAction _CurrentLimitationAction;

   /***************************************************************************
    ** Stores the BT Device ID of the Projection Device if already paired
    ***************************************************************************/
   DeviceId _ProjectionDeviceBTDeviceId;

   /***************************************************************************
    ** Stores the list<vector> of protocols to be blocked for Projection Device
    ***************************************************************************/
   ProtocolList _ProtocolsToBeBlockedForProjectionDevice;

   /***************************************************************************
    ** Stores the list<vector> of protocols to be blocked for Non-Projection Device
    ***************************************************************************/
   ProtocolList _ProtocolsToBeBlockedForNonProjectionDevices;

   /***************************************************************************
    ** BD Address of the device for which AVP is blocked
    ***************************************************************************/
   BdAddress _bdAddressOfAvpBlockedDevice;

   /***************************************************************************
    ** List which contains the BT Device ID and corresponding list of protocols.
    ** Contains objects of type DeviceProtocol class
    ***************************************************************************/
   DeviceProtocolList _deviceProtocolsWaitingToGetBlocked;

   /***************************************************************************
    ** Stores the timer ID for CPW Reconnection
    ***************************************************************************/
   timer_t _CPWReconnectionTimerID;

   /***************************************************************************
    ** Stores the timer instance for CPW Reconnection
    ***************************************************************************/
   Timer   _CPWReconnectionTimer;

   /***************************************************************************
    ** Lock for protecting the timer ID
    ***************************************************************************/
   Lock _LockCPWReconnectionTimer;

   /***************************************************************************
    ** Stores Device ID info to be disconnected
    ***************************************************************************/
   DeviceId _deviceIdWaitingForBeingDisconnected;

   /***************************************************************************
    ** Wifi lost device
    ***************************************************************************/
   bool _deviceLost;

   /***************************************************************************
    ** Stores the WBL operating frequency.
    ***************************************************************************/
   BTSWblWifiFrequency _wblOperatingFreq;
};
}//End of namespace bmcore

#endif // _BT_LIMITATION_CONTROLLER_H_

