/**
 * @defgroup BmControllerModule BM Controller
 * @ingroup BluetoothConnectionManagerCore
 * 
 * @brief This module comprises all parts related to connection controlling
 *
 * @details A detailed description is not yet available
 */

/**
 * @file BmController.h
 * 
 * @swcomponent BluetoothConnectionManagerCore
 *
 * @brief This file contains the declaration of the class BmController
 * 
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup BmControllerModule
 *
 */

#ifndef _BM_CONTROLLER_H_
#define _BM_CONTROLLER_H_

#include "BmAllTypes.h"
#include "ILocalSpm.h"
#include "BmControllerOnOffSm.h"

using namespace ::btstackif;

namespace bmcore
{
   class DeviceConnectionController;
   class ProtocolConnectionController;

   /**
    * @class BmController
    * @ingroup BmControllerModule
    * 
    * @brief The class BmController is managing all BT device and protocol connections.
    * It implements the dispatching of messages to the respective instances of the device connection
    * state machines.
    * 
    */
   class BmController : public BmControllerOnOffSm, public ILocalSpm
   {
   public:
      /**
       * @brief Constructor of class BmController which creates an instance
       *
       * No further initialization of internal components and resources
       * is performed in the constructor. See the explicit allocation
       * and initialization functions Create() and Init(). Hence, the
       * implementation is trivial.
       *
       * Synchronization: not re-entrant.
       *
       * Performance considerations: none.
       *
       * @return void
       */
      BmController(ComponentId componentID);

      /**
       * @brief Destructor of class BmController
       *
       * @return void
       */
      virtual ~BmController();

/*
      void updateLinkQuality(IN const DeviceIdList& connectedDevices, IN const DeviceIdList& previousConnectedDevices, IN const bool considerAllConnectedDevices);
*/
      void startLinkQualityRequestTimer(void);

      void cancelLinkQualityRequestTimer(void);

      void requestLinkQualityOnExpiredTimer(void);

      Result getUsedDeviceConnectionControllers(OUT DeviceConnectionControllerList& dccInstances);

      Result destroyConnectionControllers(void);

      Result onDeviceConnectionControllerReleased(IN const DeviceConnectionController* dccInstance);

      /**
       * @brief Gets the currently stored legacy pairing PIN
       *
       * Gets the legacy pairing PIN currently stored in the LegacyPin property.
       *
       * @return Legacy pairing PIN
       */

      inline bool isRequestQueueEmpty(void) const
      {
         return (_requestQueue.empty());
      }

      DeviceId getDeviceIdOfDeferredRequest(void) const;

      Result addConnectionRequestDataToQueue(IN const bool internal, IN const ConnectionRequestOrigin origin,
            IN const BdAddress& bdAddress, IN const ProtocolList& protocolList, IN const PageTimeout pageTimeout, IN const bool delayConnection = false);

      Result addConnectionRequestDataToQueue(IN const bool internal, IN const ConnectionRequestOrigin origin,
            IN const DeviceId deviceId, IN const ProtocolList& protocolList, IN const PageTimeout pageTimeout, IN const bool delayConnection = false);

      Result removeRequestDataFromQueue(void);

      Result cleanUpDeferredConnectionRequest(IN const ProtocolList& protocolList);

      BmResult handleRequest(IN const bool deferredRequest = false, IN const bool ignoreStopAutoconnection = false);

      inline void setSuppressAutoConnection(IN const bool suppressAutoConnection)
      {
         _suppressAutoConnection = suppressAutoConnection;
      }

      inline bool isAutoConnectionSuppressed(void) const
      {
         return _suppressAutoConnection;
      }

      // ***********************************
      // * BM Core interface handlers *
      // ***********************************

      BmResult doRequestTestModeLinkQuality(void);

      //BmResult doGetLinkQualityTestMode(OUT TestModeLinkQuality& linkQuality);

      Result getProtocolsForConnectionRestoring(OUT ProtocolList& protocolList, IN const DeviceId deviceId, bool ignoreProtocolsLastConnected);

      // *********************************************
      // * Overridden methods of BmControllerOnOffSm *
      // *********************************************

      Result messageNotConsumed();

      Result startAutoConnection(IN const AutoConnectionStartMode autoConnectionStartMode);

      Result stopAutoConnection(void);

      Result clearAutoConnectionBooking(void);

      Result isFirstStartUp();

      Result initializeBtStackIf();

      Result initializeProperties();

      Result restoreBtStatus();

      Result handleBmControllerActive();

      Result startAutoConnectionAttempt();

      Result disconnectAllDevices();

      Result initiateSwitchingBtOff();

      Result switchBtOff();

      Result stopAllConnectionControllers();

      Result areAllConnCtrlStopped();

      Result handleBmControllerStopped();

      Result terminateAllConnectionControllers();

      Result areAllConnCtrlTerminated();

      Result handleBmControllerTerminated();

      Result StopDeviceDiscovery();

      bool checkSupportedUUID(const DeviceId& deviceId, const ProtocolId& protocolId, const Uuid& uuid, const ConnectionRequestOrigin connectionRequestOrigin, bool isRequestInternal );

      Result getReleasableDeviceConnectionControllers(OUT DeviceConnectionControllerList& dccInstances,
            OUT DeviceId& sendBtLimitationDeactivateToDeviceId);

      BmResult disconnectProtocols(IN DeviceConnectionController* const& dccInstance, IN const ProtocolList& protocolList,
            IN const bool disconnectAll = false, IN const DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN,
            IN const bool deleteDevice = false, IN const bool ignoreStopAutoconnection = false);

      BmResult disconnectProtocols(IN const DeviceId deviceId, IN const ProtocolList& protocolList,
            IN const bool disconnectAll = false, IN const DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN,
            IN const bool deleteDevice = false, IN const bool ignoreStopAutoconnection = false);

      // ************************************
      // * Overridden methods of ILocalSpm  *
      // ************************************

      /**
       * @brief Performs an explicit allocation of the resources used by
       * the BmController instance internally
       *
       * Synchronization: not re-entrant
       *
       * Performance considerations: none
       *
       * @return void
       */
      void create();

      /**
       * @brief Performs the initialization of the BmController instance and its
       * subcomponents, i.e.:
       * - initialization of member variables
       *
       * Synchronization: re-entrant
       *
       * Performance considerations:
       *
       * @return 0 on success and an error code otherwise.
       */
      Result init(InitReason reason);

      /**
       * @brief Starts the BmController instance, implies that all other
       * components are available
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none.
       *
       * @return 0 on success and an error code otherwise
       */
      Result run();

      /**
       * @brief Stops the BmController instance
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none
       *
       * @return 0 on success and an error code otherwise
       */
      Result stop();

      /**
       * @brief Cleans up the resources used by the BmController instance
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none
       *
       * @return 0 on success and an error code otherwise
       */
      Result done();

      /**
       * @brief Returns statistic data as a string for the over all object statistics
       *
       * @param[out] stat string with statistics information to be printed
       *
       * @return 0 on success and an error code otherwise
       */
      int statistics(OUT Statistics stat);


   private:

      typedef std::vector<ProtocolConnectionController*> ProtocolConnectionControllerList;

      Result initialize();

      Result createProtocolConnectionControllers(void);

      Result initProtocolConnectionControllers(void);

      Result runProtocolConnectionControllers(void);

      Result stopProtocolConnectionControllers(void);

      Result terminateProtocolConnectionControllers(void);

      Result destroyProtocolConnectionControllers(void);

      Result createDeviceConnectionControllers(void);

      Result initDeviceConnectionControllers(void);

      Result runDeviceConnectionControllers(void);

      Result stopDeviceConnectionControllers(void);

      Result terminateDeviceConnectionControllers(void);

      Result destroyDeviceConnectionControllers(void);

      Result getUnusedDeviceConnectionController(OUT DeviceConnectionController** dcc);

      Result assignDeviceConnectionController(OUT DeviceConnectionController** dcc, IN const DeviceId deviceId,
            IN const ProtocolList& protocolList, IN const ConnectionRequestOrigin origin);

      Result assignDeviceConnectionController(OUT DeviceConnectionController** dcc, IN const BdAddress& bdAddress,
            IN const ProtocolList& protocolList, IN const ConnectionRequestOrigin origin);

      Result handleConnectionRequest(IN const bool deferredRequest = false);

      static bool linkQualityTimerCb(timer_t timerId , void *object, const void *userData);

      DeviceConnectionController* getNewDeviceConnectionController(void);

      RequestData* getNewConnectionRequestData(IN const bool internal, IN const ConnectionRequestOrigin origin,
            IN const DeviceId deviceId, IN const ProtocolList& protocolList, IN const PageTimeout pageTimeout, IN const bool delayConnection);

      BmResult releaseDccInstance(IN ConnectionRequestData* connectionRequestData, IN DeviceConnectionController* dccInstance);

      void checkAndReleaseReservedDCC();

      bool _firstStartUp;

      DeviceConnectionControllerList      _deviceConnectionControllers;
      unsigned int                        _numberConnCtrlStopping;

      DeviceConnectionControllerList      _dccInstancesToBeReleased;

      RequestDataList _requestQueue;

      bool _autoConnectionBooked;
      bool _suppressAutoConnection;

      unsigned int _linkQualityRequestTimeOut;
      Timer _linkQualityTimer;
      timer_t _linkQualityTimerId;
   };
}

#endif // _BM_CONTROLLER_H_

