/**
 * @file AutoConnectionController.cpp
 *
 * @par SW-Component
 * Bluetooth Connection Manager Core
 *
 * @brief This file contains the definition of the class AutoConnectionController
 *
 * @copyright (c) 2016 Robert Bosch GmbH
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup AutoConnectionControllerModule
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_fw.h"

#include "BmTraceClasses.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_AUTO_CONNECTION_CONTROLLER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/AutoConnectionController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BM_CORE_AUTO_CONNECTION_CONTROLLER
#endif
#endif

#include "LocalSpm.h"
#include "FunctionTracer.h"
#include "Dispatcher.h"
#include "BmVarTrace.h"
#include "AutoConnectionController.h"
#include "BmGlobalLock.h"
#include "BmCoreIfMessagesCreator.h"

namespace bmcore
{
   AutoConnectionController::AutoConnectionController(const ComponentId componentId) :
         ILocalSpm(componentId), _deviceProtocolsWaitingToGetConnected(), _deviceProtocolConnectionInProgress(),
         _atLeastOneConnAttemptInitiated(false),
         _standardAutoConnectionType(BM_STANDARD_AUTO_CONNECTION_UNKNOWN), _cpwDeviceId(0u),
         //!Fix for CID 123605: Uninitialized scalar field (UNINIT_CTOR)
         _lastPanConnectedDevice(0u),
         _panConnectionTriggered(false),
         _lastIntendedModeList(),
         _wblServiceIsAvailable(false),
         _verifiedWBLStatus(false),
         _firstUpdate(false),
         _pageTimeout(0u),
         _waitforWBLInfoTimer(),
         _waitforWBLInfoTimerId(0),
         _delayConnectionTimer(),
         _delayConnectionTimerId(),
         _delayAutoConnectionTimer(),
         _delayAutoConnectionTimerId(0),
         _disconnectedReasonDelayTimer(),
         _disconnectedReasonDelayTimerId(0),
         _expiredTimerSmEvent(this->GetSMNameFull()),
         _ignoreDeviceUsagePreference(false),
         _recreateOrderedDeviceList(false),
         _delayAutoConnection(false),
         _deviceProtocolConnectionInProgressLock()
   {
      ENTRY_INTERNAL

      ETG_TRACE_USR1(("AutoConnectionController: is being created"));

      // threadless SM
      disableAllLocks(true);

      /* Early initialization of variables */
      initMembers();
   }

   AutoConnectionController::~AutoConnectionController()
   {
      ENTRY_INTERNAL

      _waitforWBLInfoTimerId = 0;
      _delayConnectionTimerId = 0;
      _delayAutoConnectionTimerId = 0;
      _disconnectedReasonDelayTimerId = 0;
   }

   DeviceId AutoConnectionController::getDeviceIdConnectionInProgress(void)
   {
      ENTRY

      ETG_TRACE_USR1(("getDeviceIdConnectionInProgress"));

      Locker locker(&_deviceProtocolConnectionInProgressLock);

      return _deviceProtocolConnectionInProgress._deviceId;
   }

   ProtocolList AutoConnectionController::getDeviceProtocolListConnectionInProgress(void)
   {

      ETG_TRACE_USR1(("getDeviceIdConnectionInProgress : entered"));

      Locker locker(&_deviceProtocolConnectionInProgressLock);

      return _deviceProtocolConnectionInProgress._protocolList;
   }

   PageTimeout AutoConnectionController::getPageTimeout(void)
   {

      ETG_TRACE_USR1(("getPageTimeout : entered"));

      Locker locker(&_deviceProtocolConnectionInProgressLock);

      return _pageTimeout;
   }

   Result AutoConnectionController::initSm()
   {
      ENTRY

      ETG_TRACE_USR1(("initSm"));

      return static_cast<Result>(CC_ERR_INT_NO_ERROR);
   }

   Result AutoConnectionController::handleStopSm()
   {
      ENTRY

      ETG_TRACE_USR1(("handleStopSm"));

      /* Send stop done to SPM in the transition to final state in state machine */
      return stopDone(0);
   }

   Result AutoConnectionController::handleDone()
   {
      ENTRY

      ETG_TRACE_USR1(("handleDone"));

      /* Send done done to SPM in the transition to final state in state machine */
      return doneDone(0);

   }

   Result AutoConnectionController::enterIdle()
   {
      ENTRY

      ETG_TRACE_USR1(("enterIdle"));

      initMembers();

      _pageTimeout = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultConnectionPageTimeoutMilliSeconds;

      _ignoreDeviceUsagePreference = false;
      _recreateOrderedDeviceList = false;
      _delayAutoConnection = false;

      LocalSpm::getBmCoreMainController().setAutoConnectionStatusInt(BM_AUTO_CONNECTION_STATE_IDLE, 0u);

      return CC_ERR_INT_NO_ERROR;
   }

   Result AutoConnectionController::createOrderedDeviceList(IN const StandardAutoConnectionType standardAutoConnectionType)
   {
      ENTRY

      ETG_TRACE_USR1(("createOrderedDeviceList: standardAutoConnectionType = %d",
            ETG_CENUM(StandardAutoConnectionType, standardAutoConnectionType)));

      Result result(CC_ERR_INT_NO_ERROR);
      std::string methodName("createOrderedDeviceList");

      // Reset the member variables
      initMembers();

      _pageTimeout = LocalSpm::getDataProvider().getBmCoreConfiguration()._defaultConnectionPageTimeoutMilliSeconds;

      _recreateOrderedDeviceList = false;

      // Set the Auto connection type
      _standardAutoConnectionType = standardAutoConnectionType;

      ETG_TRACE_USR3(("createOrderedDeviceList: ignoreDeviceUsagePreference - %10s", _ignoreDeviceUsagePreference ? "true": "false"));

      // Set the Auto Connection status property to PREPARING
      LocalSpm::getBmCoreMainController().setAutoConnectionStatusInt(BM_AUTO_CONNECTION_STATE_PREPARING, 0u);

      ETG_TRACE_USR3(("createOrderedDeviceList: _carPlayWirelessSupported - %d",
            LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported));

      //step - 1: Added the Last connected CPW device in the deviceProtocolsWaitingToGetConnected list if available.
      // Check any BTLimitation mode - CPW is available or not. If available then ignore the CPW/TEMP_CLASSIC_BT usage preference devices
      if ((true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported) && (false == _ignoreDeviceUsagePreference))
      {
         if(false == checkCPWLimitationModeIsAvailable())
         {
            checkLastConnectedCPWDevice();
         }
         else
         {
            ETG_TRACE_USR3(("createOrderedDeviceList: BTLimitation mode is already available for CPW so ignore the last connected CPW devices"));
         }
      }
      else // i.e. true != LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported
      {
         ETG_TRACE_USR3(("createOrderedDeviceList: project not support carPlayWireless or ignore device usage preference is set"));
      }

      //step - 2: Added the connected devices in the deviceProtocolsWaitingToGetConnected list if available and
      // also added the disconnected devices in separate disconnectedDevices/disconnectedDevicesNoProfilesExpectedToBeConnected list
      DeviceIdList devicesLastConnected;
      result = LocalSpm::getDbManager().getDevicesLastConnected(devicesLastConnected);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         DeviceProtocolList disconnectedDevices;
         DeviceProtocolList disconnectedDevicesNoProfilesExpectedToBeConnected;
         ConnectionStatus connectionStatus;
         DisconnectedReason disconnectedReason;

         // Get the Device connection protocols from db. If no protocols are expected to be connected then these
         // protocols will be used for device connection
         ProtocolIdList deviceConnectionProtocols;
         result = LocalSpm::getDataProvider().getDeviceConnectionProtocols(deviceConnectionProtocols);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("createOrderedDeviceList: could not get device connection protocols from DataProvider (error = %d) ",
                  ETG_CENUM(CcErrorInternal, result)));
         }

         for (size_t devIdx = 0u; devIdx < devicesLastConnected.size(); devIdx++)
         {
            // Get the Device connection status from db. Based on the Connection status devices are added in the corresponding list
            result = LocalSpm::getDbManager().getDeviceConnectionStatus(connectionStatus, disconnectedReason, devicesLastConnected[devIdx]);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               DeviceProtocol deviceProtocol;
               deviceProtocol._deviceId = devicesLastConnected[devIdx];
               deviceProtocol._protocolList.clear();

               if (BM_CONNECTION_STATUS_DISCONNECTED == connectionStatus)
               {
                  if(true == checkDeviceTobeAddedList(deviceProtocol))
                  {
                     if (true == deviceProtocol._protocolList.empty())
                     {
                        ETG_TRACE_USR1(("createOrderedDeviceList: list of protocols expected to be connected for device with ID = %d is empty -> assuming all device protocols as to be connected",
                              deviceProtocol._deviceId));

                        for (size_t i = 0u; i < deviceConnectionProtocols.size(); ++i)
                        {
                           deviceProtocol._protocolList.push_back(Protocol(deviceConnectionProtocols[i], ""));
                        }

                        disconnectedDevicesNoProfilesExpectedToBeConnected.push_back(deviceProtocol);
                     }
                     else
                     {
                        disconnectedDevices.push_back(deviceProtocol);
                     }

                     if(checkPANProtcolInList(deviceProtocol._protocolList))
                     {
                        _lastPanConnectedDevice = deviceProtocol._deviceId;
                     }
                  }
               }
               else // BM_CONNECTION_STATUS_DISCONNECTED != connectionStatus
               {
                  if(true == checkDeviceTobeAddedList(deviceProtocol))
                  {
                     if (false == deviceProtocol._protocolList.empty())
                     {
                        _deviceProtocolsWaitingToGetConnected.push_back(deviceProtocol);

                        if(checkPANProtcolInList(deviceProtocol._protocolList))
                        {
                           _lastPanConnectedDevice = deviceProtocol._deviceId;
                        }
                     }
                     else
                     {
                        ETG_TRACE_USR1(("createOrderedDeviceList: list of protocols expected to be connected for device with ID = %d is empty -> ignoring device",
                              deviceProtocol._deviceId));
                     }
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR(("createOrderedDeviceList: could not get device connection status of device with ID = %d from DB (error = %d)",
                     devicesLastConnected[devIdx], ETG_CENUM(CcErrorInternal, result)));
            }
         }

         ETG_TRACE_USR3(("createOrderedDeviceList: ordered list of devices that are not disconnected:"));
         this->traceDeviceProtocolList(methodName + ": _deviceProtocolsWaitingToGetConnected =", _deviceProtocolsWaitingToGetConnected);

         //step - 3: Added the favorites devices in the deviceProtocolsWaitingToGetConnected list if available
         checkingFavoriteProtocols();

         //step - 3: Added the Primary HFP devices in the deviceProtocolsWaitingToGetConnected list if available
         checkingPrimaryHFPDevice();

         //step - 4: Added the disconnectedDevices list in the deviceProtocolsWaitingToGetConnected list if available
         ETG_TRACE_USR3(("createOrderedDeviceList: ordered list of devices that are disconnected:"));
         this->traceDeviceProtocolList(methodName + ": disconnectedDevices =", disconnectedDevices);

         _deviceProtocolsWaitingToGetConnected.insert(_deviceProtocolsWaitingToGetConnected.end(), disconnectedDevices.begin(), disconnectedDevices.end());

         //step - 5: Added the disconnectedDevicesNoProfilesExpectedToBeConnected list in the deviceProtocolsWaitingToGetConnected list if available
         ETG_TRACE_USR3(("createOrderedDeviceList: ordered list of devices that are disconnected, but no protocol expected to be connected:"));
         this->traceDeviceProtocolList(methodName + ": disconnectedDevicesNoProfilesExpectedToBeConnected =", disconnectedDevicesNoProfilesExpectedToBeConnected);

         _deviceProtocolsWaitingToGetConnected.insert(_deviceProtocolsWaitingToGetConnected.end(),
               disconnectedDevicesNoProfilesExpectedToBeConnected.begin(),
               disconnectedDevicesNoProfilesExpectedToBeConnected.end());
      }
      else
      {
         ETG_TRACE_ERR(("createOrderedDeviceList: could not get list of last connected devices from DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      ETG_TRACE_USR1(("createOrderedDeviceList: sorted list of devices considered for auto connection run:"));
      this->traceDeviceProtocolList(methodName + ": _deviceProtocolsWaitingToGetConnected =", _deviceProtocolsWaitingToGetConnected);

      return CC_ERR_INT_NO_ERROR;
   }

   bool AutoConnectionController::checkCPWLimitationModeIsAvailable()
   {
      bool available = false;

      BtLimitationMode btLimitationMode;

      // Get the BTLimitation mode
      LocalSpm::getBmCoreMainController().getBtLimitationMode(btLimitationMode);

      for(size_t index = 0; index < btLimitationMode._btLimitationModeInfoList.size(); index++)
      {
         if((BM_LIMITATION_COMMUNICATION_IF_WIFI_2_4 == btLimitationMode._btLimitationModeInfoList[index]._limitationMode._limitationCommunicationIf)
               ||(BM_LIMITATION_COMMUNICATION_IF_WIFI_5 == btLimitationMode._btLimitationModeInfoList[index]._limitationMode._limitationCommunicationIf))
         {
            if(BM_LIMITATION_FEATURE_CAR_PLAY == btLimitationMode._btLimitationModeInfoList[index]._limitationMode._limitationFeature)
            {
               available = true;
               break;
            }
         }
      }

      ETG_TRACE_USR3(("checkCPWLimitationModeIsAvailable: available -  %10s", available ? "true": "false"));

      return available;
   }

   bool AutoConnectionController::checkPANProtcolInList(ProtocolList& protocolList)
   {
      bool available = false;

      // Check if PAN protocol is available for the requested protocolList. If available then return true else false.
      for (size_t idxGiven = 0u; idxGiven < protocolList.size(); ++idxGiven)
      {
         if(BM_PROTOCOL_ID_PAN == protocolList[idxGiven]._protocolId)
         {
            available = true;
            break;
         }
      }

      ETG_TRACE_USR1(("checkPANProtcolInList: available = %d", available));

      return available;
   }

   bool AutoConnectionController::checkDeviceTobeAddedList(DeviceProtocol& deviceProtocol)
   {
      Result result(CC_ERR_INT_NO_ERROR);
      bool deviceTobeAdded = false;

      ETG_TRACE_USR1(("checkDeviceTobeAddedList: deviceId - %d", deviceProtocol._deviceId));

      //Ignore the device if the disconnection Reason of a device is MISSING_LINK_KEY
      result = checkDeviceDisconnectedReason(deviceProtocol._deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         // Get the Protocols to be connected for the device
         result = LocalSpm::getBmController().getProtocolsForConnectionRestoring(deviceProtocol._protocolList, deviceProtocol._deviceId, true);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            if ((true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported) && (false == _ignoreDeviceUsagePreference))
            {
               // check the device is already added in the list for CPW(LastConnectedCPWdevice)
               if (_cpwDeviceId != deviceProtocol._deviceId)
               {
                  result = LocalSpm::getDbManager().getDeviceUsagePreference(deviceProtocol._usagePreference, deviceProtocol._deviceId);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     deviceTobeAdded = true;
                     ETG_TRACE_USR1(("checkDeviceTobeAddedList: device is added in the list"));
                  }
                  else
                  {
                     ETG_TRACE_ERR(("checkDeviceTobeAddedList: could not get usage preference for device with ID = %d in DB (error = %d)",
                           deviceProtocol._deviceId, ETG_CENUM(CcErrorInternal, result)));
                  }
               }
               else
               {
                  ETG_TRACE_USR3(("checkDeviceTobeAddedList: CPW device with ID = %d has been already added to the list",
                        deviceProtocol._deviceId));
               }
            }
            else // i.e. true != LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported
            {
               deviceProtocol._usagePreference = BM_UP_CLASSIC_BT;
               deviceTobeAdded = true;
               ETG_TRACE_USR1(("checkDeviceTobeAddedList: device is added in the list"));
            }
         }
      }

      return deviceTobeAdded;
   }

   Result AutoConnectionController::checkDeviceDisconnectedReason(IN DeviceId deviceId)
   {
      Result result(CC_ERR_INT_NO_ERROR);
      DisconnectedReason disconnectedReason;

      ETG_TRACE_USR3(("checkDeviceDisconnectedReason: device = %d", deviceId));

      result = LocalSpm::getDbManager().getDeviceDisconnectedReason(disconnectedReason, deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         ETG_TRACE_USR3(("checkDeviceDisconnectedReason(deviceId = %d): disconnectedReason = %d",
               deviceId, ETG_CENUM(DisconnectedReason, disconnectedReason)));

         if(BM_DISCONNECTED_REASON_MISSING_LINK_KEY == disconnectedReason)
         {
            ETG_TRACE_USR3(("checkDeviceDisconnectedReason(deviceId = %d): disconnectedReason of the device is = %d so ignore this device for autoconnection",
                  deviceId, ETG_CENUM(DisconnectedReason, disconnectedReason)));
            result = CC_ERR_INT_NOT_ALLOWED;
         }
      }
      else
      {
         ETG_TRACE_ERR(("checkDeviceDisconnectedReason(deviceId = %d): could not get disconnected reason for device in DB (error = %d)",
               deviceId, ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   void AutoConnectionController::checkLastConnectedCPWDevice()
   {
      Result result(CC_ERR_INT_NO_ERROR);
      std::string methodName("checkingLastConnectedCPWDevice");

      ETG_TRACE_USR3(("checkingLastConnectedCPWDevice: ordered list of devices that are CPW:"));

      _cpwDeviceId = 0u;

      DeviceId deviceId = 0;

      //Get the Last connected CPW device from db
      result = LocalSpm::getDbManager().getCPWDevice(deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         //Check the deviceId is non-zero. If non-zero then device is connected for CPW last time
         if (0u != deviceId)
         {
            ETG_TRACE_USR3(("checkLastConnectedCPWDevice: CPW last connected device id - %d", deviceId));

            DeviceProtocol deviceProtocol;
            deviceProtocol._deviceId = deviceId;
            deviceProtocol._protocolList.clear();

            if(true == checkDeviceTobeAddedList(deviceProtocol))
            {
               if((BM_UP_CPW == deviceProtocol._usagePreference) || (BM_UP_TEMP_CLASSIC_BT == deviceProtocol._usagePreference))
               {
                  if (true == deviceProtocol._protocolList.empty())
                  {
                     ETG_TRACE_USR1(("checkLastConnectedCPWDevice: list of protocols expected to be connected for device with ID = %d is empty -> assuming all device protocols as to be connected",
                           deviceProtocol._deviceId));

                     // Get the Device connection protocols from db. If no protocols are expected to be connected then these
                     // protocols will be used for device connection
                     ProtocolIdList deviceConnectionProtocols;
                     result = LocalSpm::getDataProvider().getDeviceConnectionProtocols(deviceConnectionProtocols);

                     if (CC_ERR_INT_NO_ERROR != result)
                     {
                        ETG_TRACE_ERR(("checkLastConnectedCPWDevice: could not get device connection protocols from DataProvider (error = %d) ",
                              ETG_CENUM(CcErrorInternal, result)));
                     }
                     else
                     {
                        for (size_t i = 0u; i < deviceConnectionProtocols.size(); ++i)
                        {
                           deviceProtocol._protocolList.push_back(Protocol(deviceConnectionProtocols[i], ""));
                        }
                     }
                  }

                  _cpwDeviceId = deviceId;

                  _deviceProtocolsWaitingToGetConnected.push_back(deviceProtocol);

                  this->traceDeviceProtocolList(methodName + ": _deviceProtocolsWaitingToGetConnected =", _deviceProtocolsWaitingToGetConnected);
               }
               else
               {
                  ETG_TRACE_USR3(("checkLastConnectedCPWDevice: Usage Preference is not set as CPW/TEMP_CLASSIC_BT for CPW device - %d so not added in the list",
                        deviceProtocol._deviceId));
                  // Todo: Need to check to reset the CPW flag in db if the usage preference is not CPW/TEMP_CLASSIC_BT
               }
            }

         } //!End of if (0u != deviceId)
         else
         {
            ETG_TRACE_USR3(("checkLastConnectedCPWDevice: no CPW device is connected"));
         }
      } //!End of if (CC_ERR_INT_NO_ERROR == result)
      else
      {
         ETG_TRACE_ERR(("checkLastConnectedCPWDevice: could not get CPW last connected device in DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void AutoConnectionController::checkingFavoriteProtocols()
   {
      Result result(CC_ERR_INT_NO_ERROR);
      std::string methodName("checkingFavoriteProtocols");

      DeviceIdList favoriteHFPDeviceIds, favoriteAVPDeviceIds;

      favoriteHFPDeviceIds.clear();
      favoriteAVPDeviceIds.clear();

      ETG_TRACE_USR3(("checkingFavoriteProtocol: ordered list of devices that are HFP favorite"));

      // Get the favorite devices
      result = LocalSpm::getDbManager().getDevicesSortedByFavorite(favoriteHFPDeviceIds, BM_FAVORITE_TYPE_PROTOCOL_HFP);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         if (false == favoriteHFPDeviceIds.empty())
         {
            for (size_t i = 0u; i < favoriteHFPDeviceIds.size(); i++)
            {
               DeviceProtocol deviceProtocol;
               deviceProtocol._deviceId = favoriteHFPDeviceIds[i];

               deviceProtocol._protocolList.clear();
               Protocol protocol;
               protocol._protocolId = BM_PROTOCOL_ID_HFP;
               protocol._uuid = "";
               deviceProtocol._protocolList.push_back(protocol);

               result = checkDeviceDisconnectedReason(deviceProtocol._deviceId);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  deviceProtocol._usagePreference = BM_UP_CLASSIC_BT;
                  _deviceProtocolsWaitingToGetConnected.push_back(deviceProtocol);
               }
            }

            this->traceDeviceProtocolList(methodName + ": _deviceProtocolsWaitingToGetConnected =", _deviceProtocolsWaitingToGetConnected);

         } //!End of if (false == favoriteHFPDeviceIds.empty())
      }
      else
      {
         ETG_TRACE_ERR(("checkingFavoriteProtocols: could not get HFP favorites in DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      ETG_TRACE_USR3(("checkingFavoriteProtocols: ordered list of devices that are AVP favorite"));

      result = LocalSpm::getDbManager().getDevicesSortedByFavorite(favoriteAVPDeviceIds, BM_FAVORITE_TYPE_PROTOCOL_AVP);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         if (false == favoriteAVPDeviceIds.empty())
         {
            for (size_t i = 0u; i < favoriteAVPDeviceIds.size(); i++)
            {
               DeviceProtocol deviceProtocol;
               deviceProtocol._deviceId = favoriteAVPDeviceIds[i];

               deviceProtocol._protocolList.clear();
               Protocol protocol;
               protocol._protocolId = BM_PROTOCOL_ID_AVP;
               protocol._uuid = "";
               deviceProtocol._protocolList.push_back(protocol);

               result = checkDeviceDisconnectedReason(deviceProtocol._deviceId);

               if (CC_ERR_INT_NO_ERROR == result)
               {
                  deviceProtocol._usagePreference = BM_UP_CLASSIC_BT;
                  _deviceProtocolsWaitingToGetConnected.push_back(deviceProtocol);
               }
            }

            this->traceDeviceProtocolList(methodName + ": _deviceProtocolsWaitingToGetConnected =", _deviceProtocolsWaitingToGetConnected);
         }
      }
      else
      {
         ETG_TRACE_ERR(("checkingFavoriteProtocols: could not get AVP favorites in DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }
   }

   void AutoConnectionController::checkingPrimaryHFPDevice()
   {
      Result result(CC_ERR_INT_NO_ERROR);
      std::string methodName("checkingPrimaryHFPDevice");

      ETG_TRACE_USR3(("checkingPrimaryHFPDevice: ordered list of devices that are primary HFP Device"));

      DeviceId deviceId = 0;
      result = LocalSpm::getDbManager().getMasterHFPDevice(deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         if(0u != deviceId)
         {
            ETG_TRACE_USR3(("checkingPrimaryHFPDevice:  primary HFP Device Id - %d", deviceId));

            LastExpectedConnectionStatus lastExpectedConnectionStatus;
            Uuid uuid = "";

            result = LocalSpm::getDbManager().getLastExpectedProtocolConnectionStatus(lastExpectedConnectionStatus,
                  deviceId, BM_PROTOCOL_ID_HFP, uuid);

            if (CC_ERR_INT_NO_ERROR == result)
            {
               ETG_TRACE_USR3(("checkingPrimaryHFPDevice: last Expected Protocol ConnectionStatus of primary HFP Device Id - %d is %d",
                     deviceId, ETG_CENUM(LastExpectedConnectionStatus, lastExpectedConnectionStatus)));

               if(BM_LAST_EXPECTED_CONNECTION_STATUS_CONNECTED == lastExpectedConnectionStatus)
               {
                  DeviceProtocol deviceProtocol;
                  deviceProtocol._deviceId = deviceId;

                  deviceProtocol._protocolList.clear();
                  Protocol protocol;
                  protocol._protocolId = BM_PROTOCOL_ID_HFP;
                  protocol._uuid = "";
                  deviceProtocol._protocolList.push_back(protocol);

                  result = checkDeviceDisconnectedReason(deviceProtocol._deviceId);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     if((true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported) && (false == _ignoreDeviceUsagePreference))
                     {
                        result = LocalSpm::getDbManager().getDeviceUsagePreference(deviceProtocol._usagePreference, deviceProtocol._deviceId);

                        if (CC_ERR_INT_NO_ERROR == result)
                        {
                           ETG_TRACE_USR3(("checkingPrimaryHFPDevice: Usage Preference of primary HFP Device Id - %d is %d",
                                 deviceId, ETG_CENUM(UsagePreference, deviceProtocol._usagePreference)));

                           if((BM_UP_CPW != deviceProtocol._usagePreference) && (BM_UP_TEMP_CLASSIC_BT != deviceProtocol._usagePreference))
                           {
                              _deviceProtocolsWaitingToGetConnected.push_back(deviceProtocol);

                              this->traceDeviceProtocolList(methodName + ": _deviceProtocolsWaitingToGetConnected =", _deviceProtocolsWaitingToGetConnected);
                           }
                           else
                           {
                              ETG_TRACE_USR3(("checkingPrimaryHFPDevice: Usage Preference set as CPW/TEMP_CLASSIC_BT for HFP primary device - %d so not added in the list",
                                    deviceProtocol._deviceId));
                           }
                        }
                        else
                        {
                           ETG_TRACE_ERR(("checkingPrimaryHFPDevice: could not get usage preference for device with ID = %d in DB (error = %d)",
                                 deviceProtocol._deviceId, ETG_CENUM(CcErrorInternal, result)));
                        }
                     }
                     else
                     {
                        deviceProtocol._usagePreference = BM_UP_CLASSIC_BT;
                        _deviceProtocolsWaitingToGetConnected.push_back(deviceProtocol);

                        this->traceDeviceProtocolList(methodName + ": _deviceProtocolsWaitingToGetConnected =", _deviceProtocolsWaitingToGetConnected);
                     }
                  }
               }
               else
               {
                  ETG_TRACE_USR3(("checkingPrimaryHFPDevice:HFP is not the expected protocol for primary device - %d so not added in the list",
                        deviceId));
               }
            }
            else
            {
               ETG_TRACE_ERR(("checkingPrimaryHFPDevice: getProtocolsExpectedToBeConnected(deviceId = %d): Get failed (error = %d)",
                     deviceId, ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_USR3(("checkingPrimaryHFPDevice: no primary HFP Device is available"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("checkingPrimaryHFPDevice: could not get primary HFP device in DB (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }
   }

   Result AutoConnectionController::enableAutoConnectionLite()
   {
      ETG_TRACE_USR1(("enableAutoConnectionLite: delayAutoConnectionnAtEndOfBtLimitationInSec - %d",
            LocalSpm::getDataProvider().getBmCoreConfiguration()._delayAutoConnectionAtEndOfBtLimitationInSec));

      _ignoreDeviceUsagePreference = true;
      _recreateOrderedDeviceList = true;

      if(0u != LocalSpm::getDataProvider().getBmCoreConfiguration()._delayAutoConnectionAtEndOfBtLimitationInSec)
      {
         _delayAutoConnection = true;
      }
      else
      {
         _delayAutoConnection = false;
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result AutoConnectionController::checkAndRecreateOrderedDeviceList()
   {
      ETG_TRACE_USR1(("checkAndRecreateOrderedDeviceList: _recreateOrderedDeviceList = %d", _recreateOrderedDeviceList));

      if(true == _recreateOrderedDeviceList)
      {
         (void) createOrderedDeviceList(_standardAutoConnectionType);
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result AutoConnectionController::checkDelayAutoConnection()
   {
      Result result(CC_ERR_INT_NO_ERROR);

      ETG_TRACE_USR1(("checkDelayAutoConnection: _delayAutoConnection - %10s", _delayAutoConnection ? "true": "false"));

      if (false == _deviceProtocolsWaitingToGetConnected.empty())
      {
         // check for delay the autoconnection. if the _delayAutoConnection is true then start the timer for 5 sec once the timer is expired
         // then send the AUTO_CONNECT event else send AUTO_CONNECT event directly.
         if (true == _delayAutoConnection)
         {
            _delayAutoConnection = false;

            _expiredTimerSmEvent = this->GetSMNameFull();
            _expiredTimerSmEvent += "::DELAY_AUTO_CONNECTION_TIMER_CB";

            if (true == _delayAutoConnectionTimer.StartSMTimer(_delayAutoConnectionTimerId,
                  1000L * static_cast<long>(LocalSpm::getDataProvider().getBmCoreConfiguration()._delayAutoConnectionAtEndOfBtLimitationInSec), 0L, _expiredTimerSmEvent.c_str(), this))
            {
               char milliSecondsStr[200];
               snprintf(milliSecondsStr, 199, "%llu", 1000L * static_cast<long long unsigned>(LocalSpm::getDataProvider().getBmCoreConfiguration()._delayAutoConnectionAtEndOfBtLimitationInSec));
               ETG_TRACE_USR4(("checkDelayAutoConnection: started delay connection retry timer (timer ID = 0x%p, intervalMilliseconds = %50s)",
                     _delayAutoConnectionTimerId, milliSecondsStr));
            }
            else
            {
               ETG_TRACE_ERR(("checkDelayAutoConnection: starting delay connection timer failed"));

               result = this->SendEventByName("AUTO_CONNECT", 0);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("checkDelayAutoConnection: could not send event AUTO_CONNECT (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }
         }
         else
         {
            ETG_TRACE_USR4(("checkDelayAutoConnection: sending event AUTO_CONNECT to myself"));

            result = this->SendEventByName("AUTO_CONNECT", 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("checkDelayAutoConnection: could not send event AUTO_CONNECT (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }
      else
      {
         ETG_TRACE_USR4(("checkDelayAutoConnection: no devices left to be connected so sending event STOP_AUTO_CONNECTION to myself"));

         result = this->SendEventByName("STOP_AUTO_CONNECTION", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("checkDelayAutoConnection: could not send event STOP_AUTO_CONNECTION (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result AutoConnectionController::handleDelayAutoConnectionTimerCb()
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      ETG_TRACE_USR1(("handleDelayAutoConnectionTimerCb: entered"));

      // Cancel the timer and send the auto_connect event
      if(_delayAutoConnectionTimerId != 0)
      {
         _delayAutoConnectionTimer.CancelTimer(_delayAutoConnectionTimerId);

         _delayAutoConnectionTimerId = 0;
      }

      _expiredTimerSmEvent = this->GetSMNameFull();

      ETG_TRACE_USR4(("handleDelayAutoConnectionTimerCb: sending event AUTO_CONNECT to myself"));

      result = this->SendEventByName("AUTO_CONNECT", 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleDelayAutoConnectionTimerCb: could not send event AUTO_CONNECT (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result AutoConnectionController::checkWblHealthIndicator()
   {
      PageTimeout pausePagingTimeoutMilliSeconds = 0u;
      PageTimeout pageTimeout = 0u;
      Result result(CC_ERR_INT_NO_ERROR);
      bool sendEvent = true;

      ETG_TRACE_USR1(("checkWblHealthIndicator: entered"));

      if (false == _deviceProtocolsWaitingToGetConnected.empty())
      {
         result = LocalSpm::getBmCoreMainController().getProfileConnectionPageTimeout(pausePagingTimeoutMilliSeconds, pageTimeout);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            _pageTimeout = pageTimeout;

            if(0u < pausePagingTimeoutMilliSeconds)
            {
               _expiredTimerSmEvent = this->GetSMNameFull();
               _expiredTimerSmEvent += "::DELAY_CONNECTION_TIMER_CB";

               if (true == _delayConnectionTimer.StartSMTimer(_delayConnectionTimerId, static_cast<long>(pausePagingTimeoutMilliSeconds),
                     0L, _expiredTimerSmEvent.c_str(), this))
               {
                  char milliSecondsStr[200];
                  snprintf(milliSecondsStr, 199, "%llu", static_cast<long long unsigned>(pausePagingTimeoutMilliSeconds));
                  ETG_TRACE_USR4(("checkWblHealthIndicator: started delay connection retry timer (timer ID = 0x%p, intervalMilliseconds = %50s)",
                        _delayConnectionTimerId, milliSecondsStr));

                  sendEvent = false;
               }

            }
            else
            {
               ETG_TRACE_ERR(("checkWblHealthIndicator: starting delay connection timer failed so sending event CONNECT_DEVICE directly"));
            }

            if(true == sendEvent)
            {
               ETG_TRACE_USR4(("checkWblHealthIndicator: sending event CONNECT_DEVICE to myself"));

               result = this->SendEventByName("CONNECT_DEVICE", 0);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("checkWblHealthIndicator: could not send event CONNECT_DEVICE (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkWblHealthIndicator:getProfileConnectionPageTimeout: could not get ProfileConnectionPageTimeout from MainController (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_USR4(("checkWblHealthIndicator: no devices left to be connected"));

         char eventParams[20];

         result = this->ParameterAUTO_CONNECTION_FINISHED(eventParams, sizeof(eventParams), _standardAutoConnectionType);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("checkWblHealthIndicator: sending event AUTO_CONNECTION_FINISHED to myself"));

            result = this->SendEventByName("AUTO_CONNECTION_FINISHED", eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("checkWblHealthIndicator: could not send event AUTO_CONNECTION_FINISHED (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkWblHealthIndicator: could not marshal event parameters for event AUTO_CONNECTION_FINISHED (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result AutoConnectionController::checkDeviceInList()
   {
      ENTRY

      ETG_TRACE_USR1(("checkDeviceInList: entered"));

      std::string methodName("checkDeviceInList");
      Result result(CC_ERR_INT_NO_ERROR);
      ConnectionStatus connectionStatus;
      DisconnectedReason disconnectedReason;
      bool erase = false;
      bool connectDeviceasClassicBT = false;

      ETG_TRACE_USR3(("checkDeviceInList: _carPlayWirelessSupported - %d",
                  LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported));

      ETG_TRACE_USR4(("checkDeviceInList: ignoreDeviceUsagePreference = - %10s", _ignoreDeviceUsagePreference ? "true": "false"));

      //Condition check added for previously same profile it is already connected
      do
      {
         erase = false;

         if (false == _deviceProtocolsWaitingToGetConnected.empty())
         {
            if(((BM_UP_CPW == _deviceProtocolsWaitingToGetConnected[0]._usagePreference) ||
                  (BM_UP_TEMP_CLASSIC_BT == _deviceProtocolsWaitingToGetConnected[0]._usagePreference)) &&
                  (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported) && (false == _ignoreDeviceUsagePreference))
            {
               if(0u == _cpwDeviceId)
               {
                  DeviceIdList connectedDevices;

                  ETG_TRACE_USR4(("checkDeviceInList: list of device handles of connected devices:"));

                  // Property is getting updated delay due to queuing so that only getConnectedDevices from DB
                  result = LocalSpm::getDbManager().getConnectedDevices(connectedDevices);

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_USR4(("checkDeviceInList: could not get connected device in DB (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                  }
                  else
                  {
                     if (true == connectedDevices.empty())
                     {
                        ETG_TRACE_USR4(("checkDeviceInList: no connected devices found"));

                        BtLimitationMode btLimitationMode;
                        LocalSpm::getBmCoreMainController().getBtLimitationMode(btLimitationMode);

                        if(false == btLimitationMode._btLimitationModeInfoList.empty())
                        {
                           ETG_TRACE_USR4(("checkDeviceInList: BTLimitation Mode is not Idle so Connect the device as CLASSIC BT"));
                           connectDeviceasClassicBT = true;
                        }
                     }
                     else
                     {
                        ETG_TRACE_USR4(("checkDeviceInList: at least one connected device found so Connect the device as CLASSIC BT"));
                        connectDeviceasClassicBT = true;
                     }
                  }
               }
               else
               {
                  ETG_TRACE_USR1(("checkDeviceInList: CPW device is available"));
                  _cpwDeviceId = 0u;

                  // NCG3D-219484 : check the limitation Mode
                  BtLimitationMode btLimitationMode;
                  LocalSpm::getBmCoreMainController().getBtLimitationMode(btLimitationMode);

                  if(false == btLimitationMode._btLimitationModeInfoList.empty())
                  {
                     ETG_TRACE_USR4(("checkDeviceInList: BTLimitation Mode is not Idle so Connect the device as CLASSIC BT"));
                     connectDeviceasClassicBT = true;
                  }
               }
            }
            else
            {
               connectDeviceasClassicBT = true;
            }

            if(true == connectDeviceasClassicBT)
            {
               for (size_t protocolidx = 0u; protocolidx < _deviceProtocolsWaitingToGetConnected[0]._protocolList.size();)
               {
                  result = LocalSpm::getDbManager().getProtocolConnectionStatus(connectionStatus, disconnectedReason, _deviceProtocolsWaitingToGetConnected[0]._deviceId,
                        _deviceProtocolsWaitingToGetConnected[0]._protocolList[protocolidx]._protocolId, _deviceProtocolsWaitingToGetConnected[0]._protocolList[protocolidx]._uuid);

                  if (CC_ERR_INT_NO_ERROR == result)
                  {
                     if (BM_CONNECTION_STATUS_CONNECTED == connectionStatus)
                     {
                        ETG_TRACE_USR1(("checkDeviceInList: erase the protocol - %d for device - %d from list because it is already connected",
                              _deviceProtocolsWaitingToGetConnected[0]._protocolList[protocolidx]._protocolId,_deviceProtocolsWaitingToGetConnected[0]._deviceId));

                        if((BM_PROTOCOL_ID_PAN == _deviceProtocolsWaitingToGetConnected[0]._protocolList[protocolidx]._protocolId) &&
                              (_deviceProtocolsWaitingToGetConnected[0]._deviceId == _lastPanConnectedDevice))
                        {
                           ETG_TRACE_USR3(("checkDeviceInList: device - %d is already connected with PAN profile so erase the flag", _lastPanConnectedDevice));
                           _lastPanConnectedDevice = 0u;
                        }

                        _deviceProtocolsWaitingToGetConnected[0]._protocolList.erase((_deviceProtocolsWaitingToGetConnected[0]._protocolList.begin()) + protocolidx);
                     }
                     else
                     {
                        protocolidx++;
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("checkDeviceInList: could not get device connection status of profile for device with ID = %d from DB (error = %d)",
                           _deviceProtocolsWaitingToGetConnected[0]._deviceId, ETG_CENUM(CcErrorInternal, result)));
                     protocolidx++;
                  }
               }

               if(true == _deviceProtocolsWaitingToGetConnected[0]._protocolList.empty())
               {
                  ETG_TRACE_USR1(("checkDeviceInList: erase the device - %d from list because requested protocols are already connected", _deviceProtocolsWaitingToGetConnected[0]._deviceId));
                  _deviceProtocolsWaitingToGetConnected.erase(_deviceProtocolsWaitingToGetConnected.begin());
                  erase = true;
               }
            }

            ETG_TRACE_USR1(("checkDeviceInList: sorted list of devices remaining for consideration in current auto connection run:"));
            this->traceDeviceProtocolList(methodName + ": _deviceProtocolsWaitingToGetConnected =", _deviceProtocolsWaitingToGetConnected);
         }
      }while(true == erase);


      if (false == _deviceProtocolsWaitingToGetConnected.empty())
      {
         if(((BM_UP_CPW == _deviceProtocolsWaitingToGetConnected[0]._usagePreference) || (BM_UP_TEMP_CLASSIC_BT == _deviceProtocolsWaitingToGetConnected[0]._usagePreference)) &&
               (true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported) && (false == connectDeviceasClassicBT))
         {
            result = this->SendEventByName("CPW_DEVICE", (char *) 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("checkDeviceInList: could not send event CPW_DEVICE (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            result = this->SendEventByName("CLASSICBT_DEVICE", (char *) 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("checkDeviceInList: could not send event CLASSICBT_DEVICE (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }
      else
      {
         ETG_TRACE_USR1(("checkDeviceInList: no devices left to be connected"));

         char eventParams[20];

         result = this->ParameterAUTO_CONNECTION_FINISHED(eventParams, sizeof(eventParams), _standardAutoConnectionType);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            ETG_TRACE_USR4(("checkDeviceInList: sending event AUTO_CONNECTION_FINISHED to myself"));

            result = this->SendEventByName("AUTO_CONNECTION_FINISHED", eventParams);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("checkDeviceInList: could not send event AUTO_CONNECTION_FINISHED (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            ETG_TRACE_ERR(("checkDeviceInList: could not marshal event parameters for event AUTO_CONNECTION_FINISHED (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result AutoConnectionController::checkWBLServiceIsAvailable()
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      ETG_TRACE_USR4(("checkWBLServiceIsAvailable: _wblServiceIsAvailable = %d _verifiedWBLStatus = %d", _wblServiceIsAvailable, _verifiedWBLStatus));

      if((true == _wblServiceIsAvailable) || (true == _verifiedWBLStatus))
      {
         ETG_TRACE_USR4(("checkWBLServiceIsAvailable: sending event WBL_AVAILABLE to myself"));

         result = this->SendEventByName("WBL_AVAILABLE", (char *) 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("checkWBLServiceIsAvailable: could not send event WBL_AVAILABLE (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         // Cancel the timer
         if(_waitforWBLInfoTimerId != 0)
         {
            _waitforWBLInfoTimer.CancelTimer(_waitforWBLInfoTimerId);

            _waitforWBLInfoTimerId = 0;
         }

         _expiredTimerSmEvent = this->GetSMNameFull();
         _expiredTimerSmEvent += "::SERVICE_TIMEOUT";

         // Start the timer for waiting info(WBLServiceAvailablity)
         if (true == _waitforWBLInfoTimer.StartSMTimer(_waitforWBLInfoTimerId,
               1000L * static_cast<long>(WAIT_FOR_WBL_SERVICE_AVAILABLE_INFO_SEC), 0L, _expiredTimerSmEvent.c_str(), this))
         {
            char milliSecondsStr[200];
            snprintf(milliSecondsStr, 199, "%llu", (1000L * static_cast<long long unsigned>(WAIT_FOR_WBL_SERVICE_AVAILABLE_INFO_SEC)));
            ETG_TRACE_USR1(("checkWBLServiceIsAvailable(this = 0x%p): started timer for WBL service availablity (timer ID = 0x%p, intervalMilliseconds = %50s)",
                  (void *) this, _waitforWBLInfoTimerId, milliSecondsStr));
         }
         else
         {
            ETG_TRACE_ERR(("checkWBLServiceIsAvailable(this = 0x%p): starting timer for WBL service availablity is failed",
                  (void *) this));
            result = this->SendEventByName("WBL_NOT_AVAILABLE", (char *) 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("checkWBLServiceIsAvailable: could not send event WBL_NOT_AVAILABLE (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result AutoConnectionController::stopTimerAndUpdateWBLServiceAvailability()
   {
      ETG_TRACE_USR1(("stopTimerAndUpdateWBLServiceAvailability:entered"));

      Result result(CC_ERR_INT_NO_ERROR);

      if(_waitforWBLInfoTimerId != 0)
      {
         _waitforWBLInfoTimer.CancelTimer(_waitforWBLInfoTimerId);

         _waitforWBLInfoTimerId = 0;

         _expiredTimerSmEvent = this->GetSMNameFull();
      }

      if(true == _wblServiceIsAvailable)
      {
         ETG_TRACE_USR4(("stopTimerAndUpdateWBLServiceAvailability: sending event WBL_AVAILABLE to myself"));

         result = this->SendEventByName("WBL_AVAILABLE", (char *) 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("stopTimerAndUpdateWBLServiceAvailability: could not send event WBL_AVAILABLE (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         ETG_TRACE_USR4(("stopTimerAndUpdateWBLServiceAvailability: sending event WBL_NOT_AVAILABLE to myself"));

         result = this->SendEventByName("WBL_NOT_AVAILABLE", 0);
         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("stopTimerAndUpdateWBLServiceAvailability: could not send event WBL_NOT_AVAILABLE (error = %d)", ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   void AutoConnectionController::handleSmTimerCallback(const char* message)
   {
      if(0 != message)
      {
         LocalSpm::getBmCoreMainController().pushBmCoreIfMessage(getNewBmCoreIfMessage_BmCoreIfMessage_SmTimeout(message));
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   Result AutoConnectionController::checkWBLLastIntendedMode()
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);
      bool startTimer = true;

      if(false == _verifiedWBLStatus)
      {
         if(0u < _lastIntendedModeList.size())
         {
            for(BTSWblLastIntendedModeList::iterator itlastIntendedModeList = _lastIntendedModeList.begin(); itlastIntendedModeList != _lastIntendedModeList.end(); itlastIntendedModeList++)
            {
               BTSWblWiFiSetupStatus wiFiSetupStatus = itlastIntendedModeList->status;

               if(BTS_WBL_WIFI_MODE_AP == wiFiSetupStatus.mode)
               {
                  if(BTS_WBL_AP_CONFIG_CPW == wiFiSetupStatus.type)
                  {
                     if(BTS_WBL_WIFI_STATE_POWERED_ON == wiFiSetupStatus.powerState)
                     {
                        ETG_TRACE_USR4(("checkWBLLastIntendedMode: sending event APMODE_CPW to myself"));

                        result = this->SendEventByName("APMODE_CPW", (char *) 0);

                        if (CC_ERR_INT_NO_ERROR != result)
                        {
                           ETG_TRACE_ERR(("checkWBLLastIntendedMode: could not send event APMODE_CPW (error = %d)",
                                 ETG_CENUM(CcErrorInternal, result)));
                        }
                        else
                        {
                           startTimer = false;
                        }
                     }
                     else if(BTS_WBL_WIFI_STATE_POWERED_OFF == wiFiSetupStatus.powerState)
                     {
                        result = this->SendEventByName("APMODE_NOT_CPW", (char *) 0);

                        if (CC_ERR_INT_NO_ERROR != result)
                        {
                           ETG_TRACE_ERR(("checkWBLLastIntendedMode: could not send event APMODE_NOT_CPW (error = %d)",
                                 ETG_CENUM(CcErrorInternal, result)));
                        }
                        else
                        {
                           startTimer = false;
                        }
                     }
                  }
                  else
                  {
                     result = this->SendEventByName("APMODE_NOT_CPW", (char *) 0);

                     if (CC_ERR_INT_NO_ERROR != result)
                     {
                        ETG_TRACE_ERR(("checkWBLLastIntendedMode: could not send event APMODE_NOT_CPW (error = %d)",
                              ETG_CENUM(CcErrorInternal, result)));
                     }
                     else
                     {
                        startTimer = false;
                     }
                  }

                  break;
               }
            }
         }

         if(true == startTimer)
         {
            if(_waitforWBLInfoTimerId != 0)
            {
               _waitforWBLInfoTimer.CancelTimer(_waitforWBLInfoTimerId);

               _waitforWBLInfoTimerId = 0;
            }

            _expiredTimerSmEvent = this->GetSMNameFull();
            _expiredTimerSmEvent += "::LIM_TIMEOUT";

            // Start the timer for waiting info(WBLServiceAvailablity)
            if (true == _waitforWBLInfoTimer.StartSMTimer(_waitforWBLInfoTimerId,
                  1000L * static_cast<long>(WAIT_FOR_WBL_LAST_INTENDED_MODE_SEC), 0L, _expiredTimerSmEvent.c_str(), this))
            {
               char milliSecondsStr[200];
               snprintf(milliSecondsStr, 199, "%llu", (1000L * static_cast<long long unsigned>(WAIT_FOR_WBL_LAST_INTENDED_MODE_SEC)));
               ETG_TRACE_USR1(("checkWBLLastIntendedMode(this = 0x%p): started timer for WBL Last intended mode (timer ID = 0x%p, intervalMilliseconds = %50s)",
                     (void *) this, _waitforWBLInfoTimerId, milliSecondsStr));
            }
            else
            {
               ETG_TRACE_ERR(("checkWBLLastIntendedMode(this = 0x%p): starting timer for WBL service availablity is failed",
                     (void *) this));
               result = CC_ERR_INT_GENERAL_ERROR;
            }
         }
      }
      else
      {
         result = this->SendEventByName("APMODE_NOT_CPW", (char *) 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("checkWBLLastIntendedMode: could not send event APMODE_NOT_CPW (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   void AutoConnectionController::updateWblLastIntendedMode(BTSWblLastIntendedModeList lastIntendedModeList)
   {
      ETG_TRACE_USR4(("updateWblLastIntendedMode: lastIntendedModeList size = %d", lastIntendedModeList.size()));

      Result result(CC_ERR_INT_NO_ERROR);

      for(BTSWblLastIntendedModeList::iterator itlastIntendedModeList = lastIntendedModeList.begin(); itlastIntendedModeList != lastIntendedModeList.end(); itlastIntendedModeList++)
      {
         BTSWblWiFiSetupStatus wiFiSetupStatus = itlastIntendedModeList->status;

         ETG_TRACE_USR4(("updateWblLastIntendedMode: wiFiSetupStatus.mode = %d type = %d powerState = %d", wiFiSetupStatus.mode, wiFiSetupStatus.type, wiFiSetupStatus.powerState));

         if(BTS_WBL_WIFI_MODE_AP == wiFiSetupStatus.mode)
         {
            if(BTS_WBL_AP_CONFIG_CPW == wiFiSetupStatus.type)
            {
               if(BTS_WBL_WIFI_STATE_POWERED_ON == wiFiSetupStatus.powerState)
               {
                  ETG_TRACE_USR4(("checkWBLLastIntendedMode: sending event APMODE_CPW to myself"));

                  result = this->SendEventByName("APMODE_CPW", (char *) 0);

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("checkWBLLastIntendedMode: could not send event APMODE_CPW (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                  }
               }
               else if(BTS_WBL_WIFI_STATE_POWERED_OFF == wiFiSetupStatus.powerState)
               {
                  for(BTSWblLastIntendedModeList::iterator itoldlastIntendedModeList = _lastIntendedModeList.begin();
                        itoldlastIntendedModeList != _lastIntendedModeList.end(); itoldlastIntendedModeList++)
                  {
                     BTSWblWiFiSetupStatus oldwiFiSetupStatus = itoldlastIntendedModeList->status;

                     if((BTS_WBL_WIFI_MODE_AP == oldwiFiSetupStatus.mode) && (BTS_WBL_AP_CONFIG_CPW == oldwiFiSetupStatus.type))
                     {
                        if(wiFiSetupStatus.powerState != oldwiFiSetupStatus.powerState)
                        {
                           ETG_TRACE_USR4(("checkWBLLastIntendedMode: sending event APMODE_NOT_CPW to myself"));

                           result = this->SendEventByName("APMODE_NOT_CPW", (char *) 0);

                           if (CC_ERR_INT_NO_ERROR != result)
                           {
                              ETG_TRACE_ERR(("checkWBLLastIntendedMode: could not send event APMODE_NOT_CPW (error = %d)",
                                    ETG_CENUM(CcErrorInternal, result)));
                           }
                           break;
                        }
                     }
                  }
               }
            }
            else
            {
               ETG_TRACE_USR4(("checkWBLLastIntendedMode: sending event APMODE_NOT_CPW to myself"));

               result = this->SendEventByName("APMODE_NOT_CPW", (char *) 0);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("checkWBLLastIntendedMode: could not send event APMODE_NOT_CPW (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }

            break;
         }
      }

      _lastIntendedModeList = lastIntendedModeList;
   }

   void AutoConnectionController::updateWblServiceAvailability(bool wblServiceIsAvailable)
   {
      ENTRY

      _wblServiceIsAvailable = wblServiceIsAvailable;

      ETG_TRACE_USR4(("updateWblServiceAvailability: _wblServiceIsAvailable - %10s", _wblServiceIsAvailable?"true":"false"));

      if(true == _firstUpdate)
      {
         stopTimerAndUpdateWBLServiceAvailability();
      }
      else
      {
         if(true == _wblServiceIsAvailable)
         {
            stopTimerAndUpdateWBLServiceAvailability();
         }
      }
   }

   Result AutoConnectionController::checkDeviceId(IN const DeviceId deviceId)
   {
      Result result(CC_ERR_INT_NO_ERROR);
      ConnectionStatus connectionStatus;
      DisconnectedReason disconnectedReason;

      ETG_TRACE_USR4(("checkDeviceId: deviceId - %d", deviceId));

      // check the received device Id is matched with requested deviceId.
      // Send the PROCEED signal internally if deviceId is matched
      if(_deviceProtocolConnectionInProgress._deviceId == deviceId)
      {
         // Get the Device connection status from db.
         result = LocalSpm::getDbManager().getDeviceConnectionStatus(connectionStatus, disconnectedReason, deviceId);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_USR1(("checkDeviceId: could not get connected status in DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
         else
         {
            //remove the all the entries for the device Id in the autoconnection list if the disconnected reason is MISSING_LINK_KEY
            if((BM_CONNECTION_STATUS_DISCONNECTED == connectionStatus) && (BM_DISCONNECTED_REASON_MISSING_LINK_KEY == disconnectedReason))
            {
               removeDeviceIdfromList(deviceId);
            }

            // Starts the Timer If the device is DISCONNECTED and disconnected reason is not OUT_OF_RANGE
            if((BM_CONNECTION_STATUS_DISCONNECTED == connectionStatus) && (BM_DISCONNECTED_REASON_OUT_OF_RANGE != disconnectedReason))
            {
               ETG_TRACE_USR4(("checkDeviceId: disconnected reason is %d so delay the next connection",
                     ETG_CENUM(DisconnectedReason, disconnectedReason)));

               _expiredTimerSmEvent = this->GetSMNameFull();
               _expiredTimerSmEvent += "::DISCONNECTED_REASON_DELAY_TIMER_CB";

               if (true == _disconnectedReasonDelayTimer.StartSMTimer(_disconnectedReasonDelayTimerId,
                     static_cast<long>(LocalSpm::getDataProvider().getBmCoreConfiguration()._lostDeviceConnectionRetryDelayMilliSeconds),
                     0L, _expiredTimerSmEvent.c_str(), this))
               {
                  char milliSecondsStr[200];
                  snprintf(milliSecondsStr, 199, "%llu", static_cast<long long unsigned>(LocalSpm::getDataProvider().getBmCoreConfiguration()._lostDeviceConnectionRetryDelayMilliSeconds));
                  ETG_TRACE_USR1(("checkDeviceId: started delay connection retry timer (timer ID = 0x%p, intervalMilliseconds = %50s)",
                        _disconnectedReasonDelayTimerId, milliSecondsStr));
               }
               else
               {
                  ETG_TRACE_ERR(("checkDeviceId: starting delay disconnection timer failed"));

                  result = this->SendEventByName("PROCEED", 0);

                  if (CC_ERR_INT_NO_ERROR != result)
                  {
                     ETG_TRACE_ERR(("checkDeviceId: could not send event PROCEED (error = %d)",
                           ETG_CENUM(CcErrorInternal, result)));
                  }
               }
            }// End of if((BM_CONNECTION_STATUS_DISCONNECTED == connectionStatus) && (BM_DISCONNECTED_REASON_OUT_OF_RANGE != disconnectedReason))
            else
            {
               ETG_TRACE_USR4(("checkDeviceId: device is %d and disconnected reason is %d so send PROCEED event directly",
                     ETG_CENUM(ConnectionStatus, connectionStatus), ETG_CENUM(DisconnectedReason, disconnectedReason)));

               result = this->SendEventByName("PROCEED", 0);

               if (CC_ERR_INT_NO_ERROR != result)
               {
                  ETG_TRACE_ERR(("checkDeviceId: could not send event PROCEED (error = %d)",
                        ETG_CENUM(CcErrorInternal, result)));
               }
            }
         }
      }
      else
      {
         ETG_TRACE_USR4(("checkDeviceId: deviceId  - %d is not matched with _deviceProtocolConnectionInProgress._deviceId - %d so not sending PROCEED event",
               deviceId, _deviceProtocolConnectionInProgress._deviceId));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result AutoConnectionController::handleDelayConnectionTimerCb()
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      ETG_TRACE_USR1(("handleDelayConnectionTimerCb: entered"));

      // Cancel the timer and send the proceed event
      if(_delayConnectionTimerId != 0)
      {
         _delayConnectionTimer.CancelTimer(_delayConnectionTimerId);

         _delayConnectionTimerId = 0;
      }

      _expiredTimerSmEvent = this->GetSMNameFull();

      ETG_TRACE_USR4(("handleDelayConnectionTimerCb: sending event CONNECT_DEVICE to myself"));

      result = this->SendEventByName("CONNECT_DEVICE", 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleDelayConnectionTimerCb: could not send event CONNECT_DEVICE (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result AutoConnectionController::handleDisconnectedReasonDelayTimerCb()
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);

      ETG_TRACE_USR1(("handleDisconnectedReasonDelayTimerCb: entered"));

      // Cancel the timer and send the proceed event
      if(_disconnectedReasonDelayTimerId != 0)
      {
         _disconnectedReasonDelayTimer.CancelTimer(_disconnectedReasonDelayTimerId);

         _disconnectedReasonDelayTimerId = 0;
      }

      _expiredTimerSmEvent = this->GetSMNameFull();

      ETG_TRACE_USR4(("handleDisconnectedReasonDelayTimerCb: sending event PROCEED to myself"));

      result = this->SendEventByName("PROCEED", 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("handleDisconnectedReasonDelayTimerCb: could not send event PROCEED (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      return CC_ERR_INT_NO_ERROR;
   }

   void AutoConnectionController::removeDeviceIdfromList(IN const DeviceId deviceId)
   {
      std::string methodName("removeDeviceIdfromList");

      //remove the all the entries for the device Id in the autoconnection list
      DeviceProtocolList::iterator itdeviceProtocolList;
      for(itdeviceProtocolList = _deviceProtocolsWaitingToGetConnected.begin(); itdeviceProtocolList != _deviceProtocolsWaitingToGetConnected.end();)
      {
         if(deviceId == itdeviceProtocolList->_deviceId)
         {
            itdeviceProtocolList = _deviceProtocolsWaitingToGetConnected.erase(itdeviceProtocolList);
         }
         else
         {
            itdeviceProtocolList++;
         }
      }

      ETG_TRACE_USR1(("removeDeviceIdfromList: sorted list of devices remaining for consideration in current auto connection run:"));
      this->traceDeviceProtocolList(methodName + ": _deviceProtocolsWaitingToGetConnected =", _deviceProtocolsWaitingToGetConnected);
   }

   Result AutoConnectionController::connectClassicBT()
   {
      ENTRY

      Result result(CC_ERR_INT_NO_ERROR);
      BmResult bmResult(BM_RESULT_ERR_GENERAL);
      DeviceProtocol deviceProtocol;

      // Cancel the timer
      if(_waitforWBLInfoTimerId != 0)
      {
         _waitforWBLInfoTimer.CancelTimer(_waitforWBLInfoTimerId);

         _waitforWBLInfoTimerId = 0;

         _expiredTimerSmEvent = this->GetSMNameFull();
      }

      deviceProtocol = _deviceProtocolsWaitingToGetConnected[0];

      setDeviceProtocolConnectionInProgress(deviceProtocol);

      _deviceProtocolsWaitingToGetConnected.erase(_deviceProtocolsWaitingToGetConnected.begin());

      ETG_TRACE_USR1(("connectClassicBT: initiating connection of device with ID = %d", deviceProtocol._deviceId));

      if(false == deviceProtocol._protocolList.empty())
      {
         if((deviceProtocol._deviceId == _lastPanConnectedDevice) && checkPANProtcolInList(deviceProtocol._protocolList))
         {
            _panConnectionTriggered = true;
         }

         bmResult = LocalSpm::getBmCoreMainController().connectProfilesInt(deviceProtocol._deviceId, deviceProtocol._protocolList, _pageTimeout);
      }

      if (BM_RESULT_OK != bmResult)
      {
         ETG_TRACE_ERR(("connectClassicBT: could not initiate connect profiles (bmResult = %d) for deviceId = %d", bmResult, deviceProtocol._deviceId));

         result = LocalSpm::getDbManager().setDeviceDisconnectedReason(deviceProtocol._deviceId,
               BM_DISCONNECTED_REASON_CONNECTION_FAILED);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("connectClassicBT: could not set disconnected reason for device with ID = %d in DB (error = %d)",
                  deviceProtocol._deviceId, ETG_CENUM(CcErrorInternal, result)));
         }

         ETG_TRACE_USR4(("connectClassicBT: sending event PROCEED to myself"));
         result = this->SendEventByName("PROCEED", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("connectClassicBT: could not send event PROCEED (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }
      else
      {
         _atLeastOneConnAttemptInitiated = true;
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result AutoConnectionController::connectCPW()
   {
      Result result(CC_ERR_INT_NO_ERROR);

      DeviceProtocol deviceProtocol;

      // Cancel the timer
      if(_waitforWBLInfoTimerId != 0)
      {
         _waitforWBLInfoTimer.CancelTimer(_waitforWBLInfoTimerId);

         _waitforWBLInfoTimerId = 0;

         _expiredTimerSmEvent = this->GetSMNameFull();
      }

      deviceProtocol = _deviceProtocolsWaitingToGetConnected[0];

      setDeviceProtocolConnectionInProgress(deviceProtocol);

      _deviceProtocolsWaitingToGetConnected.erase(_deviceProtocolsWaitingToGetConnected.begin());

      ETG_TRACE_USR1(("connectCPW: initiating BtLimitation-CPW for the device with ID = %d", deviceProtocol._deviceId));

      BdAddress bdAddress = "";

      result = LocalSpm::getDbManager().getBdAddress(bdAddress, deviceProtocol._deviceId);

      if (CC_ERR_INT_NO_ERROR == result)
      {
         BdName bdName = "";
         LimitationMode limitationMode;
         LimitationAction limitationAction = BM_LIMITATION_ACTION_PREPARE;

         limitationMode._limitationCommunicationIf = BM_LIMITATION_COMMUNICATION_IF_WIFI;
         limitationMode._limitationFeature = BM_LIMITATION_FEATURE_CAR_PLAY;

         BmResult bmResult = LocalSpm::getBmCoreMainController().setBtLimitationModeInt(bdAddress, bdName ,limitationMode, limitationAction);

         if (BM_RESULT_OK != bmResult)
         {
            ETG_TRACE_ERR(("connectCPW: could not initiate BTLimitationMode (bmResult = %d) for deviceId = %d", bmResult, deviceProtocol._deviceId));

            ETG_TRACE_USR4(("connectCPW: sending event PROCEED to myself"));
            result = this->SendEventByName("PROCEED", 0);

            if (CC_ERR_INT_NO_ERROR != result)
            {
               ETG_TRACE_ERR(("connectCPW: could not send event PROCEED (error = %d)",
                     ETG_CENUM(CcErrorInternal, result)));
            }
         }
         else
         {
            _atLeastOneConnAttemptInitiated = true;
         }
      }
      else
      {
         ETG_TRACE_USR1(("connectCPW: could not get bdAddress for device id - %d in DB (error = %d)",
               deviceProtocol._deviceId, ETG_CENUM(CcErrorInternal, result)));

         ETG_TRACE_USR4(("connectCPW: sending event PROCEED to myself"));
         result = this->SendEventByName("PROCEED", 0);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_ERR(("connectCPW: could not send event PROCEED (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
         }
      }

      return CC_ERR_INT_NO_ERROR;
   }

   Result AutoConnectionController::autoConnectionToBeRestarted(IN const StandardAutoConnectionType standardAutoConnectionType)
   {
      ENTRY

      ETG_TRACE_USR1(("autoConnectionToBeRestarted: standardAutoConnectionType = %d",
            ETG_CENUM(StandardAutoConnectionType, standardAutoConnectionType)));

      unsigned short restartAutoConnection(0u);
      Result result(CC_ERR_INT_NO_ERROR);

      if (true == _atLeastOneConnAttemptInitiated)
      {
         DeviceIdList connectedDevices;

         ETG_TRACE_USR1(("autoConnectionToBeRestarted: list of device handles of connected devices:"));

         // Property is getting updated delay due to queuing so that only getConnectedDevices from DB
         result = LocalSpm::getDbManager().getConnectedDevices(connectedDevices);

         if (CC_ERR_INT_NO_ERROR != result)
         {
            ETG_TRACE_USR1(("autoConnectionToBeRestarted:  could not get connected device in DB (error = %d)",
                  ETG_CENUM(CcErrorInternal, result)));
            restartAutoConnection = 1u;
         }
         else
         {
            if (true == connectedDevices.empty())
            {
               ETG_TRACE_USR1(("autoConnectionToBeRestarted: no connected devices found"));

               if(true == LocalSpm::getDataProvider().getBmCoreConfiguration()._carPlayWirelessSupported)
               {
                  BtLimitationMode btLimitationMode;
                  LocalSpm::getBmCoreMainController().getBtLimitationMode(btLimitationMode);

                  if(true == btLimitationMode._btLimitationModeInfoList.empty())
                  {
                     restartAutoConnection = 1u;
                  }
                  else
                  {
                     ETG_TRACE_USR1(("autoConnectionToBeRestarted: BTLimitation Mode is not empty"));
                  }
               }
               else
               {
                  restartAutoConnection = 1u;
               }
            }
            else
            {
               ETG_TRACE_USR1(("autoConnectionToBeRestarted: at least one connected device found"));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("autoConnectionToBeRestarted: during previous auto connection run not even a single connection attempt could be initiated"));
      }

      ETG_TRACE_USR1(("autoConnectionToBeRestarted: auto connection procedure is %s",
            (0u == restartAutoConnection) ? "canceled" : "re-started"));

      ETG_TRACE_USR4(("autoConnectionToBeRestarted: sending event CLEAR_AUTO_CONNECTION_BOOKING to BmControllerOnOffSm"));
      result = LocalSpm::getBmController().SendEventByName("CLEAR_AUTO_CONNECTION_BOOKING", (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("autoConnectionToBeRestarted: could not send event CLEAR_AUTO_CONNECTION_BOOKING (error = %d)",
               ETG_CENUM(CcErrorInternal, result)));
      }

      // reset the PAN connection status
      if(!restartAutoConnection)
      {
         resetLastExpectedPANConnectionStatus();
      }

      return restartAutoConnection;
   }

   void AutoConnectionController::resetLastExpectedPANConnectionStatus()
   {
      Result result(CC_ERR_INT_NO_ERROR);
      ETG_TRACE_USR3(("resetLastExpectedConnectionStatus: _lastPanConnectedDevice - %d ", _lastPanConnectedDevice));

      if((0u != _lastPanConnectedDevice) && (true == _panConnectionTriggered))
      {
         ConnectionStatus connectionStatus = BM_CONNECTION_STATUS_UNKNOWN;
         DisconnectedReason disconnectedReason = BM_DISCONNECTED_REASON_UNKNOWN;
         Protocol protocol;
         protocol._protocolId = BM_PROTOCOL_ID_PAN;

         result = LocalSpm::getDbManager().getProtocolConnectionStatus(connectionStatus, disconnectedReason,
               _lastPanConnectedDevice, protocol._protocolId, protocol._uuid);

         if (CC_ERR_INT_NO_ERROR == result)
         {
            if (BM_CONNECTION_STATUS_DISCONNECTED == connectionStatus)
            {
               ETG_TRACE_USR1(("autoConnectionToBeRestarted(deviceId - %d):Reset Pan status flag", _lastPanConnectedDevice));

               (void)LocalSpm::getDbManager().setLastExpectedProtocolConnectionStatus(_lastPanConnectedDevice, protocol._protocolId,
                     protocol._uuid, BM_LAST_EXPECTED_CONNECTION_STATUS_DISCONNECTED);
            }
            else
            {
               ETG_TRACE_USR1(("resetLastExpectedConnectionStatus(deviceId - %d): Device is connected with PAN profile", _lastPanConnectedDevice));
            }

            _lastPanConnectedDevice = 0u;
            _panConnectionTriggered = 0u;
         }
      }
      else
      {
         ETG_TRACE_USR1(("resetLastExpectedConnectionStatus: No devices is connected with PAN profile"));
      }
   }

   Result AutoConnectionController::handleStopAutoConnection()
   {
      ENTRY

      ETG_TRACE_USR1(("handleStopAutoConnection"));

      // Cancel the timer
      if(_waitforWBLInfoTimerId != 0)
      {
         _waitforWBLInfoTimer.CancelTimer(_waitforWBLInfoTimerId);

         _waitforWBLInfoTimerId = 0;
      }

      if(_delayConnectionTimerId != 0)
      {
         _delayConnectionTimer.CancelTimer(_delayConnectionTimerId);

         _delayConnectionTimerId = 0;
      }

      if(_delayAutoConnectionTimerId != 0)
      {
         _delayAutoConnectionTimer.CancelTimer(_delayAutoConnectionTimerId);

         _delayAutoConnectionTimerId = 0;
      }

      if(_disconnectedReasonDelayTimerId != 0)
      {
         _disconnectedReasonDelayTimer.CancelTimer(_disconnectedReasonDelayTimerId);

         _disconnectedReasonDelayTimerId = 0;
      }

      _expiredTimerSmEvent = this->GetSMNameFull();

      resetLastExpectedPANConnectionStatus();

      return CC_ERR_INT_NO_ERROR;
   }

   void AutoConnectionController::create()
   {
      ENTRY

      ETG_TRACE_USR1(("create"));

      /* Create the state machine */
      AutoConnectionControllerSm::Create();

      createDone(0);
   }

   Result AutoConnectionController::init(InitReason reason)
   {
      ENTRY

      ETG_TRACE_USR1(("init: reason = %d", reason));

      (void) reason;

      /* Init the state machine */
      AutoConnectionControllerSm::Init();
      SetAnswerTimeout(2000);

      /* Register state machine with dispatcher */
      Dispatcher::GetInstance().Register(IN this);

      return initDone(0);
   }

   Result AutoConnectionController::run()
   {
      ENTRY

      ETG_TRACE_USR1(("run"));

      return runDone(0);
   }

   Result AutoConnectionController::stop()
   {
      ENTRY

      ETG_TRACE_USR1(("stop"));

      Result result(CC_ERR_INT_NO_ERROR);

      /* Send STOP message to own SM */
      result = this->SendForceEvent(STOP_SM, (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("stop: could not send event STOP_SM (error = %d)", ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   Result AutoConnectionController::done()
   {
      ENTRY

      ETG_TRACE_USR1(("done"));

      Result result(CC_ERR_INT_NO_ERROR);

      /* Deregister state machine with dispatcher */
      Dispatcher::GetInstance().DeRegister(IN this);

      /* Send DONE message to own SM */
      result = this->SendForceEvent(DONE, (char *) 0);

      if (CC_ERR_INT_NO_ERROR != result)
      {
         ETG_TRACE_ERR(("done: could not send event DONE (error = %d)", ETG_CENUM(CcErrorInternal, result)));
      }

      return result;
   }

   char* AutoConnectionController::getSmStateName(OUT tGeneralString stateName, size_t size)
   {
      ENTRY

      ETG_TRACE_USR1(("getSmStateName"));

      this->GetCurrentState((char *) stateName, size);

      return stateName;
   }

   Result AutoConnectionController::initMembers()
   {
      ENTRY

      ETG_TRACE_USR1(("initMembers"));

      _deviceProtocolsWaitingToGetConnected.clear();
      _deviceProtocolConnectionInProgress._deviceId = 0u;
      _deviceProtocolConnectionInProgress._protocolList.clear();
      _atLeastOneConnAttemptInitiated = false;
      _standardAutoConnectionType = BM_STANDARD_AUTO_CONNECTION_UNKNOWN;
      _cpwDeviceId = 0u;
      _lastPanConnectedDevice = 0u;
      _panConnectionTriggered = false;
      _verifiedWBLStatus = false;
      _expiredTimerSmEvent = this->GetSMNameFull();

      return CC_ERR_INT_NO_ERROR;
   }

   void AutoConnectionController::setDeviceProtocolConnectionInProgress(DeviceProtocol deviceProtocolConnectionInProgress)
   {
      ENTRY

      ETG_TRACE_USR1(("setDeviceProtocolConnectionInProgress"));

      Locker locker(&_deviceProtocolConnectionInProgressLock);

      _deviceProtocolConnectionInProgress = deviceProtocolConnectionInProgress;

      LocalSpm::getBmCoreMainController().setAutoConnectionStatusInt(BM_AUTO_CONNECTION_STATE_IN_PROGRESS,
            _deviceProtocolConnectionInProgress._deviceId);
   }

   void AutoConnectionController::traceDeviceProtocolList(IN const std::string& callingMethodName, IN const DeviceProtocolList& deviceProtocolList) const
   {
      ETG_TRACE_USR1(("%200s", callingMethodName.c_str()));

      for (size_t idx = 0u; idx < deviceProtocolList.size(); ++idx)
      {
         ETG_TRACE_USR1(("DeviceProtocolList[%d]: _deviceId = %d",idx, deviceProtocolList[idx]._deviceId));

         for (size_t protocolidx = 0u; protocolidx < deviceProtocolList[idx]._protocolList.size(); ++protocolidx)
         {
            ETG_TRACE_USR1(("DeviceProtocolList[%d]. _protocolList[%d]. protocolId = %d, uuid = \"%50s\"",
                  idx, protocolidx, deviceProtocolList[idx]._protocolList[protocolidx]._protocolId, deviceProtocolList[idx]._protocolList[protocolidx]._uuid.c_str()));
         }

         ETG_TRACE_USR1(("DeviceProtocolList[%d]: _usagePreference = %d",idx, deviceProtocolList[idx]._usagePreference));
      }
   }

   bool AutoConnectionController::checkLastIntendedAPModeIsCPW()
   {
      ETG_TRACE_USR1(("checkLastIntendedAPModeIsCPW: entered"));

      bool lastIntendedAPModeIsCPW = false;

      if(0u < _lastIntendedModeList.size())
      {
         for(BTSWblLastIntendedModeList::iterator itlastIntendedModeList = _lastIntendedModeList.begin(); itlastIntendedModeList != _lastIntendedModeList.end(); itlastIntendedModeList++)
         {
            BTSWblWiFiSetupStatus wiFiSetupStatus = itlastIntendedModeList->status;

            if(BTS_WBL_WIFI_MODE_AP == wiFiSetupStatus.mode)
            {
               if(BTS_WBL_AP_CONFIG_CPW == wiFiSetupStatus.type)
               {
                  if(BTS_WBL_WIFI_STATE_POWERED_ON == wiFiSetupStatus.powerState)
                  {
                     lastIntendedAPModeIsCPW = true;
                  }
               }
               break;
            }
         }
      }

      ETG_TRACE_USR1(("checkLastIntendedAPModeIsCPW: AP Mode is - %10s", lastIntendedAPModeIsCPW? "CPW": "NOT CPW"));

      return lastIntendedAPModeIsCPW;
   }
}
