/**
 * @defgroup AutoConnectionControllerModule Auto Connection Controller
 * @ingroup BmCoreModule
 *
 * @brief This module comprises all parts related to auto connection
 *
 * @details A detailed description is not yet available
 */

/**
 * @file AutoConnectionController.h
 *
 * @par SW-Component
 * Bluetooth Connection Manager Core
 *
 * @brief This file contains the declaration of the class AutoConnectionController
 *
 * @copyright (c) 2016 Robert Bosch GmbH
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details A detailed description is not yet available
 *
 * @ingroup AutoConnectionControllerModule
 */

#ifndef _AUTO_CONNECTION_CONTROLLER_H_
#define _AUTO_CONNECTION_CONTROLLER_H_

#include "AutoConnectionControllerSm.h"
#include "ILocalSpm.h"
#include "BmAllTypesInternal.h"
#include "Lock.h"
#include <algorithm>
#include <sstream>
#include "FwISmTimerCallback.h"

namespace bmcore
{
   /**
    * @class AutoConnectionController
    * @ingroup AutoConnectionControllerModule
    *
    * @brief The class AutoConnectionController implements the state machine AutoConnectionControllerSm
    *
    */
   class AutoConnectionController : public AutoConnectionControllerSm, public ILocalSpm, public ::fw::ISmTimerCallback
   {

   public:
      /**
       * @brief Constructor of class AutoConnectionController which creates an instance
       *
       * No further initialization of internal components and resources
       * is performed in the constructor. See the explicit allocation
       * and initialization functions Create() and Init(). Hence, the
       * implementation is trivial.
       *
       * Synchronization: not re-entrant.
       *
       * Performance considerations: none.
       *
       * @return void
       */
      AutoConnectionController(ComponentId componentID);

      /**
       * @brief Destructor of class PairingController
       *
       * @return void
       */
      virtual ~AutoConnectionController();

      DeviceId getDeviceIdConnectionInProgress(void);

      ProtocolList getDeviceProtocolListConnectionInProgress(void);

      PageTimeout getPageTimeout();

      void updateWblLastIntendedMode(BTSWblLastIntendedModeList lastIntendedModeList);

      void updateWblServiceAvailability(bool wblServiceIsAvailable);

      void updateWblHealthinessIndicator(BTSWblHealthinessIndicator wblHealthinessIndicator);

      // ************************************
      // * Overridden methods of AutoConnectionControllerSm *
      // ************************************

      Result initSm();

      Result handleStopSm();

      Result handleDone();

      Result enterIdle();

      Result createOrderedDeviceList(IN const StandardAutoConnectionType standardAutoConnectionType);

      Result checkWblHealthIndicator();

      Result checkDeviceInList();

      Result connectClassicBT();

      Result connectCPW();

      Result checkWBLServiceIsAvailable();

      Result checkWBLLastIntendedMode();

      Result autoConnectionToBeRestarted(IN const StandardAutoConnectionType standardAutoConnectionType);

      Result handleStopAutoConnection();

      Result stopTimerAndUpdateWBLServiceAvailability();

      Result checkDeviceId(IN const DeviceId deviceId);

      Result handleDelayConnection();

      Result enableAutoConnectionLite();

      Result checkDelayAutoConnection();

      Result handleDelayConnectionTimerCb();

      Result handleDelayAutoConnectionTimerCb();

      Result handleDisconnectedReasonDelayTimerCb();

      Result checkAndRecreateOrderedDeviceList();

      bool checkLastIntendedAPModeIsCPW();

      void handleSmTimerCallback(const char* message);

      // ************************************
      // * Overridden methods of ILocalSpm *
      // ************************************

      /**
       * @brief Creates the Pairing Controller instance
       *
       * Creates the Pairing state machine (including creation of message queue).
       * Informs LocalSpm that create is ready -> createDone(0).
       *
       * @attention: running in SPM thread context
       *
       * @return void
       */
      void create();

      /**
       * @brief Performs the initialization of the Pairing Controller instance, i.e.:
       * - initialization of member variables
       *
       * Synchronization: re-entrant
       *
       * Performance considerations:
       *
       * @return 0 on success and an error code otherwise.
       */
      Result init(InitReason reason);

      /**
       * @brief Starts the Pairing Controller instance, implies that all other
       * components are available
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none.
       *
       * @return 0 on success and an error code otherwise
       */
      Result run();

      /**
       * @brief Stops the Pairing Controller instance
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none
       *
       * @return 0 on success and an error code otherwise
       */
      Result stop();

      /**
       * @brief Cleans up the resources used by the Pairing Controller instance
       *
       * Synchronization: re-entrant
       *
       * Performance considerations: none
       *
       * @return 0 on success and an error code otherwise
       */
      Result done();

      bool isComponentSm() {return true;};

      /**
       * Returns the current state the state machine is in (for debugging of shutdown problems)
       *
       * @param[in,out] stateName buffer for storing the current state name
       * @param[in] size size of the stateName buffer
       *
       * @return pointer to stateName
       */
      char *getSmStateName(OUT tGeneralString stateName, IN size_t size);

      /**
       * @brief Returns statistic data as a string for the over all object statistics
       *
       * @param[out] stat string with statistics information to be printed
       *
       * @return 0 on success and an error code otherwise
       */
   //   int Statistics(OUT Statistics stat);

   private:

      class DeviceProtocol
      {
      public:
         DeviceProtocol(): _deviceId(0u), _protocolList(), _usagePreference(BM_UP_UNDECIDED)
         {
         }

         DeviceProtocol(IN const DeviceId deviceId, IN const ProtocolList& protocol, IN const UsagePreference& usagePreference) :
            _deviceId(deviceId),
            _protocolList(protocol),
            _usagePreference(usagePreference)
         {
         }

         DeviceProtocol(const DeviceProtocol& other) :
            _deviceId(other._deviceId), _protocolList(other._protocolList), _usagePreference(other._usagePreference)
         {
         }

         virtual ~DeviceProtocol()
         {
         }

         friend void swap(DeviceProtocol& first, DeviceProtocol& second)
         {
            using std::swap;

            swap(first._deviceId, second._deviceId);
            swap(first._protocolList, second._protocolList);
            swap(first._usagePreference, second._usagePreference);
         }

         DeviceProtocol& operator=(DeviceProtocol other)
         {
            swap(*this, other);

            return *this;
         }

         bool operator==(const DeviceProtocol& rhs) const
         {
            return ((this->_deviceId == rhs._deviceId)
                  && (this->_protocolList == rhs._protocolList)
                  && (this->_usagePreference == rhs._usagePreference));
         }

         DeviceId _deviceId;
         ProtocolList _protocolList;
         UsagePreference _usagePreference;
      };

      typedef std::vector<DeviceProtocol> DeviceProtocolList;

      /**
       * @brief Initializes members
       *
       * @return 0: success
       */
      Result initMembers();

      void setDeviceProtocolConnectionInProgress(DeviceProtocol deviceProtocolConnectionInProgress);

      void traceDeviceProtocolList(IN const std::string& callingMethodName, IN const DeviceProtocolList& deviceProtocolList) const;

      void checkLastConnectedCPWDevice();

      void checkingPrimaryHFPDevice();

      void checkingFavoriteProtocols();

      bool checkCPWLimitationModeIsAvailable();

      bool checkDeviceTobeAddedList(DeviceProtocol& deviceProtocol);

      bool checkPANProtcolInList(ProtocolList& protocolList);

      void resetLastExpectedPANConnectionStatus();

      Result checkDeviceDisconnectedReason(IN DeviceId deviceId);

      void removeDeviceIdfromList(IN const DeviceId deviceId);

      DeviceProtocolList _deviceProtocolsWaitingToGetConnected;
      DeviceProtocol     _deviceProtocolConnectionInProgress;
      bool         _atLeastOneConnAttemptInitiated;
      StandardAutoConnectionType _standardAutoConnectionType;
      DeviceId _cpwDeviceId;
      DeviceId _lastPanConnectedDevice;
      bool _panConnectionTriggered;
      BTSWblLastIntendedModeList _lastIntendedModeList;
      bool _wblServiceIsAvailable;
      bool _verifiedWBLStatus;
      bool _firstUpdate;
      PageTimeout _pageTimeout;

      Timer      _waitforWBLInfoTimer;
      timer_t    _waitforWBLInfoTimerId;

      Timer      _delayConnectionTimer;
      timer_t    _delayConnectionTimerId;

      Timer      _delayAutoConnectionTimer;
      timer_t    _delayAutoConnectionTimerId;

      Timer      _disconnectedReasonDelayTimer;
      timer_t    _disconnectedReasonDelayTimerId;

      std::string _expiredTimerSmEvent;

      bool _ignoreDeviceUsagePreference;
      bool _recreateOrderedDeviceList;
      bool _delayAutoConnection;

      LockForever _deviceProtocolConnectionInProgressLock;
   };
}

#endif //_AUTO_CONNECTION_CONTROLLER_H_

