//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_VARIANT_h)
#define FeatStd_Util_VARIANT_h

FEATSTD_LINT_FILE(967, Variant.h, "false positive - file has valid include guards.")

#include <FeatStd/Base.h>

FEATSTD_UNIT_TEST_TESTCASE_DECLARATION(VariantTest, VariantTest)

namespace FeatStd {
    namespace VariantValueType {
        /// @addtogroup FEATSTD_UTILS
        /// @{
        /**
        * InputValueType::Enum contains the four types a Variant can get
        * Float, Int32, UInt32 and Bool
        */
        enum Enum {
            FloatValue,  ///< Float
            Int32Value,  ///< Int32
            UInt32Value, ///< UInt32
            BoolValue,   ///< bool
            DoubleValue, ///< Double
            Int8Value,   ///< Int8
            Int16Value,  ///< Int16
            Int64Value,  ///< Int64
            UInt8Value,  ///< UInt8
            UInt16Value, ///< UInt16
            UInt64Value, ///< UInt64
            StringValue  ///< String
        };
    }

    class String;

    namespace Internal {
        template<typename T> struct VariantValueConverter;
    }

    /**
    * Variant class offers to contain either an Float, Int, UInt or an Boolean.
    * Also Contains Enum for the current type
    */
    class Variant
    {
    public:
        /**
        *  Default Constructor.
        */
        Variant();

        /**
        * Constructor.
        *  @param value Float value.
        */
        explicit Variant(FeatStd::Float value);

        /**
        * Constructor.
        *  @param value Double value.
        */
        explicit Variant(FeatStd::Double value);

        /**
        * Constructor.
        *  @param value Int8 value.
        */
        explicit Variant(FeatStd::Int8 value);

        /**
        * Constructor.
        *  @param value Int16 value.
        */
        explicit Variant(FeatStd::Int16 value);

        /**
        * Constructor.
        *  @param value Int32 value.
        */
        explicit Variant(FeatStd::Int32 value);

        /**
        * Constructor.
        *  @param value Int64 value.
        */
        explicit Variant(FeatStd::Int64 value);

        /**
        * Constructor.
        *  @param value UInt8 value.
        */
        explicit Variant(FeatStd::UInt8 value);

        /**
        * Constructor.
        *  @param value UInt16 value.
        */
        explicit Variant(FeatStd::UInt16 value);

        /**
        * Constructor.
        *  @param value UInt32 value.
        */
        explicit Variant(FeatStd::UInt32 value);

        /**
        * Constructor.
        *  @param value UInt64 value.
        */
        explicit Variant(FeatStd::UInt64 value);

        /**
        *  Constructor.
        *  @param value bool value.
        */
        explicit Variant(bool value);

        /**
        * Constructor.
        *  @param value String value.
        */
        explicit Variant(const FeatStd::String& value);

        /**
        *  Destructor.
        */
        ~Variant();

        /**
        *  Copy constructor.
        *  @param variant The other Variant that is copied.
        */
        Variant(const Variant& variant);

        Variant& operator=(const Variant& otherVariant);
        Variant& operator%=(const Variant& otherVariant);
        Variant& operator*=(const Variant& otherVariant);
        Variant& operator/=(const Variant& otherVariant);
        Variant& operator+=(const Variant& otherVariant);
        Variant& operator-=(const Variant& otherVariant);
        Variant operator%(const Variant& otherVariant) const;
        Variant operator*(const Variant& otherVariant) const;
        Variant operator/(const Variant& otherVariant) const;
        Variant operator+(const Variant& otherVariant) const;
        Variant operator-(const Variant& otherVariant) const;
        bool operator==(const Variant& otherVariant) const;
        bool operator!=(const Variant& otherVariant) const;
        bool operator<(const Variant& otherVariant) const;
        bool operator>(const Variant& otherVariant) const;
        bool operator<=(const Variant& otherVariant) const;
        bool operator>=(const Variant& otherVariant) const;

        /**
        *  Retrieves the current type.
        *  @return InputValueType with current variant type.
        */
        VariantValueType::Enum GetType() const;

        /**
        *  Retrieves the float value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @return Float value of the variant (the value is limited to the upper/lower bound of the return type).
        */
        FeatStd::Float GetFloat() const;
        /**
        *  Retrieves the float value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @param value the converted value
        *  @return true if the conversion is possible without limiting (the value is limited to the upper/lower bound of the return type).
        */
        bool TryGetFloat(FeatStd::Float& value) const;

        /**
        *  Retrieves the double value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @return Double value of the variant (the value is limited to the upper/lower bound of the return type).
        */
        FeatStd::Double GetDouble() const;
        /**
        *  Retrieves the double value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @param value the converted value
        *  @return true if the conversion is possible without limiting (the value is limited to the upper/lower bound of the return type).
        */
        bool TryGetDouble(FeatStd::Double& value) const;

        /**
        *  Retrieves the 8 bit signed integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @return Int8 value of the variant (the value is limited to the upper/lower bound of the return type).
        */
        FeatStd::Int8 GetInt8() const;
        /**
        *  Retrieves the 8 bit signed integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @param value the converted value
        *  @return true if the conversion is possible without limiting (the value is limited to the upper/lower bound of the return type).
        */
        bool TryGetInt8(FeatStd::Int8& value) const;

        /**
        *  Retrieves the 16 bit signed integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @return Int16 value of the variant (the value is limited to the upper/lower bound of the return type).
        */
        FeatStd::Int16 GetInt16() const;
        /**
        *  Retrieves the 16 bit signed integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @param value the converted value
        *  @return true if the conversion is possible without limiting (the value is limited to the upper/lower bound of the return type).
        */
        bool TryGetInt16(FeatStd::Int16& value) const;

        /**
        *  Retrieves the 32 bit signed integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @return Int32 value of the variant (the value is limited to the upper/lower bound of the return type).
        */
        FeatStd::Int32 GetInt32() const;
        /**
        *  Retrieves the 32 bit signed integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @param value the converted value
        *  @return true if the conversion is possible without limiting (the value is limited to the upper/lower bound of the return type).
        */
        bool TryGetInt32(FeatStd::Int32& value) const;

        /**
        *  Retrieves the 64 bit signed integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @return Int64 value of the variant (the value is limited to the upper/lower bound of the return type).
        */
        FeatStd::Int64 GetInt64() const;
        /**
        *  Retrieves the 64 bit signed integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @param value the converted value
        *  @return true if the conversion is possible without limiting (the value is limited to the upper/lower bound of the return type).
        */
        bool TryGetInt64(FeatStd::Int64& value) const;

        /**
        *  Retrieves the 8 bit unsigned integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @return UInt8 value of the variant (the value is limited to the upper/lower bound of the return type).
        */
        FeatStd::UInt8 GetUInt8() const;
        /**
        *  Retrieves the 8 bit unsigned signed integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @param value the converted value
        *  @return true if the conversion is possible without limiting (the value is limited to the upper/lower bound of the return type).
        */
        bool TryGetUInt8(FeatStd::UInt8& value) const;

        /**
        *  Retrieves the 16 bit unsigned integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @return UInt16 value of the variant (the value is limited to the upper/lower bound of the return type).
        */
        FeatStd::UInt16 GetUInt16() const;
        /**
        *  Retrieves the 16 bit unsigned signed integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @param value the converted value
        *  @return true if the conversion is possible without limiting (the value is limited to the upper/lower bound of the return type).
        */
        bool TryGetUInt16(FeatStd::UInt16& value) const;

        /**
        *  Retrieves the 32 bit unsigned integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @return UInt32 value of the variant (the value is limited to the upper/lower bound of the return type).
        */
        FeatStd::UInt32 GetUInt32() const;
        /**
        *  Retrieves the 32 bit unsigned signed integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @param value the converted value
        *  @return true if the conversion is possible without limiting (the value is limited to the upper/lower bound of the return type).
        */
        bool TryGetUInt32(FeatStd::UInt32& value) const;

        /**
        *  Retrieves the 64 bit unsigned integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @return UInt64 value of the variant (the value is limited to the upper/lower bound of the return type).
        */
        FeatStd::UInt64 GetUInt64() const;
        /**
        *  Retrieves the 64 bit unsigned signed integer value.
        *  NOTE: if a string is not converable it will be limited to the value 0.
        *  @param value the converted value
        *  @return true if the conversion is possible without limiting (the value is limited to the upper/lower bound of the return type).
        */
        bool TryGetUInt64(FeatStd::UInt64& value) const;

        /**
        *  Retrieves the bool value.
        *  NOTE: if a string is not converable it will be limited to the value false.
        *  @return bool value of the variant.
        */
        bool GetBool() const;
        /**
        *  Retrieves the bool value.
        *  NOTE: if a string is not converable it will be limited to the value false.
        *  @param value the converted value
        *  @return true if the conversion is possible.
        */
        bool TryGetBool(bool& value) const;

        /**
        *  Retrieves the String value.
        *  @return String value of the variant.
        */
        String GetString() const;

        /**
        *  Sets the given value. But keeps the configured type. If the type does not match then the value will be converted.
        *  @return true if the conversion is possible without limiting (the value is limited to the upper/lower bound of the return type).
        */
        bool SetValue(const Variant& otherVariant);

    private:
        template<typename T> friend struct Internal::VariantValueConverter;
        friend struct VariantValueOperator;
        friend struct VariantValueComperator;
        void DetachCurrent();
        void AssignFrom(const Variant& variant);
        void AssignStringFrom(const String& string);
        void AssignStringFrom(const TChar* string);

        VariantValueType::Enum m_type;

        union U {
            FeatStd::Float m_floatValue;
            FeatStd::Double m_doubleValue;

            FeatStd::Int8 m_int8Value;
            FeatStd::Int16 m_int16Value;
            FeatStd::Int32 m_int32Value;
            FeatStd::Int64 m_int64Value;

            FeatStd::UInt8 m_uint8Value;
            FeatStd::UInt16 m_uint16Value;
            FeatStd::UInt32 m_uint32Value;
            FeatStd::UInt64 m_uint64Value;

            bool m_boolValue;

            // same mechanism as for FeatStd::String is used
            // hence, m_stringValue may be just a TChar* in ROM
            // but it can also be a shared instance of FeatStd::Internal::StringData
            TChar* m_stringValue;

            U() {} // union cannot be anonymous because FeatStd::Float might be fixed-point, which is a non-trivial type.
            ~U() {}
        } m_value;
    };
    /// @}
}

#endif // FeatStd_Util_VARIANT_h
