//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef FeatStd_Util_ValidationHelperBase_h__
#define FeatStd_Util_ValidationHelperBase_h__


#include <FeatStd/MemoryManagement/SharedPointer.h>
#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSection.h>
#endif

namespace FeatStd{
    /// @addtogroup FEATSTD_UTILS
    /// @{
    /**
    *  @brief ValidationHelperBase is a base class to generate a defined interface
    *         for validation handling, so basically the validation strategy can be replaced at any time
    *         the validator is thread-safe per default (When flag FEATSTD_THREADSAFETY_ENABLED is set).
    */
    class ValidationHelperBase{
    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(ValidationHelperBase);
        /**
        * enters state of invalidation
        */
        virtual void SetInvalid();

        /**
        * enters state of validation
        */
        virtual void SetValid();

        /**
        * @return current validation state
        */
        virtual bool IsValid() const;

        /**
        * Resets the validator - so the validator is back to a valid state
        */
        virtual void Reset();

        /**
        * Acknowledge that the validation is checked and handled
        * Only useful for complexer validation handlings
        */
        virtual void ConfirmValidHandled();

        virtual ~ValidationHelperBase(){}
    protected:
        ValidationHelperBase();

        /**
        * Implements the logic: enters state of invalidation
        */
        virtual void SetInvalidImplementation() = 0;
        /**
        * Implements the logic: enters state of validation
        */
        virtual void SetValidImplementation() = 0;
        /**
        * Implements the logic: Resets the validator - so the validator is back to a valid state
        */
        virtual void ResetImplementation() = 0;
        /**
        * Implements the logic: 
        * @return current validation state
        */
        virtual bool IsValidImplementation() const = 0;

        /**
        * Implements the logic:
        * Acknowledge that the validation is checked and handled
        * Only useful for complexer validation handlings
        */
        virtual void ConfirmValidHandledImplementation() = 0;

#ifdef FEATSTD_THREADSAFETY_ENABLED
        mutable FeatStd::Internal::CriticalSection m_lock;
#endif
        FEATSTD_MAKE_CLASS_UNCOPYABLE(ValidationHelperBase);
    private:
        FEATSTD_SHARED_POINTER_DECLARATION();
    };
    /// @}
}
#endif // ValidationHelperBase_h__
