//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_VLQEncoding_h)
#define FeatStd_Util_VLQEncoding_h

#include <FeatStd/Util/Traits.h>
#include <FeatStd/Diagnostics/Debug.h>

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_UTILS
/// @{
/**
 *  @brief
 */
class VLQEncoding
{
public:
    /**
     *  Encodes an integer value with VLQ-Algorithm:
     *  Bit 7 (MSB) implies whether an additional byte is needed to represent the value.
     *  Bit 6-0 are Data-Bits
     *  Endianess: Little Endian
     *  Maximum of bytes used to encode: 5 (sizeof(Int32)+1)
     *  Minimum of bytes used to encode: 1
     *  Negative numbers are encoded in 5 bytes!
     *  Implementation matches functionality of .NET "7-bit encoded int"
     *  @param value which will be encoded
     *  @param buf is the buffer in which the encoded bytes will be
     *  @param maxLength provided buffersize; if size too little the rest of value is truncated
     *  @return UInt8 amount of used bytes in buffer
     */
    static UInt8 Write7BitEncodedInt(Int32 value, UInt8 * buf, SizeType const maxLength);
    /**
    *  Encodes an 64-bit value with VLQ-Algorithm:
    *  Bit 7 (MSB) implies whether an additional byte is needed to represent the value.
    *  Bit 6-0 are Data-Bits
    *  Endianess: Little Endian
    *  Maximum of bytes used to encode: 10 (sizeof(Int64)+2)
    *  Minimum of bytes used to encode: 1
    *  Negative numbers are encoded in 5 bytes!
    *  Implementation does NOT match functionality of .NET "7-bit encoded int"
    *  @param value which will be encoded
    *  @param buf is the buffer in which the encoded bytes will be
    *  @param maxLength provided buffersize; if size too little the rest of value is truncated
    *  @return UInt8 amount of used bytes in buffer
    */
    static UInt8 Write7BitEncodedInt64(Int64 value, UInt8 * buf, SizeType const maxLength);

    /**
    *  Decodes an integer value with VLQ-Algorithm:
    *  Bit 7 (MSB) implies whether an additional byte is needed to represent the value.
    *  Bit 6-0 are Data-Bits
    *  Endianess: Little Endian
    *  Maximum of bytes used to decode: 5 (sizeof(Int32)+1)
    *  Minimum of bytes used to decode: 1
    *  Negative numbers are decoded in 5 bytes!
    *  Implementation matches functionality of .NET "7-bit encoded int"
    *  @param result value which will be decoded
    *  @param buf is the buffer in which the encoded bytes will be
    *  @param bufSize provided buffersize; if size too little the rest of value is truncated
    *  @return UInt8 amount of used bytes in buffer
    */
    static UInt8 Read7BitEncodedInt(Int32& result, UInt8 const * buf, SizeType const bufSize);

    /**
    *  Decodes an 64-bit value with VLQ-Algorithm:
    *  Bit 7 (MSB) implies whether an additional byte is needed to represent the value.
    *  Bit 6-0 are Data-Bits
    *  Endianess: Little Endian
    *  Maximum of bytes used to decode: 10 (sizeof(Int64)+2)
    *  Minimum of bytes used to decode: 1
    *  Negative numbers are decoded in 10 bytes!
    *  Implementation does NOT match functionality of .NET "7-bit encoded int"
    *  @param result value which will be decoded
    *  @param buf is the buffer in which the encoded bytes will be
    *  @param bufSize provided buffersize; if size too little the rest of value is truncated
    *  @return UInt8 amount of used bytes in buffer
    */
    static UInt8 Read7BitEncodedInt64(Int64& result, UInt8 const * buf, SizeType const bufSize);
};
/// @}
}}
#endif
