//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Utils_Traits_h)
#define FeatStd_Utils_Traits_h

#include <FeatStd/Base.h>
#include <FeatStd/Diagnostics/Debug.h>

#ifdef _MSC_VER
//disable warning decorated name length exceeded, name was truncated
#pragma warning( disable : 4503 )
#endif

FEATSTD_SUPPRESS_MULTI_WARNING_BEGIN(1931, "sizeof function is valid C++ construct (emitting the size of the return value)")

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_UTILS
/// @{
    namespace TypeTraitsPrivate {
        class Negative { };
        class Positive { Negative n[2]; };
    }

    /**
        *  Determines if given type has const qualifier.
        *  @param T type to inspect
        */
    template<typename T>
    struct IsConstType {
        enum { Value = false };
    };

    template<typename T>
    struct IsConstType<const T> {
        enum { Value = true };
    };

    /** trait to test if the given type is const
    @tparam T the type to analyse */
    template<typename T> struct IsConst {
        typedef T NonConstType;
        enum { Value = false };
    };

    template<typename T> struct IsConst<const T> {
        typedef T NonConstType;
        enum { Value = true };
    };

    /** trait to test if the given type is volatile
    @tparam T the type to analyse */
    template<typename T> struct IsVolatile {
        typedef T NonVolatileType;
        enum { Value = false };
    };

    template<typename T> struct IsVolatile<volatile T> {
        typedef T NonVolatileType;
        enum { Value = true };
    };

    /**
    <summary>
    Type equality check.
    Value will be true if X is same type to Y, false otherwise.
    </summary>
    @param X first type in comparison
    @param Y second type in comparison
    */
    template<typename X, typename Y> struct TypeEqual {
        static const bool Value = false;
    };

    template<typename T> struct TypeEqual<T, T> {
        static const bool Value = true;
    };

    //@{
    /** array identification */
    template<typename T> struct ArrayTrait {
        enum {
            IsArray = false,
            Dimensions = 0,
            FlatElementCount = 1
        };
    };

    template<typename T, int N> struct ArrayTrait<T[N]> {
        typedef T ElementType;
        enum {
            IsArray = true,
            Dimensions = 1,
            ElementsDim0 = N,
            FlatElementCount = N
        };
    };

    template<typename T, int N, int M> struct ArrayTrait<T[N][M]> {
        typedef T ElementType;
        enum {
            IsArray = true,
            Dimensions = 2,
            ElementsDim0 = N,
            ElementsDim1 = M,
            FlatElementCount = M * N
        };
    };

    template<typename T, int N, int M, int O> struct ArrayTrait<T[N][M][O]> {
        typedef T ElementType;
        enum {
            IsArray = true,
            Dimensions = 3,
            ElementsDim0 = N,
            ElementsDim1 = M,
            ElementsDim2 = O,
            FlatElementCount = M * N * O
        };
    };
    //@}

    /** returns the number if items in a one dimensional array */
    template <typename T, SizeType N> inline SizeType ArrayItemCount(const T(&)[N]) {
        return N;
    }

    //@{
    /** pointer identification */
    template<typename T> struct PointerTrait {
        typedef T ReferredType;
        enum { IsPointer = false };
    };

    template<typename T> struct PointerTrait<T*> {
        typedef T ReferredType;
        enum { IsPointer = true };
    };
    //@}

    //@{
    /** reference identification */
    template<typename T> struct ReferenceTrait {
        typedef T ReferredType;
        enum { IsReference = false };
    };

    template<typename T> struct ReferenceTrait<T&> {
        typedef T ReferredType;
        enum { IsReference = true };
    };
    //@}


    //@{
    /** logical and operation on the given bool template parameters */
    template<bool b1, bool b2 = true, bool b3 = true, bool b4 = true, bool b5 = true, bool b6 = true> struct LogicalAnd {
        enum {
            Value = false       ///< always false if specialization with LogicalAnd<true, true, true, true, true, true> does not apply
        };
    };

    template<> struct LogicalAnd<true, true, true, true, true, true> {
        enum {
            Value = true        ///< specialization of true case
        };
    };
    //@}

    //@{
    /** logical or operation on the given bool template parameters */
    template<bool b1, bool b2 = false, bool b3 = false, bool b4 = false, bool b5 = false, bool b6 = false> struct LogicalOr {
        enum {
            Value = true        ///< evaluates to the logical or operation on the defined template parameters
        };
    };

    template<> struct LogicalOr<false, false, false, false, false, false> {
        enum {
            Value = false       ///< specialization of false case
        };
    };
    //@}

    //@{
    /** logical not operation on the given bool template parameters
        Some compiler issue warnings if a logical operation is applied to a constant value. Use this template
        to avoid these warnings. */
    template<bool value> struct LogicalNot {
        enum {
            Value = true        ///< evaluates to the logical or operation on the defined template parameters
        };
    };

    template<> struct LogicalNot<true> {
        enum {
            Value = false       ///< specialization of false case
        };
    };
    //@}

    /** creates MakeSigned / MakeUnsigned traits for the given type with const and volatile qualifiers
        @param TraitName MakeSigned or MakeUnsigned
        @param SrcType the source type
        @param TargetType the complementary type (signed or unsigned) */
    #define FEATSTD_DEF_SIGN_TRAIT(TraitName, SrcType, TargetType) \
        template<> struct TraitName<SrcType> { typedef TargetType Type; }; \
        template<> struct TraitName<const SrcType> { typedef const TargetType Type; }; \
        template<> struct TraitName<volatile SrcType> { typedef volatile TargetType Type; }; \
        template<> struct TraitName<volatile const SrcType> { typedef volatile const TargetType Type; }

    //@{
    /** defines unsigned counterpart of given signed type
        MakeUnsigned<T>::Type returns the complementary unsigned type of T, or T if
        T is already unsigned or has no sign attribute
        @param T signed data type */
    template<typename T> struct MakeUnsigned { typedef T Type; };
    FEATSTD_DEF_SIGN_TRAIT(MakeUnsigned, signed char, unsigned char);
    FEATSTD_DEF_SIGN_TRAIT(MakeUnsigned, short, unsigned short);
    FEATSTD_DEF_SIGN_TRAIT(MakeUnsigned, int, unsigned int);
    FEATSTD_DEF_SIGN_TRAIT(MakeUnsigned, long, unsigned long);
    FEATSTD_DEF_SIGN_TRAIT(MakeUnsigned, Int64, UInt64);
    //@}

    //@{
    /** defines signed counterpart of given unsigned type
        MakeSigned<T>::Type returns the complementary signed type of T, or T if
        T is already unsigned or has no sign attribute
        @param T unsigned data type */
    template<typename T> struct MakeSigned { typedef T Type; };
    FEATSTD_DEF_SIGN_TRAIT(MakeSigned, unsigned char, signed char);
    FEATSTD_DEF_SIGN_TRAIT(MakeSigned, unsigned short, short);
    FEATSTD_DEF_SIGN_TRAIT(MakeSigned, unsigned int, int);
    FEATSTD_DEF_SIGN_TRAIT(MakeSigned, unsigned long, long);
    FEATSTD_DEF_SIGN_TRAIT(MakeSigned, UInt64, Int64);
    //@}

    #undef FEATSTD_DEF_SIGN_TRAIT

    /** defines IsSigned / IsUnsigned traits with all qualifier combinations */
    #define FEATSTD_DEF_IS_SIGN_TRAIT(TraitName, Type) \
        template<> struct TraitName<Type> { enum { Value = true }; }; \
        template<> struct TraitName<const Type> { enum { Value = true }; }; \
        template<> struct TraitName<volatile Type> { enum { Value = true }; }; \
        template<> struct TraitName<const volatile Type> { enum { Value = true }; }

    //@{
    /** defines if a type is signed
        @param T any integral / floating point data type */
    template<typename T> struct IsSigned { enum { Value = false }; };
    FEATSTD_DEF_IS_SIGN_TRAIT(IsSigned, char);
    FEATSTD_DEF_IS_SIGN_TRAIT(IsSigned, signed char);
    FEATSTD_DEF_IS_SIGN_TRAIT(IsSigned, short);
    FEATSTD_DEF_IS_SIGN_TRAIT(IsSigned, int);
    FEATSTD_DEF_IS_SIGN_TRAIT(IsSigned, long);
    FEATSTD_DEF_IS_SIGN_TRAIT(IsSigned, Int64);
    FEATSTD_DEF_IS_SIGN_TRAIT(IsSigned, float);
    FEATSTD_DEF_IS_SIGN_TRAIT(IsSigned, double);
    //@}

    //@{
    /** defines if a type is unsigned
        @param T any integral / floating point data type */
    template<typename T> struct IsUnsigned { enum { Value = false }; };
    FEATSTD_DEF_IS_SIGN_TRAIT(IsUnsigned, unsigned char);
    FEATSTD_DEF_IS_SIGN_TRAIT(IsUnsigned, unsigned short);
    FEATSTD_DEF_IS_SIGN_TRAIT(IsUnsigned, unsigned int);
    FEATSTD_DEF_IS_SIGN_TRAIT(IsUnsigned, unsigned long);
    FEATSTD_DEF_IS_SIGN_TRAIT(IsUnsigned, UInt64);
    //@}

    #undef FEATSTD_DEF_IS_SIGN_TRAIT

    /** defines TypeName<T> functions for all qualifier combinations of the given type */
    #define FEATSTD_DEF_TYPENAME_TRAIT(type) \
        template<> inline const Char* TypeName<type>() { return FEATSTD_STRINGIZE(type); } \
        template<> inline const Char* TypeName<const type>() { return FEATSTD_STRINGIZE(type); } \
        template<> inline const Char* TypeName<volatile type>() { return FEATSTD_STRINGIZE(type); } \
        template<> inline const Char* TypeName<const volatile type>() { return FEATSTD_STRINGIZE(type); }

    //@{
    /** returns name of the given type
        should only be us in debug builds
        @return name of the given type T (any unqualified integral / floating point data type) */
    template<typename T> const Char* TypeName();
    FEATSTD_DEF_TYPENAME_TRAIT(bool)
    FEATSTD_DEF_TYPENAME_TRAIT(Char)
    FEATSTD_DEF_TYPENAME_TRAIT(UInt8)
    FEATSTD_DEF_TYPENAME_TRAIT(UInt16)
    FEATSTD_DEF_TYPENAME_TRAIT(UInt32)
    FEATSTD_DEF_TYPENAME_TRAIT(UInt64)
    FEATSTD_DEF_TYPENAME_TRAIT(Int8)
    FEATSTD_DEF_TYPENAME_TRAIT(Int16)
    FEATSTD_DEF_TYPENAME_TRAIT(Int32)
    FEATSTD_DEF_TYPENAME_TRAIT(Int64)
    FEATSTD_DEF_TYPENAME_TRAIT(Float)
    FEATSTD_DEF_TYPENAME_TRAIT(Double)
    //@}

    #undef FEATSTD_DEF_TYPENAME_TRAIT

    #define FEATSTD_DEF_TYPE_CATEGORY(TraitName, type) \
        template<> struct TraitName<type> { enum { Value = true }; }; \
        template<> struct TraitName<const type> { enum { Value = true }; }; \
        template<> struct TraitName<volatile type> { enum { Value = true }; }; \
        template<> struct TraitName<const volatile type> { enum { Value = true }; }

    //@{
    /** Checks if given type is an integral type */
    template<typename T> struct IsIntegral { enum { Value = false }; };
    FEATSTD_DEF_TYPE_CATEGORY(IsIntegral, bool);
    FEATSTD_DEF_TYPE_CATEGORY(IsIntegral, char);
    FEATSTD_DEF_TYPE_CATEGORY(IsIntegral, signed char);
    FEATSTD_DEF_TYPE_CATEGORY(IsIntegral, unsigned char);
    FEATSTD_DEF_TYPE_CATEGORY(IsIntegral, short);
    FEATSTD_DEF_TYPE_CATEGORY(IsIntegral, unsigned short);
    FEATSTD_DEF_TYPE_CATEGORY(IsIntegral, int);
    FEATSTD_DEF_TYPE_CATEGORY(IsIntegral, unsigned int);
    FEATSTD_DEF_TYPE_CATEGORY(IsIntegral, long);
    FEATSTD_DEF_TYPE_CATEGORY(IsIntegral, unsigned long);
    FEATSTD_DEF_TYPE_CATEGORY(IsIntegral, Int64);
    FEATSTD_DEF_TYPE_CATEGORY(IsIntegral, UInt64);
    //@}

    //@{
    /** Checks if given type is a floating point type */
    template<typename T> struct IsFloatingPoint { enum { Value = false }; };
    FEATSTD_DEF_TYPE_CATEGORY(IsFloatingPoint, float);
    FEATSTD_DEF_TYPE_CATEGORY(IsFloatingPoint, double);
    //@}
    #undef FEATSTD_DEF_TYPE_CATEGORY

        /** Defines struct FixedTypeSizeDefinition that forces types to be of 
         *      constant size.
         */
        #define FEATSTD_DEF_FIXED_TYPE_SIZE(Type, Size) \
            template <> struct FixedTypeSizeDefinition<Type> { \
                enum {Value = true}; \
                void fixedTypeAssert() { FEATSTD_COMPILETIME_ASSERT(sizeof(Type) == Size); } \
            }
        /** Mark specializations as having fixed size */
        template <typename T> struct FixedTypeSizeDefinition { enum {Value = false}; };
        FEATSTD_DEF_FIXED_TYPE_SIZE(UInt8, 1);
        FEATSTD_DEF_FIXED_TYPE_SIZE(UInt16, 2);
        FEATSTD_DEF_FIXED_TYPE_SIZE(UInt32, 4);
        FEATSTD_DEF_FIXED_TYPE_SIZE(UInt64, 8);
        FEATSTD_DEF_FIXED_TYPE_SIZE(Int8, 1);
        FEATSTD_DEF_FIXED_TYPE_SIZE(Int16, 2);
        FEATSTD_DEF_FIXED_TYPE_SIZE(Int32, 4);
        FEATSTD_DEF_FIXED_TYPE_SIZE(Int64, 8);
        /** For types with variating size template specialization might fail if
         *  if overlaps a fixed type. This type should be used in template 
         *  initialization to guard against duplicated specialization errors.
         */
        template <typename T, bool fixed = FixedTypeSizeDefinition<T>::Value>
        struct SpecializationTypeReplacement { struct Type {}; };
        template <typename T> 
        struct SpecializationTypeReplacement<T, false> { typedef T Type; };
        /** Specializations of SpecializationTypeReplacement for FeatStd types */
        typedef SpecializationTypeReplacement<Char>::Type CharSpecializationReplacement;
        typedef SpecializationTypeReplacement<Int>::Type IntSpecializationReplacement;
        typedef SpecializationTypeReplacement<UInt>::Type UIntSpecializationReplacement;

        /** Specialization of SizeTypeSpecializationTypeReplacement for SizeType */
        template <typename T, bool isTypeEqual = TypeEqual<SizeType, UInt8 >::Value
            || TypeEqual<SizeType, UInt16>::Value
            || TypeEqual<SizeType, UInt32>::Value
            || TypeEqual<SizeType, UInt64>::Value
            || TypeEqual<SizeType, Int8  >::Value
            || TypeEqual<SizeType, Int16 >::Value
            || TypeEqual<SizeType, Int32 >::Value
            || TypeEqual<SizeType, Int64 >::Value
            || TypeEqual<SizeType, Int   >::Value
            || TypeEqual<SizeType, UInt  >::Value
        >
        struct SizeTypeSpecializationTypeReplacement { typedef T Type; };
        template <typename T>
        struct SizeTypeSpecializationTypeReplacement<T, true> { struct Type {}; };
        typedef SizeTypeSpecializationTypeReplacement<SizeType>::Type SizeTypeSpecializationReplacement;

    namespace TraitsPrivate {
        template<typename T> struct IsCallByValueType {
            enum {
                Value = LogicalOr<IsIntegral<T>::Value,
                                    IsFloatingPoint<T>::Value,
                                    PointerTrait<T>::IsPointer,
                                    ReferenceTrait<T>::IsReference,
                                    ArrayTrait<T>::IsArray
                                    >::Value
            };
        };
    }

    template<typename T, bool callByValue = TraitsPrivate::IsCallByValueType<T>::Value > struct ArgType {
        typedef T ConstRef;
        typedef T Ref;
    };

    template<typename T> struct ArgType<T, false> {
        typedef const T& ConstRef;
        typedef T& Ref;
    };

    //@{
    /** removes volatile qualifier from given type
        @param T data type */
    template<typename T> struct VolatileRemover { typedef T Type; };
    template<typename T> struct VolatileRemover<volatile T> { typedef T Type; };
    //@}

    //@{
    /** removes const qualifier from given type
        @param T data type */
    template<typename T> struct ConstRemover { typedef T Type; };
    template<typename T> struct ConstRemover<const T> { typedef T Type; };
    //@}

    /** removes any qualifier from given type
        @param T data type */
    template<typename T> struct QualifierRemover {
        typedef typename ConstRemover<typename VolatileRemover<T>::Type>::Type Type;
    };

    /**
        <summary>Determines pure type and removes reference if there is any.</summary>
        */
    template<typename T> struct PureType {
        typedef typename QualifierRemover<typename PointerTrait<typename ReferenceTrait<T>::ReferredType>::ReferredType>::Type Type;
    };

    namespace TypeTraitsPrivate {
        /** Helper type for IsDerivedFrom and IsAbstract*/
        template <typename Base, typename Derived> struct SignatureChecker {
            template <typename T> static TypeTraitsPrivate::Positive Check(Derived const volatile *, T);
            static TypeTraitsPrivate::Negative Check(Base const volatile *, int);
        };
    }

    /**
        <summary>Trait that determines if Derived is a derived type from Base.</summary>
        */
    template <typename Derived, typename Base> class IsDerivedFrom {
        private:
            struct Host {
                operator Base const volatile *() const;
                operator Derived const volatile *();
            };

        public:
            enum {
                Value = (sizeof(TypeTraitsPrivate::SignatureChecker<Base, Derived>::Check(Host(), 0)) == sizeof(TypeTraitsPrivate::Positive))
            };

            static inline bool IsDerived() { return bool(Value); }
    };

    /**
        <summary>Trait that determines if the given type is an abstract type or not.</summary>
        */
    template<typename T> class IsAbstract {
        private:
            enum {
                EnforceInstantionOfT = sizeof(T)
            };

            template<class X> static TypeTraitsPrivate::Negative Fn(X (*)[1]);
            template<class X> static TypeTraitsPrivate::Positive Fn(...);

        public:
            enum {
                Value = (sizeof(Fn<T>(0)) == sizeof(TypeTraitsPrivate::Positive))
            };

            static inline bool IsAbstractType() {
                return bool(Value);
            }
    };

    template<> class IsAbstract<void> {
        public:
            enum {
                Value = true
            };

            static inline bool IsAbstractType() {
                return true;
            }
    };


    /** decorates DstType with the SrcType type decorators (const and volatile)
        The decorated DstType type is avalable with DecoratedType.
        @tparam SrcType the source type
        @tparam DstType the destination type */
    template<typename SrcType, typename DstType> struct TypeDecorationTransferer {
        typedef DstType DecoratedType;
    };

    template<typename SrcType, typename DstType> struct TypeDecorationTransferer<const SrcType, DstType> {
        typedef const DstType DecoratedType;
    };

    template<typename SrcType, typename DstType> struct TypeDecorationTransferer<volatile SrcType, DstType> {
        typedef volatile DstType DecoratedType;
    };

    template<typename SrcType, typename DstType> struct TypeDecorationTransferer<const volatile SrcType, DstType> {
        typedef const volatile DstType DecoratedType;
    };

    /** Value is set to true if result != 0, to false if result == 0
        Multi tool chain issues warnings if constant expressions are used in conditional statements.
        Thus this template is used to silence MULTI toolchain.
        @tparam result value to check for 0 equality */
    template<Int result> struct MapValueToTrueOrFalse {
        enum { Value = true };
    };

    template<> struct MapValueToTrueOrFalse<0> {
        enum { Value = false };
    };

    /** determines if the given value is power of 2 
        @tparam value the value to check */
    template<UInt32 value> struct IsPowerOfTwo {
        enum { 
            Value = !(value & (value - 1)) 
        };
    };

    template<> struct IsPowerOfTwo<0> {
        enum { 
            Value = false 
        };
    };

    namespace TraitsPrivate {
        // const pointer -> const pointer
        template<typename To, typename From, bool constType> struct AlignedCastHelper {
            static inline To Cast(From from) {
                return static_cast<To>(static_cast<const void*>(from));
            }
        };

        // pointer -> pointer
        template<typename To, typename From> struct AlignedCastHelper<To, From, false> {
            static inline To Cast(From from) {
                return static_cast<To>(static_cast<void*>(from));
            }
        };
    }

/// @}
}}

namespace FeatStd { 
/// @addtogroup FEATSTD_UTILS
/// @{
    /**
     *  Use this template to cast e.g. UInt8* to UInt32* to avoid GCC warning "cast increases required alignment of target type".
     */
    template<typename To, typename From> inline To aligned_cast(From from) {
        // fail on non pointer types
        FEATSTD_COMPILETIME_ASSERT(Internal::PointerTrait<From>::IsPointer);
        FEATSTD_COMPILETIME_ASSERT(Internal::PointerTrait<To>::IsPointer);

        typedef typename Internal::PointerTrait<From>::ReferredType DerefFrom;
        return Internal::TraitsPrivate::AlignedCastHelper<To, From, Internal::IsConstType<DerefFrom>::Value>::Cast(from);
    }
/// @}
}

/// @addtogroup FEATSTD_UTILS
/// @{
/** Use this macro in global scope to enable debugging output of custom types names
    @param type any type */
#define FEATSTD_DEF_TYPE_NAME(type) \
    namespace FeatStd { namespace Internal { \
        template<> inline const Char* TypeName< type >() { \
            return FEATSTD_STRINGIZE(type); \
        } \
    }}

/// @}

FEATSTD_SUPPRESS_MULTI_WARNING_END()

#endif // FEATSTD_Utils_Traits_h
