//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextEncoding.h"

#include <FeatStd/Util/FeatLimits.h>
#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Platform/Memory.h>
#include <FeatStd/Platform/String.h>


namespace FeatStd { namespace Internal {
    // ------------------------------------------------------------------------
    static inline bool IsSingleByteCode(register Char c)
    {
        return (ToUInt32(c) & 0x80U) == 0U;
    }

    // ------------------------------------------------------------------------
    static inline bool IsDualByteCode(register Char c)
    {
        return (ToUInt32(c) & 0xE0U) == 0xC0U;
    }

    // ------------------------------------------------------------------------
    static inline bool IsTrippleByteCode(register Char c)
    {
        return (ToUInt32(c) & 0xF0U) == 0xE0U;
    }

    // ------------------------------------------------------------------------
    static inline bool IsQuadByteCode(register Char c)
    {
        return (ToUInt32(c) & 0xF8U) == 0xF0;
    }

    // ------------------------------------------------------------------------
    static inline UInt32 GetNumberOfBytes(register Char c)
    {
        UInt32 n;
        if (c == '\0') {
            n = 0;
        }
        else if (IsSingleByteCode(c)) {
            n = 1;
        }
        else if (IsDualByteCode(c)) {
            n = 2;
        }
        else if (IsTrippleByteCode(c)) {
            n = 3;
        }
        else if (IsQuadByteCode(c)) {
            n = 4;
        }
        else {
            n = 1;
        }
        return n;
    }

    // ------------------------------------------------------------------------
    static inline bool IsValidByteCode(const Char *codePoint, UInt32 count)
    {
        bool valid = true;
        for (UInt32 i = 1U; i < count; i++) {
            if ((ToUInt32(codePoint[i]) & 0xC0U) != 0x80U) {
                valid = false;
                break;
            }
        }
        return valid;
    }

    // ------------------------------------------------------------------------
    UInt32 Utf8Encoding::CodeLength(const Char *codePoint)
    {
        UInt32 n = GetNumberOfBytes(*codePoint);
        if (!IsValidByteCode(codePoint, n)) {
            n = 1;
        }
        return n;
    }

    // ------------------------------------------------------------------------
    SizeType AsciiEncoding::Copy(Char *dst, SizeType dstCharCount, const Char *src)
    {
        if (dstCharCount > 0) {
            String::CopyPartial(dst, src, dstCharCount - 1);
            dst[dstCharCount - 1] = Char(0);
        }
        return dstCharCount;
    }

    // ------------------------------------------------------------------------
    SizeType AsciiEncoding::Copy(Char *dst, SizeType dstCharCount, const Char *src, SizeType codePointCount)
    {
        if (dstCharCount < codePointCount) {
            codePointCount = dstCharCount;
        }
        if (codePointCount > 0) {
            String::CopyPartial(dst, src, codePointCount - 1);
            dst[codePointCount - 1] = Char(0);
        }
        return codePointCount;
    }

    // ------------------------------------------------------------------------
    SizeType Utf8Encoding::Copy(Char *dst, SizeType dstCharCount, const Char *src)
    {
        SizeType srcCharCount = String::Length(src);

        if (srcCharCount < dstCharCount) {
            String::Copy(dst, src);
            return srcCharCount;
        }
        else {
            return Copy(dst, dstCharCount, src, Internal::Limits<SizeType>::Max());
        }
    }

    // ------------------------------------------------------------------------
    SizeType Utf8Encoding::Copy(Char *dst, SizeType dstCharCount, const Char *src, SizeType codePointCount)
    {
        const Char *p = src;
        SizeType n = 0;
        for (;;) {
            if ((codePointCount == 0) || (*p =='\0')) {
                break;
            }
            UInt32 cl = CodeLength(p);
            if ((n + cl) >= dstCharCount) {
                break;
            }
            n += static_cast<SizeType>(cl);
            p += cl;
            --codePointCount;
        }
        if (n > 0) {
            Memory::Copy(dst, src, n);
            dst[n] = '\0';
        }
        return n;
    }

    // ------------------------------------------------------------------------
    bool Utf8Encoding::Match(const Char *s1, const Char *s2, UInt32 codePointCount)
    {
        if (s1 == s2) {
            return true;
        }

        bool match = false;
        for (;;) {
            if (codePointCount == 0) {
                match = true;
                break;
            }
            if ((*s1 == '\0') || (*s2 == '\0')) {
                match = (*s1 == '\0') && (*s2 == '\0');
                break;
            }

            UInt32 cl = CodeLength(s1);
            match = (cl == CodeLength(s2)) &&
                    (FeatStd::Internal::String::ComparePartial(s1, s2, cl) == 0);
            if (!match) {
                break;
            }

            s1 += cl;
            s2 += cl;
            --codePointCount;
        }
        return match;
    }

    template<Int SizeOfTChar, bool TCharIsUtfEncoded> struct CharacterEncodingCheck {
        // reports error (Value undeclared identifier) if invalid character encoding is defined
    };

    // Specialization for UTF-8 encoding (sizeof(TChar) == 1, UTF encoding true)
    template<> struct CharacterEncodingCheck<1, true> {
        enum { Value = true };
    };

    // ------------------------------------------------------------------------
    UInt32 Utf8Encoding::CodePointCount(const Char *string)
    {
//! [FEATSTD_COMPILETIME_ASSERT]
        // Courier supports UTF8 encoding only. Thus enforce Candera to be configured
        // for UTF8 encoding.
        FEATSTD_COMPILETIME_ASSERT((CharacterEncodingCheck<sizeof(FeatStd::TChar), bool(FeatStd::TCharIsUtfEncoded)>::Value));
//! [FEATSTD_COMPILETIME_ASSERT]

        UInt32 n = 0;
        while (*string != '\0') {
            string += CodeLength(string);
            ++n;
        }
        return n;
    }

    // ------------------------------------------------------------------------
    UInt32 Utf8Encoding::Advance(const Char *&codePoint)
    {
        if (*codePoint =='\0') {
            return 0;
        }

        UInt32 n = CodeLength(codePoint);
        codePoint += n;
        return n;
    }

    // ------------------------------------------------------------------------
    UInt32 Utf8Encoding::Ucs4(const Char *codePoint)
    {
        UInt32 ucs4 = ~static_cast<UInt32>(0);
        if (IsSingleByteCode(*codePoint)) {
            ucs4 = ToUInt32(codePoint[0]);
        }
        else if (IsDualByteCode(*codePoint)) {
            if (IsValidByteCode(codePoint, 2)) {
                ucs4 = (((ToUInt32(codePoint[0]) & 0x1FU) << 6U) |
                         (ToUInt32(codePoint[1]) & 0x3FU));
            }
        }
        else if (IsTrippleByteCode(*codePoint)) {
            if (IsValidByteCode(codePoint, 3)) {
                ucs4 = (((ToUInt32(codePoint[0]) & 0x0FU) << 12U) |
                        ((ToUInt32(codePoint[1]) & 0x3FU) << 6U) |
                         (ToUInt32(codePoint[2]) & 0x3FU));
            }
        }
        else {
            if (IsValidByteCode(codePoint, 4)) {
                ucs4 = (((ToUInt32(codePoint[0]) & 0x07U) << 18U) |
                        ((ToUInt32(codePoint[1]) & 0x3FU) << 12U) |
                        ((ToUInt32(codePoint[2]) & 0x3FU) <<  6U) |
                         (ToUInt32(codePoint[3]) & 0x3FU));
            }
        }
        return ucs4;
    }


    UInt32 Utf8Encoding::NCodeLength(const Char *codePoint, UInt32 bufferSize)
    {
        UInt32 n;
        if (bufferSize == 0) {
            n = 0;
        }
        else {
            n = GetNumberOfBytes(*codePoint);
            if (n > bufferSize) {
                n = 1;
            }
        }
        if (!IsValidByteCode(codePoint, n)) {
            n = 1;
        }
        return n;
    }

    UInt32 Utf8Encoding::NUcs4(const Char *codePoint, UInt32 codeLength)
    {
        UInt32 ucs4 = ~static_cast<UInt32>(0);
        switch(codeLength) {
            case 0:
                ucs4 = 0U;
                break;
            case 1:
                if (IsSingleByteCode(*codePoint)) {
                    ucs4 = ToUInt32(codePoint[0]);
                }
                break;
            case 2:
                ucs4 = (((ToUInt32(codePoint[0]) & 0x1FU) << 6U) |
                        (ToUInt32(codePoint[1]) & 0x3FU));
                break;
            case 3:
                ucs4 = (((ToUInt32(codePoint[0]) & 0x0FU) << 12U) |
                        ((ToUInt32(codePoint[1]) & 0x3FU) << 6U) |
                         (ToUInt32(codePoint[2]) & 0x3FU));
                break;
            case 4:
                ucs4 = (((ToUInt32(codePoint[0]) & 0x07U) << 18U) |
                        ((ToUInt32(codePoint[1]) & 0x3FU) << 12U) |
                        ((ToUInt32(codePoint[2]) & 0x3FU) <<  6U) |
                         (ToUInt32(codePoint[3]) & 0x3FU));
                break;
            default:
                break;
        }
        return ucs4;
    }
}}
