//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <FeatStd/Util/String.h>

#include "StringData.h"
#include "StringDataFactory.h"

#include <FeatStd/Util/TextEncoding.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

#include <FeatStd/MemoryManagement/Heap.h>

namespace FeatStd {

using FeatStd::Internal::StringData;
using FeatStd::Internal::TextEncoding;
using FeatStd::Internal::StringDataBase;
using FeatStd::Internal::StringDataFactory;
using FeatStd::Internal::TextIterator;


extern bool IsInRom(const void*);

#if !defined(PLATFORM_HAS_ISINROM_FUNCTION)
    bool IsInRom(const void*)
    {
        return false;
    }
#endif

/******************************************************************************
 *  Constructors
 ******************************************************************************/
String::String() :
    m_text(StringData::GetEmptyText())
{
}

String::String(const TChar* string) :
    m_text(StringData::GetEmptyText())
{
    if (string == 0) {
        return;
    }

    if (IsInRom(string)) {
        m_text = const_cast<TChar*>(string);
    }
    else {
        StringData* strData = AllocateStaticData(string, TextEncoding::CharCount(string));
        (void) AttachTo(strData);
    }
}

String::String(const TChar *string, SizeType length, bool isCodePoints) :
m_text(StringData::GetEmptyText())
{
    if (string != 0) {
        TextIterator it(string);
        SizeType charCount = 0;              //number of bytes (NOT characters)
        if (isCodePoints){
            while ((!it.EndOfString()) && (length > 0)) {
                charCount += static_cast<SizeType>(it.Advance());
                --length;
            }
        }else{
            while ((!it.EndOfString()) && (length > 0)) {
                UInt32 bytes=it.Advance();
                if(length>=bytes){          //truncate partial characters
                    charCount += static_cast<SizeType>(bytes);
                    length-=static_cast<SizeType>(bytes);
                }
            }
        }
        StringData* strData = AllocateStaticData(string,charCount);
        (void)AttachTo(strData);
    }
}

String::String(const String& other) :
    m_text(StringData::GetEmptyText())
{
    if (IsInRom(other.m_text)) {
        FEATSTD_LINT_NEXT_EXPRESSION(1554, "direct assignment is okay")
            m_text = other.m_text;
    }
    else {
        (void)AttachTo(other.GetData());
    }
}

String::String(StringDataBase* strData) :
    m_text(StringData::GetEmptyText())
{
    (void) AttachTo(strData);
}

String::String(SizeType hash) :
    m_text(StringData::GetEmptyText())
{
    StringDataBase* strData = StringDataFactory::GetInstance().CreateLocalizableStringData(hash);
    (void) AttachTo(strData);
}

String::String(const TextId& id) :
    m_text(StringData::GetEmptyText())
{
    SizeType hash = id.GetHashValue();
    StringDataBase* strData = StringDataFactory::GetInstance().CreateLocalizableStringData(hash);
    (void) AttachTo(strData);
}

/******************************************************************************
 *  Destructor
 ******************************************************************************/
String::~String()
{
    DetachCurrent();
    m_text = 0;
}

/******************************************************************************
 *  operator =
 ******************************************************************************/
String& String::operator=(const String& other)
{
    if ((&other != this) && (m_text != other.m_text)) {
        DetachCurrent();
        if (!AttachTo(other.GetData())) {
            m_text = StringData::GetEmptyText();
        }
    }
    return *this;
}

String& String::operator=(const TChar* other)
{
    if (other != m_text) {
        if (other == 0) {
            ClearData();
        }
        else {
            DetachCurrent();
            if (IsInRom(other)) {
                m_text = const_cast<TChar*>(other);
            }
            else {
                StringData* strData = AllocateStaticData(other, TextEncoding::CharCount(other));
                (void) AttachTo(strData);
            }
        }
    }
    return *this;
}

Int32 String::Compare(const String& other) const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(GetCriticalSection());
    FeatStd::Internal::CriticalSectionLocker lockOther(other.GetCriticalSection());
#endif
    return Internal::String::CompareStrings(GetCString(), other.GetCString());
}

Int32 String::ComparePartial(const String& other, SizeType nChars) const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(GetCriticalSection());
    FeatStd::Internal::CriticalSectionLocker lockOther(other.GetCriticalSection());
#endif
    return Internal::String::ComparePartial(GetCString(), other.GetCString(), nChars);
}


/******************************************************************************
 *  operator ==
 ******************************************************************************/
bool String::operator==(const String& other) const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(GetCriticalSection());
    FeatStd::Internal::CriticalSectionLocker otherLock(other.GetCriticalSection());
#endif
    return (m_text == other.m_text) || (TextEncoding::Match(GetCString(), other.GetCString()));
}

/******************************************************************************
 *  operator !=
 ******************************************************************************/
bool String::operator!=(const String& other) const
{
    return !(*this == other);
}

/******************************************************************************
 *  GetCString
 ******************************************************************************/
const TChar* String::GetCString() const
{
    static const TChar s_nullString = '\0';

    if (IsInRom(m_text)) {
        return m_text;
    }

    if (m_text != 0) {
        const TChar* text = GetData()->GetCString();
        if (0 != text) {
            return text;
        }
    }

    return &s_nullString;
}

/******************************************************************************
 *  CharCount
 ******************************************************************************/
SizeType String::GetCharCount() const
{
    if (IsInRom(m_text)) {
        return TextEncoding::CharCount(m_text);
    }
    else {
        return GetData()->GetCharCount();
    }
}

/******************************************************************************
 *  CodePointCount
 ******************************************************************************/
UInt32 String::GetCodePointCount() const
{
    if (IsInRom(m_text)) {
        return TextEncoding::CodePointCount(m_text);
    }
    else {
        return GetData()->GetCodePointCount();
    }
}

/******************************************************************************
 *  GetId
 ******************************************************************************/
SizeType String::GetId() const
{
    return GetData()->GetId();
}

/******************************************************************************
 *  HasChanged
 ******************************************************************************/
bool String::HasChanged() const
{
    return GetData()->HasChanged();
}

/******************************************************************************
 *  IsEmpty
 ******************************************************************************/
bool String::IsEmpty() const
{
    return GetData()->IsEmpty();
}

#ifdef FEATSTD_THREADSAFETY_ENABLED
/******************************************************************************
*  GetCriticalSection
******************************************************************************/
FeatStd::Internal::CriticalSection* String::GetCriticalSection() const
{
    return GetData()->GetCriticalSection();
}
#endif

/******************************************************************************
 *  AttachTo
 ******************************************************************************/
bool String::AttachTo(TChar*& text, Internal::StringDataBase* strData)
{
    if (strData != 0) {
        if (!IsInRom(strData)) {
            strData->IncRefCount();
        }
        text = strData->GetText();
        FEATSTD_DEBUG_ASSERT(text != 0);
        return true;
    }
    else {
        text = StringData::GetEmptyText();
        return false;
    }
}

bool String::AttachTo(StringDataBase* strData)
{
    return AttachTo(m_text, strData);
}

/******************************************************************************
 *  DetachFrom
 ******************************************************************************/
void String::DetachFrom(TChar*& text, Internal::StringDataBase* strData)
{
    if ((strData != 0) && (!IsInRom(strData))) {
        strData->DecRefCount();
    }
    text = StringData::GetEmptyText();
}

void String::DetachFrom(StringDataBase* strData) const
{
    if ((strData != 0) && (!IsInRom(strData))) {
        strData->DecRefCount();
    }
}

/******************************************************************************
 *  DetachCurrent
 ******************************************************************************/
void String::DetachCurrent() const
{
    if (!IsInRom(m_text)) {
        DetachFrom(GetData());
    }
}

/******************************************************************************
 *  Allocate
 ******************************************************************************/
StringData* String::AllocateStaticData(const TChar *string, SizeType stringCharCount) const
{
    StringData *sd;
    if (stringCharCount > 0) {
            sd = StringData::Create(stringCharCount, string);
    }
    else {
        sd = StringData::GetEmptyStringData();
    }
    return (sd != 0) ? sd : StringData::GetEmptyStringData();
}

/******************************************************************************
 *  Clear
 ******************************************************************************/
void String::ClearData()
{
    DetachCurrent();
    m_text = StringData::GetEmptyText();
}

/******************************************************************************
 *  GetStrData
 ******************************************************************************/
StringDataBase* String::GetData() const
{
    return StringData::TextToStringData(m_text);
}

/******************************************************************************
*  GetCStringAccess
******************************************************************************/
CStringAccess String::GetCStringAccess() const
{
    return CStringAccess(*this);
}

/******************************************************************************
*  CStringAccess
******************************************************************************/
CStringAccess::CStringAccess(const String& str) : m_length(0), m_copiedstring(0)
{
    InitString(str);
}

CStringAccess::~CStringAccess()
{
    ClearString();
}

void CStringAccess::InitString(const String& str)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    if (0 != str.GetCriticalSection())
    {
        str.GetCriticalSection()->Obtain();
    }
#endif
    CopyString(static_cast<UInt32>(str.GetCharCount()), str.GetCString());
#ifdef FEATSTD_THREADSAFETY_ENABLED
    if (0 != str.GetCriticalSection())
    {
        str.GetCriticalSection()->Release();
    }
#endif
}

void CStringAccess::ClearString()
{
    const UInt8 c_DeletePattern = 64U; // '@'
    Internal::Memory::Set(m_copiedstring, c_DeletePattern, m_length);
    FEATSTD_DELETE_ARRAY(m_copiedstring);
}

CStringAccess& CStringAccess::operator=(const CStringAccess& string)
{
    if (&string != this)
    {
        ClearString();
        CopyString(string.m_length, string.m_copiedstring);
    }
    return *this;
}

CStringAccess::CStringAccess(const CStringAccess& cstr)
{
    CopyString(cstr.m_length, cstr.m_copiedstring);
}

void CStringAccess::CopyString(UInt32 length, const TChar* str)
{
    m_copiedstring = FEATSTD_NEW_ARRAY(TChar, static_cast<SizeType>(length)+static_cast<SizeType>(1));
    m_length = length;
    Internal::Memory::Copy(m_copiedstring, str, static_cast<SizeType>(m_length));
    m_copiedstring[m_length] = '\0';
}

}
