//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_StateMachine2_h)
#define FeatStd_Util_StateMachine2_h

#ifndef FEATSTD_STATEMACHINE_PANIC

#include <FeatStd/Diagnostics/Debug.h>
#define FEATSTD_STATEMACHINE_PANIC() FEATSTD_DEBUG_FAIL();

#endif

namespace FeatStd {
    namespace Internal {
    template<typename _EventType, typename _DataContext, typename _ActionType, typename _ConditionType>
    class StateMachine;

    template<typename _EventType, typename _DataContext, typename _ActionType, typename _ConditionType>
    class HistoryStateMachine;

    class TransitionBase {
    };

    typedef const char* StateIdentifier;
    typedef const char* TransitionIdentifier;

    /**
    * @brief Simple state class with no inner states. Actions for entry, exit and update can be assigned (null pointers are accepted and handled correctly).
    */
    template <
        typename _EventType = int,
        typename _DataContextType = void*,
        typename _ActionType = void(*)(_DataContextType& dataContext),
        typename _ConditionType = bool(*)(const _EventType& event, const _DataContextType& dataContext, const TransitionBase& transition)
    >
    class State {
    public:
        /**
        * The concrete type of the event.
        */
        typedef _EventType EventType;

        /**
        * Data type for additional data that is available to actions and conditions.
        */
        typedef _DataContextType DataContextType;

        /**
        * Callback type for action function
        */
        typedef _ActionType ActionType;

        /**
        * Callback type for condition function. The event has to be evaluated.
        */
        typedef _ConditionType ConditionType;

        /**
            * Base type of all state parents.
            */
        typedef StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType> StateMachineType;

        /**
        * Describe a transition by this class. The condition will be evaluated for each transition in the order of the transition sequence.
        */
        class Transition : public TransitionBase {
        public:
            Transition(const TransitionIdentifier& name, ConditionType condition, State* target, const ActionType& action);

        protected:
            friend class State;

            virtual bool OnEvent(const EventType& event, DataContextType& dataContext);

        private:
            TransitionIdentifier m_name;
            ConditionType m_condition;
            State* m_target;
            ActionType m_action;
        };

        /**
        * Save function to call an action (null pointer will be handled correctly).
        */
        static void ExecuteAction(const ActionType& action, DataContextType& dataContext);

        State(const StateIdentifier& name, const ActionType& entry, const ActionType& exit, const ActionType& update);

        void InitTransitions(SizeType numberOfTransitions, Transition** transitions)
        {
            FEATSTD_DEBUG_ASSERT(0 == m_transitions);
            m_numberOfTransitions = numberOfTransitions;
            m_transitions = transitions;
        }

        /**
        * Update action will be called for each active state and sub state.
        */
        virtual void OnUpdate(DataContextType& dataContext);

        /**
        * Pass a event to the state/state machine to handle the event. If the event is consumed true will be returned and false otherwise.
        */
        virtual bool OnEvent(const EventType& event, DataContextType& dataContext);

        /**
        * Returns the parent state machine;
        */
        StateMachineType* GetParent() const;

        /**
        * The OnInit method has to be called to initialize the state machine. Until this call no action will be called.
        */
        inline virtual void OnInit(_DataContextType& dataContext);

        /**
        * The OnFinalize method has to be called to finalize the state machine.
        */
        inline virtual void OnFinalize(_DataContextType& dataContext);

    protected:
        /**
        * Entry action will be called whenever the state gets active (also valid if entered via history sub state).
        */
        virtual void OnEntry(DataContextType& dataContext);

        /**
        * Exit action will be called whenever the state gets inactive (also valid if it is a history sub state).
        */
        virtual void OnExit(DataContextType& dataContext);

        /**
         * This method will be called when a transition has been performed. Its purpose is to change the current state of the concrete state machine implementation.
         */
        virtual void OnStateChange(State* nextState, _DataContextType& dataContext, Transition* transition);

        /**
         * Executes the action of the given transition.
         */
        static void ExecuteTransitionAction(const Transition* transition, DataContextType& dataContext);

    private:
        FEATSTD_MAKE_CLASS_STATIC(State);
        FEATSTD_MAKE_CLASS_UNCOPYABLE(State);

        friend class StateMachine<EventType, DataContextType, ActionType, ConditionType>;

        SizeType GetStateDepth() const;
        StateMachineType* GetParent(SizeType count) const;

        const Char* m_name;
        ActionType m_entry;
        ActionType m_exit;
        ActionType m_update;
        SizeType m_numberOfTransitions;
        Transition** m_transitions;
        StateMachineType* m_parent;
    };

    /**
    * @brief Use the StateMachine class to define a top state machine or a sub state machine.
    * Each activation will set the init state as the current state.
    */
    template <
        typename _EventType = int,
        typename _DataContextType = void*,
        typename _ActionType = void(*)(_DataContextType& dataContext),
        typename _ConditionType = bool(*)(const _EventType& event, const _DataContextType& dataContext, const TransitionBase& transition)
    >
    class StateMachine : public State< _EventType, _DataContextType, _ActionType, _ConditionType > {
    public:
        typedef State<_EventType, _DataContextType, _ActionType, _ConditionType> StateType;
        typedef StateType Base;
        typedef typename Base::StateMachineType StateMachineType;

        StateMachine(const char* name, const _ActionType& entry, const _ActionType& exit, const _ActionType& update) :
            Base(name, entry, exit, update),
            m_initState(0),
            m_currentState(0),
            m_performInitStateTransition(true)
        {
        }

        void InitStates(StateType& initState, SizeType numberOfStates, StateType** states);

        /**
         * The OnInit method has to be called to initialize the state machine. Until this call no action will be called.
         */
        inline virtual void OnInit(_DataContextType& dataContext);

        /**
         * The OnFinalize method has to be called to finalize the state machine.
         */
        inline virtual void OnFinalize(_DataContextType& dataContext);

        inline virtual void OnUpdate(_DataContextType& dataContext) override;
        inline virtual bool OnEvent(const _EventType& event, _DataContextType& dataContext) override;

    protected:
        virtual void OnEntry(_DataContextType& dataContext) override;
        virtual void OnExit(_DataContextType& dataContext) override;
        virtual void OnStateChange(StateType* nextState, _DataContextType& dataContext, typename StateType::Transition* transition);
        StateType* GetCurrentState() const;

    private:
        FEATSTD_MAKE_CLASS_STATIC(StateMachine);
        FEATSTD_MAKE_CLASS_UNCOPYABLE(StateMachine);

        friend class State<_EventType, _DataContextType, _ActionType, _ConditionType>;
        friend class HistoryStateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>;

        StateType* m_initState;
        StateType* m_currentState;
        bool m_performInitStateTransition;
    };

    /**
    * @brief Use the HistoryStateMachine class to define a top state machine or a sub state machine.
    * A HistoryStateMachine will remember its last internal state. The init state will be only set for the first activation.
    * Afterward the history state will be used.
    */
    template <
        typename _EventType = int,
        typename _DataContextType = void*,
        typename _ActionType = void(*)(_DataContextType& dataContext),
        typename _ConditionType = bool(*)(const _EventType& event, const _DataContextType& dataContext, const TransitionBase& transition)
    >
    class HistoryStateMachine : public StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType> {
    public:
        typedef StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType> Base;
        typedef typename Base::StateMachineType StateMachineType;

        HistoryStateMachine(const char* name, const _ActionType& entry, const _ActionType& exit, const _ActionType& update) :
            Base(name, entry, exit, update),
            m_historyState(0)
        {
        }

    protected:
        inline virtual void OnEntry(_DataContextType& dataContext) override;
        inline virtual void OnExit(_DataContextType& dataContext) override;
        using StateMachineType::OnStateChange;

    private:
        FEATSTD_MAKE_CLASS_STATIC(HistoryStateMachine);
        FEATSTD_MAKE_CLASS_UNCOPYABLE(HistoryStateMachine);

        using StateMachineType::m_currentState;
        using StateMachineType::m_performInitStateTransition;
            
        typename StateMachineType::StateType* m_historyState;
    };

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    State<_EventType, _DataContextType, _ActionType, _ConditionType>::Transition::Transition(const TransitionIdentifier& name, ConditionType condition, State* target, const ActionType& action) :
        m_name(name),
        m_condition(condition),
        m_target(target),
        m_action(action)
    {
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    bool State<_EventType, _DataContextType, _ActionType, _ConditionType>::Transition::OnEvent(const EventType& event, DataContextType& dataContext)
    {
        return ((!m_condition) || (*m_condition)(event, dataContext, *this));
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void State<_EventType, _DataContextType, _ActionType, _ConditionType>::ExecuteAction(const ActionType& action, DataContextType& dataContext)
    {
        if (!!action) {
            action(dataContext);
        }
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    State<_EventType, _DataContextType, _ActionType, _ConditionType>::State(const StateIdentifier& name, const ActionType& entry, const ActionType& exit, const ActionType& update) :
        m_name(name),
        m_entry(entry),
        m_exit(exit),
        m_update(update),
        m_numberOfTransitions(0),
        m_transitions(0),
        m_parent(0)
    {
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void State<_EventType, _DataContextType, _ActionType, _ConditionType>::OnUpdate(DataContextType& dataContext)
    {
        ExecuteAction(m_update, dataContext);
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    bool State<_EventType, _DataContextType, _ActionType, _ConditionType>::OnEvent(const EventType& event, DataContextType& dataContext)
    {
        for (SizeType i = 0; i < m_numberOfTransitions; ++i) {
            Transition* transition = m_transitions[i];
            if ((0 != transition) && transition->OnEvent(event, dataContext)) {
                State* state = transition->m_target;
                if (0 == state) {
                    state = this;
                }
                OnStateChange(state, dataContext, transition);
                return true;
            }
        }
        return false;
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>* State<_EventType, _DataContextType, _ActionType, _ConditionType>::GetParent() const
    {
        return m_parent;
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void State<_EventType, _DataContextType, _ActionType, _ConditionType>::OnInit(DataContextType& /*dataContext*/)
    {
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void State<_EventType, _DataContextType, _ActionType, _ConditionType>::OnFinalize(DataContextType& dataContext)
    {
        OnExit(dataContext);
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void State<_EventType, _DataContextType, _ActionType, _ConditionType>::OnEntry(DataContextType& dataContext)
    {
        ExecuteAction(m_entry, dataContext);
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void State<_EventType, _DataContextType, _ActionType, _ConditionType>::OnExit(DataContextType& dataContext)
    {
        ExecuteAction(m_exit, dataContext);
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void State<_EventType, _DataContextType, _ActionType, _ConditionType>::ExecuteTransitionAction(const typename State<_EventType, _DataContextType, _ActionType, _ConditionType>::Transition* transition, DataContextType& dataContext)
    {
        if (0 != transition) {
            ExecuteAction(transition->m_action, dataContext);
        }
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void State<_EventType, _DataContextType, _ActionType, _ConditionType>::OnStateChange(State* nextState, _DataContextType& dataContext, typename State<_EventType, _DataContextType, _ActionType, _ConditionType>::Transition* transition)
    {
        if (0 != m_parent) {
            m_parent->OnStateChange(nextState, dataContext, transition);
        }
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    SizeType State<_EventType, _DataContextType, _ActionType, _ConditionType>::GetStateDepth() const
    {
        SizeType stateDepth = 1;
        StateMachineType* state = m_parent;
        while (0 != state) {
            ++stateDepth;
            state = state->m_parent;
        }
        return stateDepth;
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>* State<_EventType, _DataContextType, _ActionType, _ConditionType>::GetParent(SizeType count) const
    {
        if (count > 0) {
            StateMachineType* state = m_parent;
            --count;
            while ((count > 0) && (0 != state)) {
                state = state->m_parent;
                --count;
            }
            return state;
        }
        return 0;
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::OnInit(_DataContextType& dataContext)
    {
        if (0 == m_currentState) {
            Base::OnEntry(dataContext);
            OnStateChange(m_initState, dataContext, 0);
        }
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::OnFinalize(_DataContextType& dataContext)
    {
        if (0 != m_currentState) {
            m_currentState->OnFinalize(dataContext);
            m_currentState = 0;
            Base::OnExit(dataContext);
        }
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::OnUpdate(_DataContextType& dataContext)
    {
        Base::OnUpdate(dataContext);
        if (0 != m_currentState) {
            m_currentState->OnUpdate(dataContext);
        }
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    bool StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::OnEvent(const _EventType& event, _DataContextType& dataContext)
    {
        if (0 != m_currentState) {
            if (m_currentState->OnEvent(event, dataContext)) {
                return true;
            }
        }
        return Base::OnEvent(event, dataContext);
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::OnEntry(_DataContextType& dataContext)
    {
        if ((0 == m_currentState) && (0 != m_initState) && m_performInitStateTransition) {
            m_performInitStateTransition = false;
            OnStateChange(m_initState, dataContext, 0);
        }
        else {
            Base::OnEntry(dataContext);
        }
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::OnExit(_DataContextType& dataContext)
    {
        if (0 != m_currentState) {
            m_currentState->OnExit(dataContext);
            m_currentState = 0;
        }
        m_performInitStateTransition = true;
        Base::OnExit(dataContext);
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::OnStateChange(StateType* _nextState, _DataContextType& dataContext, typename StateType::Transition* transition)
    {
        StateType* nextState = _nextState;
        if (nextState == this) {
            nextState = m_currentState;
        }
        if (nextState != m_currentState) {
            StateMachineType* commonParent = 0;
            if ((0 != m_currentState) && (0 != nextState)) {
                if (m_currentState->m_parent == nextState->m_parent) {
                    commonParent = m_currentState->m_parent;
                }
                StateMachineType* parent = m_currentState->m_parent;
                while ((0 != parent) && (0 == commonParent)) {
                    if (parent == nextState) {
                        commonParent = parent;
                    }
                    parent = parent->m_parent;
                }
                if (0 == commonParent) {
                    parent = nextState->m_parent;
                    while ((0 != parent) && (0 == commonParent)) {
                        if (parent == m_currentState) {
                            commonParent = parent;
                        }
                        parent = parent->m_parent;
                    }
                }
                if (0 == commonParent) {
                    SizeType currentStateDepth = m_currentState->GetStateDepth();
                    SizeType nextStateDepth = nextState->GetStateDepth();
                    SizeType minStateDepth = (currentStateDepth < nextStateDepth) ? currentStateDepth : nextStateDepth;
                    StateMachineType* currentStateParent = m_currentState->GetParent(1 + currentStateDepth - minStateDepth);
                    StateMachineType* nextStateParent = nextState->GetParent(1 + nextStateDepth - minStateDepth);

                    while ((currentStateParent != nextStateParent) && (0 != currentStateParent) && (0 != nextStateParent)) {
                        currentStateParent = currentStateParent->m_parent;
                        nextStateParent = nextStateParent->m_parent;
                    }
                    if (currentStateParent == nextStateParent) {
                        commonParent = nextStateParent;
                    }
                }
            }
            if (0 != m_currentState) {
                StateMachineType* parent = m_currentState->m_parent;
                StateType* state = m_currentState;
                if (parent != commonParent) {
                    if ((0 != commonParent) && (0 != commonParent->m_currentState)) {
                        commonParent->m_currentState->OnExit(dataContext);
                    }
                    else {
                        FEATSTD_STATEMACHINE_PANIC();
                    }
                }
                else {
                    state->OnExit(dataContext);
                }
            }
            this->ExecuteTransitionAction(transition, dataContext);
            if (0 != nextState) {
                StateType* currentState = nextState;
                StateMachineType* parentState = nextState->m_parent;
                if (0 != parentState) {
                    if (0 == commonParent) {
                        do {
                            parentState->m_currentState = currentState;
                            parentState->m_performInitStateTransition = false;
                            currentState = parentState;
                            if (0 != parentState->m_parent) {
                                parentState = parentState->m_parent;
                            }
                        } while (0 != parentState->m_parent);
                    }
                    else {
                        do {
                            parentState->m_currentState = currentState;
                            parentState->m_performInitStateTransition = false;
                            currentState = parentState;
                            parentState = (parentState != commonParent) ? parentState->m_parent : 0;
                        } while (0 != parentState);
                        parentState = commonParent;
                    }
                    if (0 == parentState) {
                        FEATSTD_STATEMACHINE_PANIC();
                    }
                    while (0 != parentState) {
                        if (0 != parentState->m_currentState) {
                            parentState->m_currentState->OnEntry(dataContext);
                        }
                        else {
                            FEATSTD_STATEMACHINE_PANIC();
                        }
                        parentState = (parentState->m_currentState != nextState) ? static_cast<StateMachineType*>(parentState->m_currentState) : 0;
                    }
                }
                else {
                    FEATSTD_STATEMACHINE_PANIC();
                }
            }
        }
        else {
            this->ExecuteTransitionAction(transition, dataContext);
        }
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    inline
        typename StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::StateType* StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::GetCurrentState() const
    {
            return m_currentState;
        }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::InitStates(StateType& initState, SizeType numberOfStates, StateType** states)
    {
        FEATSTD_DEBUG_ASSERT(0 == m_initState);
        FEATSTD_DEBUG_ASSERT(0 != states);
        m_initState = &initState;
#ifdef FEATSTD_DEBUG
        bool foundInitState = false;
#endif
        for (SizeType i = 0; i < numberOfStates; ++i) {
#ifdef FEATSTD_DEBUG
            if (states[i] == m_initState) {
                foundInitState = true;
            }
#endif
            if ((0 != states[i]) && (0 == states[i]->m_parent)) {
                states[i]->m_parent = this;
            }
            else {
                FEATSTD_STATEMACHINE_PANIC();
            }
        }
        FEATSTD_DEBUG_ASSERT(this == initState.m_parent);
        FEATSTD_DEBUG_ASSERT(foundInitState == true);
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void HistoryStateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::OnEntry(_DataContextType& dataContext)
    {
        if ((0 == m_currentState) && (0 != m_historyState) && m_performInitStateTransition) {
            m_performInitStateTransition = false;
            OnStateChange(m_historyState, dataContext, 0);
        }
        else {
            Base::OnEntry(dataContext);
        }
    }

    template < typename _EventType, typename _DataContextType, typename _ActionType, typename _ConditionType >
    void  HistoryStateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::OnExit(_DataContextType& dataContext)
    {
        m_historyState = StateMachine<_EventType, _DataContextType, _ActionType, _ConditionType>::GetCurrentState();
        Base::OnExit(dataContext);
    }
}   //namespace Internal
}   //namespace FeatStd

#endif  // FeatStd_Util_StateMachine_h
