//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_PodUtil_h)
#define FeatStd_Util_PodUtil_h

#include <FeatStd/Util/Traits.h>
#include <FeatStd/Util/Alignment.h>
#include <FeatStd/Util/PointerUtil.h>
#include <FeatStd/MemoryManagement/TypeTraits.h>


/// @addtogroup FEATSTD_UTILS
/// @{

/** Macro to determine the offset of a member in POD type
    FeatStd version of StdC library offsetof macro.
    Applying this operation to non-POD types might yield undefined results. Therefore
    the Container type is checked at compile time. A POD type must be explicitly tagged 
    with FeatStdTypeTrait.
    @code 
    // declare a POD type
    struct Foo {
        UInt32 x;
        UInt32 y;
    };

    // let FeatStd type system know Foo is a POD (must be defined in global scope)
    template<> struct FeatStdTypeTrait<Foo> { enum { IsPodType = true }; };

    // determine offset of member y in Foo
    SizeType offset = FEATSTD_OFFSET_OF(Foo, y);
    @endcode
    @param Container the POD type (typically a struct) 
    @param memberName a member of the POD type */
#define FEATSTD_OFFSET_OF(Container, memberName) (FeatStd::Internal::PointerToScalar<FeatStd::SizeType>(&(FeatStd::Internal::PodUtilPrivate::NullPtrToObject< Container, false >()->memberName)))

/** Macro to determine the offset of a member in POD type
    FeatStd version of StdC library offsetof macro.
    Applying this operation to non-POD types might yield undefined results. This version of 
    the macro will not check POD type trait.
    @code 
    // declare a POD type
    struct Foo {
        UInt32 x;
        UInt32 y;
    };

    // determine offset of member y in Foo
    SizeType offset = FEATSTD_OFFSET_OF(Foo, y);
    @endcode
    @param Container the POD type (typically a struct) 
    @param memberName a member of the POD type */
#define FEATSTD_UNCHECKED_OFFSET_OF(Container, memberName) (FeatStd::Internal::PointerToScalar<FeatStd::SizeType>(&(FeatStd::Internal::PodUtilPrivate::NullPtrToObject< Container, true >()->memberName)))

/** Macro to determine the compound container type from a member pointer
    Applying this operation to non-POD types might yield undefined results. Therefore
    the Container type is checked at compile time. A POD type must be explicitly tagged 
    with FeatStdTypeTrait.
    @code 
    // declare a POD type
    struct Foo {
        UInt32 x;
        UInt32 y;
    };

    // let FeatStd type system know Foo is a POD (must be defined in global scope)
    template<> struct FeatStdTypeTrait<Foo> { enum { IsPodType = true }; };

    Foo foo;
    void *pointerToMember = &foo.y;
    Foo *foo1 = FEATSTD_CONTAINER_OF(pointerToMember, Foo, y);
    assert(foo1 == &foo);
    @endcode
    @param memberPtr pointer to a member of a Container object 
    @param Container the POD type (typically a struct) 
    @param memberName a member of the POD type */
#define FEATSTD_CONTAINER_OF(memberPtr, Container, memberName) (FeatStd::Internal::PodUtilPrivate::ContainerFromMemberPtr< Container, false >((memberPtr), FEATSTD_OFFSET_OF(Container, memberName)))

/** Macro to determine the compound container type from a member pointer
    Applying this operation to non-POD types might yield undefined results. This version
    of the macro will not check POD type trait.
    @code 
    // declare a POD type
    struct Foo {
        UInt32 x;
        UInt32 y;
    };

    Foo foo;
    void *pointerToMember = &foo.y;
    Foo *foo1 = FEATSTD_CONTAINER_OF(pointerToMember, Foo, y);
    assert(foo1 == &foo);
    @endcode
    @param memberPtr pointer to a member of a Container object 
    @param Container the POD type (typically a struct) 
    @param memberName a member of the POD type */
#define FEATSTD_UNCHECKED_CONTAINER_OF(memberPtr, Container, memberName) (FeatStd::Internal::PodUtilPrivate::ContainerFromMemberPtr< Container, true >((memberPtr), FEATSTD_UNCHECKED_OFFSET_OF(Container, memberName)))

/// @}

namespace FeatStd { namespace Internal { namespace PodUtilPrivate {
/// @addtogroup FEATSTD_UTILS
/// @{

    
    /** Helper function for FEATSTD_OFFSET_OF macro
        checks if given type is a POD object and returns a null pointer of
        the Container type. 
        @tparam POD compound type
        @tparam noPodCheck controls of POD type trait is checked 
        @return 0 (null) pointer of type Container */
    template<typename Container, bool noPodCheck> inline Container const volatile* NullPtrToObject() 
    {
        
        FEATSTD_COMPILETIME_ASSERT((FeatStd::TypeTraits::Internal::OrCondition<noPodCheck, FeatStd::TypeTraits::Internal::TypeTrait<Container>::IsPodType>::Value));
        return reinterpret_cast<Container const volatile*>(0);
    }

    /** Helper function for macro FEATSTD_CONTAINER_OF 
        checks the given Container type if it is a POD type and returns pointer to 
        the compound object.
        @tparam Container the POD compound type 
        @tparam noPodCheck controls of POD type trait is checked 
        @tparam T type of the member pointer passed to the function
        @param memberPtr the pointer to the member of the Container object
        @param memberOffset offset of the member referenced by memberPtr in the Container object 
        @return pointer to the Container object containing the member referenced by memberPtr */
    template<typename Container, bool noPodCheck, typename T> 
    inline typename FeatStd::Internal::TypeDecorationTransferer<T, Container>::DecoratedType* ContainerFromMemberPtr(T *memberPtr, SizeType memberOffset)
    {
        FEATSTD_DEBUG_ASSERT(memberPtr != 0);
        FEATSTD_COMPILETIME_ASSERT((FeatStd::TypeTraits::Internal::OrCondition<noPodCheck, FeatStd::TypeTraits::Internal::TypeTrait<Container>::IsPodType>::Value));
        typedef typename FeatStd::Internal::TypeDecorationTransferer<T, UInt8>::DecoratedType Intermediate;
        typedef typename FeatStd::Internal::TypeDecorationTransferer<T, Container>::DecoratedType Dest;

        return FeatStd::Internal::aligned_reinterpret_cast<Dest*>(reinterpret_cast<Intermediate*>(memberPtr) - memberOffset);
    }




/// @}
}}}

#endif
