//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_ObjectBufferTypeProvider_h)
#define FeatStd_Util_ObjectBufferTypeProvider_h

#include <FeatStd/Util/Alignment.h>

namespace FeatStd { 

    /** ObjectBufferTypeProvider defines a type that is capable to host an entity of the given type.
        In place construction of objects requires to define a buffer large enough to store 
        the object. Size is not the the only issue, but certain types might also require
        specific alignments (eg. long double on ARM requires 8 byte alignment).
        Whereever objects are created in place, ObjectBufferTypeProvider shall be used
        to get a type that defines the buffer type. 

        Default buffer type is FeatStd::SizeType that should at least guarantee sizeof(FeatStd::SizeType)*8 bit alignment.
        If a type requires a certain alignment, be sure to define a specialization of 
        ObjectBufferTypeProvider for that type. The type shall always be declared as
        array, even if the size is only on item.

        @param T the type to get the buffer type for */
    template<typename T> struct ObjectBufferTypeProvider {
        typedef FeatStd::SizeType Type[FeatStd::Internal::AlignedTypeSize<T, sizeof(FeatStd::SizeType)>::cAlignedSize / sizeof(FeatStd::SizeType)];
    };

    /** specialization for 8 bit types */
    template<> struct ObjectBufferTypeProvider<char> {
        typedef UInt8 Type[1];
    };

    /** specialization for 8 bit types */
    template<> struct ObjectBufferTypeProvider<unsigned char> {
        typedef UInt8 Type[1];
    };

    /** specialization for 8 bit types */
    template<> struct ObjectBufferTypeProvider<signed char> {
        typedef UInt8 Type[1];
    };

    /** specialization for 16 bit types */
    template<> struct ObjectBufferTypeProvider<short> {
        typedef UInt16 Type[1];
    };

    /** specialization for 16 bit types */
    template<> struct ObjectBufferTypeProvider<unsigned short> {
        typedef UInt16 Type[1];
    };

}

#endif
