//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_LifeTimeManaged_h)
#define FeatStd_Util_LifeTimeManaged_h

#include <stdio.h>
#include <FeatStd/Base.h>

namespace FeatStd {
    namespace Internal {
        /// @addtogroup FEATSTD_UTILS
        /// @{

        /** Controls initialization and destruction of a singleton
            LifeTimeManaged controls the lifetime of the supplied singleton. It guarantees
            that the singleton will be initialized before control enters main function, and
            the singleton will be destroyed after control leaves main function. Destruction of
            the controlled singleton is aligned with inverse order of static object construction.

            As initizalization and destruction of the singleton is done out of main function
            scope, both operations are performed in a single threaded environment.

            The singleton passed as template parameter must expose an OnBeginOfLifeTime and an
            OnEndOfLifeTime method. The methods have following signature:
            @code
                struct SingletonSample {
                    static void OnBeginOfLifeTime();
                    static void OnEndOfLifeTime();
                };
            @endcode
            @tparam Singleton the singleton type or a type interfacing LifeTimeManaged with the singleton */
        template<typename Singleton> 
        class LifeTimeManaged {
            public:
                /** dtor - destroy singleton */
                virtual ~LifeTimeManaged();

                /** start the life time of the managed singleton
                    Global object construction sequence is undefined. It may therefore happen,
                    that construction of a global object already accesses services offered by
                    the managed singleton (e.g. if a custom heap is managed by LifeTimeManaged,
                    allocations may happen before LifeTimeManaged has a chance to invoke
                    Singleton::OnBeginOfLifeTime.
                    The managed singleton will therefore initialize on demand and has to signal
                    this event with StartLifeTime to LifeTimeManaged. LifeTimeManaged will then
                    sort singleton destruction (Singleton::OnEndOdLifeTime) into global object dtor
                    list.
                    mInit guard prevents indirect, recursive invocation of StartLifeTime which
                    would lead to undefined behavior. */
                static void StartLifeTime();

            private:
                /** InitEnforcer is a helper type
                    InitEnforcer enforces invokation of Singleton::OnBeginOfLifeTime before main is entered.
                    To achieve this, LifeTimeManaged defines a static instance of InitEnforcer. This static
                    instance will be initialized as part of global object initialization. */
                struct InitEnforcer {
                    InitEnforcer();

                    LifeTimeManaged& GetInstance();
                };

                static InitEnforcer sInitEnforcer;          ///< enforces the initialization of LifeTimeManaged before main()
                static bool mInit;                          ///< guards indirect recursive initialization

                LifeTimeManaged();

                void Dummy();
        };

        template<typename Singleton>
        LifeTimeManaged<Singleton>::LifeTimeManaged()
        {
            // trigger singleton initialization
            Singleton::OnBeginOfLifeTime();
        }

        template<typename Singleton>
        LifeTimeManaged<Singleton>::~LifeTimeManaged()
        {
            Singleton::OnEndOfLifeTime();
        }

        template<typename Singleton>
        void LifeTimeManaged<Singleton>::StartLifeTime()
        {
            if (!mInit) {
                mInit = true;
                sInitEnforcer.GetInstance().Dummy();
            }
        }

        template<typename Singleton>
        LifeTimeManaged<Singleton>::InitEnforcer::InitEnforcer()
        {
            // enforce construction of LifeTimeManaged
            if (!LifeTimeManaged::mInit) {
                LifeTimeManaged::mInit = true;
                GetInstance().Dummy();
            }
        }

        template<typename Singleton>
        LifeTimeManaged<Singleton>& LifeTimeManaged<Singleton>::InitEnforcer::GetInstance()
        {
            // sLifeTimeManaged will be constructed on first invocation of GetInstance.
            // This will either happen within the constructor of InitEnforcer, or
            // explicitly with StartOfLifeTime. Construction of sLifeTimeManaged adds
            // the object to the global object dtor list. Thus the sLifeTimeManaged will
            // be automatically destructed in reverse order of global object construction
            // after main function terminated. The ctor of LiveTimeManaged will then invoke
            // Singleton::OnEndOfLifeTime

            static LifeTimeManaged sLifeTimeManaged;
            return sLifeTimeManaged;
        }

        template<typename Singleton>
        void LifeTimeManaged<Singleton>::Dummy()
        {
            // this function is intended to be empty. it tricks compiler optimizers
            // not to eliminate sInitEnforcer construction.
        }

        template<typename T> typename LifeTimeManaged<T>::InitEnforcer LifeTimeManaged<T>::sInitEnforcer;
        template<typename T> bool LifeTimeManaged<T>::mInit = false;

        /// @}
    }
}
#endif
