//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_Hash_h)
#define FeatStd_Util_Hash_h

#include <FeatStd/Base.h>

#include <FeatStd/Util/FeatLimits.h>

namespace FeatStd {
/// @addtogroup FEATSTD_UTILS
/// @{
    /** Functions for computing hashes for strings and binary buffers. */
    class Hash
    {
        public:
            static const Char cHashIndicator = '#';
            static const Char cIndexIndicator = '!';
            static const UInt32 cInvalidHashValue = Internal::Limits<UInt32>::cMaxValue;
            static const UInt32 cInitialHashValue = 0;
            static const UInt32 cInitialSeed = 11;

            /**
             *  Calculates hash for a string.
             *  @param  name    The string.
             *  @param  nameLen The string length (optional).
             *  @return number  The computed hash.
             */            
            static UInt32 CalcHash(register const Char *name,
                                   SizeType nameLen = Internal::Limits<UInt32>::cMaxValue);

            /**
             *  Calculates hash for a binary buffer.
             *  @param  buffer    The byte buffer.
             *  @param  bufferLen The length of the buffer.
             *  @return number    The computed hash.
             */            
            static UInt32 CalcBinBufferHash(register const UInt8 * buffer, SizeType bufferLen);

            /**
             *  Starts/Updates the hash value with a binary buffer. To finalize the calculation FinishBinBufferHash has to be called
             *  @param  buffer The begin of the buffer.
             *  @param  end    The end of the buffer.
             *  @param  hash   The current hash value (initial value should be set to cInitialHashValue).
             *  @return True if the update succeeded and false if the ha
             */
            static void UpdateBinBufferHash(register const UInt8 * buffer, register const UInt8 * end, UInt32& hash);

            /**
             *  Finish the hash value with a binary buffer.
             *  @param  hash   The current hash value.
             *  @param  seed   The current seed value (initial value should be set to cInitialSeed).
             *  @return True if the update succeeded and false if the ha
             */
            static bool FinishBinBufferHash(UInt32& hash, UInt32& seed);
    };
/// @}
}
#endif
