//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Guid.h"
#include <FeatStd/Platform/Memory.h>
#include <FeatStd/MemoryManagement/Heap.h>

namespace FeatStd {
    namespace Internal {
        static FeatStd::UInt8 HexToChar(FeatStd::Char c) {
            FeatStd::UInt8 retval = 0;
            switch (c) {
            case '0': retval = 0; break;
            case '1': retval = 1; break;
            case '2': retval = 2; break;
            case '3': retval = 3; break;
            case '4': retval = 4; break;
            case '5': retval = 5; break;
            case '6': retval = 6; break;
            case '7': retval = 7; break;
            case '8': retval = 8; break;
            case '9': retval = 9; break;
            case 'a': case 'A': retval = 10; break;
            case 'b': case 'B': retval = 11; break;
            case 'c': case 'C': retval = 12; break;
            case 'd': case 'D': retval = 13; break;
            case 'e': case 'E': retval = 14; break;
            case 'f': case 'F': retval = 15; break;
            default: break;
            }
            return retval;
        }

        static FeatStd::Char ValueToCharacter(FeatStd::UInt8 value)
        {
            if (value > 9){
                value += static_cast<FeatStd::UInt8>('A') - 0xA;
            }
            else {
                value += value + static_cast<FeatStd::UInt8>('0');
            }
            return static_cast<FeatStd::Char>(value);
        }

        Guid::Guid()
        {
            FeatStd::Internal::Memory::Set(m_data, 0x00, GuidSize);
        }

        Guid::Guid(const Guid& other)
        {
            Memory::Copy(m_data, other.m_data, GuidSize);
        }

        Guid::Guid(const FeatStd::Char*  humanReadableGuid)
        {
            SizeType j = 0;
            SizeType i = 0;
            while (j < GuidSize){
                if (humanReadableGuid[i] != '-'){
                    FeatStd::UInt8 h = FeatStd::Internal::HexToChar(humanReadableGuid[i]);
                    ++i;
                    FeatStd::UInt8 l = FeatStd::Internal::HexToChar(humanReadableGuid[i]);

                    m_data[j] = l | static_cast<FeatStd::UInt8>((h << 4));
                    ++j;
                }
                ++i;
            }
        }

        Guid::~Guid()
        {
#ifdef CANDERA_DEBUG
            FeatStd::Internal::Memory::Set(m_data, 0x0F, GuidSize);
#endif
        }

        void Guid::ToString(FeatStd::Char* dest, const FeatStd::SizeType size) const
        {
            SizeType j = 0;
            SizeType i = 0;
            while ((j < GuidSize) && (i < size)){
                switch (i){
                    //put the '-' at the appropriate positions
                case 8: dest[i] = '-'; ++i; break;
                case 13: dest[i] = '-'; ++i; break;
                case 18: dest[i] = '-'; ++i; break;
                case 23: dest[i] = '-'; ++i; break;
                default:
                    //get upper 4 bit and convert to human readable character
                    dest[i] = FeatStd::Internal::ValueToCharacter((m_data[j] & 0xF0U) >> 0x04U);
                    ++i;
                    //get lower 4 bit and convert to human readable character
                    dest[i] = FeatStd::Internal::ValueToCharacter(m_data[j] & 0x0FU);
                    ++i; ++j;
                    break;
                }
            }
            dest[i] = '\0';
        }

        void Guid::CopyBytes(Guid const & src, FeatStd::UInt8* dest, FeatStd::SizeType destSize)
        {
            FeatStd::SizeType guidsize(static_cast<FeatStd::SizeType>(GuidSize));
            Memory::Copy(dest, src.m_data, guidsize > destSize ? destSize : guidsize);
        }

        bool Guid::operator<(const Guid & other) const
        {
            bool retVal = false;
            for (SizeType i = 0; (i < GuidSize) && (!retVal); ++i){
                if (m_data[i] < other.m_data[i]){
                    retVal = true; //all preceding compares were equal -> terminate
                }
                else {
                    if (m_data[i] > other.m_data[i]){
                        break; //all preceding compares were equal -> this > other ->terminate
                    }
                }
            }
            return retVal;
        }

        bool Guid::IsNil() const
        {
            bool retVal = true;
            for (FeatStd::UInt8 i = 0; retVal && (i < GuidSize); ++i){
                retVal = !(m_data[i] != 0x00);
            }
            return retVal;
        }

        Guid& FeatStd::Internal::Guid::operator=(const Guid & other)
        {
            // check for self-assignment
            if (&other != this){
                Memory::Copy(m_data, other.m_data, GuidSize);
            }
            return *this;
        }
    }
}
