//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Utils_Limits_h)
#define FeatStd_Utils_Limits_h

#include <FeatStd/Base.h>
#include <FeatStd/Util/Traits.h>
#include <FeatStd/Util/WarningMacros.h>

#include <climits>
#include <cfloat>


namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_UTILS
/// @{

    FEATSTD_LINT_MACRO(970, FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT, "[MISRA C++ Rule 3-9-2] - plain type is needed to specify this types properties (limits are defined on native types)")
    FEATSTD_LINT_MACRO(1960, FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT, "MISRA C++ 2008 2-13-4: Lower case literal suffix - dialed in from system include limits.h")

    /** internal macro to create specializations of type NativeTypeLimit
        @param type the type to specify limits for
        @param minVal the minimum value the specified type can store
        @param maxVal the maximum value the specified type can store */
    #define FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT(type, minVal, maxVal) \
        template<> struct NativeTypeLimit<type> { \
            typedef type ThisType; \
            static const ThisType cMinValue = ThisType(minVal); \
            static const ThisType cMaxValue = ThisType(maxVal); \
            static const bool cIsSignedType = ThisType(minVal) < ThisType(0); \
            static inline bool IsSignedType() { return cIsSignedType; } \
            static inline ThisType Min() { return cMinValue; } \
            static inline ThisType Max() { return cMaxValue; } \
        }

    //@{
    /** defines limits (cMinValue and cMaxValue) for standard C++ integral data types */
    template<typename T> struct NativeTypeLimit {
    };

    //                                native type       minVal      maxVal
    FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT(char,             CHAR_MIN,   CHAR_MAX);
    FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT(signed char,      SCHAR_MIN,  SCHAR_MAX);
    FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT(unsigned char,    0,          UCHAR_MAX);
    FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT(short,            SHRT_MIN,   SHRT_MAX);
    FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT(unsigned short,   0,          USHRT_MAX);
    FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT(int,              INT_MIN,    INT_MAX);
    FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT(unsigned int,     0,          UINT_MAX);
    FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT(long,             LONG_MIN,   LONG_MAX);
    FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT(unsigned long,    0,          ULONG_MAX);
    FEATSTD_LINT_NEXT_EXPRESSION(1960, "Violates MISRA C++ 2008 Required Rule 2-13-4: dialed in from system header file")
    FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT(Int64,            LLONG_MIN,  LLONG_MAX);
    FEATSTD_LINT_NEXT_EXPRESSION(1960, "Violates MISRA C++ 2008 Required Rule 2-13-4: dialed in from system header file")
    FEATSTD_PRIVATE_NATIVE_TYPE_LIMIT(UInt64,           0,          ULLONG_MAX);

    //@}

    namespace LimitsPrivate {
        /** dummy type to express absence of LargerType or SmallerType */
        struct InvalidType { };

        /** internal macro to define partial specializations of type LimitsHelper
            @param type the type to specify the LimitsHelper specialization for
            @param largerType the next largest type
            @param smallerType the next smaller type */
        #define FEATSTD_PRIVATE_DEF_LIMIT(type, largerType, smallerType) \
            template<> struct LimitsHelper< type > : public NativeTypeLimit< type > { \
                enum { VariableSize = false }; \
                typedef type ThisType; \
                typedef largerType LargerType; \
                typedef smallerType SmallerType; \
            }

        //@{
        /** LimitsHelper type provides minimum and maximum values of selected types */
        template<typename T> struct LimitsHelper {
            // defaults for types that may vary in size depending on platform
            enum { VariableSize = true };

            /** LargerType defines a type capable to store (T::Max + 1) */
            typedef InvalidType LargerType;
            /** SmallerType defines a type capable to store (T::Min - 1) */
            typedef InvalidType SmallerType;

            /** Min returns the smallest number T can store
                @return smallest number T can store */
            static inline T Min();

            /** Max returns the largest number T can store
                @return largest number T can store */
            static inline T Max();
        };

        //                        type      largerType              smallerType         unsignedType
        FEATSTD_PRIVATE_DEF_LIMIT(UInt64,   InvalidType,            UInt32);
        FEATSTD_PRIVATE_DEF_LIMIT(Int64,    InvalidType,            Int32);
        FEATSTD_PRIVATE_DEF_LIMIT(UInt32,   UInt64,                 UInt16);
        FEATSTD_PRIVATE_DEF_LIMIT(Int32,    Int64,                  Int16);
        FEATSTD_PRIVATE_DEF_LIMIT(UInt16,   UInt32,                 UInt8);
        FEATSTD_PRIVATE_DEF_LIMIT(Int16,    Int32,                  Int8);
        FEATSTD_PRIVATE_DEF_LIMIT(UInt8,    UInt16,                 InvalidType);
        FEATSTD_PRIVATE_DEF_LIMIT(Int8,     Int16,                  InvalidType);

        template<> struct LimitsHelper<Float> {
            typedef double LargerType;
            typedef InvalidType SmallerType;
            enum { VariableSize = false };
            static inline Float Min() { return FLT_MIN; }
            static inline Float Max() { return FLT_MAX; }
            static inline Float Epsilon() { return FLT_EPSILON; }
        };

        template<> struct LimitsHelper<Double> {
            typedef InvalidType LargerType;
            typedef float SmallerType;
            enum { VariableSize = false };
            static inline Double Min() { return DBL_MIN; }
            static inline Double Max() { return DBL_MAX; }
            static inline Double Epsilon() { return DBL_EPSILON; }
        };
        //@}

        template<typename T, bool variableSize> struct LimitsSelectorHelper {
            typedef NativeTypeLimit<T> Type;
        };

        template<typename T> struct LimitsSelectorHelper<T, false> {
            typedef LimitsHelper<T> Type;
        };

        template<typename T> struct LimitsSelector {
            typedef typename QualifierRemover<T>::Type UnqualifiedType;
            typedef typename LimitsSelectorHelper<UnqualifiedType, LimitsHelper<UnqualifiedType>::VariableSize>::Type Type;
        };

        /**
            Checks if n is in range cMin to cMax.
            Some compilers issue warnings for condition n >= cMin where cMin == 0 and the n is an unsigned
            type (unsigned types are always >= zero).
         */
        template<typename N, N n, N cMin, N cMax, bool isSigned = IsUnsigned<N>::Value> struct RangeCheck {
            enum {
                cIsInRange = ((n >= cMin) && (n <= cMax))
            };
        };

        template<typename N, N n, N cMin, N cMax> struct RangeCheck<N, n, cMin, cMax, true> {
            enum {
                cIsInRange = (n <= cMax)
            };
        };

        //@{
        /** TaylorTypeHelper is a helper class to UnsignedIntTypeTylor and TaylorSignedIntType
            TaylorTypeHelper determines for a given number n of type N the smallest integer type
            that can store n.
            @param N the type of the source quantity
            @param n the value of the source quantity
            @param T the type passed in to check if it can store n
            @param LargerType exit condition for partial template specialization */
        template<typename N, N n, typename T, typename SmallerType, bool cond = false> struct TaylorTypeHelper {
            typedef typename LimitsSelector<T>::Type LimitsType;
            enum {
                cIsInRange = RangeCheck<N, n, LimitsType::cMinValue, LimitsType::cMaxValue>::cIsInRange
            };
            typedef typename TaylorTypeHelper<N, n, typename LimitsType::LargerType, T, cIsInRange>::Type Type;
        };

        template<typename N, N n, typename T, typename SmallerType> struct TaylorTypeHelper<N, n, T, SmallerType, true> {
            typedef SmallerType Type;
        };
        //@}

        template<UInt32 byteCount> struct ByteCountMapperHelper {
        };

        template<typename Unsigned> struct ByteCountMapperHelperBase {
            typedef Unsigned UnsignedType;
            typedef typename MakeSigned<Unsigned>::Type SignedType;
        };

        template<> struct ByteCountMapperHelper<1> : ByteCountMapperHelperBase<UInt8> { };
        template<> struct ByteCountMapperHelper<2> : ByteCountMapperHelperBase<UInt16> { };
        template<> struct ByteCountMapperHelper<3> : ByteCountMapperHelperBase<UInt32> { };
        template<> struct ByteCountMapperHelper<4> : ByteCountMapperHelperBase<UInt32> { };
        template<> struct ByteCountMapperHelper<5> : ByteCountMapperHelperBase<UInt64> { };
        template<> struct ByteCountMapperHelper<6> : ByteCountMapperHelperBase<UInt64> { };
        template<> struct ByteCountMapperHelper<7> : ByteCountMapperHelperBase<UInt64> { };
        template<> struct ByteCountMapperHelper<8> : ByteCountMapperHelperBase<UInt64> { };
    }   // namespace LimitsPrivate

    /** Limits type provides minimum and maximum values of selected types */
    template<typename T> struct Limits : public LimitsPrivate::LimitsSelector<T>::Type {
    };

    /** determines the smallest unsigned integer data type capable to store the given value
        @param n the value for which smallest possible type shall be determined
        @code
        typedef typename TaylorUnsignedIntType<0>::Type T1;         // T1 will be UInt8
        typedef typename TaylorUnsignedIntType<70000>::Type T2;     // T2 will be UInt32
        @endcode */
    template<UInt64 n> struct TaylorUnsignedIntType {
        typedef typename LimitsPrivate::TaylorTypeHelper<UInt64, n, UInt8, LimitsPrivate::InvalidType>::Type Type;
    };

    /** determines the smallest signed integer data type capable to store the given value
        @param n the value for which smallest possible type shall be determined
        @code
        typedef typename TaylorSignedIntType<0>::Type T1;         // T1 will be Int8
        typedef typename TaylorSignedIntType<-70000>::Type T2;    // T2 will be Int32
        @endcode */
    template<Int64 n> struct TaylorSignedIntType {
        typedef typename LimitsPrivate::TaylorTypeHelper<Int64, n, Int8, LimitsPrivate::InvalidType>::Type Type;
    };

    /** trait that defines an integer type capable to store at least bitCount number of bits
        @tparam bitCount number of bits the type shall be capable to store */
    template<UInt32 bitCount> struct BitCountToIntegerTypeMapper {
        typedef typename LimitsPrivate::ByteCountMapperHelper<(bitCount + 7) / 8>::UnsignedType UnsignedType;
        typedef typename LimitsPrivate::ByteCountMapperHelper<(bitCount + 7) / 8>::SignedType SignedType;
    };

    /** trait that defines an integer type which size is at least byteCount number of bytes
        sizeof(ByteCountToIntegerTypeMapper<N>::UnsignedType) >=
        @tparam byteCount */
    template<UInt32 byteCount> struct ByteCountToIntegerTypeMapper {
        typedef typename LimitsPrivate::ByteCountMapperHelper<byteCount>::UnsignedType UnsignedType;
        typedef typename LimitsPrivate::ByteCountMapperHelper<byteCount>::SignedType SignedType;
    };

    /**
     * Compute no. of bits necessary to represent a value
     */
    template <UInt16 N> struct BitsNeeded {
        FEATSTD_LINT_NEXT_EXPRESSION(572, "False positive: N is of type UInt8 which has precision greater than 1.")
        static const UInt8 result = 1 + BitsNeeded<(N >> 1)>::result;
    };
    template <> struct BitsNeeded<0> {
        static const UInt8 result = 0;
    };
/// @}
}}
#endif
