//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_CharBuffer_h)
#define FeatStd_Util_CharBuffer_h

#include <FeatStd/Base.h>
#include <FeatStd/Diagnostics/Debug.h>

#include <FeatStd/Diagnostics/ErrorHandling.h>

#include <FeatStd/Platform/VarArg.h>
#include <FeatStd/Platform/String.h>


namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_UTILS
/// @{
/**
 * Helper class to provide a handy way for primitive string operations.
 * @remark Format uses vsnprintf. Which means that the user is responsible for
 *         a sufficient buffer Size!
 * @param   Size the number of bytes that can be stored in the buffer.
 */
template <SizeType Size>
class CharBuffer
{
public:
    /// Initializes buffer to length zero!
    CharBuffer() {
        FEATSTD_COMPILETIME_ASSERT(Size > 0);

        mBuffer[0] = 0x00;
    }
    ~CharBuffer() { }

    SizeType Capacity() const { return Size; }
    const Char* c_str() const { return mBuffer; }

    SizeType Length() const {
        return FeatStd::Internal::String::Length(mBuffer);
    }

    FEATSTD_LINT_IGNORE_RETURN_VALUE(FeatStd::Internal::CharBuffer<*>::Format) // vsnprintf
    UInt32 Format(const Char* format, va_list argList)
    {
        const Int32 lLen = Format(mBuffer, Size, format, argList);
        FEATSTD_DEBUG_ASSERT(lLen >= 0);
        const UInt32 lULen(static_cast<UInt32>(lLen));
        FEATSTD_DEBUG_ASSERT(lULen < Size); // PANIC otherwise in Format(...) above!

        return lULen;
    }

    FEATSTD_LINT_IGNORE_RETURN_VALUE(FeatStd::Internal::CharBuffer<*>::Format) // vsnprintf
    UInt32 Format(const Char* format, ...)
    {
        FEATSTD_VALIST lArgs;
        FEATSTD_VASTART(lArgs, format);
        const Int32 lInfoLen = Format(format, lArgs);
        FEATSTD_VAEND(lArgs);

        return lInfoLen;
    }

    /**
     *  Implemented here to avoid Integrity compile problems.
     *  Ideally this function lies in a static base class.
     *  @param buffer The buffer that is used.
     *  @param bufferLength The buffer length of the buffer.
     *  @param format The format of the buffer.
     *  @param argList The additional parameters.
     *  @return The total number of characters written
     */
    static UInt32 Format(Char* buffer, SizeType bufferLength, const Char* format, va_list argList)
    {
        FEATSTD_DEBUG_ASSERT(buffer != 0);
        if (buffer == 0) {
            return 0; // Robustness
        }
        FEATSTD_DEBUG_ASSERT(bufferLength > 0);
        if (bufferLength <= 0) {
            return 0; // Robustness
        }
        FEATSTD_DEBUG_ASSERT(format != 0);
        if (format == 0) {
            return 0; // Robustness
        }
        const Int32 lLen = Impl::String::VStringPrintf(buffer, bufferLength, format, argList);

        FEATSTD_DEBUG_ASSERT(lLen >= 0);
        return static_cast<UInt32>(lLen);
    }

private:
    Char mBuffer[Size];

    FEATSTD_MAKE_CLASS_UNCOPYABLE(CharBuffer);
};
/// @}
}}
#endif
