//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_BitKit_h)
#define FeatStd_Util_BitKit_h

#include <FeatStd/Util/Traits.h>
#include <FeatStd/Util/FeatLimits.h>
#include <FeatStd/Diagnostics/Debug.h>

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_UTILS
/// @{
/**
 *  @brief
 */
class BitKit
{
public:
    /**
     *  Tests in source the bit given bit
     *  @param source the value to be tested
     *  @param offset the bit to be tested
     *  @return true if the bit is set, false otherwise
     */
    template<typename T> inline static bool TestBit(T source, UInt32 offset) {
        FEATSTD_DEBUG_ASSERT(offset < (sizeof(T) * 8));
        FEATSTD_COMPILETIME_ASSERT(FeatStd::Internal::IsUnsigned<T>::Value);
        return (source & (T(1U) << offset)) != 0;
    }

    /**
     *  Sets the specified bit in the given value
     *  @param source the value to change
     *  @param offset the bit to set
     */
    template<typename T> inline static void SetBit(T &source, UInt32 offset) {
        FEATSTD_DEBUG_ASSERT(offset < (sizeof(T) * 8));
        FEATSTD_COMPILETIME_ASSERT(FeatStd::Internal::IsUnsigned<T>::Value);
        source |= (T(1U) << offset);
    }

    /**
     *  Clears the specified bit in the given value
     *  @param source the value to change
     *  @param offset the bit to clear
     */
    template<typename T> inline static void ClearBit(T &source, UInt32 offset) {
        FEATSTD_DEBUG_ASSERT(offset < (sizeof(T) * 8));
        FEATSTD_COMPILETIME_ASSERT(FeatStd::Internal::IsUnsigned<T>::Value);
        source &= ~(T(1U) << offset);
    }

    /**
     *  Toggles the specified bit in the given value
     *  @param source the value to change
     *  @param offset the bit to toggle
     */
    template<typename T> inline static void ToggleBit(T &source, UInt32 offset) {
        FEATSTD_DEBUG_ASSERT(offset < (sizeof(T) * 8));
        FEATSTD_COMPILETIME_ASSERT(FeatStd::Internal::IsUnsigned<T>::Value);
        source ^= (T(1U) << offset);
    }

    /**
     *  Counts the number of bits set in source
     *  @param source the value which bit shall be counted
     *  @return the number of bits set in source
     */
    static UInt8 CountBits(UInt32 source);

    /**
     *  extracts number bits starting from offset from the given value
     *  @param source the value to extract bits from 
     *  @param offset the start index of the bits to extract
     *  @param number the number of bits to extract
     *  @return the extracted bits
     */
    template<typename T> static T ExtractBits(T source, UInt32 offset, UInt32 number) {
        FEATSTD_DEBUG_ASSERT(offset < (sizeof(T) * 8));
        FEATSTD_DEBUG_ASSERT(number <= (sizeof(T) * 8));
        FEATSTD_DEBUG_ASSERT((offset + number) <= (sizeof(T) * 8));

        return (source >> offset) & (Limits<T>::cMaxValue >> (((sizeof(T) * 8) - number)));
    }
};
/// @}
}}
#endif
