//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Util_Alignment_h)
#define FeatStd_Util_Alignment_h

#include <FeatStd/Util/FeatLimits.h>
#include <FeatStd/Util/Traits.h>
#include <FeatStd/Diagnostics/Debug.h>

namespace FeatStd { namespace Internal { 

    namespace AlignmentPrivate {

        /** helper to enforce byteAlignment to be a power of two value value */
        template<FeatStd::SizeType byteAlignment, bool validAlignment = IsPowerOfTwo<byteAlignment>::Value> struct Helper {
            enum {
                cByteAlignment = byteAlignment
            };
        };

        /** specialization for non power of two alignment values */
        template<FeatStd::SizeType byteAlignment> struct Helper<byteAlignment, false> {
        };
    }

    /** ByteAlignment class exposes functions to align address or sizes to a specific byte alignment
        @tparam alignment the byte alignment of interest 
        @tparam validAlign denotes of the given byte alignment is a valid (power of 2) value */
    template<UInt32 alignment> class ByteAlignment {
        public:
            enum {
                cByteAlignment = AlignmentPrivate::Helper<alignment>::cByteAlignment
            };

            /** aligns the given pointer 
                @tparam T the pointer type
                @param ptr the pointer to align
                @return the aligned pointer */
            template<typename T> static inline T* AlignAddress(T *ptr) {                
                return reinterpret_cast<T*>(reinterpret_cast<TransferType>(ptr) & cUnmask);
            }

            /** rounds up the given quantity to the aligned size >= quantity
                @param byteSize the quantity
                @return the aligned quantity */
            template<typename T> static inline T RoundUp(const T &byteSize) {
                return (byteSize + T(cByteAlignment - 1)) & cUnmask;
            }

            /** rounds down the given quantity to the aligned size <= quantity
                @param byteSize the quantity
                @return the aligned quantity */
            template<typename T> static inline T RoundDown(const T &byteSize) {
                return byteSize & T(cUnmask);
            }

            /** checks if the given pointer is aligned 
                @param ptr the pointer to check 
                @return true, if the pointer address is aligned, false otherwise */
            static inline bool CheckAddress(const void *ptr) {
                return (reinterpret_cast<TransferType>(ptr) & cMask) == 0;
            }

            /** Checks if the given quantity is aligned
                @tparam T the type of the quantity parameter
                @param val the quantity
                @return true if quantity is aligned, false otherwise*/
            template<typename T> static inline bool CheckSize(const T &val) {
                return (val & cMask) == 0;
            }

        private:
            /** scalar type capable to store a pointer address. needed for pointer address manipulation */
            typedef typename ByteCountToIntegerTypeMapper<sizeof(void*)>::UnsignedType TransferType;

            static const TransferType cMask = TransferType(cByteAlignment - 1);
            static const TransferType cUnmask= ~cMask;
    };

    /** specialization for trivial single byte alignment */
    template<> class ByteAlignment<1> {
        public:
            template<typename T> static inline T* AlignAddress(T *ptr) { return ptr; }
            template<typename T> static inline T RoundUp(const T &byteSize) { return byteSize; }
            template<typename T> static inline T RoundDown(const T &byteSize) { return byteSize; }
            static inline bool CheckAddress(const void*) { return true; }
            template<typename T> static inline bool CheckSize(const T&) { return true; }
    };

    // ========================================================================

    /** BitAlignment class - generic definition for valid (multiple of 8) bit alignments */
    template<UInt32 bits, bool validBitCount = (bits % 8 == 0)> class BitAlignment  : public ByteAlignment<bits / 8> { 
    };

    /** specialization for invalid alignments */
    template<UInt32 bits> class BitAlignment<bits, false> { 
    };

    // ========================================================================

    /** The default alignment is defined by the size of a pointer */
    typedef ByteAlignment<sizeof(void*)> DefaultAlignment;

    /** 32 bit alignment */
    typedef BitAlignment<32> Align32Bit;

    /** 64 bit alignment */
    typedef BitAlignment<64> Align64Bit;

    // ========================================================================

    /** static size alignment calculation. can be used in declarations */
    template<UInt32 value, FeatStd::SizeType byteAlignment = 4> struct AlignedByteSize {
        enum {
            cByteAlignment = AlignmentPrivate::Helper<byteAlignment>::cByteAlignment
        };

        static const UInt32 cAlignedUpperValue = ((value + cByteAlignment - 1) / cByteAlignment) * cByteAlignment;
        static const UInt32 cAlignedLowerValue = (value / cByteAlignment) * cByteAlignment;
    };

    /** default alignment (32bit) */
    template<UInt32 value> struct AlignedByteSize<value, 0> {
        enum {
            cByteAlignment = DefaultAlignment::cByteAlignment
        };

        static const UInt32 cAlignedUpperValue = ((value + 3) / 4) * 4;
        static const UInt32 cAlignedLowerValue = (value / 4) * 4;
    };

    // ========================================================================

    /** defines aligned size for the given type and alignment
        @tparam T the type which size should be determined 
        @tparam byteAlignment the alignment that should be applied to the size. 
                0 means default alignment (use alignment as tool chain aligns objects in an array) */
    template<typename T, FeatStd::SizeType byteAlignment = 0> struct AlignedTypeSize {
        static const SizeType cAlignedSize = AlignedByteSize<(sizeof(T[2]) / 2), byteAlignment>::cAlignedUpperValue;
    };

    template<typename T> struct AlignedTypeSize<T, 0> {
        static const SizeType cAlignedSize = sizeof(T[2]) / 2;
    };


    // ========================================================================

    /// @{
    /** cast operator that takes care about alignment 
        Some compilers (ARM GCC) issue a warning when a pointer is casted to another pointer
        type with more strict alignment requirements (eg. reinterpret_cast<UInt32*>(UInt8*))
        two static casts with an intermediate cast to const avoid these warnings
        
        Correct alignment of the pointers must be ensured by the caller of the operator. 
        The cast operator will not check the pointers for correct alignment!

        @tparam D the resulting type of the cast operation
        @param s the source pointer
        @returns the destination pointer of type D
    */
FEATSTD_SUPPRESS_GCC_WARNING_BEGIN("-Wstrict-aliasing", "Correct alignment of the pointers must be ensured by the caller of the operator.")

    // pointer type variant
    template<typename D> inline D aligned_reinterpret_cast(void *s)
    {
        return static_cast<D>(s);
    }

    // const pointer type variant
    template<typename D> inline D aligned_reinterpret_cast(const void *s)
    {
        return static_cast<D>(s);
    }

    // volatile pointer type variant
    template<typename D> inline D aligned_reinterpret_cast(volatile void *s)
    {
        return static_cast<D>(s);
    }

    // const volatile pointer type variant
    template<typename D> inline D aligned_reinterpret_cast(const volatile void *s)
    {
        return static_cast<D>(s);
    }

    // specific variant for pointer to member
    template<typename D, typename S, typename M> inline D aligned_reinterpret_cast(M S::* s)
    {
        typedef typename FeatStd::Internal::TypeDecorationTransferer<S, void>::DecoratedType C;
        return *static_cast<D*>(static_cast<C*>(&s));
    }
    
FEATSTD_SUPPRESS_GCC_WARNING_END()
    /// @}

}}
#endif
