//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef FeatStd_Platform_TcpIp_h
#define FeatStd_Platform_TcpIp_h

#include <FeatStd/Base.h>
#include <FeatStd/Platform/ICommNode.h>
#include <FeatStd/Platform/CommunicationSettings.h>

#include FEATSTD_PLATFORM_INCLUDE(TcpIp)

namespace FeatStd{
    namespace Internal{
        /// @addtogroup FEATSTD_PLATFORM
        /// @{
        /**<em>TcpIp</em> is a communication abstraction class for tcp-ip connections.
        It supports ICommNode interface.
        Features which are supported by this class: TCP/IP, Single Server - Single Client connection,
                                                   can be server as well as client, custom ip and port
        */
        class TcpIp : private Impl::TcpIp, public ICommNode{
            typedef Impl::TcpIp Base;
        public:
            /** <em>Configure</em> passes an undefined structure pointer through to concrete communication type via static typecast. \p
            This will add new settings infromation. To activate these settings Close() an opened connection
            and reopen it with Open() \p
            The structure contains all settings that can be changed by given communication type \p
            For further settings details and the structures @see FeatStd/Platform/CommunicationSettings.h
            @param configData   pointer to settings structure
            @return             <em>true</em> if copying all data succeeded
            <em>false</em> if error occurred while copying the settings  */
            bool Configure(void * configData);
            /** <em>Configure</em> passes an defined structure pointer through to concrete communication type. \p
            This will add new settings infromation. To activate these settings Close() an opened connection
            and reopen it with Open() \p
            The structure contains all settings that can be changed by given communication type \p
            For further settings details and the structures @see FeatStd/Platform/CommunicationSettings.h
            @param configData   pointer to tcp/ip specific settings structure
            @return             <em>true</em> if copying all data succeeded
            <em>false</em> if error occurred while copying the settings  */
            bool Configure(TcpIpSettings const * configData);
            /** <em>Open</em> opens connection of given communication type without handshake
            @return             <em>true</em>  if opening succeeded
            <em>false</em> if error occurred while opening connection */
            bool Open(){ return Open(0); }
            /** <em>Open</em> opens connection of given communication type
            @param handshake    handshake function pointer to establish a connection
            @return             <em>true</em>  if opening succeeded
            <em>false</em> if error occurred while opening connection */
            bool Open(ConnectionHandshake handshake);
            /** <em>Connect</em> opens connection of given communication type to be compatible with ICommNode
            use Open instead.
            @param handshake    optional handshake function pointer to establish a connection
            @return             <em>true</em>  if opening succeeded
            <em>false</em> if error occurred while opening connection */
            bool Connect(ConnectionHandshake handshake = 0);
            /** <em>Close</em> closes current communication
            @return             <em>true</em> if closing succeeded
            <em>false</em> if error occurred while closing connection */
            bool Close();
            /** <em>IsOpen</em> checks whether connection is established or not. To be compatible with ICommNode
            use Connected instead.
            @return             <em>true</em> if connection is established
            <em>false</em> if connection is not established  */
            bool IsOpen();
            /** <em>Connected</em> checks whether connection is established or not
            @return             <em>true</em> if connection is established
            <em>false</em> if connection is not established  */
            bool Connected();
            /** <em>DataAvailable</em> checks whether data is available or not
            @param available    out-parameter which is set to true when data is available else it is false
            @return             <em>true</em> if setting available worked
            <em>false</em> if error occurred while retrieving available state  */
            bool DataAvailable(bool& available);
            /** <em>Read</em> reads given amount of bytes (as a maximum).\p
            Further configuration is set by settings.
            Currently supported are read pending only data and read amount of bytes given by length
            @param buffer       buffer in which retrieved data will be stored.
            @param length       Amount of bytes to be read
            @return             <0: Error occurred while reading data \p >=0: Amount of bytes read */
            Int32 Read(UInt8*buffer, UInt32 length);
            /** <em>Write</em> writes given buffer into the communication stream
            @param buffer       buffer which is sent
            @param length       Amount of bytes to be written
            @return             <0: Error occurred while writing data \p >=0: Amount of bytes written  */
            Int32 Write(const UInt8* buffer, UInt32 length);

            /** <em>GetConfiguration</em>
            @return             returns current configuration of  communication node  */
            void const * GetConfiguration() const;

            /** <em>CopySettings</em> safe copy of settings 
            @param to    destination structure
            @param from  source structure
            @return             <em>true</em> if copy settings worked
                                <em>false</em> if error occurred  */
            static bool CopySettings(TcpIpSettings * const to, TcpIpSettings const * const from){
                bool copyable = from != 0 && to != 0;
                if (copyable)
                {
                    *to = *from;
                }
                return copyable;
            }
        };
        inline void const * TcpIp::GetConfiguration() const{
            return static_cast<void const *>(Base::GetConfiguration());
        }
        inline bool TcpIp::Configure(void * configData)
        {
            return Configure(static_cast<TcpIpSettings*>(configData));
        }
        inline bool TcpIp::Configure(TcpIpSettings const * configData)
        {
            return Base::Configure(configData);
        }

        inline bool TcpIp::DataAvailable(bool& available)
        {
            return Base::DataAvailable(available);
        }

        inline Int32 TcpIp::Read(UInt8* buffer, UInt32 length)
        {
            return Base::Read(buffer,length);
        }

        inline Int32 TcpIp::Write(const UInt8* buffer, UInt32 length)
        {
            return Base::Write(buffer,length);
        }

        inline bool TcpIp::Close()
        {
            return Base::Close();
        }

        inline bool TcpIp::Open(ConnectionHandshake handshake)
        {
            return Connect(handshake);
        }
        inline bool TcpIp::Connect(ConnectionHandshake handshake)
        {
            return Base::Connect(handshake);
        }
        inline bool TcpIp::IsOpen()
        {
            return Connected();
        }
        inline bool TcpIp::Connected()
        {
            return Base::Connected();
        }
        /// @}
    }
}



#endif
