//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_String_h)
#define FeatStd_Platform_String_h

#include <FeatStd/Platform/Base.h>
#include <FeatStd/Platform/CodePointIterator.h>
#include <FeatStd/Platform/VarArg.h>

#include FEATSTD_PLATFORM_INCLUDE(String)

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{
    /** Generic implementation of string functions using <string.h>. */
    struct String : private FeatStd::Internal::Impl::String {
        /**
         * Compares two null-terminated character strings given.
         * return zero if the strings are identical, positive value if the left char string is greater, and negative value
         * if the left char string is less than right char string.
         * @param lhs First string to be compared.
         * @param rhs Second String to be compared.
         * @return    Zero if the strings are identical, positive value if the left char string is greater, and negative value
         *            if the left char string is less than right char string
         */
        static inline Int32 CompareStrings(const Char* lhs, const Char* rhs) {
            return (lhs != rhs) ? Impl::String::CompareStrings(lhs, rhs) : 0;
        }

        /**
         * Compares two strings formed by the first n characters from two given strings.
         * return zero if the strings are identical, positive value if the left char string is greater, and negative value
         * if the left char string is less than right char string.
         * @param lhs    First string to be compared.
         * @param rhs    Second String to be compared.
         * @param nChars Maximum number of characters to compare.
         * @return       Zero if the strings are identical, positive value if the left char string is greater, and negative value
         *               if the left char string is less than right char string.
         */
        static inline Int32 ComparePartial(const Char* lhs, const Char* rhs, SizeType nChars) {
            return (lhs != rhs) ? Impl::String::ComparePartial(lhs, rhs, nChars) : 0;
        }

        /**
         * Copy string from to to
         * @param to   Destination string to copy to.
         * @param from Source string to copy from.
         */
        static inline void Copy(Char *to, const Char *from) {
            Impl::String::Copy(to, from);
        }

        /**
         * Copy nChars characters of string from to string to
         * @param to     Destination string to copy to.
         * @param from   Source string to copy from.
         * @param nChars Maximum number of characters to be copied from source.
         */
        static inline void CopyPartial(Char *to, const Char *from, SizeType nChars) {
            Impl::String::CopyPartial(to, from, nChars);
        }

        /**
         * Returns the length of string str
         * @param str String to retrieve length from.
         * @return    The length of string str.
         */
        static inline SizeType Length(const Char *str) {
            return Impl::String::Length(str);
        }

        /**
         * Returns the length of string str
         * @param str String to retrieve length from.
         * @param size Maximum size of the string to prevent buffer overflow.
         * @return    The length of string str.
         */
        static inline SizeType NLength(const Char *str, SizeType size) {
            return Impl::String::NLength(str, size);
        }

        /**
        * Converts a text string of a hexadecimal representation of a number to an integer.
        * @param str string of a a hexadecimal representation of a number.
        * @return Parsed integer
        */
        static inline UInt32 HexStringToInteger(const Char* str) {
            return Impl::String::HexStringToInteger(str);
        }

        /**
        * Returns the length of code points in string
        * @param string String to retrieve length from.
        * @return The length code points of the string.
        */
        static inline SizeType TLength(const TChar *string) {
            return Impl::String::TLength(string);
        }

        /** safe version of sprintf 
            The function will return a failure if the number of bytes to be written
            exceeds bufferSize.
            @param buffer buffer to receive the output
            @param bufferSize the size of the buffer in bytes
            @param fmt a printf style format string
            @return number of bytes written to buffer or -1 in case of failure */
        static Int StringPrintf(Char *buffer, SizeType bufferSize, const Char *fmt, ...) {
            FeatStd::Internal::VarArg::VaList argp;
            FEATSTD_VASTART(argp, fmt);
            Int rc = Impl::String::VStringPrintf(buffer, bufferSize, fmt, argp);
            FEATSTD_VAEND(argp);
            return rc;
        }

        /** Platform abstraction of sscanf
            @param buffer buffer containing the data
            @param format a scanf style format string
            @return number of bytes read from buffer or -1 in case of failure */
        static Int StringScanf(const FeatStd::Char *buffer, const FeatStd::Char* format, ...) {
            FeatStd::Internal::VarArg::VaList argp;
            FEATSTD_VASTART(argp, format);
            Int rc = Impl::String::VStringScanf(buffer, format, argp);
            FEATSTD_VAEND(argp);
            return rc;
        }

        /** CodePointIterator defines the class that can be used to iterate through TString code points. */
        typedef Impl::String::CodePointIterator CodePointIterator;
    };
/// @}
}}
#endif
