//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#include <FeatStd/Platform/Thread.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#include <FeatStd/Platform/Memory.h>
#include <FeatStd/Platform/String.h>
#include <FeatStd/Util/PointerUtil.h>
#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Diagnostics/Log.h>

#include <FeatStd/MemoryManagement/Heap.h>

#include <sched.h>

FEATSTD_LINT_FILE(829, errno.h, "platform dependent file")
//lint -efile(829, errno.h)  platform dependent file
#include <errno.h>
FEATSTD_LINT_FILE(829, signal.h, "platform dependent file")
#include <signal.h>
FEATSTD_LINT_FILE(829, time.h, "platform dependent file")
#include <time.h>

#include <unistd.h>


#ifdef __linux__
#include <sys/resource.h>
#include <sys/syscall.h>
#include <sys/types.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,8)
#include <sys/prctl.h>
#endif
#endif


namespace FeatStd { namespace Internal { namespace Posix {

FEATSTD_LOG_SET_REALM(FeatStd::Diagnostics::LogRealm::FeatStdPlatform);

#define LOG_SYSCALL_ERROR(syscall, error)   \
    FEATSTD_LOG_ERROR( \
            "Thread %s[%d]: ::%s() failed (Error %d - %s)", mName, GetId(), syscall, error, \
            (error == EAGAIN)  ? "EAGAIN" : \
            (error == EINVAL) ? "EINVAL" : \
            (error == EPERM) ? "EPERM" : \
            (error == EACCES) ? "EACCES" : \
            (error == ESRCH) ? "ESRCH" : \
            (error == ENOMEM) ? "ENOMEM" : \
            "?" \
            )
#define LOG_SYSCALL_ERROR_STATIC(syscall)   \
    FEATSTD_LOG_ERROR("SysCall ::%s() failed (Error %d - %s)", syscall, errno, \
            (errno == EAGAIN)  ? "EAGAIN" : \
            (errno == EINVAL) ? "EINVAL" : \
            (errno == EPERM) ? "EPERM" : \
            (errno == EACCES) ? "EACCES" : \
            (errno == ESRCH) ? "ESRCH" : \
            (errno == ENOMEM) ? "ENOMEM" : \
            "?" \
            );

    FEATSTD_LINT_CURRENT_SCOPE(1401, "Violates MISRA C++ 2008 Required Rule 8-5-1: false positive, not initialized is wrong")
    FEATSTD_LINT_CURRENT_SCOPE(1960, "Violates MISRA C++ 2008 Required Rule 5-18-1: false positive, comma operator is not used")
    FEATSTD_LINT_CURRENT_SCOPE(774, "Violates MISRA C++ 2008 Required Rule 0-1-2: false positive as wrongly identified as comma operator")
    FEATSTD_LINT_CURRENT_SCOPE(948, "Violates MISRA C++ 2008 Required Rule 0-1-9: false positive as wrongly identified as comma operator")
    FEATSTD_LINT_CURRENT_SCOPE(1013, "false positive, tv_sec and tv_nsec are members of timespec")
    // ------------------------------------------------------------------------
    SizeType PosixThread::GetCurrentId()
    {
        FEATSTD_LINT_NEXT_EXPRESSION(1924, "Violates MISRA C++ Required Rule 5-2-4: C-style cast is needed because of varying typedefs for pthread_t on Linux & Integrity")
        FEATSTD_LINT_NEXT_EXPRESSION(923, "Cast to unsigned int type is required.")
        return (SizeType)::pthread_self();
    }

    // ------------------------------------------------------------------------
    SizeType PosixThread::GetCurrentSystemId()
    {
#ifdef __linux__
        return (SizeType)syscall(SYS_gettid);
#else
        return GetCurrentId();
#endif
    }

    // ------------------------------------------------------------------------
    void PosixThread::Sleep(UInt32 timeMs)
    {
        bool lRc = (0 == ::usleep(timeMs * 1000U));
        if (!lRc) {
            LOG_SYSCALL_ERROR_STATIC("usleep");
        }
    }

    // ------------------------------------------------------------------------
    void PosixThread::NanoSleep(UInt32 seconds, UInt32 nanoSeconds)
    {
        FEATSTD_LINT_SYMBOL(550, seconds, "[MISRA C++ Rule 0-1-4] seconds is not accessed is a false positive")
        FEATSTD_LINT_SYMBOL(550, nanoSeconds, "[MISRA C++ Rule 0-1-4] nanoSeconds is not accessed is a false positive")
        FEATSTD_LINT_SYMBOL(550, timeSpec, "[MISRA C++ Rule 0-1-4] timeSpec is not accessed is a false positive")

        struct timespec timeSpec;
        timeSpec.tv_sec = seconds;
        timeSpec.tv_nsec = nanoSeconds;

        Int rc = static_cast<Int>(::nanosleep( &timeSpec, 0 ));
        if (rc != 0) {
            LOG_SYSCALL_ERROR_STATIC("nanosleep");
        }
    }

    // ------------------------------------------------------------------------
    bool PosixThread::Yield()
    {
        bool lRc = (0 == ::sched_yield());
        if (!lRc) {
            LOG_SYSCALL_ERROR_STATIC("sched_yield");
        }
        return (lRc);
    }
    // ------------------------------------------------------------------------
    void * PosixThread::StartProc(void * arg)
    {
        {
            // cancel can happen any time
            Int rc = ::pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, 0);
            if (rc != 0) {
                LOG_SYSCALL_ERROR_STATIC("pthread_setcancelstate");
            }
        }

        FEATSTD_DEBUG_ASSERT(0 != arg);
        if (0 == arg) {
            FEATSTD_LOG_FATAL("StartProc has no argument!");
            FEATSTD_LINT_NEXT_EXPRESSION(923, "Cast to pointer type is required.")
            return reinterpret_cast<void *>(-4);
        }

        PosixThread * lThread = FeatStd::Internal::PointerToPointer<PosixThread*>(arg);

        FEATSTD_DEBUG_ASSERT(0 != lThread);
        if (0 == lThread) {
            FEATSTD_LOG_FATAL("StartProc argument is invalid (lThread = 0)!");
            FEATSTD_LINT_NEXT_EXPRESSION(923, "Cast to pointer type is required.")
            return reinterpret_cast<void *>(-2);
        }

        // cache the ID of this thread.
        lThread->mCachedSystemId = lThread->GetCurrentSystemId();
        // Call user thread function
        lThread->OnExecute();
        FEATSTD_LOG_INFO("Thread '%s' (Id %d) is running", lThread->GetName(), lThread->GetId());


        {
            CriticalSectionLocker lLock(&lThread->mCs);

            if (lThread->mNice != 0)
            {
                lThread->SetNiceLevelImpl();
            }
            lThread->SetStatus(ThreadStatus::Running);
        }

        UInt32 lExitCode = lThread->ThreadFn();


        lThread->OnTerminate(lExitCode);
        lThread->mCachedSystemId = 0;

        FEATSTD_LOG_INFO("%s: leaving StartProc\n", lThread->mName);
        lThread->SetStatus(ThreadStatus::Terminated);

        FEATSTD_LINT_NEXT_EXPRESSION(923, "Cast to pointer type is required.")
        return reinterpret_cast<void *>(lExitCode);
    }

    // ------------------------------------------------------------------------
    PosixThread::PosixThread() :
        mJoined(false),
        mStatus(ThreadStatus::Invalid),
        mThread(0),
        mCachedSystemId(0),
        mNice(0),
        mNiceCallback(static_cast<SetNiceCallback>(0))
    {
        FeatStd::Internal::Memory::Set(mName, '\0', sizeof(mName));
    }

    // ------------------------------------------------------------------------
    PosixThread::~PosixThread()
    {
        ThreadStatus::Enum state = GetStatus();
        if ((state == ThreadStatus::Started) || (state == ThreadStatus::Running)) {
            FEATSTD_LOG_WARN("DTOR called on running thread '%s' (Id %u)", mName, GetId());
            FEATSTD_LINT_NEXT_EXPRESSION(534, "No need of return value of Kill() in ~Win32Thread()")
            Kill(0);
        }
        mJoined = false;
        mCachedSystemId = 0;
    }

    /** Map posix thread policy and priority to generic thread priority
     *  @param policy POSIX thread policy
     *  @param priority POSIX thread priority
     *  @return Thread genericy priority
     */
    static ThreadPriority::Enum PriorityPosixToGeneric(const int policy, const int priority);

    /** Map generic thread priority to posix thread policy and priority
     *  @param thPriority Thread generic priority
     *  @param policy[out] POSIX thread policy
     *  @param priority[out] POSIX thread priority
     *  @return Actual generic priority that could be mapped to POSIX
     *
     */
    static ThreadPriority::Enum PriorityGenericToPosix(ThreadPriority::Enum thPriority, int &policy, int &priority);

    // ------------------------------------------------------------------------
    bool PosixThread::Run(ThreadPriority::Enum thPriority, SetPriorityCallback Callback)
    {
        FEATSTD_LOG_DEBUG("Priority = %d", thPriority);
        {
            CriticalSectionLocker lLock(&mCs);
            if ((mStatus == ThreadStatus::Started) || (mStatus == ThreadStatus::Running)) {
                FEATSTD_LOG_WARN("Thread %s[%u] was already started", mName, GetId());
                return false;
            }
            mStatus = ThreadStatus::Started;
        }

        if (mName[0] == '\0') {
            static Int lThreadCount = 0;
            if (0 > sprintf(mName, ThreadInvariant::ThreadNameFormat, lThreadCount)) {
                FEATSTD_LOG_WARN("Setting default thread name (Courier-0x%x) failed (now using name %s)", lThreadCount, mName);
            }
            FEATSTD_DEBUG_ASSERT(FeatStd::Internal::String::Length(mName)<=ThreadInvariant::NameLength);
            FEATSTD_DEBUG_ASSERT(FeatStd::Internal::String::Length(mName)<ThreadInvariant::NameBufferLength);
            lThreadCount++;
        }

        FEATSTD_LOG_INFO("Thread %s is starting", mName);

        OnStart();

        Int lResult = 0;
        {
            pthread_attr_t lAttr;
            FEATSTD_LINT_NEXT_EXPRESSION(838, "Violates MISRA C++ 2008 Required Rule 0-1-9: previously assigned value")
            lResult = ::pthread_attr_init(&lAttr);

            if (0 != lResult) {
                LOG_SYSCALL_ERROR("pthread_attr_init", lResult);
            } else {
                bool useCallback = (Callback != static_cast<SetPriorityCallback>(0));
                int posixPolicy = SCHED_OTHER;
                struct ::sched_param posixPriority;
                posixPriority.sched_priority = 0;

                if (thPriority <= ThreadPriority::REAL_TIME)
                {
                    // Set thread priority
                    ThreadPriority::Enum thFinalPriority = PriorityGenericToPosix(thPriority, posixPolicy, posixPriority.sched_priority);
                    FEATSTD_LOG_DEBUG("thFinalPriority = %d, posixPolicy = %s, posixPriority = %d",
                            thFinalPriority,
                            (posixPolicy == SCHED_FIFO)  ? "SCHED_FIFO" :
                            (posixPolicy == SCHED_RR)    ? "SCHED_RR" :
                            (posixPolicy == SCHED_OTHER) ? "SCHED_OTHER" :
#ifdef __linux__
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,23)
                            (posixPolicy == SCHED_IDLE) ? "SCHED_IDLE" :
#endif
#endif
                            "(?)",
                            posixPriority.sched_priority
                            );

                    if (thFinalPriority != thPriority)
                    {
                        FEATSTD_LOG_WARN("Can't set priority to %d, updated priority to %d", thPriority, thFinalPriority);
                    }

                    if (!useCallback)
                    {
                        lResult = ::pthread_attr_setschedpolicy(&lAttr, posixPolicy);
                        if (0 != lResult)
                        {
                            LOG_SYSCALL_ERROR("pthread_attr_setschedpolicy", lResult);
                        }
                        lResult = ::pthread_attr_setschedparam(&lAttr, &posixPriority);
                        if (0 != lResult)
                        {
                            LOG_SYSCALL_ERROR("pthread_attr_setschedparam", lResult);
                        }
                        lResult = ::pthread_attr_setinheritsched(&lAttr, PTHREAD_EXPLICIT_SCHED);
                        if (0 != lResult)
                        {
                            LOG_SYSCALL_ERROR("pthread_attr_setschedparam", lResult);
                        }
                    }
                }

                lResult = ::pthread_create(&mThread, &lAttr, &StartProc, this);
                if (lResult == EPERM)
                {
                    FEATSTD_LOG_WARN("No permission to set thread priority, will create thread with default priority.");
                    lResult = ::pthread_attr_destroy(&lAttr);
                    if (0 == lResult)
                    {
                        lResult = ::pthread_attr_init(&lAttr);
                        // Attempt to create thread again
                        if (0 == lResult)
                        {
                            lResult = ::pthread_create(&mThread, &lAttr, &StartProc, this);
                        }
                    }
                }
                static_cast<void>(::pthread_attr_destroy(&lAttr));

                if (useCallback)
                {
                    FEATSTD_LOG_INFO("Calling callback to set thread priority");
                    Callback(GetId(), posixPolicy, &posixPriority);
                }
            }
        }

        bool lRc = (0 == lResult);
        if (lRc) {
            FEATSTD_LOG_INFO("Thread %s[%u] started", mName, GetId());
#ifdef __linux__
#if defined(__GLIBC__) && (__GLIBC__ > 2 || (__GLIBC__ == 2 && __GLIBC_MINOR__ >= 12))
            static_cast<void>(pthread_setname_np(mThread, mName));
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,8)
            prctl (PR_SET_NAME, mName, 0, 0, 0);
#endif
#endif
        } else {
            LOG_SYSCALL_ERROR("pthread_create", lResult);
            SetStatus(ThreadStatus::Invalid);
        }
        return (lRc);
    }

    ThreadPriority::Enum PosixThread::GetPriority() const
    {
        CriticalSectionLocker lLock(const_cast<CriticalSection *>(&mCs));

        if ((mStatus != ThreadStatus::Started) && (mStatus != ThreadStatus::Running))
        {
            FEATSTD_LOG_ERROR("Thread is terminated/invalid, can't set priority");
            return ThreadPriority::ERROR_RETURN;
        }

        int posixPolicy = 0;
        struct ::sched_param posixPriority;
        int inheritsched = 0;

#if defined(__linux__) && defined(__GLIBC__) && (__GLIBC__ > 2 || (__GLIBC__ == 2 && __GLIBC_MINOR__ >= 2))
        pthread_attr_t attr;
        int lResult = ::pthread_getattr_np(mThread, &attr);
        if (0 != lResult)
        {
            LOG_SYSCALL_ERROR("pthread_getattr_np", lResult);
        }

        static_cast<void>(::pthread_attr_getschedpolicy(&attr, &posixPolicy));
        static_cast<void>(::pthread_attr_getschedparam(&attr, &posixPriority));
        static_cast<void>(::pthread_attr_getinheritsched(&attr, &inheritsched));
#else
        int lResult = pthread_getschedparam(mThread, &posixPolicy, &posixPriority);
        if (0 != lResult)
        {
            LOG_SYSCALL_ERROR("pthread_getattr_np", lResult);
        }
#endif
        // Debug attributes
        FEATSTD_LOG_DEBUG("GetPriority: posixPolicy = %s, posixPriority = %d, inheritsched = %s",
                (posixPolicy == SCHED_FIFO)  ? "SCHED_FIFO" :
                    (posixPolicy == SCHED_RR)    ? "SCHED_RR" :
                    (posixPolicy == SCHED_OTHER) ? "SCHED_OTHER" :
#ifdef __linux__
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,23)
                    (posixPolicy == SCHED_IDLE) ? "SCHED_IDLE" :
#endif
#endif
                    "(?)",
                posixPriority.sched_priority,
                (inheritsched == PTHREAD_INHERIT_SCHED)  ? "INHERIT" :
                    (inheritsched == PTHREAD_EXPLICIT_SCHED) ? "EXPLICIT" :
                    "(?)"
                );

        return PriorityPosixToGeneric(posixPolicy, posixPriority.sched_priority);
    }

    ThreadPriority::Enum PosixThread::SetPriority(ThreadPriority::Enum thPriority, SetPriorityCallback Callback)
    {
        if (thPriority > ThreadPriority::REAL_TIME)
        {
            FEATSTD_LOG_WARN("Invalid priority value, exiting.");
            return ThreadPriority::ERROR_RETURN;
        }

        CriticalSectionLocker lLock(const_cast<CriticalSection *>(&mCs));

        if ((mStatus != ThreadStatus::Started) && (mStatus != ThreadStatus::Running))
        {
            FEATSTD_LOG_ERROR("Thread is terminated/invalid, can't set priority");
            return ThreadPriority::ERROR_RETURN;
        }

        int posixPolicy = SCHED_OTHER;
        struct ::sched_param posixPriority;
        posixPriority.sched_priority = 0;

        ThreadPriority::Enum thFinalPriority = PriorityGenericToPosix(thPriority, posixPolicy, posixPriority.sched_priority);
        FEATSTD_LOG_DEBUG("thFinalPriority = %d, posixPolicy = %d, posixPriority = %d", thFinalPriority, posixPolicy, posixPriority.sched_priority);

        if (thFinalPriority != thPriority)
        {
            FEATSTD_LOG_WARN("Can't set priority %d, updated priority to %d", thPriority, thFinalPriority);
        }

        if (Callback != static_cast<SetPriorityCallback>(0))
        {
            FEATSTD_LOG_INFO("Using callback function to set thread priority");
            Callback(GetId(), posixPolicy, &posixPriority);
            return thFinalPriority;
        }

        int lResult = ::pthread_setschedparam(mThread, posixPolicy, &posixPriority);
        if (0 != lResult)
        {
            if (EPERM == lResult)
            {
                // No permission to set thread priority, return current priority
                FEATSTD_LOG_WARN("No permission to set thread priority");
                thFinalPriority = ThreadPriority::NO_PERMISSION_RETURN;
            }
            else
            {
                LOG_SYSCALL_ERROR("pthread_setschedparam", lResult);
                thFinalPriority = ThreadPriority::ERROR_RETURN;
            }
        }

        return thFinalPriority;
    }

    ThreadPriority::Enum PriorityPosixToGeneric(const int policy, const int priority)
    {
        ThreadPriority::Enum threadPriority = ThreadPriority::LOW;
        if (policy == SCHED_OTHER)
        {
            threadPriority = ThreadPriority::LOW;
        }
        else if ((policy == SCHED_RR) || (policy == SCHED_FIFO))
        {
            int min = ::sched_get_priority_min(policy);
            int max = ::sched_get_priority_max(policy);

            if (priority == max)
            {
                threadPriority = ThreadPriority::REAL_TIME;
            }
            else if (priority > (min + 2))
            {
                threadPriority = ThreadPriority::HIGH;
            }
            else if (priority == (min + 2))
            {
                threadPriority = ThreadPriority::ABOVE_NORMAL;
            }
            else if (priority == (min + 1))
            {
                threadPriority = ThreadPriority::NORMAL;
            }
            else
            {
                threadPriority = ThreadPriority::BELOW_NORMAL;
            }
        }
#ifdef __linux__
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,23)
        else if (policy == SCHED_IDLE)
        {
            threadPriority = ThreadPriority::IDLE;
        }
#endif
#endif

        // Defaults to low
        FEATSTD_LOG_DEBUG("Generic thread priority: %d", threadPriority);
        return threadPriority;
    }

    ThreadPriority::Enum PriorityGenericToPosix(ThreadPriority::Enum thPriority, int &policy, int &priority)
    {
        ThreadPriority::Enum finalPriority = thPriority;

        // Defaults
        policy = SCHED_OTHER;
        priority = 0;

#ifdef __linux__
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,23)
        if (thPriority == ThreadPriority::IDLE)
        {
            FEATSTD_LOG_WARN("Thread priority IDLE not supported by kernel, setting priority to LOW");
            finalPriority = ThreadPriority::LOW;
        }
#endif
#endif

        int min = ::sched_get_priority_min(SCHED_FIFO);
        int max = ::sched_get_priority_max(SCHED_FIFO);

        if (thPriority == ThreadPriority::REAL_TIME)
        {
            policy = SCHED_FIFO;
            priority = max;
        }
        else if (thPriority == ThreadPriority::HIGH)
        {
            policy = SCHED_FIFO;
            priority = max - 1;
        }
        else if (thPriority == ThreadPriority::ABOVE_NORMAL)
        {
            policy = SCHED_FIFO;
            priority = min + 2;
        }
        else if (thPriority == ThreadPriority::NORMAL)
        {
            policy = SCHED_FIFO;
            priority = min + 1;
        }
        else if (thPriority == ThreadPriority::BELOW_NORMAL)
        {
            policy = SCHED_FIFO;
            priority = min;
        }
        else if (thPriority == ThreadPriority::LOW)
        {
            policy = SCHED_OTHER;
            priority = 0;
        }
#ifdef __linux__
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,23)
        else if (thPriority == ThreadPriority::IDLE)
        {
            policy = SCHED_IDLE;
            priority = 0;
        }
#endif
#endif

        return finalPriority;
    }

    ///
    Int16 PosixThread::GetNiceLevel()
    {
#ifdef __linux__
        CriticalSectionLocker lLock(&mCs);

        if ((mStatus == ThreadStatus::Started) || (mStatus == ThreadStatus::Running))
        {
            // Get actual nice level
            errno = 0;
            int nice = ::getpriority(PRIO_PROCESS, GetSystemId());
            if (errno != 0)
            {
                LOG_SYSCALL_ERROR_STATIC("getpriority");
                return 0;
            }
            return (Int16)nice;
        }
#endif

        return mNice;
    }

    ///
    void PosixThread::SetNiceLevel(
            Int16 nice,
            SetNiceCallback Callback
            )
    {
#ifdef __linux__
        CriticalSectionLocker lLock(&mCs);

        mNice = nice;
        mNiceCallback = Callback;

        if ((mStatus == ThreadStatus::Started) || (mStatus == ThreadStatus::Running))
        {
            SetNiceLevelImpl();
        }
#else
        FEATSTD_UNUSED2(nice, Callback);
        FEATSTD_LOG_WARN("Can't set Nice Value on non-Linux Posix. This call will have no effect.");
#endif
    }

    void PosixThread::SetNiceLevelImpl()
    {
#ifdef __linux__
        // Set nice level
        if (mNiceCallback != 0)
        {
            FEATSTD_LOG_DEBUG("Using callback to set nice level");
            mNiceCallback(static_cast<pid_t>(GetSystemId()), mNice);
        }
        else
        {
            int rc = ::setpriority(PRIO_PROCESS, GetSystemId(), mNice);
            if (rc != 0)
            {
                LOG_SYSCALL_ERROR_STATIC("setpriority");
                mNice = 0;
            }
        }
#endif
    }

    // ------------------------------------------------------------------------
    void PosixThread::SetName(const Char * name)
    {
        if (0 != name) {
            FeatStd::Internal::Memory::Set(mName, '\0', ThreadInvariant::NameBufferLength);
            ::strncat(mName, name, ThreadInvariant::NameLength);
            if (mStatus == ThreadStatus::Running) {
#ifdef __linux__
#if defined(__GLIBC__) && (__GLIBC__ > 2 || (__GLIBC__ == 2 && __GLIBC_MINOR__ >= 12))
                static_cast<void>(pthread_setname_np(mThread, mName));
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,8)
                prctl (PR_SET_NAME, mName, 0, 0, 0);
#endif
#endif
            }
        }
    }

    // ------------------------------------------------------------------------
    SizeType PosixThread::GetId() const
    {
        FEATSTD_LINT_NEXT_EXPRESSION(1924, "Violates MISRA C++ Required Rule 5-2-4: C-style cast is needed because of varying typedefs for pthread_t on Linux & Integrity")
        FEATSTD_LINT_NEXT_EXPRESSION(923, "Cast to unsigned int type is required.")
        return (SizeType)mThread;
    }

    // ------------------------------------------------------------------------
    void PosixThread::OnStart()
    {
    }

    // ------------------------------------------------------------------------
    void PosixThread::OnExecute()
    {
    }

    // ------------------------------------------------------------------------
    void PosixThread::OnTerminate(Int exitCode)
    {
        FEATSTD_UNUSED(exitCode);
    }

    // ------------------------------------------------------------------------
    bool PosixThread::Kill(Int exitCode)
    {
        ThreadStatus::Enum state = GetStatus();
        if ((state != ThreadStatus::Started) && (state != ThreadStatus::Running)) {
            FEATSTD_LOG_INFO("Thread '%s' was not started", mName);
            return false;
        }

        SetStatus(ThreadStatus::Terminated);
        if (::pthread_equal(::pthread_self(), mThread) != 0) {
            OnTerminate(exitCode);
        }

        Int lResult = ::pthread_cancel(mThread);
        bool lRc = (0 == lResult);
        if (!lRc) {
            LOG_SYSCALL_ERROR("pthread_kill", lResult);
        }

        return lRc;
    }

    // ------------------------------------------------------------------------
    ThreadStatus::Enum PosixThread::GetStatus() const
    {
        CriticalSectionLocker lLock(const_cast<CriticalSection *>(&mCs));
        return mStatus;
    }

    // ------------------------------------------------------------------------
    void PosixThread::SetStatus(ThreadStatus::Enum status)
    {
        CriticalSectionLocker lLock(&mCs);
        mStatus = status;
    }

    // ------------------------------------------------------------------------
    bool PosixThread::Join()
    {
        if (0 == mThread) {
            FEATSTD_LOG_FATAL("::Join() mThread is null!");
            return false;
        }

        bool retVal = false;

        if (mJoined) {
            // Joining with a thread that has previously been joined results in undefined behavior.
            return true;
        }

        Int joinStatus = ::pthread_join(mThread, NULL);
        switch (joinStatus)
        {
            case 0: // No error
                retVal = true;
                mJoined = true;
                break;

            case EINVAL:    // The implementation has detected that the value specified by thread does not refer to a joinable thread.
            case ESRCH:     // No thread could be found corresponding to that specified by the given thread ID.
            case EDEADLK:   // A deadlock was detected or the value of thread specifies the calling thread.
            default:
                FEATSTD_LOG_FATAL("join() returned error code: %d", joinStatus);
                retVal = false;
                break;
        }
        return retVal;
    }

    // ------------------------------------------------------------------------
    SizeType PosixThread::GetSystemId() const
    {
        return mCachedSystemId;
    }

}}}
