//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_Memory_h)
#define FeatStd_Platform_Memory_h

#include <FeatStd/Platform/Base.h>

#include FEATSTD_PLATFORM_INCLUDE(Memory)

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{
/**
* @brief Abstraction layer for memory methods.
*/
struct Memory : private Impl::Memory
{
    typedef Impl::Memory Base;

public:
    /**
     *  Copies bytes between buffers.
     *  @param  destination     The destination buffer.
     *  @param  source          The source buffer.
     *  @param  count           The count in bytes.
     */
    static inline void Copy(void* destination, const void* source, SizeType count)
    {
        Base::Copy(destination, source, count);
    }

    /**
    * Moves bytes between buffers (std::memmove semantics).
    * @param destination      The destination buffer.
    * @param source           The source buffer.
    * @param count            The count in bytes.
    */
    static inline void Move(void* destination, const void* source, SizeType count) {
        Base::Move(destination, source, count);
    }

    /**
     *  Sets buffers to a specified value.
     *  @param  destination     The destination buffer.
     *  @param  value           The value to set.
     *  @param  count           The count in bytes.
     */
    static inline void Set(void* destination, UInt8 value, SizeType count)
    {
        Base::Set(destination, value, count);
    }

    /**
     *  Compares two memory buffers
     *  @param  l               left-hand buffer
     *  @param  r               right-hand buffer
     *  @param  nBytes          number of bytes to compare
     *  @return                 <0: l less than r \p >0: l greater than r, \p 0: l identical to r
     */
    static inline Int32 Compare(const void *l, const void *r, SizeType nBytes)
    {
        return Base::Compare(l, r, nBytes);
    }

#if !defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)

    /**
     *  Allocates nbytes memory
     *  @param  nbytes          number of bytes which have been allocated
     *  @return                 pointer to memory. 0, if no memory way allocated.
     */
    static inline void* MemAlloc(SizeType nbytes)
    {
        return Base::MemAlloc(nbytes);
    }

    /**
     *  Reallocate memory
     *  functionality according to standard C realloc function
     *  @param  p               buffer to allocated memory
     *  @param  nbytes          number of bytes which have been allocated
     *  @return                 pointer to memory. 0, if no memory allocated.
     */
    static inline void* MemRealloc(void *p, SizeType nbytes)
    {
        return Base::MemRealloc(p, nbytes);
    }

    /**
     *  Releases previously allocated memory.
     *  @param  p               pointer to the buffer to be released
     */
    static inline void MemFree(void *p)
    {
        Base::MemFree(p);
    }

    /**
     *  Estimates the largest available block of memory which is available for allocation.
     *  This function might not be supported on all platforms. If unsupported, the function
     *  shall return a reasonable default value.
     *  @return                 The largest available block in memory in bytes.
     */
    static inline SizeType MemAvailable()
    {
        return Base::MemAvailable();
    }

#endif
};
/// @}
}}
#endif
