//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_Math_h)
#define FeatStd_Platform_Math_h

#include <FeatStd/Platform/Base.h>
#include "FractionalNumberMath.h"

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{
    /** @brief Math platform abstraction interface */
    struct Math : private Impl::Math
    {
        /**
         * Calculates the square root for simple precision types.
         * @param f        A non-negative value.
         * @return         The square-root of f. If f is negative, SquareRoot returns an indefinite value.
         */
        static inline Float SquareRoot(Float f) { return ::FractionalNumber::Math::SquareRoot(f); }

        /**
         * Calculates the square root for double precision types.
         * @param d        A non-negative value.
         * @return         The square-root of d. If d is negative, SquareRoot returns an indefinite value.
         */
        static inline Double SquareRoot(Double d) { return ::FractionalNumber::Math::SquareRoot(d); }

        /**
         * Computes the sine of a given value in radians.
         * @param val      A float value in radians.
         * @return         The sine of the given float value.
         */
        static inline Float Sine(Float val) { return ::FractionalNumber::Math::Sine(val); }

        /**
         * Computes the arcsine of a given value.
         * @param val A float value inside [-1, 1] interval.
         * @return the arcsine of the given value in radians.
         */
        static inline Float ASine(Float val) { return ::FractionalNumber::Math::ASine(val); }

        /**
         * Computes the cosine of a given value in radians.
         * @param val      A float value in radians.
         * @return         The cosine of the given float value.
         */
        static inline Float Cosine(Float val) { return ::FractionalNumber::Math::Cosine(val); }

        /**
         * Computes the arccosine of a given value.
         * @param val A float value inside [-1, 1] interval.
         * @return the arccosine of the given value in radians.
         */
        static inline Float ACosine(Float val) { return ::FractionalNumber::Math::ACosine(val); }

        /**
         * Computes the tangent of a given value in radians.
         * @param val      A float value in radians.
         * @return         The tangent of the given float value.
         */
        static inline Float Tangent(Float val) { return ::FractionalNumber::Math::Tangent(val); }

        /**
         * Computes the atan of a given value.
         * @param val A float value.
         * @return the atan of the given value in radians.
         */
        static inline Float ATangent(Float val) { return ::FractionalNumber::Math::ATangent(val); }

        /**
         * Computes the arctangent 2 of the given values.
         * @param y Opposite side.
         * @param x Adjacent side.
         * @return the arctangent 2 of the given values in radians.
         */
        static inline Float ATangent2(Float y, Float x) { return ::FractionalNumber::Math::ATangent2(y,x); }

        /**
         * Computes a base raised to the power of a exponent.
         * @param base     The base.
         * @param exponent     The exponent.
         * @return         The base raised to the power of the exponent.
         */
        static inline Float Power(Float base, Float exponent) { return ::FractionalNumber::Math::Power(base, exponent); }

        /**
         * Computes the base 10 logarithm of the value given.
         * @param value    The value to compute the base 10 logarithm from.
         * @return         The base 10 logarithm of the value given.
         */
        static inline Float Log10(Float value) { return ::FractionalNumber::Math::Log10(value); }

        /**
         * Computes the absolute value of the value given.
         * @param value    The value to compute the absolute value from.
         * @return         The absolute value of the value given.
         */
        static inline UInt32 Absolute(Int32 value) { return ::FractionalNumber::Math::Absolute(value); }

        /**
        * Computes the absolute value of the value given.
        * @param value    The value to compute the absolute value from.
        * @return         The absolute value of the value given.
        */
        static inline Float Absolute(Float value) { return ::FractionalNumber::Math::Absolute(value); }

        /**
        * Computes the largest integer smaller than the value given.
        * @param value    The value to compute the floor value from.
        * @return         The floor value of the value given.
        */
        static inline Float Floor(Float value) { return ::FractionalNumber::Math::Floor(value); }

        /**
        * Computes the largest integer smaller than the value given.
        * @param value    The value to compute the floor value from.
        * @return         The floor value of the value given.
        */
        static inline Double Floor(Double value) { return ::FractionalNumber::Math::Floor(value); }

        /**
        * Computes the smallest integer larger than the value given.
        * @param value    The value to compute the ceiling value from.
        * @return         The ceiling value of the value given.
        */
        static inline Float Ceil(Float value) { return ::FractionalNumber::Math::Ceil(value); }

        /** Returns the largest floating point number
            @return largest floating point number */
        static inline Float MaxFloat() { return ::FractionalNumber::Math::MaxFloat(); }

        /** Returns the smallest positive floating point number
            @return smallest floating point number */
        static inline Float MinFloat() { return ::FractionalNumber::Math::MinFloat(); }

        /** Returns the smallest positive number such that 1.0+number !=1.0
            @return Smallest positive number such that 1.0+number !=1.0 */
        static inline Float EpsilonFloat() { return ::FractionalNumber::Math::EpsilonFloat(); }

        /** Returns the smallest positive number acceptable as a relative error in math operations
            @return Smallest positive number acceptable as a relative error in math operations */
        static inline Float EpsilonFloat3D() { return ::FractionalNumber::Math::EpsilonFloat3D(); }

        /**
        * Computes the value in radians corresponding to a value in degrees.
        * @param degree   The value in degrees.
        * @return         The value in radians.
        */
        static inline Float DegreeToRadian(Float degree) { return ::FractionalNumber::Math::DegreeToRadian(degree); }

        /**
        * Computes the value in degrees corresponding to a value in radians.
        * @param radian   The value in radians.
        * @return         The value in degrees.
        */
        static inline Float RadianToDegree(Float radian) { return ::FractionalNumber::Math::RadianToDegree(radian); }

        /**
        * Compares two Float values.
        * @param a         The first value.
        * @param b         The second value.
        * @param relativeEpsilon   The accepted inaccuracy on comparison.
        *                  If 99.999% accuracy is needed then pass a epsilon value of 0.00001.
        * @param absoluteEpsilon   The accepted inaccuracy on comparison.
        *                  If 99.999% accuracy is needed then pass a epsilon value of 0.00001.
        * @return          true if the given Float values are equal in respect to the given epsilon, otherwise false.
        */
        static inline bool FloatAlmostEqual(Float a, Float b, Float relativeEpsilon = EpsilonFloat(), Float absoluteEpsilon = EpsilonFloat3D()) { return ::FractionalNumber::Math::FloatAlmostEqual(a, b,relativeEpsilon, absoluteEpsilon); }

        /**
        * Returns the remainder of x divided by y.
        * @param x  The division numerator.
        * @param y  The division denominator.
        * @return The remainder of x divided by y.
        */
        static inline Double FMod(Double x, Double y) { return ::FractionalNumber::Math::FMod(x, y); }

        /**
        * Check a Float for being infinite (INF).
        * @param value  The value to check.
        * @return true, if the value is infinite.
        */
        static inline bool IsInfinite(Float value) { return ::FractionalNumber::Math::IsInfinite(value); }

        /**
        * Check a Float for being not a number or undefined (NAN, IND).
        * @param value  The value to check.
        * @return true, if the value is undefined / not a number.
        */
        static inline bool IsNan(Float value) { return ::FractionalNumber::Math::IsNan(value); }

        /**
        * Check a Float for being valid, which means not infinite nor undefined.
        * @param value  The value to check.
        * @return true, if the value is a valid float.
        */
        static inline bool IsValid(Float value) { return ::FractionalNumber::Math::IsValid(value); }
    };
/// @}
}}
#endif
