//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_Generic_GenericString_h)
#define FeatStd_Platform_Generic_GenericString_h

#include <FeatStd/Platform/Base.h>

#include <wchar.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <FeatStd/Platform/CodePointIterator.h>
#include <FeatStd/Platform/VarArg.h>
#include <FeatStd/Util/PointerUtil.h>

namespace FeatStd { namespace Internal { namespace Generic {
    namespace Private {
    /** TLength returns the number of characters in a TChar string
        number of chars must not be mixed with number of bytes
        @param string pointer to string
        @return number of characters in the string
        */
        template<typename T, bool utfEncoded> FeatStd::SizeType TLength(const T *string) {
            FeatStd::Internal::CodePointIterator<FeatStd::TChar, utfEncoded> it(string);
            FeatStd::SizeType n = 0;
            while(*it != 0) {
                it.Advance();
                ++n;
            }
            return n;
        }

        template<> inline FeatStd::SizeType TLength<FeatStd::Char, false>(const FeatStd::Char *string) {
            return strlen(string);
        }

        template<> inline FeatStd::SizeType TLength<FeatStd::Ucs2Char, false>(const FeatStd::Ucs2Char *string) {
            return wcslen(string);
        }
    }

    /// @addtogroup FEATSTD_PLATFORM_GENERIC
    /// @{
    /** Generic implementation of FeatStd string functions using STDC <string.h>. */
    struct GenericString {
        /**
         * Compares two null-terminated character string given.
         * return zero if the strings are identical, positive value if the left char string is greater, and negative value
         * if the left char string is less than right char string.
         * @param lhs First string to be compared.
         * @param rhs Second String to be compared.
         * @return Zero if the strings are identical, positive value if the left char string is greater, and negative value
         * if the left char string is less than right char string.
         */
        static inline FeatStd::Int32 CompareStrings(const FeatStd::Char* lhs, const FeatStd::Char* rhs) {
            return static_cast<FeatStd::Int32>(strcmp(static_cast<const char*>(lhs), static_cast<const char*>(rhs)));
        }

        /**
         * Compares two strings formed by the first n characters from two given strings.
         * return zero if the strings are identical, positive value if the left char string is greater, and negative value
         * if the left char string is less than right char string.
         * @param lhs First string to be compared.
         * @param rhs Second String to be compared.
         * @param n   Maximum number of characters to compare.
         * @return    Zero if the strings are identical, positive value if the left char string is greater, and negative value
         * if the left char string is less than right char string.
         */
        static inline FeatStd::Int32 ComparePartial(const FeatStd::Char* lhs, const FeatStd::Char* rhs, FeatStd::SizeType n) {
            return static_cast<FeatStd::Int32>(strncmp(static_cast<const char*>(lhs), static_cast<const char*>(rhs), n));
        }

        /**
         * Copy string from to to
         * @param to   Destination string to copy to.
         * @param from Source string to copy from.
         */
        static inline void Copy(FeatStd::Char *to, const FeatStd::Char *from) {
            strcpy(static_cast<char *>(to), static_cast<const char*>(from));
        }

        /**
         * Copy n characters of string from to string to
         * @param to   Destination string to copy to.
         * @param from Source string to copy from.
         * @param n    Maximum number of characters to be copied from source.
         * @return A FeatStd::Char Pointer.
         */
        static inline FeatStd::Char* CopyPartial(FeatStd::Char *to, const FeatStd::Char *from, FeatStd::SizeType n) {
            return static_cast<FeatStd::Char*>(strncpy(static_cast<char*>(to), static_cast<const char*>(from), n));
        }

        /**
         * Returns the length of string str
         * @param str String to retrieve length from.
         * @return    The length of string str.
         */
        static inline FeatStd::SizeType Length(const FeatStd::Char *str) {
            return strlen(static_cast<const char*>(str));
        }

        /**
         * Returns the length of string str
         * @param str String to retrieve length from.
         * @param size Maximum size of the string to prevent buffer overflow.
         * @return    The length of string str.
         */
        static inline FeatStd::SizeType NLength(const FeatStd::Char *str, FeatStd::SizeType size) {
#if defined(__MINGW32__) || defined(__CC_ARM) || defined(FEATSTD_OS_QNX)
            // strnlen not available in mingw
            // see http://sourceforge.net/p/mingw/bugs/1912/
            const FeatStd::Char *end = (const FeatStd::Char *)memchr(str, '\0', size);
            return (end) ? (size_t)(PointerSub(end,(FeatStd::SizeType)str)) : size;
#else
            return strnlen(static_cast<const char*>(str), size);
#endif
        }

        /**
        * Converts a text string of a hexadecimal representation of a number to an integer.
        * @param str string of a a hexadecimal representation of a number.
        * @return Parsed integer, In case of failure 0 is returned (but 0 may also be a
                  valid value).
        */
        static inline UInt32 HexStringToInteger(const FeatStd::Char* str) {
            UInt integer = 0;
            if (sscanf(str, "%x", &integer) == EOF) {
                integer = 0;
            }
            return static_cast<UInt32>(integer);
        }

        /**
        * Platform abstraction of sscanf
        */
        static inline Int VStringScanf(const FeatStd::Char *buffer, const FeatStd::Char* format, FeatStd::Internal::VarArg::VaList argp) {
            return vsscanf(buffer, format, argp);
        }

        /** CodePointIterator defines the class that can be used to iterate through TString code points. */
        typedef FeatStd::Internal::CodePointIterator<FeatStd::TChar, FeatStd::TCharIsUtfEncoded> CodePointIterator;

        static inline FeatStd::SizeType TLength(const FeatStd::TChar *string) {
            return Private::TLength<FeatStd::TChar, FeatStd::TCharIsUtfEncoded>(string);
        }

        /** safe version of sprintf */
        static Int VStringPrintf(Char *buffer, SizeType bufferSize, const Char *fmt, FeatStd::Internal::VarArg::VaList argp) {
           // vs 2014 always adds a \0 at the end of the buffer - regardless of the size vs fmt input ratio.
            return vsnprintf(buffer, bufferSize, fmt, argp);
        }

    };
    /// @}
}}}

namespace FeatStd { namespace Internal { namespace Impl {
    typedef FeatStd::Internal::Generic::GenericString String;
}}}


#endif
