//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_File_h)
#define FeatStd_Platform_File_h

#include <FeatStd/Platform/Base.h>

#include FEATSTD_PLATFORM_INCLUDE(File)

namespace FeatStd {
/// @addtogroup FEATSTD_PLATFORM
/// @{
/**
    * @brief Typedef for the file handle. Implemented as a pointer to a FILE struct for now
    */
typedef FeatStd::Internal::Impl::FileHandle FileHandle;

/**
 * @brief Incapsulates file handling routines
 */
struct File : public Internal::Impl::File
{
    typedef Internal::Impl::File Base;

    /** Import FileHandle type from platform specific */
    using Base::SeekMark;

    /** import SeekMark::Begin - move file pointer relative to begin of file */
    using Base::Begin;

    /** import SeekMark::Current - move file pointer relative to current position of file pointer */
    using Base::Current;

    /** import SeekMark::End - move file pointer relative to end of file */
    using Base::End;

    /** import Mode::ReadBinary - open a file for binary reading */
    using Base::ReadBinary;

    /** import Mode::WriteBinary - open a file for binary writing */
    using Base::WriteBinary;

    /**
     * Opens the file for binary reading.
     * To check for open status, use IsFileOpened()
     * @param filePath Path of file to open.
     * @param mode     Access Mode to open file, ReadBinary or WriteBinary.
     * @return         Handle to opened file. 0 if opening failed.
     */
    static inline FileHandle Open(const Char* filePath, Mode mode = ReadBinary) {
        return Base::Open(filePath, mode);
    }

    /**
     * To be used after Open() in order to check if the file was successfuly opened
     * @param handle Handle to file to check if it's open.
     * @return True if handle is not 0.
     */
    static inline bool IsFileOpened(FileHandle handle) {
        return Base::IsFileOpened(handle);
    }

    /**
     * Reads from a given file.
     * @param file      Handle to file to read from.
     * @param buffer    Pre allocated buffer to copy read data into.
     * @param blockSize Size in bytes of each element to read.
     * @param count     Count of elements to read.
     * @return          Amount of read data in bytes.
     */
    static inline SizeType Read(FileHandle file, void* buffer,
                        SizeType blockSize, SizeType count) {
        return Base::Read(file, buffer, blockSize, count);
    }

    /**
     * Writes to a given file.
     * @param file      Handle to file to write into.
     * @param buffer    Buffer containing data to be written.
     * @param blockSize Size in bytes of each element to be written.
     * @param count     Count of elements to be written.
     * @return          Amount of data written in bytes.
     */
    static inline SizeType Write(FileHandle file, const void* buffer,
                        SizeType blockSize, SizeType count) {
        return Base::Write(file, buffer, blockSize, count);
    }

    /**
     * returns current file pointer position
     * @param hdl Handle to file from which position indicator shall be retrieved.
     * @return Current file pointer position.
     */
    static inline OffsetType Tell(FileHandle hdl) {
        return Base::Tell(hdl);
    }

    /**
     * Seeks the file by offset bytes
     * @param file   Handle to file to seek.
     * @param offset Offset to which position indicator should be set (@see Tell).
     * @param mark   Whether offset shall be added to the beginning or the current position.
     * @return True if succeeded.
     */
    static inline bool Seek(FileHandle file, OffsetType offset, SeekMark mark) {
        return Base::Seek(file, offset, mark);
    }

    /**
     * Closes the file associated to the handle and sets the parameter to the null handle
     * @param handle Handle to file to close.
     * @return True if file was closed, false on failure.
     */
    static inline bool Close(FileHandle& handle) {
        return Base::Close(handle);
    }
};
/// @}
}
#endif
